# Flagpy - World Flag Identifier
Flagpy is a python library that allows the user to identify the flags of countries from all over the world. It also has support for seeing how similar two different flags are and finding which two countries have the most/least similar flag.
## Installation
Flagpy can be installed simply using pip.
```sh
$ pip install flagpy
```
## Usage
### Identify
Flagpy can take in a url which links to an image and then determine which flag is being displayed in the image. It is most accurate when the flag takes up the whole image. The type of method to find the closest flag can be specified, but the default (and most accurate) method is "mse". Supported methods as of right now are "mse", "ssim", and "hash".
```python
>>> import flagpy as fp
>>> fp.identify('https://upload.wikimedia.org/wikipedia/commons/thumb/d/d9/Flag_of_Canada_%28Pantone%29.svg/1200px-Flag_of_Canada_%28Pantone%29.svg.png')
'Canada'
>>> fp.identify('https://upload.wikimedia.org/wikipedia/commons/thumb/d/d9/Flag_of_Canada_%28Pantone%29.svg/1200px-Flag_of_Canada_%28Pantone%29.svg.png', method = 'ssim')
'Canada'
>>> fp.identify('https://upload.wikimedia.org/wikipedia/commons/thumb/d/d9/Flag_of_Canada_%28Pantone%29.svg/1200px-Flag_of_Canada_%28Pantone%29.svg.png', method = 'hash')
'Canada'
```

### Closest/Farthest Flag
Flagpy can take in a name of a country and return which country's flag is most/least similar to the given country's flag. Once again, the method to find the most/least similar flag can be specified, but the default is "mse". The name of the country must match one of the items in flagpy's list of countries, which can be acquired with get_country_list().
```python
>>> fp.get_country_list()
['Afghanistan', 'Albania', ... , 'Zambia', 'Zimbabwe']
>>> fp.closest_flag("India")
'Niger'
>>> fp.closest_flag('India', method = 'ssim')
'Luxembourg'
>>> fp.farthest_flag('India')
'Estonia'
>>> fp.farthest_flag('India', method = 'hash')
'Armenia'
```
### Displaying Flags
Flagpy has support for both displaying a country's flag and storing it as an Image object.
```python
>>> fp.display('Ivory Coast')
```
![alt text](https://upload.wikimedia.org/wikipedia/commons/thumb/f/fe/Flag_of_C%C3%B4te_d%27Ivoire.svg/300px-Flag_of_C%C3%B4te_d%27Ivoire.svg.png "Ivory Coast Flag")
```python
>>> img = fp.get_flag_img('Yemen')
>>> img.show()
```
![alt text](https://upload.wikimedia.org/wikipedia/commons/thumb/8/89/Flag_of_Yemen.svg/300px-Flag_of_Yemen.svg.png "Yemen Flag")
### Difference Between Flags
Flagpy allows the user to find the quantitative difference between two countries' flags. There are three possible methods to calculate this difference: mean-squared error (the default), structural similarity index measurement, and hash difference. For both mse and hash, smaller values indicate higher similarity between the two flags, and a difference of 0 means the images are identical. However, the hash difference will always be an integer. For ssim, the value must be between -1 and 1, so higher values indicate a higher similarity.
```python
>>> fp.flag_dist('Denmark', 'Germany')
57595.77333333333
>>> fp.flag_dist('Denmark', 'Germany', method = 'ssim')
0.25852880532772604
>>> fp.flag_dist('Denmark', 'Germany', method = 'hash')
40
```
### Flag DataFrame
Flagpy can supply the user with a pandas DataFrame of all the countries and their flag image (scraped from [Wikipedia](https://en.wikipedia.org/wiki/Gallery_of_sovereign_state_flags)).
```python
>>> df = fp.get_flag_df()
>>> df.shape
(195, 1)
>>> df["flag"].loc["The United States"]
array([[[ 51,  51, 102],
        [ 51,  51, 102],
        [ 51,  51, 102],
        ...
        [153,  51,  51],
        [153,  51,  51],
        [153,  51,  51]]]
```
