import requests
from typing import Optional
from vesselapi.models import operations
from . import utils

class Webhooks:
    _client: requests.Session
    _security_client: requests.Session
    _server_url: str
    _language: str
    _sdk_version: str
    _gen_version: str

    def __init__(self, client: requests.Session, security_client: requests.Session, server_url: str, language: str, sdk_version: str, gen_version: str) -> None:
        self._client = client
        self._security_client = security_client
        self._server_url = server_url
        self._language = language
        self._sdk_version = sdk_version
        self._gen_version = gen_version

    
    def create(self, request: operations.PostWebhookRequest) -> operations.PostWebhookResponse:
        r"""Create Webhook
        Create a new webhook for a given connection
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/connection/webhook"
        
        headers = {}
        req_content_type, data, json, files = utils.serialize_request_body(request)
        if req_content_type != "multipart/form-data" and req_content_type != "multipart/mixed":
            headers["content-type"] = req_content_type
        
        client = self._security_client
        
        r = client.request("POST", url, data=data, json=json, files=files, headers=headers)
        content_type = r.headers.get("Content-Type")

        res = operations.PostWebhookResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.PostWebhookResponseBody])
                res.response_body = out

        return res

    
    def delete(self, request: operations.DeleteWebhookRequest) -> operations.DeleteWebhookResponse:
        r"""Remove Webhook
        Removes a webhook for a given connection and id
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/connection/webhook"
        
        headers = {}
        req_content_type, data, json, files = utils.serialize_request_body(request)
        if req_content_type != "multipart/form-data" and req_content_type != "multipart/mixed":
            headers["content-type"] = req_content_type
        
        client = self._security_client
        
        r = client.request("DELETE", url, data=data, json=json, files=files, headers=headers)
        content_type = r.headers.get("Content-Type")

        res = operations.DeleteWebhookResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            pass

        return res

    
    def find(self, request: operations.GetOneWebhookRequest) -> operations.GetOneWebhookResponse:
        r"""Get Webhook
        Retrieve information about a webhook for a given connection and id
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/connection/webhook"
        
        query_params = utils.get_query_params(request.query_params)
        
        client = self._security_client
        
        r = client.request("GET", url, params=query_params)
        content_type = r.headers.get("Content-Type")

        res = operations.GetOneWebhookResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.GetOneWebhookResponseBody])
                res.response_body = out

        return res

    