import requests
from typing import Optional
from vesselapi.models import operations
from . import utils

class EngAccounts:
    _client: requests.Session
    _security_client: requests.Session
    _server_url: str
    _language: str
    _sdk_version: str
    _gen_version: str

    def __init__(self, client: requests.Session, security_client: requests.Session, server_url: str, language: str, sdk_version: str, gen_version: str) -> None:
        self._client = client
        self._security_client = security_client
        self._server_url = server_url
        self._language = language
        self._sdk_version = sdk_version
        self._gen_version = gen_version

    
    def create(self, request: operations.PostEngagementAccountRequest) -> operations.PostEngagementAccountResponse:
        r"""Create Account
        Create a new account.
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/engagement/account"
        
        headers = {}
        req_content_type, data, json, files = utils.serialize_request_body(request)
        if req_content_type != "multipart/form-data" and req_content_type != "multipart/mixed":
            headers["content-type"] = req_content_type
        
        client = self._security_client
        
        r = client.request("POST", url, data=data, json=json, files=files, headers=headers)
        content_type = r.headers.get("Content-Type")

        res = operations.PostEngagementAccountResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.PostEngagementAccountResponseBody])
                res.response_body = out

        return res

    
    def find(self, request: operations.GetOneEngagementAccountRequest) -> operations.GetOneEngagementAccountResponse:
        r"""Get Account
        Retrieve a single Account by Id
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/engagement/account"
        
        query_params = utils.get_query_params(request.query_params)
        
        client = self._security_client
        
        r = client.request("GET", url, params=query_params)
        content_type = r.headers.get("Content-Type")

        res = operations.GetOneEngagementAccountResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.GetOneEngagementAccountResponseBody])
                res.response_body = out

        return res

    
    def list(self, request: operations.GetAllEngagementAccountsRequest) -> operations.GetAllEngagementAccountsResponse:
        r"""Get All Accounts
        Retrieve all Accounts
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/engagement/accounts"
        
        query_params = utils.get_query_params(request.query_params)
        
        client = self._security_client
        
        r = client.request("GET", url, params=query_params)
        content_type = r.headers.get("Content-Type")

        res = operations.GetAllEngagementAccountsResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.GetAllEngagementAccountsResponseBody])
                res.response_body = out

        return res

    
    def update(self, request: operations.PutEngagementAccountRequest) -> operations.PutEngagementAccountResponse:
        r"""Update Account
        Update an existing account.
        """
        
        base_url = self._server_url
        
        url = base_url.removesuffix("/") + "/engagement/account"
        
        headers = {}
        req_content_type, data, json, files = utils.serialize_request_body(request)
        if req_content_type != "multipart/form-data" and req_content_type != "multipart/mixed":
            headers["content-type"] = req_content_type
        
        client = self._security_client
        
        r = client.request("PATCH", url, data=data, json=json, files=files, headers=headers)
        content_type = r.headers.get("Content-Type")

        res = operations.PutEngagementAccountResponse(status_code=r.status_code, content_type=content_type)
        
        if r.status_code == 200:
            if utils.match_content_type(content_type, "application/json"):
                out = utils.unmarshal_json(r.text, Optional[operations.PutEngagementAccountResponseBody])
                res.response_body = out

        return res

    