#   Copyright 2019 AUI, Inc. Washington DC, USA
#
#   Licensed under the Apache License, Version 2.0 (the "License");
#   you may not use this file except in compliance with the License.
#   You may obtain a copy of the License at
#
#       http://www.apache.org/licenses/LICENSE-2.0
#
#   Unless required by applicable law or agreed to in writing, software
#   distributed under the License is distributed on an "AS IS" BASIS,
#   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#   See the License for the specific language governing permissions and
#   limitations under the License.


########################
def rebin(ds, **kwargs):
    """
    Rebin an n-dimensional image across any single (spatial or spectral) axis
    
    .. todo::
        Check for and apply all masks and regions

        Accept arguments that control which DataArray is rebinned

        Improve performance when framework client has processes=True

    .. note::
        The new Dataset generated by the current implementation of this function will lose its metadata attributes. See https://github.com/pydata/xarray/issues/3376

    Parameters
    ----------
    ds : xarray.core.dataset.Dataset
        input Image
    factor : int, optional
        scaling factor for binning, Default=1
    axis : str, optional
        dataset dimension upon which to rebin, Default='chan'
    **kwargs
        Arbitrary keyword arguments
    
    Returns
    -------
    xarray.core.dataset.Dataset
        output Image
    """
    
    if 'factor' in kwargs.keys():
        factor = kwargs['factor']
    else:
        factor = 1
        
    if 'axis' in  kwargs.keys():
        axis = kwargs['axis']
        if axis in ds.dims:
            pass
        else:
            print("Requested axis not a dimension of input dataset.")
    else:
        print("Defaulting to rebinning dataset by frequency dimension")
        axis = 'chan'

    # works best with threads
    try:
        new_ds = ds.coarsen({axis:factor}).mean(keep_attrs=True)
    except ValueError as exception:
        print(f"{exception}")
        print(f"Using requested scaling factor and padding with NaN")
        new_ds = ds.coarsen({axis:factor}, boundary='pad').mean(keep_attrs=True)

    return new_ds



