"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
let stack;
let role;
let api;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
    });
    api = new appsync.GraphqlApi(stack, 'API', {
        name: 'demo',
        schema: appsync.Schema.fromAsset(path_1.join(__dirname, 'appsync.test.graphql')),
        authorizationConfig: {
            defaultAuthorization: {
                authorizationType: appsync.AuthorizationType.IAM,
            },
        },
    });
});
describe('grant Permissions', () => {
    test('IamResource throws error when custom is called with no arguments', () => {
        //THEN
        expect(() => {
            api.grant(role, appsync.IamResource.custom(), 'appsync:GraphQL');
        }).toThrowError('At least 1 custom ARN must be provided.');
    });
    test('grant provides custom permissions when called with `custom` argument', () => {
        // WHEN
        api.grant(role, appsync.IamResource.custom('types/Mutation/fields/addTest'), 'appsync:GraphQL');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Mutation/fields/addTest',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grant provides [type parameter]/* permissions when called with `type` argument', () => {
        // WHEN
        api.grant(role, appsync.IamResource.ofType('Mutation'), 'appsync:GraphQL');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Mutation/*',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grant provides fields/[field param] permissions when called with `type` and `field` argument', () => {
        // WHEN
        api.grant(role, appsync.IamResource.ofType('Mutation', 'addTest'), 'appsync:GraphQL');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Mutation/fields/addTest',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grant provides all permissions when called with IamResource.all()', () => {
        // WHEN
        api.grant(role, appsync.IamResource.all(), 'appsync:GraphQL');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/*',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grant provides multiple permissions using one IamResource custom call', () => {
        // WHEN
        api.grant(role, appsync.IamResource.custom('I', 'am', 'custom'), 'appsync:GraphQL');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: [
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/I',
                                    ]],
                            },
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/am',
                                    ]],
                            },
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/custom',
                                    ]],
                            },
                        ],
                    },
                ],
            },
        });
    });
    test('grant provides multiple permissions using one IamResource ofType call', () => {
        // WHEN
        api.grant(role, appsync.IamResource.ofType('I', 'am', 'custom'), 'appsync:GraphQL');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: [
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/I/fields/am',
                                    ]],
                            },
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/I/fields/custom',
                                    ]],
                            },
                        ],
                    },
                ],
            },
        });
    });
});
describe('grantMutation Permissions', () => {
    test('grantMutation provides Mutation/* permissions when called with no `fields` argument', () => {
        // WHEN
        api.grantMutation(role);
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Mutation/*',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grantMutation provides fields/[field param] permissions when called with `fields` argument', () => {
        // WHEN
        api.grantMutation(role, 'addTest');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Mutation/fields/addTest',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grantMutation provides multiple permissions when called with `fields` argument', () => {
        // WHEN
        api.grantMutation(role, 'addTest', 'removeTest');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: [
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/Mutation/fields/addTest',
                                    ]],
                            },
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/Mutation/fields/removeTest',
                                    ]],
                            },
                        ],
                    },
                ],
            },
        });
    });
});
describe('grantQuery Permissions', () => {
    test('grantQuery provides Query/* permissions when called without the `fields` argument', () => {
        // WHEN
        api.grantQuery(role);
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Query/*',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grantQuery provides fields/[field param] permissions when called with `fields` arugment', () => {
        // WHEN
        api.grantQuery(role, 'getTest');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Query/fields/getTest',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grantQuery provides multiple permissions when called with `fields` argument', () => {
        // WHEN
        api.grantQuery(role, 'getTests', 'getTest');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: [
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/Query/fields/getTests',
                                    ]],
                            },
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/Query/fields/getTest',
                                    ]],
                            },
                        ],
                    },
                ],
            },
        });
    });
});
describe('grantSubscription Permissions', () => {
    test('grantSubscription provides Subscription/* permissions when called without `fields` argument', () => {
        // WHEN
        api.grantSubscription(role);
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Subscription/*',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grantSubscription provides fields/[field param] when called with `field` argument', () => {
        api.grantSubscription(role, 'subscribe');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Subscription/fields/subscribe',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grantSubscription provides multiple permissions when called with `fields` argument', () => {
        // WHEN
        api.grantSubscription(role, 'subscribe', 'custom');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: [
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/Subscription/fields/subscribe',
                                    ]],
                            },
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/Subscription/fields/custom',
                                    ]],
                            },
                        ],
                    },
                ],
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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