"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
const t = require("./scalar-type-defintions");
const out = 'enum Test {\n  test1\n  test2\n  test3\n}\n';
let stack;
let api;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    api = new appsync.GraphqlApi(stack, 'api', {
        name: 'api',
    });
});
describe('testing Enum Type properties', () => {
    test('EnumType configures properly', () => {
        // WHEN
        const test = new appsync.EnumType('Test', {
            definition: ['test1', 'test2', 'test3'],
        });
        api.addType(test);
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
        expect(stack).not.toHaveResource('AWS::AppSync::Resolver');
    });
    test('EnumType can addField', () => {
        // WHEN
        const test = new appsync.EnumType('Test', {
            definition: ['test1', 'test2'],
        });
        api.addType(test);
        test.addField({ fieldName: 'test3' });
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}`,
        });
    });
    test('EnumType can be a GraphqlType', () => {
        // WHEN
        const test = new appsync.EnumType('Test', {
            definition: ['test1', 'test2', 'test3'],
        });
        api.addType(test);
        api.addType(new appsync.ObjectType('Test2', {
            definition: { enum: test.attribute() },
        }));
        const obj = 'type Test2 {\n  enum: Test\n}\n';
        // THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${out}${obj}`,
        });
    });
    test('errors when enum type is configured with white space', () => {
        // THEN
        expect(() => {
            new appsync.EnumType('Test', {
                definition: ['test 1', 'test2', 'test3'],
            });
        }).toThrowError('Enum Type values cannot have whitespace. Received: test 1');
    });
    test('errors when the fieldName in addField has white space', () => {
        // WHEN
        const test = new appsync.EnumType('Test', {
            definition: [],
        });
        // THEN
        expect(() => {
            test.addField({ fieldName: ' ' });
        }).toThrowError('Enum Type values cannot have whitespace. Received:  ');
    });
    test('errors when enum type is configured with field options', () => {
        // WHEN
        const test = new appsync.EnumType('Test', {
            definition: [],
        });
        // THEN
        expect(() => {
            test.addField({ fieldName: 'test', field: t.string });
        }).toThrowError('Enum Type fields consist of strings. Use the fieldName option instead of the field option.');
    });
    test('errors when enum type is missing fieldName option', () => {
        // WHEN
        const test = new appsync.EnumType('Test', {
            definition: [],
        });
        // THEN
        expect(() => {
            test.addField({});
        }).toThrowError('When adding a field to an Enum Type, you must configure the fieldName option.');
    });
});
//# sourceMappingURL=data:application/json;base64,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