"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schema = void 0;
const fs_1 = require("fs");
const core_1 = require("@aws-cdk/core");
const appsync_generated_1 = require("./appsync.generated");
const private_1 = require("./private");
const schema_intermediate_1 = require("./schema-intermediate");
;
/**
 * (experimental) The Schema for a GraphQL Api.
 *
 * If no options are configured, schema will be generated
 * code-first.
 *
 * @experimental
 */
class Schema {
    /**
     * @experimental
     */
    constructor(options) {
        if (options === null || options === void 0 ? void 0 : options.filePath) {
            this.mode = private_1.SchemaMode.FILE;
            this.definition = fs_1.readFileSync(options.filePath).toString('utf-8');
        }
        else {
            this.mode = private_1.SchemaMode.CODE;
            this.definition = '';
        }
        this.types = [];
    }
    /**
     * (experimental) Generate a Schema from file.
     *
     * @param filePath the file path of the schema file.
     * @returns `SchemaAsset` with immutable schema defintion
     * @experimental
     */
    static fromAsset(filePath) {
        return new Schema({ filePath });
    }
    /**
     * (experimental) Called when the GraphQL Api is initialized to allow this object to bind to the stack.
     *
     * @param api The binding GraphQL Api.
     * @experimental
     */
    bind(api) {
        if (!this.schema) {
            this.schema = new appsync_generated_1.CfnGraphQLSchema(api, 'Schema', {
                apiId: api.apiId,
                definition: this.mode === private_1.SchemaMode.CODE ?
                    core_1.Lazy.stringValue({
                        produce: () => this.types.reduce((acc, type) => { return `${acc}${type._bindToGraphqlApi(api).toString()}\n`; }, `${this.declareSchema()}${this.definition}`),
                    })
                    : this.definition,
            });
        }
        return this.schema;
    }
    /**
     * (experimental) Escape hatch to add to Schema as desired.
     *
     * Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema.
     * @param delimiter the delimiter between schema and addition.
     * @default - ''
     * @experimental
     */
    addToSchema(addition, delimiter) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot append to schema because schema definition mode is not configured as CODE.');
        }
        const sep = delimiter !== null && delimiter !== void 0 ? delimiter : '';
        this.definition = `${this.definition}${sep}${addition}\n`;
    }
    /**
     * (experimental) Add a query field to the schema's Query. CDK will create an Object Type called 'Query'. For example,.
     *
     * type Query {
     *    fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query.
     * @param field the resolvable field to for this query.
     * @experimental
     */
    addQuery(fieldName, field) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add query. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.query) {
            this.query = new schema_intermediate_1.ObjectType('Query', { definition: {} });
            this.addType(this.query);
        }
        ;
        this.query.addField({ fieldName, field });
        return this.query;
    }
    /**
     * (experimental) Add a mutation field to the schema's Mutation. CDK will create an Object Type called 'Mutation'. For example,.
     *
     * type Mutation {
     *    fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation.
     * @param field the resolvable field to for this Mutation.
     * @experimental
     */
    addMutation(fieldName, field) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add mutation. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.mutation) {
            this.mutation = new schema_intermediate_1.ObjectType('Mutation', { definition: {} });
            this.addType(this.mutation);
        }
        ;
        this.mutation.addField({ fieldName, field });
        return this.mutation;
    }
    /**
     * (experimental) Add a subscription field to the schema's Subscription. CDK will create an Object Type called 'Subscription'. For example,.
     *
     * type Subscription {
     *    fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription.
     * @param field the resolvable field to for this Subscription.
     * @experimental
     */
    addSubscription(fieldName, field) {
        var _a, _b;
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error(`Unable to add subscription. Schema definition mode must be ${private_1.SchemaMode.CODE}. Received: ${this.mode}`);
        }
        if (!this.subscription) {
            this.subscription = new schema_intermediate_1.ObjectType('Subscription', { definition: {} });
            this.addType(this.subscription);
        }
        const directives = (_b = (_a = field.fieldOptions) === null || _a === void 0 ? void 0 : _a.directives) === null || _b === void 0 ? void 0 : _b.filter((directive) => directive.mutationFields);
        if (directives && directives.length > 1) {
            throw new Error(`Subscription fields must not have more than one @aws_subscribe directives. Received: ${directives.length}`);
        }
        this.subscription.addField({ fieldName, field });
        return this.subscription;
    }
    /**
     * (experimental) Add type to the schema.
     *
     * @param type the intermediate type to add to the schema.
     * @experimental
     */
    addType(type) {
        if (this.mode !== private_1.SchemaMode.CODE) {
            throw new Error('API cannot add type because schema definition mode is not configured as CODE.');
        }
        this.types.push(type);
        return type;
    }
    /**
     * Set the root types of this schema if they are defined.
     *
     * For example:
     * schema {
     *   query: Query
     *   mutation: Mutation
     *   subscription: Subscription
     * }
     */
    declareSchema() {
        if (!this.query && !this.mutation && !this.subscription) {
            return '';
        }
        const list = ['query', 'mutation', 'subscription'];
        return private_1.shapeAddition({
            prefix: 'schema',
            fields: list.map((key) => { var _a; return this[key] ? `${key}: ${(_a = this[key]) === null || _a === void 0 ? void 0 : _a.name}` : ''; })
                .filter((field) => field != ''),
        }) + '\n';
    }
}
exports.Schema = Schema;
//# sourceMappingURL=data:application/json;base64,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