import numpy as np

class KrylovSpace:
  '''Calculates various objects related to the krylov space of some square matrix. (Only works if all eigenvalues are different atm.)

  The krylov space is a vector space generated by a square matrix and the cayley-hamilton theorem
  which states that every square matrix is a root of its own characteristic polynomial. From this
  it follows that A^n is a linear combination of 1=A^0, A, A^2, ..., A^(n-1) and therefore every
  polynomial in A is such a linear combination.

  Attributes
  ----------
  M : array-like
    The matrix which generates the krylov space
  eigvals : array-like
    The eigenvalues of M
  basis : list[array-like]
    Basis of krylov generated by M
  beta : array-like
    A helper vector for solving the recurrence equation. See the documentation for details
  lambdaCoeffs : array-like
    Coefficients of M^n as a vector in the krylov space.
  funcCoeffs : array-like
    A tensor for calculating any matrix function efficiently
  '''

  def __init__(self, M: np.ndarray, eigvals: np.ndarray|None = None):
    self.M = M
    self.eigvals = np.linalg.eigvals(M) if not eigvals else eigvals
    self.basis = KrylovSpace.Basis(self.M)
    self.beta = KrylovSpace.Beta(eigvals=self.eigvals)
    self.lambdaCoeffs = KrylovSpace.Lambda(eigvals=self.eigvals)
    self.funcCoeffs = KrylovSpace.FunctionCoeffs(self.basis, self.beta, self.lambdaCoeffs, self.eigvals)

  @staticmethod
  def Basis(M: np.ndarray, rank: int|None = None):
    '''The list A^0, A^1, ..., A^`rank`. If rank is `None` use the dimension of the matrix.

    Parameters
    ----------
    M : array-like
      A square matrix
    rank : int | None = None
      The rank of `M`
    '''
    if rank is None: rank = M.shape[0]
    ret: list[np.ndarray] = [np.eye(M.shape[0]), M]
    while len(ret) < rank-1:
      ret.append(ret[-1]@M)
    return ret

  @staticmethod
  def Beta0(M: np.ndarray | None =None, eigvals: np.ndarray | None =None):
    '''A vector which is perpendicular to e_i*ev_i^m for m=0..n-2 where ev_i are the eigenvalues and n the number of eigenvalues

    Parameters
    ----------
    M : array-like | None = None
      A square matrix
    eigvals : array-like | None = None
      The eigenvalues of `M`
    Either `eigvals` or `M` should be specified.
    '''
    if eigvals is None:
      if M is None:
        raise ValueError('M or eigvals have to be specified')
      else:
        eigvals = np.linalg.eigvals(M)
    else:
      eigvals = np.array(eigvals)
    nEigvals = eigvals.shape[0]
    if nEigvals == 1: return 1.
    if nEigvals == 2: return np.array([1,-1], dtype=eigvals.dtype)
    Q = ([np.ones_like(eigvals), eigvals] + [eigvals**k for k in range(2, nEigvals-1)])[::-1]
    ret = [np.linalg.det([ei]+Q) for ei in np.eye(nEigvals)]
    return np.array(ret)

  @staticmethod
  def Beta(beta0: np.ndarray|None=None, eigvals: np.ndarray|None=None, M:np.ndarray|None=None):
    '''beta0/(beta0 @ e_i*ev_i^(n-1)), see also method `KrylovSpace.Beta0`.

    Parameters
    ----------
    beta0 : array-like | None = None
      Beta0
    eigvals : array-like | None = None
      The eigenvalues of `M`
    M : array-like | None = None
      A square matrix
    Either `eigvals` or `M` should be specified. Optionally specify beta0.
    '''
    if eigvals is None:
      if M is None:
        raise ValueError('M or eigvals have to be specified')
      else:
        eigvals = np.linalg.eigvals(M)
    else:
      eigvals = np.array(eigvals)
    if beta0 is None:
      beta0 = KrylovSpace.Beta0(M, eigvals)
    else:
      beta0 = np.array(beta0)

    nEigvals = eigvals.shape[0]
    lms = eigvals**(nEigvals-1)
    ret: np.ndarray = beta0 / (beta0 @ lms)
    return ret

  @staticmethod
  def Lambda(eigvals: np.ndarray|None=None, M: np.ndarray|None=None, inds: list[int]|int|None=None):
    '''The coefficients of M^n as a vector in the krylov space. There are also equal to the non-trivial negated coeffecients of the characteristic polynomial of `M`
    
    Parameters
    ----------
    eigvals : array-like | None = None
      The eigenvalues of `M`
    M : array-like | None = None
      A square matrix
    inds : list[int] | int | None
      The indices for which the coefficients should be calculated. If `None` all coefficients are calculated.
    Either `eigvals` or `M` should be specified. Optionally specify `inds`.
    '''
    if eigvals is None:
      if M is None:
        raise ValueError('M or eigvals have to be specified')
      else:
        eigvals = np.linalg.eigvals(M)
    else:
      eigvals = np.array(eigvals)
    nEigvals = eigvals.shape[0]
    if inds is None:
      inds = range(nEigvals)
    def getLambdaI(k:int) -> float|complex:
      def multiI():
        js = list(range(nEigvals-k))
        while True:
          yield js
          i = next((i for i in range(1, len(js)+1) if js[-i] < nEigvals-i), None)
          if i is None:
            break
          else:
            js[-i]+=1
            for j in range(-i+1, 0):
              js[j] = js[j-1]+1
          # if js[-1] < nEigvals-1:
          #   js[-1]+=1
          # elif js[-2] < nEigvals-2:
          #   js[-2]+=1
          #   js[-1]=js[-2]+1
          # #...
          # else:
          #   break
      arr = []
      for js in multiI():
        arr.append(-np.prod(-eigvals[js]))
      return np.sum(arr)

    if hasattr(inds, '__iter__'):
      return np.array([getLambdaI(i) for i in inds])
    else:
      return getLambdaI(inds)
    
  @staticmethod
  def FunctionCoeffs(basis: np.ndarray|None=None, beta:np.ndarray|None=None, lambdaCoeffs: np.ndarray|None=None, eigvals: np.ndarray|None=None, **kwargs):
    '''A rank 3 tensor which collects all contraction to efficiently calculate the matrix via tensor contraction.
    In total `f(M)_ij = FunctionCoeffs(..)_ijk f(eigvals_k)`

    Parameters
    ----------
    basis : array-like | None = None
      The list A^0, A^1, ..., A^(n-1)
    beta : array-like | None = None
      Beta
    lambdaCoeffs : array-like | None = None
      Lambda
    eigvals : array-like | None = None
      The eigenvalues of `M`
    Either `eigvals` or `M` should be specified. Optionally specify `basis`, `beta` and `lambdaCoeffs`.
    '''
    if eigvals is None:
      M = kwargs.get('M', None)
      if M is None:
        raise ValueError('M or eigvals have to be specified')
      else:
        eigvals = np.linalg.eigvals(M)
    else:
      eigvals = np.array(eigvals)
    nEigvals = eigvals.shape[0]
    if basis is None:
      basis = KrylovSpace.Basis(eigvals=eigvals, **kwargs)
    if beta is None:
      beta = KrylovSpace.Beta(eigvals=eigvals, **kwargs)
    if lambdaCoeffs is None:
      lambdaCoeffs = KrylovSpace.Lambda(eigvals=eigvals, **kwargs)
    eigvalTensor = np.array([eigvals**q for q in range(-nEigvals, 0)]).T
    #eigvalTensorK = lambda k: np.array([eigvals**q for q in range(-k-1, 0)]).T
    eigvalTensorK = lambda k: eigvalTensor[:, nEigvals-k-1:]
    ret : np.ndarray = np.sum([np.outer(basis[k], beta*(eigvalTensorK(k) @ lambdaCoeffs[:k+1])).reshape((nEigvals,)*3) for k in range(nEigvals)], axis=0)
    #ret = np.sum([np.einsum('ij,k,km,m->ijk', basis[k], beta, eigvalTensorK(k), lambdaCoeffs[:k+1]) for k in range(nEigvals)], axis=0)
    return ret

  def __repr__(self) -> str:
    return 'KrylovSpace'+repr(dict(
      M = self.M,
      Eigvals = self.eigvals,
      Basis = self.basis,
      Beta0 = KrylovSpace.Beta0(self.eigvals),
      Beta = self.beta,
      Lambda = self.lambdaCoeffs, 
      Coeffs = self.funcCoeffs,
    ))

