
#include "gtest/gtest.h"

#include "PROPOSAL/density_distr/density_exponential.h"
#include "PROPOSAL/density_distr/density_homogeneous.h"
#include "PROPOSAL/density_distr/density_polynomial.h"
#include "PROPOSAL/density_distr/density_splines.h"

using namespace PROPOSAL;

TEST(Comparison, Comparison_equal)
{
    Axis* ax_A = new RadialAxis();
    Axis* ax_B = new RadialAxis();
    EXPECT_TRUE(*ax_A == *ax_B);

    Axis* ax_C = new CartesianAxis();
    CartesianAxis ax_D;
    EXPECT_TRUE(*ax_C == ax_D);

    CartesianAxis ax_E;
    EXPECT_TRUE(ax_E == ax_D);

    Density_distr* A = new Density_homogeneous(0.3);
    Density_distr* B = new Density_homogeneous(0.3);
    EXPECT_TRUE(*A == *B);

    double sigma = 1.;
    double d0 = 0.;
    double massDensity = 1.;
    Density_distr* C = new Density_exponential(ax_D, sigma, d0, massDensity);
    Density_exponential D(ax_D, sigma, d0, massDensity);
    EXPECT_TRUE(D == *C);

    delete A;
    delete B;
    delete C;
    delete ax_A;
    delete ax_B;
    delete ax_C;
}

TEST(Comparison, Comparison_not_equal)
{
    Cartesian3D faxis_default(1,0,0);
    Cartesian3D faxis_new(0,1,0);
    Cartesian3D fp0_default(0,0,0);
    Cartesian3D fp0_new(1,1,1);
    CartesianAxis ax_A;
    CartesianAxis ax_B(faxis_default, fp0_new);
    CartesianAxis ax_C(faxis_new, fp0_default);
    EXPECT_TRUE(ax_A != ax_B);
    EXPECT_TRUE(ax_A != ax_C);

    Density_distr* B = new Density_exponential(ax_A, 1, 0, 1);
    Density_distr* C = new Density_exponential(ax_A, 1, 1, 1);
    Density_distr* D = new Density_exponential(ax_A, 2, 0, 1);
    EXPECT_TRUE(*B != *D);
    EXPECT_TRUE(*C != *D);
    EXPECT_TRUE(*B != *C);

    std::vector<double> vecA = {1,2};
    std::vector<double> vecB = {2,3};
    Polynom poly_A(vecA);
    Polynom poly_B(vecB);
    Density_polynomial E(ax_A, poly_A, 1);
    Density_polynomial F(ax_B, poly_A, 1);
    Density_polynomial G(ax_A, poly_B, 1);
    EXPECT_TRUE(E != F);
    EXPECT_TRUE(E != G);

    Linear_Spline spl_A(vecA, vecA);
    Linear_Spline spl_B(vecA, vecB);
    Density_splines H(ax_A, spl_A, 1);
    Density_splines I(ax_B, spl_A, 1);
    Density_splines J(ax_A, spl_B, 1);
    EXPECT_TRUE(H != I);
    EXPECT_TRUE(H != J);

    delete B;
    delete D;
}

TEST(Assignment, Copyconstructor)
{
    CartesianAxis ax_A;
    CartesianAxis ax_B = ax_A;
    CartesianAxis ax_C(ax_A);
    EXPECT_TRUE(ax_A == ax_B);
    EXPECT_TRUE(ax_A == ax_C);

    std::vector<double> vecA = {1,2};
    Polynom poly_A(vecA);
    Density_polynomial A(ax_A, poly_A, 1);
    Density_polynomial B = A;
    Density_polynomial C(A);
    EXPECT_TRUE(A == B);
    EXPECT_TRUE(A == C);
}

int main(int argc, char** argv)
{
    ::testing::InitGoogleTest(&argc, argv);
    return RUN_ALL_TESTS();
}
