"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CatalogBuilder = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const catalog_builder_1 = require("./catalog-builder");
const constants_1 = require("./constants");
/**
 * Builds or re-builds the `catalog.json` object in the designated bucket.
 */
class CatalogBuilder extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const handler = new catalog_builder_1.CatalogBuilder(this, 'Default', {
            description: `Creates the catalog.json object in ${props.bucket.bucketName}`,
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                AWS_EMF_ENVIRONMENT: 'Local',
            },
            logRetention: (_a = props.logRetention) !== null && _a !== void 0 ? _a : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            reservedConcurrentExecutions: 1,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.PASS_THROUGH,
        });
        this.function = handler;
        // This function may invoke itself in case it needs to continue it's work in
        // a "child" invocation. We must hence allow it to invoke itself. We cannot
        // use grantInvoke as this would (naturally) cause a circular reference
        // (Function -> Role -> Function).
        handler.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [core_1.Stack.of(this).formatArn({
                    arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                    service: 'lambda',
                    resource: 'function',
                    resourceName: '*',
                })],
        }));
        // allow the catalog builder to use the client.
        props.denyList.grantRead(handler);
        props.bucket.grantReadWrite(this.function);
        // Monitor the derivative of the catalog size, and alarm if the catalog
        // loses more than 5 items. Catalog elements can disappear if they are added
        // to the deny-list, or if they get un-published from the origin repository.
        // Such cases are normal and shouldn't typically result in a significant
        // contraction of the catalog size.
        const catalogSizeChange = new aws_cloudwatch_1.MathExpression({
            expression: 'DIFF(FILL(m1, REPEAT))',
            period: core_1.Duration.minutes(15),
            usingMetrics: { m1: this.metricRegisteredPackageMajorVersions() },
        });
        const alarmShrinkingCatalog = catalogSizeChange.createAlarm(this, 'ShrinkingCatalogAlarm', {
            alarmName: `${this.node.path}/ShrinkingCatalog`,
            alarmDescription: [
                'The number of packages registered in the catalog.json object has shrunk by more than 5',
                'elements. There might be a mass extinction event going on. This should be investigated',
                'as soon as possible.',
                '',
                `Catalog.json: ${deep_link_1.s3ObjectUrl(props.bucket, 'catalog.json')}`,
                `Catalog Builder: ${deep_link_1.lambdaFunctionUrl(handler)}`,
                '',
                `RUNBOOK: ${runbook_url_1.RUNBOOK_URL}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: -5,
        });
        props.monitoring.addHighSeverityAlarm('Catalog Size Shrunk', alarmShrinkingCatalog);
    }
    metricMissingConstructFrameworkCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(15),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingConstructFrameworkCount" /* MISSING_CONSTRUCT_FRAMEWORK_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricMissingConstructFrameworkVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(15),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingConstructFrameworkVersionCount" /* MISSING_CONSTRUCT_FRAMEWORK_VERSION_COUNT */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricRegisteredPackageMajorVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(15),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "RegisteredPackagesMajorVersion" /* REGISTERED_PACKAGES_MAJOR_VERSION */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
}
exports.CatalogBuilder = CatalogBuilder;
//# sourceMappingURL=data:application/json;base64,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