# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.base.exchange import Exchange
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import AuthenticationError
from ccxt.base.errors import ArgumentsRequired
from ccxt.base.errors import RateLimitExceeded
from ccxt.base.errors import InvalidNonce
from ccxt.base.decimal_to_precision import TICK_SIZE
from ccxt.base.precise import Precise


class coinbase(Exchange):

    def describe(self):
        return self.deep_extend(super(coinbase, self).describe(), {
            'id': 'coinbase',
            'name': 'Coinbase',
            'countries': ['US'],
            'rateLimit': 400,  # 10k calls per hour
            'version': 'v2',
            'userAgent': self.userAgents['chrome'],
            'headers': {
                'CB-VERSION': '2018-05-30',
            },
            'has': {
                'CORS': True,
                'spot': True,
                'margin': False,
                'swap': False,
                'future': False,
                'option': False,
                'addMargin': False,
                'cancelOrder': None,
                'createDepositAddress': True,
                'createOrder': None,
                'createReduceOnlyOrder': False,
                'createStopLimitOrder': False,
                'createStopMarketOrder': False,
                'createStopOrder': False,
                'fetchAccounts': True,
                'fetchBalance': True,
                'fetchBidsAsks': None,
                'fetchBorrowRate': False,
                'fetchBorrowRateHistories': False,
                'fetchBorrowRateHistory': False,
                'fetchBorrowRates': False,
                'fetchBorrowRatesPerSymbol': False,
                'fetchClosedOrders': None,
                'fetchCurrencies': True,
                'fetchDepositAddress': None,
                'fetchDeposits': True,
                'fetchFundingHistory': False,
                'fetchFundingRate': False,
                'fetchFundingRateHistory': False,
                'fetchFundingRates': False,
                'fetchIndexOHLCV': False,
                'fetchL2OrderBook': False,
                'fetchLedger': True,
                'fetchLeverage': False,
                'fetchLeverageTiers': False,
                'fetchMarginMode': False,
                'fetchMarkets': True,
                'fetchMarkOHLCV': False,
                'fetchMyBuys': True,
                'fetchMySells': True,
                'fetchMyTrades': None,
                'fetchOHLCV': False,
                'fetchOpenInterestHistory': False,
                'fetchOpenOrders': None,
                'fetchOrder': None,
                'fetchOrderBook': False,
                'fetchOrders': None,
                'fetchPosition': False,
                'fetchPositionMode': False,
                'fetchPositions': False,
                'fetchPositionsRisk': False,
                'fetchPremiumIndexOHLCV': False,
                'fetchTicker': True,
                'fetchTickers': True,
                'fetchTime': True,
                'fetchTrades': None,
                'fetchTradingFee': False,
                'fetchTradingFees': False,
                'fetchTransactions': None,
                'fetchWithdrawals': True,
                'reduceMargin': False,
                'setLeverage': False,
                'setMarginMode': False,
                'setPositionMode': False,
                'withdraw': None,
            },
            'urls': {
                'logo': 'https://user-images.githubusercontent.com/1294454/40811661-b6eceae2-653a-11e8-829e-10bfadb078cf.jpg',
                'api': {
                    'rest': 'https://api.coinbase.com',
                },
                'www': 'https://www.coinbase.com',
                'doc': [
                    'https://developers.coinbase.com/api/v2',
                    'https://docs.cloud.coinbase.com/advanced-trade-api/docs/welcome',
                ],
                'fees': [
                    'https://support.coinbase.com/customer/portal/articles/2109597-buy-sell-bank-transfer-fees',
                    'https://www.coinbase.com/advanced-fees',
                ],
                'referral': 'https://www.coinbase.com/join/58cbe25a355148797479dbd2',
            },
            'requiredCredentials': {
                'apiKey': True,
                'secret': True,
            },
            'api': {
                'v2': {
                    'public': {
                        'get': [
                            'currencies',
                            'time',
                            'exchange-rates',
                            'users/{user_id}',
                            'prices/{symbol}/buy',
                            'prices/{symbol}/sell',
                            'prices/{symbol}/spot',
                        ],
                    },
                    'private': {
                        'get': [
                            'accounts',
                            'accounts/{account_id}',
                            'accounts/{account_id}/addresses',
                            'accounts/{account_id}/addresses/{address_id}',
                            'accounts/{account_id}/addresses/{address_id}/transactions',
                            'accounts/{account_id}/transactions',
                            'accounts/{account_id}/transactions/{transaction_id}',
                            'accounts/{account_id}/buys',
                            'accounts/{account_id}/buys/{buy_id}',
                            'accounts/{account_id}/sells',
                            'accounts/{account_id}/sells/{sell_id}',
                            'accounts/{account_id}/deposits',
                            'accounts/{account_id}/deposits/{deposit_id}',
                            'accounts/{account_id}/withdrawals',
                            'accounts/{account_id}/withdrawals/{withdrawal_id}',
                            'payment-methods',
                            'payment-methods/{payment_method_id}',
                            'user',
                            'user/auth',
                        ],
                        'post': [
                            'accounts',
                            'accounts/{account_id}/primary',
                            'accounts/{account_id}/addresses',
                            'accounts/{account_id}/transactions',
                            'accounts/{account_id}/transactions/{transaction_id}/complete',
                            'accounts/{account_id}/transactions/{transaction_id}/resend',
                            'accounts/{account_id}/buys',
                            'accounts/{account_id}/buys/{buy_id}/commit',
                            'accounts/{account_id}/sells',
                            'accounts/{account_id}/sells/{sell_id}/commit',
                            'accounts/{account_id}/deposits',
                            'accounts/{account_id}/deposits/{deposit_id}/commit',
                            'accounts/{account_id}/withdrawals',
                            'accounts/{account_id}/withdrawals/{withdrawal_id}/commit',
                        ],
                        'put': [
                            'accounts/{account_id}',
                            'user',
                        ],
                        'delete': [
                            'accounts/{id}',
                            'accounts/{account_id}/transactions/{transaction_id}',
                        ],
                    },
                },
                'v3': {
                    'private': {
                        'get': [
                            'brokerage/accounts',
                            'brokerage/accounts/{account_uuid}',
                            'brokerage/orders/historical/batch',
                            'brokerage/orders/historical/fills',
                            'brokerage/orders/historical/{order_id}',
                            'brokerage/products',
                            'brokerage/products/{product_id}',
                            'brokerage/products/{product_id}/candles',
                            'brokerage/products/{product_id}/ticker',
                            'brokerage/transaction_summary',
                        ],
                        'post': [
                            'brokerage/orders',
                            'brokerage/orders/batch_cancel',
                        ],
                    },
                },
            },
            'fees': {
                'trading': {
                    'taker': self.parse_number('0.006'),
                    'maker': self.parse_number('0.004'),
                    'tierBased': True,
                    'percentage': True,
                    'tiers': {
                        'taker': [
                            [self.parse_number('0'), self.parse_number('0.006')],
                            [self.parse_number('10000'), self.parse_number('0.004')],
                            [self.parse_number('50000'), self.parse_number('0.0025')],
                            [self.parse_number('100000'), self.parse_number('0.002')],
                            [self.parse_number('1000000'), self.parse_number('0.0018')],
                            [self.parse_number('15000000'), self.parse_number('0.0016')],
                            [self.parse_number('75000000'), self.parse_number('0.0012')],
                            [self.parse_number('250000000'), self.parse_number('0.0008')],
                            [self.parse_number('400000000'), self.parse_number('0.0005')],
                        ],
                        'maker': [
                            [self.parse_number('0'), self.parse_number('0.004')],
                            [self.parse_number('10000'), self.parse_number('0.0025')],
                            [self.parse_number('50000'), self.parse_number('0.0015')],
                            [self.parse_number('100000'), self.parse_number('0.001')],
                            [self.parse_number('1000000'), self.parse_number('0.0008')],
                            [self.parse_number('15000000'), self.parse_number('0.0006')],
                            [self.parse_number('75000000'), self.parse_number('0.0003')],
                            [self.parse_number('250000000'), self.parse_number('0.0')],
                            [self.parse_number('400000000'), self.parse_number('0.0')],
                        ],
                    },
                },
            },
            'stablePairs': ['BUSD-USD', 'CBETH-ETH', 'DAI-USD', 'GUSD-USD', 'GYEN-USD', 'PAX-USD', 'PAX-USDT', 'USDC-EUR', 'USDC-GBP', 'USDT-EUR', 'USDT-GBP', 'USDT-USD', 'USDT-USDC', 'WBTC-BTC'],
            'precisionMode': TICK_SIZE,
            'exceptions': {
                'exact': {
                    'two_factor_required': AuthenticationError,  # 402 When sending money over 2fa limit
                    'param_required': ExchangeError,  # 400 Missing parameter
                    'validation_error': ExchangeError,  # 400 Unable to validate POST/PUT
                    'invalid_request': ExchangeError,  # 400 Invalid request
                    'personal_details_required': AuthenticationError,  # 400 User’s personal detail required to complete self request
                    'identity_verification_required': AuthenticationError,  # 400 Identity verification is required to complete self request
                    'jumio_verification_required': AuthenticationError,  # 400 Document verification is required to complete self request
                    'jumio_face_match_verification_required': AuthenticationError,  # 400 Document verification including face match is required to complete self request
                    'unverified_email': AuthenticationError,  # 400 User has not verified their email
                    'authentication_error': AuthenticationError,  # 401 Invalid auth(generic)
                    'invalid_authentication_method': AuthenticationError,  # 401 API access is blocked for deleted users.
                    'invalid_token': AuthenticationError,  # 401 Invalid Oauth token
                    'revoked_token': AuthenticationError,  # 401 Revoked Oauth token
                    'expired_token': AuthenticationError,  # 401 Expired Oauth token
                    'invalid_scope': AuthenticationError,  # 403 User hasn’t authenticated necessary scope
                    'not_found': ExchangeError,  # 404 Resource not found
                    'rate_limit_exceeded': RateLimitExceeded,  # 429 Rate limit exceeded
                    'internal_server_error': ExchangeError,  # 500 Internal server error
                },
                'broad': {
                    'request timestamp expired': InvalidNonce,  # {"errors":[{"id":"authentication_error","message":"request timestamp expired"}]}
                },
            },
            'commonCurrencies': {
                'CGLD': 'CELO',
            },
            'options': {
                'fetchCurrencies': {
                    'expires': 5000,
                },
                'accounts': [
                    'wallet',
                    'fiat',
                    # 'vault',
                ],
                'advanced': True,  # set to True if using any v3 endpoints from the advanced trade API
                'fetchMarkets': 'fetchMarketsV3',  # 'fetchMarketsV3' or 'fetchMarketsV2'
                'fetchTicker': 'fetchTickerV3',  # 'fetchTickerV3' or 'fetchTickerV2'
                'fetchTickers': 'fetchTickersV3',  # 'fetchTickersV3' or 'fetchTickersV2'
            },
        })

    def fetch_time(self, params={}):
        """
        fetches the current integer timestamp in milliseconds from the exchange server
        :param dict params: extra parameters specific to the coinbase api endpoint
        :returns int: the current integer timestamp in milliseconds from the exchange server
        """
        response = self.v2PublicGetTime(params)
        #
        #     {
        #         "data": {
        #             "epoch": 1589295679,
        #             "iso": "2020-05-12T15:01:19Z"
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        return self.safe_timestamp(data, 'epoch')

    def fetch_accounts(self, params={}):
        """
        fetch all the accounts associated with a profile
        :param dict params: extra parameters specific to the coinbase api endpoint
        :returns dict: a dictionary of `account structures <https://docs.ccxt.com/en/latest/manual.html#account-structure>` indexed by the account type
        """
        self.load_markets()
        request = {
            'limit': 100,
        }
        response = self.v2PrivateGetAccounts(self.extend(request, params))
        #
        #     {
        #         "id": "XLM",
        #         "name": "XLM Wallet",
        #         "primary": False,
        #         "type": "wallet",
        #         "currency": {
        #             "code": "XLM",
        #             "name": "Stellar Lumens",
        #             "color": "#000000",
        #             "sort_index": 127,
        #             "exponent": 7,
        #             "type": "crypto",
        #             "address_regex": "^G[A-Z2-7]{55}$",
        #             "asset_id": "13b83335-5ede-595b-821e-5bcdfa80560f",
        #             "destination_tag_name": "XLM Memo ID",
        #             "destination_tag_regex": "^[-~]{1,28}$"
        #         },
        #         "balance": {
        #             "amount": "0.0000000",
        #             "currency": "XLM"
        #         },
        #         "created_at": null,
        #         "updated_at": null,
        #         "resource": "account",
        #         "resource_path": "/v2/accounts/XLM",
        #         "allow_deposits": True,
        #         "allow_withdrawals": True
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_accounts(data, params)

    def parse_account(self, account):
        #
        #     {
        #         "id": "XLM",
        #         "name": "XLM Wallet",
        #         "primary": False,
        #         "type": "wallet",
        #         "currency": {
        #             "code": "XLM",
        #             "name": "Stellar Lumens",
        #             "color": "#000000",
        #             "sort_index": 127,
        #             "exponent": 7,
        #             "type": "crypto",
        #             "address_regex": "^G[A-Z2-7]{55}$",
        #             "asset_id": "13b83335-5ede-595b-821e-5bcdfa80560f",
        #             "destination_tag_name": "XLM Memo ID",
        #             "destination_tag_regex": "^[-~]{1,28}$"
        #         },
        #         "balance": {
        #             "amount": "0.0000000",
        #             "currency": "XLM"
        #         },
        #         "created_at": null,
        #         "updated_at": null,
        #         "resource": "account",
        #         "resource_path": "/v2/accounts/XLM",
        #         "allow_deposits": True,
        #         "allow_withdrawals": True
        #     }
        #
        currency = self.safe_value(account, 'currency', {})
        currencyId = self.safe_string(currency, 'code')
        code = self.safe_currency_code(currencyId)
        return {
            'id': self.safe_string(account, 'id'),
            'type': self.safe_string(account, 'type'),
            'code': code,
            'info': account,
        }

    def create_deposit_address(self, code, params={}):
        """
        create a currency deposit address
        :param str code: unified currency code of the currency for the deposit address
        :param dict params: extra parameters specific to the coinbase api endpoint
        :returns dict: an `address structure <https://docs.ccxt.com/en/latest/manual.html#address-structure>`
        """
        accountId = self.safe_string(params, 'account_id')
        params = self.omit(params, 'account_id')
        if accountId is None:
            self.load_accounts()
            for i in range(0, len(self.accounts)):
                account = self.accounts[i]
                if account['code'] == code and account['type'] == 'wallet':
                    accountId = account['id']
                    break
        if accountId is None:
            raise ExchangeError(self.id + ' createDepositAddress() could not find the account with matching currency code, specify an `account_id` extra param')
        request = {
            'account_id': accountId,
        }
        response = self.v2PrivatePostAccountsAccountIdAddresses(self.extend(request, params))
        #
        #     {
        #         "data": {
        #             "id": "05b1ebbf-9438-5dd4-b297-2ddedc98d0e4",
        #             "address": "coinbasebase",
        #             "address_info": {
        #                 "address": "coinbasebase",
        #                 "destination_tag": "287594668"
        #             },
        #             "name": null,
        #             "created_at": "2019-07-01T14:39:29Z",
        #             "updated_at": "2019-07-01T14:39:29Z",
        #             "network": "eosio",
        #             "uri_scheme": "eosio",
        #             "resource": "address",
        #             "resource_path": "/v2/accounts/14cfc769-e852-52f3-b831-711c104d194c/addresses/05b1ebbf-9438-5dd4-b297-2ddedc98d0e4",
        #             "warnings": [
        #                 {
        #                     "title": "Only send EOS(EOS) to self address",
        #                     "details": "Sending any other cryptocurrency will result in permanent loss.",
        #                     "image_url": "https://dynamic-assets.coinbase.com/deaca3d47b10ed4a91a872e9618706eec34081127762d88f2476ac8e99ada4b48525a9565cf2206d18c04053f278f693434af4d4629ca084a9d01b7a286a7e26/asset_icons/1f8489bb280fb0a0fd643c1161312ba49655040e9aaaced5f9ad3eeaf868eadc.png"
        #                 },
        #                 {
        #                     "title": "Both an address and EOS memo are required to receive EOS",
        #                     "details": "If you send funds without an EOS memo or with an incorrect EOS memo, your funds cannot be credited to your account.",
        #                     "image_url": "https://www.coinbase.com/assets/receive-warning-2f3269d83547a7748fb39d6e0c1c393aee26669bfea6b9f12718094a1abff155.png"
        #                 }
        #             ],
        #             "warning_title": "Only send EOS(EOS) to self address",
        #             "warning_details": "Sending any other cryptocurrency will result in permanent loss.",
        #             "destination_tag": "287594668",
        #             "deposit_uri": "eosio:coinbasebase?dt=287594668",
        #             "callback_url": null
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        tag = self.safe_string(data, 'destination_tag')
        address = self.safe_string(data, 'address')
        return {
            'currency': code,
            'tag': tag,
            'address': address,
            'info': response,
        }

    def fetch_my_sells(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch sells
        :param str|None symbol: not used by coinbase fetchMySells()
        :param int|None since: timestamp in ms of the earliest sell, default is None
        :param int|None limit: max number of sells to return, default is None
        :param dict params: extra parameters specific to the coinbase api endpoint
        :returns dict: a `list of order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        # they don't have an endpoint for all historical trades
        request = self.prepare_account_request(limit, params)
        self.load_markets()
        query = self.omit(params, ['account_id', 'accountId'])
        sells = self.v2PrivateGetAccountsAccountIdSells(self.extend(request, query))
        return self.parse_trades(sells['data'], None, since, limit)

    def fetch_my_buys(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch buys
        :param str|None symbol: not used by coinbase fetchMyBuys()
        :param int|None since: timestamp in ms of the earliest buy, default is None
        :param int|None limit: max number of buys to return, default is None
        :param dict params: extra parameters specific to the coinbase api endpoint
        :returns dict: a list of  `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        # they don't have an endpoint for all historical trades
        request = self.prepare_account_request(limit, params)
        self.load_markets()
        query = self.omit(params, ['account_id', 'accountId'])
        buys = self.v2PrivateGetAccountsAccountIdBuys(self.extend(request, query))
        return self.parse_trades(buys['data'], None, since, limit)

    def fetch_transactions_with_method(self, method, code=None, since=None, limit=None, params={}):
        request = self.prepare_account_request_with_currency_code(code, limit, params)
        self.load_markets()
        query = self.omit(params, ['account_id', 'accountId'])
        response = getattr(self, method)(self.extend(request, query))
        return self.parse_transactions(response['data'], None, since, limit)

    def fetch_withdrawals(self, code=None, since=None, limit=None, params={}):
        """
        fetch all withdrawals made from an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch withdrawals for
        :param int|None limit: the maximum number of withdrawals structures to retrieve
        :param dict params: extra parameters specific to the coinbase api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        # fiat only, for crypto transactions use fetchLedger
        return self.fetch_transactions_with_method('v2PrivateGetAccountsAccountIdWithdrawals', code, since, limit, params)

    def fetch_deposits(self, code=None, since=None, limit=None, params={}):
        """
        fetch all deposits made to an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch deposits for
        :param int|None limit: the maximum number of deposits structures to retrieve
        :param dict params: extra parameters specific to the coinbase api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        # fiat only, for crypto transactions use fetchLedger
        return self.fetch_transactions_with_method('v2PrivateGetAccountsAccountIdDeposits', code, since, limit, params)

    def parse_transaction_status(self, status):
        statuses = {
            'created': 'pending',
            'completed': 'ok',
            'canceled': 'canceled',
        }
        return self.safe_string(statuses, status, status)

    def parse_transaction(self, transaction, market=None):
        #
        # fiat deposit
        #
        #     {
        #         "id": "f34c19f3-b730-5e3d-9f72",
        #         "status": "completed",
        #         "payment_method": {
        #             "id": "a022b31d-f9c7-5043-98f2",
        #             "resource": "payment_method",
        #             "resource_path": "/v2/payment-methods/a022b31d-f9c7-5043-98f2"
        #         },
        #         "transaction": {
        #             "id": "04ed4113-3732-5b0c-af86-b1d2146977d0",
        #             "resource": "transaction",
        #             "resource_path": "/v2/accounts/91cd2d36-3a91-55b6-a5d4-0124cf105483/transactions/04ed4113-3732-5b0c-af86"
        #         },
        #         "user_reference": "2VTYTH",
        #         "created_at": "2017-02-09T07:01:18Z",
        #         "updated_at": "2017-02-09T07:01:26Z",
        #         "resource": "deposit",
        #         "resource_path": "/v2/accounts/91cd2d36-3a91-55b6-a5d4-0124cf105483/deposits/f34c19f3-b730-5e3d-9f72",
        #         "committed": True,
        #         "payout_at": "2017-02-12T07:01:17Z",
        #         "instant": False,
        #         "fee": {"amount": "0.00", "currency": "EUR"},
        #         "amount": {"amount": "114.02", "currency": "EUR"},
        #         "subtotal": {"amount": "114.02", "currency": "EUR"},
        #         "hold_until": null,
        #         "hold_days": 0,
        #         "hold_business_days": 0,
        #         "next_step": null
        #     }
        #
        # fiat_withdrawal
        #
        #     {
        #         "id": "cfcc3b4a-eeb6-5e8c-8058",
        #         "status": "completed",
        #         "payment_method": {
        #             "id": "8b94cfa4-f7fd-5a12-a76a",
        #             "resource": "payment_method",
        #             "resource_path": "/v2/payment-methods/8b94cfa4-f7fd-5a12-a76a"
        #         },
        #         "transaction": {
        #             "id": "fcc2550b-5104-5f83-a444",
        #             "resource": "transaction",
        #             "resource_path": "/v2/accounts/91cd2d36-3a91-55b6-a5d4-0124cf105483/transactions/fcc2550b-5104-5f83-a444"
        #         },
        #         "user_reference": "MEUGK",
        #         "created_at": "2018-07-26T08:55:12Z",
        #         "updated_at": "2018-07-26T08:58:18Z",
        #         "resource": "withdrawal",
        #         "resource_path": "/v2/accounts/91cd2d36-3a91-55b6-a5d4-0124cf105483/withdrawals/cfcc3b4a-eeb6-5e8c-8058",
        #         "committed": True,
        #         "payout_at": "2018-07-31T08:55:12Z",
        #         "instant": False,
        #         "fee": {"amount": "0.15", "currency": "EUR"},
        #         "amount": {"amount": "13130.69", "currency": "EUR"},
        #         "subtotal": {"amount": "13130.84", "currency": "EUR"},
        #         "idem": "e549dee5-63ed-4e79-8a96",
        #         "next_step": null
        #     }
        #
        subtotalObject = self.safe_value(transaction, 'subtotal', {})
        feeObject = self.safe_value(transaction, 'fee', {})
        id = self.safe_string(transaction, 'id')
        timestamp = self.parse8601(self.safe_value(transaction, 'created_at'))
        updated = self.parse8601(self.safe_value(transaction, 'updated_at'))
        type = self.safe_string(transaction, 'resource')
        amount = self.safe_number(subtotalObject, 'amount')
        currencyId = self.safe_string(subtotalObject, 'currency')
        currency = self.safe_currency_code(currencyId)
        feeCost = self.safe_number(feeObject, 'amount')
        feeCurrencyId = self.safe_string(feeObject, 'currency')
        feeCurrency = self.safe_currency_code(feeCurrencyId)
        fee = {
            'cost': feeCost,
            'currency': feeCurrency,
        }
        status = self.parse_transaction_status(self.safe_string(transaction, 'status'))
        if status is None:
            committed = self.safe_value(transaction, 'committed')
            status = 'ok' if committed else 'pending'
        return {
            'info': transaction,
            'id': id,
            'txid': id,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'network': None,
            'address': None,
            'addressTo': None,
            'addressFrom': None,
            'tag': None,
            'tagTo': None,
            'tagFrom': None,
            'type': type,
            'amount': amount,
            'currency': currency,
            'status': status,
            'updated': updated,
            'fee': fee,
        }

    def parse_trade(self, trade, market=None):
        #
        #     {
        #         "id": "67e0eaec-07d7-54c4-a72c-2e92826897df",
        #         "status": "completed",
        #         "payment_method": {
        #             "id": "83562370-3e5c-51db-87da-752af5ab9559",
        #             "resource": "payment_method",
        #             "resource_path": "/v2/payment-methods/83562370-3e5c-51db-87da-752af5ab9559"
        #         },
        #         "transaction": {
        #             "id": "441b9494-b3f0-5b98-b9b0-4d82c21c252a",
        #             "resource": "transaction",
        #             "resource_path": "/v2/accounts/2bbf394c-193b-5b2a-9155-3b4732659ede/transactions/441b9494-b3f0-5b98-b9b0-4d82c21c252a"
        #         },
        #         "amount": {"amount": "1.00000000", "currency": "BTC"},
        #         "total": {"amount": "10.25", "currency": "USD"},
        #         "subtotal": {"amount": "10.10", "currency": "USD"},
        #         "created_at": "2015-01-31T20:49:02Z",
        #         "updated_at": "2015-02-11T16:54:02-08:00",
        #         "resource": "buy",
        #         "resource_path": "/v2/accounts/2bbf394c-193b-5b2a-9155-3b4732659ede/buys/67e0eaec-07d7-54c4-a72c-2e92826897df",
        #         "committed": True,
        #         "instant": False,
        #         "fee": {"amount": "0.15", "currency": "USD"},
        #         "payout_at": "2015-02-18T16:54:00-08:00"
        #     }
        #
        symbol = None
        totalObject = self.safe_value(trade, 'total', {})
        amountObject = self.safe_value(trade, 'amount', {})
        subtotalObject = self.safe_value(trade, 'subtotal', {})
        feeObject = self.safe_value(trade, 'fee', {})
        id = self.safe_string(trade, 'id')
        timestamp = self.parse8601(self.safe_value(trade, 'created_at'))
        if market is None:
            baseId = self.safe_string(amountObject, 'currency')
            quoteId = self.safe_string(totalObject, 'currency')
            if (baseId is not None) and (quoteId is not None):
                base = self.safe_currency_code(baseId)
                quote = self.safe_currency_code(quoteId)
                symbol = base + '/' + quote
        orderId = None
        side = self.safe_string(trade, 'resource')
        type = None
        costString = self.safe_string(subtotalObject, 'amount')
        amountString = self.safe_string(amountObject, 'amount')
        cost = self.parse_number(costString)
        amount = self.parse_number(amountString)
        price = self.parse_number(Precise.string_div(costString, amountString))
        feeCost = self.safe_number(feeObject, 'amount')
        feeCurrencyId = self.safe_string(feeObject, 'currency')
        feeCurrency = self.safe_currency_code(feeCurrencyId)
        fee = {
            'cost': feeCost,
            'currency': feeCurrency,
        }
        return {
            'info': trade,
            'id': id,
            'order': orderId,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'symbol': symbol,
            'type': type,
            'side': side,
            'takerOrMaker': None,
            'price': price,
            'amount': amount,
            'cost': cost,
            'fee': fee,
        }

    def fetch_markets(self, params={}):
        """
        retrieves data on all markets for coinbase
        :param dict params: extra parameters specific to the exchange api endpoint
        :returns [dict]: an array of objects representing market data
        """
        method = self.safe_string(self.options, 'fetchMarkets', 'fetchMarketsV3')
        return getattr(self, method)(params)

    def fetch_markets_v2(self, params={}):
        response = self.fetch_currencies_from_cache(params)
        currencies = self.safe_value(response, 'currencies', {})
        exchangeRates = self.safe_value(response, 'exchangeRates', {})
        data = self.safe_value(currencies, 'data', [])
        dataById = self.index_by(data, 'id')
        rates = self.safe_value(self.safe_value(exchangeRates, 'data', {}), 'rates', {})
        baseIds = list(rates.keys())
        result = []
        for i in range(0, len(baseIds)):
            baseId = baseIds[i]
            base = self.safe_currency_code(baseId)
            type = 'fiat' if (baseId in dataById) else 'crypto'
            # https://github.com/ccxt/ccxt/issues/6066
            if type == 'crypto':
                for j in range(0, len(data)):
                    quoteCurrency = data[j]
                    quoteId = self.safe_string(quoteCurrency, 'id')
                    quote = self.safe_currency_code(quoteId)
                    result.append({
                        'id': baseId + '-' + quoteId,
                        'symbol': base + '/' + quote,
                        'base': base,
                        'quote': quote,
                        'settle': None,
                        'baseId': baseId,
                        'quoteId': quoteId,
                        'settleId': None,
                        'type': 'spot',
                        'spot': True,
                        'margin': False,
                        'swap': False,
                        'future': False,
                        'option': False,
                        'active': None,
                        'contract': False,
                        'linear': None,
                        'inverse': None,
                        'contractSize': None,
                        'expiry': None,
                        'expiryDatetime': None,
                        'strike': None,
                        'optionType': None,
                        'precision': {
                            'amount': None,
                            'price': None,
                        },
                        'limits': {
                            'leverage': {
                                'min': None,
                                'max': None,
                            },
                            'amount': {
                                'min': None,
                                'max': None,
                            },
                            'price': {
                                'min': None,
                                'max': None,
                            },
                            'cost': {
                                'min': self.safe_number(quoteCurrency, 'min_size'),
                                'max': None,
                            },
                        },
                        'info': quoteCurrency,
                    })
        return result

    def fetch_markets_v3(self, params={}):
        response = self.v3PrivateGetBrokerageProducts(params)
        #
        #     [
        #         {
        #             "product_id": "TONE-USD",
        #             "price": "0.01523",
        #             "price_percentage_change_24h": "1.94109772423025",
        #             "volume_24h": "19773129",
        #             "volume_percentage_change_24h": "437.0170530929949",
        #             "base_increment": "1",
        #             "quote_increment": "0.00001",
        #             "quote_min_size": "1",
        #             "quote_max_size": "10000000",
        #             "base_min_size": "26.7187147229469674",
        #             "base_max_size": "267187147.2294696735908216",
        #             "base_name": "TE-FOOD",
        #             "quote_name": "US Dollar",
        #             "watched": False,
        #             "is_disabled": False,
        #             "new": False,
        #             "status": "online",
        #             "cancel_only": False,
        #             "limit_only": False,
        #             "post_only": False,
        #             "trading_disabled": False,
        #             "auction_mode": False,
        #             "product_type": "SPOT",
        #             "quote_currency_id": "USD",
        #             "base_currency_id": "TONE",
        #             "fcm_trading_session_details": null,
        #             "mid_market_price": ""
        #         },
        #         ...
        #     ]
        #
        fees = self.v3PrivateGetBrokerageTransactionSummary(params)
        #
        #     {
        #         "total_volume": 0,
        #         "total_fees": 0,
        #         "fee_tier": {
        #             "pricing_tier": "",
        #             "usd_from": "0",
        #             "usd_to": "10000",
        #             "taker_fee_rate": "0.006",
        #             "maker_fee_rate": "0.004"
        #         },
        #         "margin_rate": null,
        #         "goods_and_services_tax": null,
        #         "advanced_trade_only_volume": 0,
        #         "advanced_trade_only_fees": 0,
        #         "coinbase_pro_volume": 0,
        #         "coinbase_pro_fees": 0
        #     }
        #
        feeTier = self.safe_value(fees, 'fee_tier', {})
        data = self.safe_value(response, 'products', [])
        result = []
        for i in range(0, len(data)):
            market = data[i]
            id = self.safe_string(market, 'product_id')
            baseId = self.safe_string(market, 'base_currency_id')
            quoteId = self.safe_string(market, 'quote_currency_id')
            base = self.safe_currency_code(baseId)
            quote = self.safe_currency_code(quoteId)
            marketType = self.safe_string_lower(market, 'product_type')
            tradingDisabled = self.safe_value(market, 'trading_disabled')
            result.append({
                'id': id,
                'symbol': base + '/' + quote,
                'base': base,
                'quote': quote,
                'settle': None,
                'baseId': baseId,
                'quoteId': quoteId,
                'settleId': None,
                'type': marketType,
                'spot': (marketType == 'spot'),
                'margin': None,
                'swap': False,
                'future': False,
                'option': False,
                'active': not tradingDisabled,
                'contract': False,
                'linear': None,
                'inverse': None,
                'taker': 0.00001 if self.in_array(id, self.stablePairs) else self.safe_number(feeTier, 'taker_fee_rate'),
                'maker': 0.0 if self.in_array(id, self.stablePairs) else self.safe_number(feeTier, 'maker_fee_rate'),
                'contractSize': None,
                'expiry': None,
                'expiryDatetime': None,
                'strike': None,
                'optionType': None,
                'precision': {
                    'amount': self.safe_number(market, 'base_increment'),
                    'price': self.safe_number(market, 'quote_increment'),
                },
                'limits': {
                    'leverage': {
                        'min': None,
                        'max': None,
                    },
                    'amount': {
                        'min': self.safe_number(market, 'base_min_size'),
                        'max': self.safe_number(market, 'base_max_size'),
                    },
                    'price': {
                        'min': None,
                        'max': None,
                    },
                    'cost': {
                        'min': self.safe_number(market, 'quote_min_size'),
                        'max': self.safe_number(market, 'quote_max_size'),
                    },
                },
                'info': market,
            })
        return result

    def fetch_currencies_from_cache(self, params={}):
        options = self.safe_value(self.options, 'fetchCurrencies', {})
        timestamp = self.safe_integer(options, 'timestamp')
        expires = self.safe_integer(options, 'expires', 1000)
        now = self.milliseconds()
        if (timestamp is None) or ((now - timestamp) > expires):
            currencies = self.v2PublicGetCurrencies(params)
            exchangeRates = self.v2PublicGetExchangeRates(params)
            self.options['fetchCurrencies'] = self.extend(options, {
                'currencies': currencies,
                'exchangeRates': exchangeRates,
                'timestamp': now,
            })
        return self.safe_value(self.options, 'fetchCurrencies', {})

    def fetch_currencies(self, params={}):
        """
        fetches all available currencies on an exchange
        :param dict params: extra parameters specific to the coinbase api endpoint
        :returns dict: an associative dictionary of currencies
        """
        response = self.fetch_currencies_from_cache(params)
        currencies = self.safe_value(response, 'currencies', {})
        #
        #     {
        #         "data":[
        #             {"id":"AED","name":"United Arab Emirates Dirham","min_size":"0.01000000"},
        #             {"id":"AFN","name":"Afghan Afghani","min_size":"0.01000000"},
        #             {"id":"ALL","name":"Albanian Lek","min_size":"0.01000000"},
        #             {"id":"AMD","name":"Armenian Dram","min_size":"0.01000000"},
        #             {"id":"ANG","name":"Netherlands Antillean Gulden","min_size":"0.01000000"},
        #             ...
        #         ],
        #     }
        #
        exchangeRates = self.safe_value(response, 'exchangeRates', {})
        #
        #     {
        #         "data":{
        #             "currency":"USD",
        #             "rates":{
        #                 "AED":"3.67",
        #                 "AFN":"78.21",
        #                 "ALL":"110.42",
        #                 "AMD":"474.18",
        #                 "ANG":"1.75",
        #                 ...
        #             },
        #         }
        #     }
        #
        data = self.safe_value(currencies, 'data', [])
        dataById = self.index_by(data, 'id')
        rates = self.safe_value(self.safe_value(exchangeRates, 'data', {}), 'rates', {})
        keys = list(rates.keys())
        result = {}
        for i in range(0, len(keys)):
            key = keys[i]
            type = 'fiat' if (key in dataById) else 'crypto'
            currency = self.safe_value(dataById, key, {})
            id = self.safe_string(currency, 'id', key)
            name = self.safe_string(currency, 'name')
            code = self.safe_currency_code(id)
            result[code] = {
                'id': id,
                'code': code,
                'info': currency,  # the original payload
                'type': type,
                'name': name,
                'active': True,
                'deposit': None,
                'withdraw': None,
                'fee': None,
                'precision': None,
                'limits': {
                    'amount': {
                        'min': self.safe_number(currency, 'min_size'),
                        'max': None,
                    },
                    'withdraw': {
                        'min': None,
                        'max': None,
                    },
                },
            }
        return result

    def fetch_tickers(self, symbols=None, params={}):
        """
        fetches price tickers for multiple markets, statistical calculations with the information calculated over the past 24 hours each market
        :param [str]|None symbols: unified symbols of the markets to fetch the ticker for, all market tickers are returned if not assigned
        :param dict params: extra parameters specific to the coinbase api endpoint
        :returns dict: an array of `ticker structures <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        method = self.safe_string(self.options, 'fetchTickers', 'fetchTickersV3')
        if method == 'fetchTickersV3':
            return self.fetch_tickers_v3(symbols, params)
        return self.fetch_tickers_v2(symbols, params)

    def fetch_tickers_v2(self, symbols=None, params={}):
        self.load_markets()
        symbols = self.market_symbols(symbols)
        request = {
            # 'currency': 'USD',
        }
        response = self.v2PublicGetExchangeRates(self.extend(request, params))
        #
        #     {
        #         "data":{
        #             "currency":"USD",
        #             "rates":{
        #                 "AED":"3.6731",
        #                 "AFN":"103.163942",
        #                 "ALL":"106.973038",
        #             }
        #         }
        #     }
        #
        data = self.safe_value(response, 'data', {})
        rates = self.safe_value(data, 'rates', {})
        quoteId = self.safe_string(data, 'currency')
        result = {}
        baseIds = list(rates.keys())
        delimiter = '-'
        for i in range(0, len(baseIds)):
            baseId = baseIds[i]
            marketId = baseId + delimiter + quoteId
            market = self.safe_market(marketId, None, delimiter)
            symbol = market['symbol']
            result[symbol] = self.parse_ticker(rates[baseId], market)
        return self.filter_by_array(result, 'symbol', symbols)

    def fetch_tickers_v3(self, symbols=None, params={}):
        self.load_markets()
        symbols = self.market_symbols(symbols)
        response = self.v3PrivateGetBrokerageProducts(params)
        #
        #     {
        #         'products': [
        #             {
        #                 "product_id": "TONE-USD",
        #                 "price": "0.01523",
        #                 "price_percentage_change_24h": "1.94109772423025",
        #                 "volume_24h": "19773129",
        #                 "volume_percentage_change_24h": "437.0170530929949",
        #                 "base_increment": "1",
        #                 "quote_increment": "0.00001",
        #                 "quote_min_size": "1",
        #                 "quote_max_size": "10000000",
        #                 "base_min_size": "26.7187147229469674",
        #                 "base_max_size": "267187147.2294696735908216",
        #                 "base_name": "TE-FOOD",
        #                 "quote_name": "US Dollar",
        #                 "watched": False,
        #                 "is_disabled": False,
        #                 "new": False,
        #                 "status": "online",
        #                 "cancel_only": False,
        #                 "limit_only": False,
        #                 "post_only": False,
        #                 "trading_disabled": False,
        #                 "auction_mode": False,
        #                 "product_type": "SPOT",
        #                 "quote_currency_id": "USD",
        #                 "base_currency_id": "TONE",
        #                 "fcm_trading_session_details": null,
        #                 "mid_market_price": ""
        #             },
        #             ...
        #         ],
        #         "num_products": 549
        #     }
        #
        data = self.safe_value(response, 'products', [])
        result = {}
        for i in range(0, len(data)):
            entry = data[i]
            marketId = self.safe_string(entry, 'product_id')
            market = self.safe_market(marketId, None, '-')
            symbol = market['symbol']
            result[symbol] = self.parse_ticker(entry, market)
        return self.filter_by_array(result, 'symbol', symbols)

    def fetch_ticker(self, symbol, params={}):
        """
        fetches a price ticker, a statistical calculation with the information calculated over the past 24 hours for a specific market
        :param str symbol: unified symbol of the market to fetch the ticker for
        :param dict params: extra parameters specific to the coinbase api endpoint
        :returns dict: a `ticker structure <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        method = self.safe_string(self.options, 'fetchTicker', 'fetchTickerV3')
        if method == 'fetchTickerV3':
            return self.fetch_ticker_v3(symbol, params)
        return self.fetch_ticker_v2(symbol, params)

    def fetch_ticker_v2(self, symbol, params={}):
        self.load_markets()
        market = self.market(symbol)
        request = self.extend({
            'symbol': market['id'],
        }, params)
        spot = self.v2PublicGetPricesSymbolSpot(request)
        #
        #     {"data":{"base":"BTC","currency":"USD","amount":"48691.23"}}
        #
        ask = self.v2PublicGetPricesSymbolBuy(request)
        #
        #     {"data":{"base":"BTC","currency":"USD","amount":"48691.23"}}
        #
        bid = self.v2PublicGetPricesSymbolSell(request)
        #
        #     {"data":{"base":"BTC","currency":"USD","amount":"48691.23"}}
        #
        spotData = self.safe_value(spot, 'data', {})
        askData = self.safe_value(ask, 'data', {})
        bidData = self.safe_value(bid, 'data', {})
        bidAskLast = {
            'bid': self.safe_number(bidData, 'amount'),
            'ask': self.safe_number(askData, 'amount'),
            'price': self.safe_number(spotData, 'amount'),
        }
        return self.parse_ticker(bidAskLast, market)

    def fetch_ticker_v3(self, symbol, params={}):
        self.load_markets()
        market = self.market(symbol)
        request = {
            'product_id': market['id'],
            'limit': 1,
        }
        response = self.v3PrivateGetBrokerageProductsProductIdTicker(self.extend(request, params))
        #
        #     {
        #         "trades": [
        #             {
        #                 "trade_id": "10209805",
        #                 "product_id": "BTC-USDT",
        #                 "price": "19381.27",
        #                 "size": "0.1",
        #                 "time": "2023-01-13T20:35:41.865970Z",
        #                 "side": "BUY",
        #                 "bid": "",
        #                 "ask": ""
        #             }
        #         ]
        #     }
        #
        data = self.safe_value(response, 'trades', [])
        return self.parse_ticker(data[0], market)

    def parse_ticker(self, ticker, market=None):
        #
        # fetchTickerV2
        #
        #     {
        #         "bid": 20713.37,
        #         "ask": 20924.65,
        #         "price": 20809.83
        #     }
        #
        # fetchTickerV3
        #
        #     {
        #         "trade_id": "10209805",
        #         "product_id": "BTC-USDT",
        #         "price": "19381.27",
        #         "size": "0.1",
        #         "time": "2023-01-13T20:35:41.865970Z",
        #         "side": "BUY",
        #         "bid": "",
        #         "ask": ""
        #     }
        #
        # fetchTickersV2
        #
        #     "48691.23"
        #
        # fetchTickersV3
        #
        #     [
        #         {
        #             "product_id": "TONE-USD",
        #             "price": "0.01523",
        #             "price_percentage_change_24h": "1.94109772423025",
        #             "volume_24h": "19773129",
        #             "volume_percentage_change_24h": "437.0170530929949",
        #             "base_increment": "1",
        #             "quote_increment": "0.00001",
        #             "quote_min_size": "1",
        #             "quote_max_size": "10000000",
        #             "base_min_size": "26.7187147229469674",
        #             "base_max_size": "267187147.2294696735908216",
        #             "base_name": "TE-FOOD",
        #             "quote_name": "US Dollar",
        #             "watched": False,
        #             "is_disabled": False,
        #             "new": False,
        #             "status": "online",
        #             "cancel_only": False,
        #             "limit_only": False,
        #             "post_only": False,
        #             "trading_disabled": False,
        #             "auction_mode": False,
        #             "product_type": "SPOT",
        #             "quote_currency_id": "USD",
        #             "base_currency_id": "TONE",
        #             "fcm_trading_session_details": null,
        #             "mid_market_price": ""
        #         },
        #         ...
        #     ]
        #
        marketId = self.safe_string(ticker, 'product_id')
        last = self.safe_number(ticker, 'price')
        return self.safe_ticker({
            'symbol': self.safe_symbol(marketId, market),
            'timestamp': None,
            'datetime': None,
            'bid': self.safe_number(ticker, 'bid'),
            'ask': self.safe_number(ticker, 'ask'),
            'last': last,
            'high': None,
            'low': None,
            'bidVolume': None,
            'askVolume': None,
            'vwap': None,
            'open': None,
            'close': last,
            'previousClose': None,
            'change': None,
            'percentage': self.safe_number(ticker, 'price_percentage_change_24h'),
            'average': None,
            'baseVolume': None,
            'quoteVolume': None,
            'info': ticker,
        }, market)

    def parse_balance(self, response, params={}):
        balances = self.safe_value(response, 'data', [])
        accounts = self.safe_value(params, 'type', self.options['accounts'])
        result = {'info': response}
        for b in range(0, len(balances)):
            balance = balances[b]
            type = self.safe_string(balance, 'type')
            if self.in_array(type, accounts):
                value = self.safe_value(balance, 'balance')
                if value is not None:
                    currencyId = self.safe_string(value, 'currency')
                    code = self.safe_currency_code(currencyId)
                    total = self.safe_string(value, 'amount')
                    free = total
                    account = self.safe_value(result, code)
                    if account is None:
                        account = self.account()
                        account['free'] = free
                        account['total'] = total
                    else:
                        account['free'] = Precise.string_add(account['free'], total)
                        account['total'] = Precise.string_add(account['total'], total)
                    result[code] = account
        return self.safe_balance(result)

    def fetch_balance(self, params={}):
        """
        query for balance and get the amount of funds available for trading or funds locked in orders
        :param dict params: extra parameters specific to the coinbase api endpoint
        :returns dict: a `balance structure <https://docs.ccxt.com/en/latest/manual.html?#balance-structure>`
        """
        self.load_markets()
        request = {
            'limit': 100,
        }
        response = self.v2PrivateGetAccounts(self.extend(request, params))
        #
        #     {
        #         "pagination":{
        #             "ending_before":null,
        #             "starting_after":null,
        #             "previous_ending_before":null,
        #             "next_starting_after":"6b17acd6-2e68-5eb0-9f45-72d67cef578b",
        #             "limit":100,
        #             "order":"desc",
        #             "previous_uri":null,
        #             "next_uri":"/v2/accounts?limit=100\u0026starting_after=6b17acd6-2e68-5eb0-9f45-72d67cef578b"
        #         },
        #         "data":[
        #             {
        #                 "id":"94ad58bc-0f15-5309-b35a-a4c86d7bad60",
        #                 "name":"MINA Wallet",
        #                 "primary":false,
        #                 "type":"wallet",
        #                 "currency":{
        #                     "code":"MINA",
        #                     "name":"Mina",
        #                     "color":"#EA6B48",
        #                     "sort_index":397,
        #                     "exponent":9,
        #                     "type":"crypto",
        #                     "address_regex":"^(B62)[A-Za-z0-9]{52}$",
        #                     "asset_id":"a4ffc575-942c-5e26-b70c-cb3befdd4229",
        #                     "slug":"mina"
        #                 },
        #                 "balance":{"amount":"0.000000000","currency":"MINA"},
        #                 "created_at":"2022-03-25T00:36:16Z",
        #                 "updated_at":"2022-03-25T00:36:16Z",
        #                 "resource":"account",
        #                 "resource_path":"/v2/accounts/94ad58bc-0f15-5309-b35a-a4c86d7bad60",
        #                 "allow_deposits":true,
        #                 "allow_withdrawals":true
        #             },
        #         ]
        #     }
        #
        return self.parse_balance(response, params)

    def fetch_ledger(self, code=None, since=None, limit=None, params={}):
        """
        fetch the history of changes, actions done by the user or operations that altered balance of the user
        :param str|None code: unified currency code, default is None
        :param int|None since: timestamp in ms of the earliest ledger entry, default is None
        :param int|None limit: max number of ledger entrys to return, default is None
        :param dict params: extra parameters specific to the coinbase api endpoint
        :returns dict: a `ledger structure <https://docs.ccxt.com/en/latest/manual.html#ledger-structure>`
        """
        self.load_markets()
        currency = None
        if code is not None:
            currency = self.currency(code)
        request = self.prepare_account_request_with_currency_code(code, limit, params)
        query = self.omit(params, ['account_id', 'accountId'])
        # for pagination use parameter 'starting_after'
        # the value for the next page can be obtained from the result of the previous call in the 'pagination' field
        # eg: instance.last_json_response.pagination.next_starting_after
        response = self.v2PrivateGetAccountsAccountIdTransactions(self.extend(request, query))
        return self.parse_ledger(response['data'], currency, since, limit)

    def parse_ledger_entry_status(self, status):
        types = {
            'completed': 'ok',
        }
        return self.safe_string(types, status, status)

    def parse_ledger_entry_type(self, type):
        types = {
            'buy': 'trade',
            'sell': 'trade',
            'fiat_deposit': 'transaction',
            'fiat_withdrawal': 'transaction',
            'exchange_deposit': 'transaction',  # fiat withdrawal(from coinbase to coinbasepro)
            'exchange_withdrawal': 'transaction',  # fiat deposit(to coinbase from coinbasepro)
            'send': 'transaction',  # crypto deposit OR withdrawal
            'pro_deposit': 'transaction',  # crypto withdrawal(from coinbase to coinbasepro)
            'pro_withdrawal': 'transaction',  # crypto deposit(to coinbase from coinbasepro)
        }
        return self.safe_string(types, type, type)

    def parse_ledger_entry(self, item, currency=None):
        #
        # crypto deposit transaction
        #
        #     {
        #         id: '34e4816b-4c8c-5323-a01c-35a9fa26e490',
        #         type: 'send',
        #         status: 'completed',
        #         amount: {amount: '28.31976528', currency: 'BCH'},
        #         native_amount: {amount: '2799.65', currency: 'GBP'},
        #         description: null,
        #         created_at: '2019-02-28T12:35:20Z',
        #         updated_at: '2019-02-28T12:43:24Z',
        #         resource: 'transaction',
        #         resource_path: '/v2/accounts/c01d7364-edd7-5f3a-bd1d-de53d4cbb25e/transactions/34e4816b-4c8c-5323-a01c-35a9fa26e490',
        #         instant_exchange: False,
        #         network: {
        #             status: 'confirmed',
        #             hash: '56222d865dae83774fccb2efbd9829cf08c75c94ce135bfe4276f3fb46d49701',
        #             transaction_url: 'https://bch.btc.com/56222d865dae83774fccb2efbd9829cf08c75c94ce135bfe4276f3fb46d49701'
        #         },
        #         from: {resource: 'bitcoin_cash_network', currency: 'BCH'},
        #         details: {title: 'Received Bitcoin Cash', subtitle: 'From Bitcoin Cash address'}
        #     }
        #
        # crypto withdrawal transaction
        #
        #     {
        #         id: '459aad99-2c41-5698-ac71-b6b81a05196c',
        #         type: 'send',
        #         status: 'completed',
        #         amount: {amount: '-0.36775642', currency: 'BTC'},
        #         native_amount: {amount: '-1111.65', currency: 'GBP'},
        #         description: null,
        #         created_at: '2019-03-20T08:37:07Z',
        #         updated_at: '2019-03-20T08:49:33Z',
        #         resource: 'transaction',
        #         resource_path: '/v2/accounts/c6afbd34-4bd0-501e-8616-4862c193cd84/transactions/459aad99-2c41-5698-ac71-b6b81a05196c',
        #         instant_exchange: False,
        #         network: {
        #             status: 'confirmed',
        #             hash: '2732bbcf35c69217c47b36dce64933d103895277fe25738ffb9284092701e05b',
        #             transaction_url: 'https://blockchain.info/tx/2732bbcf35c69217c47b36dce64933d103895277fe25738ffb9284092701e05b',
        #             transaction_fee: {amount: '0.00000000', currency: 'BTC'},
        #             transaction_amount: {amount: '0.36775642', currency: 'BTC'},
        #             confirmations: 15682
        #         },
        #         to: {
        #             resource: 'bitcoin_address',
        #             address: '1AHnhqbvbYx3rnZx8uC7NbFZaTe4tafFHX',
        #             currency: 'BTC',
        #             address_info: {address: '1AHnhqbvbYx3rnZx8uC7NbFZaTe4tafFHX'}
        #         },
        #         idem: 'da0a2f14-a2af-4c5a-a37e-d4484caf582bsend',
        #         application: {
        #             id: '5756ab6e-836b-553b-8950-5e389451225d',
        #             resource: 'application',
        #             resource_path: '/v2/applications/5756ab6e-836b-553b-8950-5e389451225d'
        #         },
        #         details: {title: 'Sent Bitcoin', subtitle: 'To Bitcoin address'}
        #     }
        #
        # withdrawal transaction from coinbase to coinbasepro
        #
        #     {
        #         id: '5b1b9fb8-5007-5393-b923-02903b973fdc',
        #         type: 'pro_deposit',
        #         status: 'completed',
        #         amount: {amount: '-0.00001111', currency: 'BCH'},
        #         native_amount: {amount: '0.00', currency: 'GBP'},
        #         description: null,
        #         created_at: '2019-02-28T13:31:58Z',
        #         updated_at: '2019-02-28T13:31:58Z',
        #         resource: 'transaction',
        #         resource_path: '/v2/accounts/c01d7364-edd7-5f3a-bd1d-de53d4cbb25e/transactions/5b1b9fb8-5007-5393-b923-02903b973fdc',
        #         instant_exchange: False,
        #         application: {
        #             id: '5756ab6e-836b-553b-8950-5e389451225d',
        #             resource: 'application',
        #             resource_path: '/v2/applications/5756ab6e-836b-553b-8950-5e389451225d'
        #         },
        #         details: {title: 'Transferred Bitcoin Cash', subtitle: 'To Coinbase Pro'}
        #     }
        #
        # withdrawal transaction from coinbase to gdax
        #
        #     {
        #         id: 'badb7313-a9d3-5c07-abd0-00f8b44199b1',
        #         type: 'exchange_deposit',
        #         status: 'completed',
        #         amount: {amount: '-0.43704149', currency: 'BCH'},
        #         native_amount: {amount: '-51.90', currency: 'GBP'},
        #         description: null,
        #         created_at: '2019-03-19T10:30:40Z',
        #         updated_at: '2019-03-19T10:30:40Z',
        #         resource: 'transaction',
        #         resource_path: '/v2/accounts/c01d7364-edd7-5f3a-bd1d-de53d4cbb25e/transactions/badb7313-a9d3-5c07-abd0-00f8b44199b1',
        #         instant_exchange: False,
        #         details: {title: 'Transferred Bitcoin Cash', subtitle: 'To GDAX'}
        #     }
        #
        # deposit transaction from gdax to coinbase
        #
        #     {
        #         id: '9c4b642c-8688-58bf-8962-13cef64097de',
        #         type: 'exchange_withdrawal',
        #         status: 'completed',
        #         amount: {amount: '0.57729420', currency: 'BTC'},
        #         native_amount: {amount: '4418.72', currency: 'GBP'},
        #         description: null,
        #         created_at: '2018-02-17T11:33:33Z',
        #         updated_at: '2018-02-17T11:33:33Z',
        #         resource: 'transaction',
        #         resource_path: '/v2/accounts/c6afbd34-4bd0-501e-8616-4862c193cd84/transactions/9c4b642c-8688-58bf-8962-13cef64097de',
        #         instant_exchange: False,
        #         details: {title: 'Transferred Bitcoin', subtitle: 'From GDAX'}
        #     }
        #
        # deposit transaction from coinbasepro to coinbase
        #
        #     {
        #         id: '8d6dd0b9-3416-568a-889d-8f112fae9e81',
        #         type: 'pro_withdrawal',
        #         status: 'completed',
        #         amount: {amount: '0.40555386', currency: 'BTC'},
        #         native_amount: {amount: '1140.27', currency: 'GBP'},
        #         description: null,
        #         created_at: '2019-03-04T19:41:58Z',
        #         updated_at: '2019-03-04T19:41:58Z',
        #         resource: 'transaction',
        #         resource_path: '/v2/accounts/c6afbd34-4bd0-501e-8616-4862c193cd84/transactions/8d6dd0b9-3416-568a-889d-8f112fae9e81',
        #         instant_exchange: False,
        #         application: {
        #             id: '5756ab6e-836b-553b-8950-5e389451225d',
        #             resource: 'application',
        #             resource_path: '/v2/applications/5756ab6e-836b-553b-8950-5e389451225d'
        #         },
        #         details: {title: 'Transferred Bitcoin', subtitle: 'From Coinbase Pro'}
        #     }
        #
        # sell trade
        #
        #     {
        #         id: 'a9409207-df64-585b-97ab-a50780d2149e',
        #         type: 'sell',
        #         status: 'completed',
        #         amount: {amount: '-9.09922880', currency: 'BTC'},
        #         native_amount: {amount: '-7285.73', currency: 'GBP'},
        #         description: null,
        #         created_at: '2017-03-27T15:38:34Z',
        #         updated_at: '2017-03-27T15:38:34Z',
        #         resource: 'transaction',
        #         resource_path: '/v2/accounts/c6afbd34-4bd0-501e-8616-4862c193cd84/transactions/a9409207-df64-585b-97ab-a50780d2149e',
        #         instant_exchange: False,
        #         sell: {
        #             id: 'e3550b4d-8ae6-5de3-95fe-1fb01ba83051',
        #             resource: 'sell',
        #             resource_path: '/v2/accounts/c6afbd34-4bd0-501e-8616-4862c193cd84/sells/e3550b4d-8ae6-5de3-95fe-1fb01ba83051'
        #         },
        #         details: {
        #             title: 'Sold Bitcoin',
        #             subtitle: 'Using EUR Wallet',
        #             payment_method_name: 'EUR Wallet'
        #         }
        #     }
        #
        # buy trade
        #
        #     {
        #         id: '63eeed67-9396-5912-86e9-73c4f10fe147',
        #         type: 'buy',
        #         status: 'completed',
        #         amount: {amount: '2.39605772', currency: 'ETH'},
        #         native_amount: {amount: '98.31', currency: 'GBP'},
        #         description: null,
        #         created_at: '2017-03-27T09:07:56Z',
        #         updated_at: '2017-03-27T09:07:57Z',
        #         resource: 'transaction',
        #         resource_path: '/v2/accounts/8902f85d-4a69-5d74-82fe-8e390201bda7/transactions/63eeed67-9396-5912-86e9-73c4f10fe147',
        #         instant_exchange: False,
        #         buy: {
        #             id: '20b25b36-76c6-5353-aa57-b06a29a39d82',
        #             resource: 'buy',
        #             resource_path: '/v2/accounts/8902f85d-4a69-5d74-82fe-8e390201bda7/buys/20b25b36-76c6-5353-aa57-b06a29a39d82'
        #         },
        #         details: {
        #             title: 'Bought Ethereum',
        #             subtitle: 'Using EUR Wallet',
        #             payment_method_name: 'EUR Wallet'
        #         }
        #     }
        #
        # fiat deposit transaction
        #
        #     {
        #         id: '04ed4113-3732-5b0c-af86-b1d2146977d0',
        #         type: 'fiat_deposit',
        #         status: 'completed',
        #         amount: {amount: '114.02', currency: 'EUR'},
        #         native_amount: {amount: '97.23', currency: 'GBP'},
        #         description: null,
        #         created_at: '2017-02-09T07:01:21Z',
        #         updated_at: '2017-02-09T07:01:22Z',
        #         resource: 'transaction',
        #         resource_path: '/v2/accounts/91cd2d36-3a91-55b6-a5d4-0124cf105483/transactions/04ed4113-3732-5b0c-af86-b1d2146977d0',
        #         instant_exchange: False,
        #         fiat_deposit: {
        #             id: 'f34c19f3-b730-5e3d-9f72-96520448677a',
        #             resource: 'fiat_deposit',
        #             resource_path: '/v2/accounts/91cd2d36-3a91-55b6-a5d4-0124cf105483/deposits/f34c19f3-b730-5e3d-9f72-96520448677a'
        #         },
        #         details: {
        #             title: 'Deposited funds',
        #             subtitle: 'From SEPA Transfer(GB47 BARC 20..., reference CBADVI)',
        #             payment_method_name: 'SEPA Transfer(GB47 BARC 20..., reference CBADVI)'
        #         }
        #     }
        #
        # fiat withdrawal transaction
        #
        #     {
        #         id: '957d98e2-f80e-5e2f-a28e-02945aa93079',
        #         type: 'fiat_withdrawal',
        #         status: 'completed',
        #         amount: {amount: '-11000.00', currency: 'EUR'},
        #         native_amount: {amount: '-9698.22', currency: 'GBP'},
        #         description: null,
        #         created_at: '2017-12-06T13:19:19Z',
        #         updated_at: '2017-12-06T13:19:19Z',
        #         resource: 'transaction',
        #         resource_path: '/v2/accounts/91cd2d36-3a91-55b6-a5d4-0124cf105483/transactions/957d98e2-f80e-5e2f-a28e-02945aa93079',
        #         instant_exchange: False,
        #         fiat_withdrawal: {
        #             id: 'f4bf1fd9-ab3b-5de7-906d-ed3e23f7a4e7',
        #             resource: 'fiat_withdrawal',
        #             resource_path: '/v2/accounts/91cd2d36-3a91-55b6-a5d4-0124cf105483/withdrawals/f4bf1fd9-ab3b-5de7-906d-ed3e23f7a4e7'
        #         },
        #         details: {
        #             title: 'Withdrew funds',
        #             subtitle: 'To HSBC BANK PLC(GB74 MIDL...)',
        #             payment_method_name: 'HSBC BANK PLC(GB74 MIDL...)'
        #         }
        #     }
        #
        amountInfo = self.safe_value(item, 'amount', {})
        amount = self.safe_string(amountInfo, 'amount')
        direction = None
        if Precise.string_lt(amount, '0'):
            direction = 'out'
            amount = Precise.string_neg(amount)
        else:
            direction = 'in'
        currencyId = self.safe_string(amountInfo, 'currency')
        code = self.safe_currency_code(currencyId, currency)
        #
        # the address and txid do not belong to the unified ledger structure
        #
        #     address = None
        #     if item['to']:
        #         address = self.safe_string(item['to'], 'address')
        #     }
        #     txid = None
        #
        fee = None
        networkInfo = self.safe_value(item, 'network', {})
        # txid = network['hash']  # txid does not belong to the unified ledger structure
        feeInfo = self.safe_value(networkInfo, 'transaction_fee')
        if feeInfo is not None:
            feeCurrencyId = self.safe_string(feeInfo, 'currency')
            feeCurrencyCode = self.safe_currency_code(feeCurrencyId, currency)
            feeAmount = self.safe_number(feeInfo, 'amount')
            fee = {
                'cost': feeAmount,
                'currency': feeCurrencyCode,
            }
        timestamp = self.parse8601(self.safe_value(item, 'created_at'))
        id = self.safe_string(item, 'id')
        type = self.parse_ledger_entry_type(self.safe_string(item, 'type'))
        status = self.parse_ledger_entry_status(self.safe_string(item, 'status'))
        path = self.safe_string(item, 'resource_path')
        accountId = None
        if path is not None:
            parts = path.split('/')
            numParts = len(parts)
            if numParts > 3:
                accountId = parts[3]
        return {
            'info': item,
            'id': id,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'direction': direction,
            'account': accountId,
            'referenceId': None,
            'referenceAccount': None,
            'type': type,
            'currency': code,
            'amount': self.parse_number(amount),
            'before': None,
            'after': None,
            'status': status,
            'fee': fee,
        }

    def find_account_id(self, code):
        self.load_markets()
        self.load_accounts()
        for i in range(0, len(self.accounts)):
            account = self.accounts[i]
            if account['code'] == code:
                return account['id']
        return None

    def prepare_account_request(self, limit=None, params={}):
        accountId = self.safe_string_2(params, 'account_id', 'accountId')
        if accountId is None:
            raise ArgumentsRequired(self.id + ' prepareAccountRequest() method requires an account_id(or accountId) parameter')
        request = {
            'account_id': accountId,
        }
        if limit is not None:
            request['limit'] = limit
        return request

    def prepare_account_request_with_currency_code(self, code=None, limit=None, params={}):
        accountId = self.safe_string_2(params, 'account_id', 'accountId')
        if accountId is None:
            if code is None:
                raise ArgumentsRequired(self.id + ' prepareAccountRequestWithCurrencyCode() method requires an account_id(or accountId) parameter OR a currency code argument')
            accountId = self.find_account_id(code)
            if accountId is None:
                raise ExchangeError(self.id + ' prepareAccountRequestWithCurrencyCode() could not find account id for ' + code)
        request = {
            'account_id': accountId,
        }
        if limit is not None:
            request['limit'] = limit
        return request

    def sign(self, path, api=[], method='GET', params={}, headers=None, body=None):
        version = api[0]
        signed = api[1] == 'private'
        pathPart = 'api/v3' if (version == 'v3') else 'v2'
        fullPath = '/' + pathPart + '/' + self.implode_params(path, params)
        query = self.omit(params, self.extract_params(path))
        savedPath = fullPath
        if method == 'GET':
            if query:
                fullPath += '?' + self.urlencode(query)
        url = self.urls['api']['rest'] + fullPath
        if signed:
            authorization = self.safe_string(self.headers, 'Authorization')
            if authorization is not None:
                headers = {
                    'Authorization': authorization,
                    'Content-Type': 'application/json',
                }
            elif self.token:
                headers = {
                    'Authorization': 'Bearer ' + self.token,
                    'Content-Type': 'application/json',
                }
            else:
                self.check_required_credentials()
                nonce = str(self.nonce())
                payload = ''
                if method != 'GET':
                    if query:
                        body = self.json(query)
                        payload = body
                auth = None
                if version == 'v3':
                    auth = nonce + method + savedPath + payload
                else:
                    auth = nonce + method + fullPath + payload
                signature = self.hmac(self.encode(auth), self.encode(self.secret))
                headers = {
                    'CB-ACCESS-KEY': self.apiKey,
                    'CB-ACCESS-SIGN': signature,
                    'CB-ACCESS-TIMESTAMP': nonce,
                    'Content-Type': 'application/json',
                }
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def handle_errors(self, code, reason, url, method, headers, body, response, requestHeaders, requestBody):
        if response is None:
            return  # fallback to default error handler
        feedback = self.id + ' ' + body
        #
        #    {"error": "invalid_request", "error_description": "The request is missing a required parameter, includes an unsupported parameter value, or is otherwise malformed."}
        #
        # or
        #
        #    {
        #      "errors": [
        #        {
        #          "id": "not_found",
        #          "message": "Not found"
        #        }
        #      ]
        #    }
        #
        errorCode = self.safe_string(response, 'error')
        if errorCode is not None:
            errorMessage = self.safe_string(response, 'error_description')
            self.throw_exactly_matched_exception(self.exceptions['exact'], errorCode, feedback)
            self.throw_broadly_matched_exception(self.exceptions['broad'], errorMessage, feedback)
            raise ExchangeError(feedback)
        errors = self.safe_value(response, 'errors')
        if errors is not None:
            if isinstance(errors, list):
                numErrors = len(errors)
                if numErrors > 0:
                    errorCode = self.safe_string(errors[0], 'id')
                    errorMessage = self.safe_string(errors[0], 'message')
                    if errorCode is not None:
                        self.throw_exactly_matched_exception(self.exceptions['exact'], errorCode, feedback)
                        self.throw_broadly_matched_exception(self.exceptions['broad'], errorMessage, feedback)
                        raise ExchangeError(feedback)
        advancedTrade = self.options['advanced']
        data = self.safe_value(response, 'data')
        if (data is None) and (not advancedTrade):
            raise ExchangeError(self.id + ' failed due to a malformed response ' + self.json(response))
