# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['fastapi_sqla']

package_data = \
{'': ['*']}

install_requires = \
['fastapi>=0.61', 'pydantic<2', 'sqlalchemy<2', 'structlog>=20']

extras_require = \
{'asyncpg': ['asyncpg>=0.25.0,<0.26.0'],
 'aws_rds_iam': ['boto3>=1.20.50,<2.0.0'],
 'tests': ['alembic>=1.4.3,<2.0.0',
           'asgi_lifespan>=1.0.1,<2.0.0',
           'black>=20.8b1,<21.0',
           'Faker>=4.5.0,<5.0.0',
           'httpx>=0.15.4,<0.16.0',
           'isort>=5.5.3,<6.0.0',
           'pdbpp>=0.10.2,<0.11.0',
           'psycopg2>=2.8.6,<3.0.0',
           'pylama>=7.7.1,<8.0.0',
           'pytest>=6.0.2,<7.0.0',
           'pytest-asyncio>=0.14.0,<0.15.0',
           'pytest-cov>=2.10.1,<3.0.0',
           'pytest-watch>=4.2.0,<5.0.0',
           'tox>=3.23.0,<4.0.0']}

entry_points = \
{'pytest11': ['fastapi-sqla = fastapi_sqla._pytest_plugin']}

setup_kwargs = {
    'name': 'fastapi-sqla',
    'version': '2.2.0',
    'description': 'A highly opinionated SQLAlchemy extension for FastAPI.',
    'long_description': '# fastapi-sqla\n\n[![Conventional Commits](https://img.shields.io/badge/Conventional%20Commits-1.0.0-brightgreen.svg)](https://conventionalcommits.org)\n[![CircleCI](https://circleci.com/gh/dialoguemd/fastapi-sqla.svg?style=svg&circle-token=998482f269270ee521aa54f2accbee2e22943743)](https://circleci.com/gh/dialoguemd/fastapi-sqla)\n[![codecov](https://codecov.io/gh/dialoguemd/fastapi-sqla/branch/master/graph/badge.svg?token=BQHLryClIn)](https://codecov.io/gh/dialoguemd/fastapi-sqla)\n\nA highly opinionated [SQLAlchemy] extension for [FastAPI]:\n\n* Setup using environment variables to connect on DB;\n* `fastapi_sqla.Base` a declarative base class to reflect DB tables at startup;\n* `fastapi_sqla.Session` a dependency to get an sqla session;\n* `fastapi_sqla.open_session` a context manager to get an sqla session;\n* `fastapi_sqla.asyncio_support.AsyncSession` a dependency to get an async sqla session ;\n* `fastapi_sqla.asyncio_support.open_session` a context manager to get an async sqla\n  session;\n* Automated commit/rollback of sqla session at the end of request before returning\n  response;\n* Pagination utilities;\n* Pytest fixtures;\n\n# Configuration\n\n## Environment variables:\n\nThe keys of interest in `os.environ` are prefixed with `sqlalchemy_`.\nEach matching key (after the prefix is stripped) is treated as though it were the\ncorresponding keyword argument to [`sqlalchemy.create_engine`]\ncall.\n\nThe only required key is `sqlalchemy_url`, which provides the database URL.\n\n### `asyncio` support using [`asyncpg`]\n\nSQLAlchemy `>= 1.4` supports `asyncio`.\nTo enable `asyncio` support against a Postgres DB, install `asyncpg`:\n\n```bash\npip install asyncpg\n```\n\nAnd define environment variable `async_sqlalchemy_url` with `postgres+asyncpg` scheme:\n\n```bash\nexport async_sqlalchemy_url=postgresql+asyncpg://postgres@localhost\n```\n\n## Setup the app:\n\n```python\nimport fastapi_sqla\nfrom fastapi import FastAPI\n\napp = FastAPI()\nfastapi_sqla.setup(app)\n```\n\n# SQLAlchemy\n\n## Adding a new entity class:\n\n```python\nfrom fastapi_sqla import Base\n\n\nclass Entity(Base):\n    __tablename__ = "table-name-in-db"\n```\n\n## Getting an sqla session\n\n### Using dependency injection\n\nUse [FastAPI dependency injection] to get a session as a parameter of a path operation\nfunction.\nSQLAlchemy session is committed before response is returned or rollbacked if any\nexception occurred:\n\n```python\nfrom fastapi import APIRouter, Depends\nfrom fastapi_sqla import Session\nfrom fastapi_sqla.asyncio_support import AsyncSession\n\nrouter = APIRouter()\n\n\n@router.get("/example")\ndef example(session: Session = Depends()):\n    return session.execute("SELECT now()").scalar()\n\n\n@router.get("/async_example")\nasync def async_example(session: AsyncSession = Depends()):\n    return await session.scalar("SELECT now()")\n```\n\n### Using a context manager\n\nWhen needing a session outside of a path operation, like when using\n[FastAPI background tasks], use `fastapi_sqla.open_session` context manager.\nSQLAlchemy session is committed when exiting context or rollbacked if any exception\noccurred:\n\n```python\nfrom fastapi import APIRouter, BackgroundTasks\nfrom fastapi_sqla import open_session\nfrom fastapi_sqla import asyncio_support\n\nrouter = APIRouter()\n\n\n@router.get("/example")\ndef example(bg: BackgroundTasks):\n    bg.add_task(run_bg)\n    bg.add_task(run_async_bg)\n\n\ndef run_bg():\n    with open_session() as session:\n        session.execute("SELECT now()").scalar()\n\n\nasync def run_async_bg():\n    async with asyncio_support.open_session() as session:\n        await session.scalar("SELECT now()")\n```\n\n## Pagination\n\n```python\nfrom fastapi import APIRouter, Depends\nfrom fastapi_sqla import Base, Page, Paginate\nfrom pydantic import BaseModel\nfrom sqlalchemy import select\n\nrouter = APIRouter()\n\n\nclass User(Base):\n    __tablename__ = "user"\n\n\nclass UserModel(BaseModel):\n    id: int\n    name: str\n\n    class Config:\n        orm_mode = True\n\n\n@router.get("/users", response_model=Page[UserModel])\ndef all_users(paginate: Paginate = Depends()):\n    return paginate(select(User))\n```\n\nBy default:\n\n* It returns pages of 10 items, up to 100 items;\n* Total number of items in the collection is queried using [`Query.count`] for legacy\n  orm queries and the equivalent for 2.0 style queries.\n* Response example for `/users?offset=40&limit=10`:\n\n    ```json\n    {\n        "data": [\n            {\n                "id": 41,\n                "name": "Pat Thomas"\n            },\n            {\n                "id": 42,\n                "name": "Mulatu Astatke"\n            }\n        ],\n        "meta": {\n            "offset": 40,\n            "total_items": 42,\n            "total_pages": 5,\n            "page_number": 5\n        }\n    }\n    ```\n\n### Paginating non-scalar results\n\nTo paginate a query which doesn\'t return [scalars], specify `scalars=False` when invoking\n`paginate`:\n\n```python\nfrom fastapi import APIRouter, Depends\nfrom fastapi_sqla import Base, Page, Paginate\nfrom pydantic import BaseModel\nfrom sqlalchemy import func, select\nfrom sqlalchemy.orm import relationship\n\nrouter = APIRouter()\n\n\nclass User(Base):\n    __tablename__ = "user"\n    notes = relationship("Note")\n\n\nclass Note(Base):\n    __tablename__ = "note"\n\n\nclass UserModel(BaseModel):\n    id: int\n    name: str\n    notes_count: int\n\n\n@router.get("/users", response_model=Page[UserModel])\ndef all_users(paginate: Paginate = Depends()):\n    query = (\n        select(User.id, User.name, func.count(Note.id).label("notes_count"))\n        .join(Note)\n        .group_by(User)\n    )\n    return paginate(query, scalars=False)\n```\n\n\n### Customize pagination\n\nYou can customize:\n- Minimum and maximum number of items per pages;\n- How the total number of items in the collection is queried;\n\nTo customize pagination, create a dependency using `fastapi_sqla.Pagination`:\n\n```python\nfrom fastapi import APIRouter, Depends\nfrom fastapi_sqla import Base, Page, Pagination, Session\nfrom pydantic import BaseModel\nfrom sqlalchemy import func, select\n\nrouter = APIRouter()\n\n\nclass User(Base):\n    __tablename__ = "user"\n\n\nclass UserModel(BaseModel):\n    id: int\n    name: str\n\n\ndef query_count(session: Session = Depends()) -> int:\n    return session.execute(select(func.count()).select_from(User)).scalar()\n\n\nPaginate = Pagination(min_page_size=5, max_page_size=500, query_count=query_count)\n\n\n@router.get("/users", response_model=Page[UserModel])\ndef all_users(paginate: Paginate = Depends()):\n    return paginate(select(User))\n```\n\n# Pytest fixtures\n\nThis library provides a set of utility fixtures, through its PyTest plugin, which is\nautomatically installed with the library.\n\nBy default, no records are actually written to the database when running tests.\nThere currently is no way to change this behaviour.\n\n## `sqla_modules`\n\nYou must define this fixture, in order for the plugin to reflect table metadata in your\nSQLAlchemy entities. It should just import all of the application\'s modules which contain\nSQLAlchemy models.\n\nExample:\n\n```python\n# tests/conftest.py\nfrom pytest import fixture\n\n\n@fixture\ndef sqla_modules():\n    from app import sqla  # noqa\n```\n\n## `db_url`\n\nThe DB url to use.\n\nWhen `CI` key is set in environment variables, it defaults to using `postgres` as the\nhost name:\n\n```\npostgresql://postgres@postgres/postgres\n```\n\nIn other cases, the host is set to `localhost`:\n\n```\npostgresql://postgres@localhost/postgres\n```\n\nOf course, you can override it by overloading the fixture:\n\n```python\nfrom pytest import fixture\n\n\n@fixture(scope="session")\ndef db_url():\n    return "postgresql://postgres@localhost/test_database"\n```\n\n## `async_sqlalchemy_url`\n\nDB url to use when using `asyncio` support. Defaults to `db_url` fixture with\n`postgresql+asyncpg://` scheme.\n\n\n## `session` & `async_session`\n\nSqla sessions to create db fixture:\n* All changes done at test setup or during the test are rollbacked at test tear down;\n* No record will actually be written in the database;\n* Changes in one regular session need to be committed to be available from other regular\n  sessions;\n* Changes in one async session need to be committed to be available from other async\n  sessions;\n* Changes from regular sessions are not available from `async` session and vice-versa\n  even when committed;\n\nExample:\n```python\nfrom pytest import fixture\n\n\n@fixture\ndef patient(session):\n    from er.sqla import Patient\n    patient = Patient(first_name="Bob", last_name="David")\n    session.add(patient)\n    session.commit()\n    return patient\n\n\n@fixture\nasync def doctor(async_session):\n    from er.sqla import Doctor\n    doctor = Doctor(name="who")\n    async_session.add(doctor)\n    await async_session.commit()\n    return doctor\n```\n\n## `db_migration`\n\nA session scope fixture that runs `alembic upgrade` at test session setup and\n`alembic downgrade` at tear down.\n\nIt depends on `alembic_ini_path` fixture to get the path of `alembic.ini` file.\n\nTo use in a test or test module:\n\n```python\nfrom pytest import mark\n\npytestmark = mark.usefixtures("db_migration")\n```\n\nTo use globally, add to [pytest options]:\n\n ```ini\n [pytest]\n usefixtures =\n     db_migration\n ```\n\nOr depends on it in top-level `conftest.py` and mark it as _auto-used_:\n\n```python\nfrom pytest import fixture\n\n\n@fixture(scope="session", autouse=True)\ndef db_migration(db_migration):\n    pass\n```\n\n## `alembic_ini_path`\n\nIt returns the path of  `alembic.ini` configuration file. By default, it returns\n`./alembic.ini`.\n\n\n# Development\n\n## Prerequisites\n\n- **Python >=3.9**\n- [**Poetry**](https://poetry.eustace.io/) to install package dependencies.\n- A postgres DB reachable at `postgresql://postgres@localhost/postgres`\n\n\n## Setup\n\n```bash\n$ poetry install --extras tests --extras asyncpg --extras aws_rds_iam\n```\n\n## Running tests\n\n```bash\n$ poetry run pytest\n```\n\n#### Runing tests on multiple environments\n\n```bash\n$ poetry run tox\n```\n\n[`sqlalchemy.create_engine`]: https://docs.sqlalchemy.org/en/13/core/engines.html?highlight=create_engine#sqlalchemy.create_engine\n[`Query.count`]: https://docs.sqlalchemy.org/en/13/orm/query.html#sqlalchemy.orm.query.Query.count\n[pytest options]: https://docs.pytest.org/en/stable/reference.html#confval-usefixtures\n[FastAPI]: https://fastapi.tiangolo.com/\n[FastAPI dependency injection]: https://fastapi.tiangolo.com/tutorial/dependencies/\n[FastAPI background tasks]: https://fastapi.tiangolo.com/tutorial/background-tasks/\n[SQLAlchemy]: http://sqlalchemy.org/\n[`asyncpg`]: https://magicstack.github.io/asyncpg/current/\n[scalars]: (https://docs.sqlalchemy.org/en/14/core/connections.html?highlight=scalars#sqlalchemy.engine.Result.scalars),\n',
    'author': 'Hadrien David',
    'author_email': 'hadrien.david@dialogue.co',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/dialoguemd/fastapi-sqla',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'entry_points': entry_points,
    'python_requires': '>=3.7,<4.0',
}


setup(**setup_kwargs)
