/*
 * Decompiled with CFR 0.152.
 */
package loci.common;

import java.io.BufferedInputStream;
import java.io.DataInputStream;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.FileAttribute;
import java.util.regex.Pattern;
import loci.common.HandleException;
import loci.common.StreamHandle;
import loci.common.services.DependencyException;
import loci.common.services.S3ClientService;
import loci.common.services.S3ClientServiceException;
import loci.common.services.S3ClientStat;
import loci.common.services.ServiceFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class S3Handle
extends StreamHandle {
    public static final String DEFAULT_S3_PROTOCOL = "https";
    private static final Logger LOGGER = LoggerFactory.getLogger(S3Handle.class);
    protected static final Pattern SCHEME_PARSER = Pattern.compile("s3(\\+\\p{Alnum}+)?(://.*)?");
    private final StreamHandle.Settings settings;
    private final URI uri;
    private final String accessKey;
    private final String secretKey;
    private final String bucket;
    private final String server;
    private final int port;
    private final String path;
    private S3ClientService s3Client;
    private S3ClientStat stat;
    private boolean isBucket;
    private Throwable objectNotFound;
    protected static final int S3_MAX_FORWARD_SEEK = 0x100000;

    public static boolean canHandleScheme(String url) {
        return SCHEME_PARSER.matcher(url).matches();
    }

    public S3Handle(String url) throws IOException {
        this(url, true, null);
    }

    public S3Handle(String uristr, boolean initialize, StreamHandle.Settings s2) throws IOException {
        String[] pathparts;
        this.settings = s2 == null ? new StreamHandle.Settings() : s2;
        try {
            this.uri = new URI(uristr);
        }
        catch (URISyntaxException e) {
            throw new RuntimeException("Invalid URI " + uristr, e);
        }
        String auth = this.uri.getUserInfo();
        String accessKey = null;
        String secretKey = null;
        if (auth != null) {
            String[] authparts = auth.split(":", 2);
            accessKey = authparts[0];
            if (authparts.length > 1) {
                secretKey = authparts[1];
            }
        }
        this.accessKey = accessKey;
        this.secretKey = secretKey;
        String scheme = this.uri.getScheme();
        String protocol = scheme.equals("s3") ? DEFAULT_S3_PROTOCOL : (scheme.startsWith("s3+") ? scheme.substring(3) : scheme);
        this.server = protocol + "://" + this.uri.getHost();
        this.port = this.uri.getPort() == -1 ? 0 : this.uri.getPort();
        String fullpath = this.uri.getPath();
        if (fullpath == null || fullpath.length() == 0) {
            fullpath = "/";
        }
        this.bucket = (pathparts = fullpath.split("/", 3))[1].length() > 0 ? pathparts[1] : null;
        this.path = pathparts.length > 2 && pathparts[2].length() > 0 ? pathparts[2] : null;
        this.isBucket = false;
        this.stat = null;
        if (initialize) {
            this.connect();
            try {
                this.initialize();
            }
            catch (S3ClientServiceException e) {
                this.objectNotFound = e;
                LOGGER.debug("Object not found: [{}] {}", (Object)this, (Object)e);
            }
            LOGGER.trace("isBucket:{} stat:{}", (Object)this.isBucket, (Object)this.stat);
        }
    }

    protected void connect() throws IOException {
        String appName = "Bio-Formats";
        String appVersion = "dev";
        try {
            ServiceFactory factory = new ServiceFactory();
            this.s3Client = factory.getInstance(S3ClientService.class);
            this.s3Client.initialize(this.server, this.port, this.accessKey, this.secretKey, "Bio-Formats", "dev");
        }
        catch (S3ClientServiceException e) {
            throw new IOException(String.format("Failed to connect: %s", this), e);
        }
        catch (DependencyException e) {
            throw new IOException(String.format("S3 requires additional dependencies: %s", this), e);
        }
        LOGGER.trace("connected: server:{} port:{}", (Object)this.server, (Object)this.port);
    }

    protected void initialize() throws IOException, S3ClientServiceException {
        if (this.path == null) {
            this.isBucket = this.s3Client.bucketExists(this.bucket);
        } else {
            this.isBucket = false;
            this.stat = this.s3Client.statObject(this.bucket, this.path);
            this.resetStream();
        }
    }

    public String getServer() {
        return this.server;
    }

    public int getPort() {
        return this.port;
    }

    public String getBucket() {
        return this.bucket;
    }

    public String getPath() {
        return this.path;
    }

    public static String cacheObject(String url, StreamHandle.Settings s2) throws IOException, HandleException {
        String cacheroot = s2.getRemoteCacheRootDir();
        if (cacheroot == null) {
            throw new HandleException("Remote cache root dir is not set");
        }
        S3Handle s3 = new S3Handle(url, true, s2);
        String cacheobj = s3.getCacheKey();
        Path cachepath = Paths.get(cacheroot, cacheobj);
        if (Files.exists(cachepath, new LinkOption[0])) {
            LOGGER.debug("Found existing cache for {} at {}", (Object)s3, (Object)cachepath);
        } else {
            LOGGER.debug("Caching {} to {}", (Object)s3, (Object)cachepath);
            s3.downloadObject(cachepath);
            LOGGER.debug("Downloaded {}", (Object)cachepath);
        }
        return cachepath.toString();
    }

    public String getCacheKey() {
        String cachekey = this.getServer().replace("://", "/") + "/" + this.getPort() + "/" + this.getBucket() + "/" + this.getPath();
        return cachekey;
    }

    protected void downloadObject(Path destination) throws HandleException, IOException {
        LOGGER.trace("destination:{}", (Object)destination);
        if (this.stat == null || this.objectNotFound != null) {
            throw new IOException("Object not found " + this, this.objectNotFound);
        }
        if (this.path == null) {
            throw new HandleException("Download path=null not allowed");
        }
        Files.createDirectories(destination.getParent(), new FileAttribute[0]);
        try {
            this.s3Client.getObject(this.bucket, this.path, destination.toString());
        }
        catch (S3ClientServiceException e) {
            throw new HandleException("Download failed " + this.toString(), e);
        }
    }

    public boolean isBucket() {
        return this.isBucket;
    }

    @Override
    public long length() throws IOException {
        if (this.stat == null || this.objectNotFound != null) {
            throw new DelayedObjectNotFound(this);
        }
        return this.length;
    }

    @Override
    public void seek(long pos) throws IOException {
        LOGGER.trace("{}", (Object)pos);
        if (this.stat == null || this.objectNotFound != null) {
            throw new DelayedObjectNotFound(this);
        }
        long diff = pos - this.fp;
        if (diff < 0L || diff > 0x100000L) {
            this.resetStream(pos);
        } else {
            super.seek(pos);
        }
    }

    @Override
    protected void resetStream() throws IOException {
        this.resetStream(0L);
    }

    @Override
    public boolean exists() throws IOException {
        return this.objectNotFound == null && (this.isBucket || this.stat != null);
    }

    protected void resetStream(long offset) throws IOException {
        LOGGER.trace("Resetting {}", (Object)offset);
        if (this.stat == null || this.objectNotFound != null) {
            throw new DelayedObjectNotFound(this);
        }
        try {
            this.length = this.stat.length();
            this.stream = new DataInputStream(new BufferedInputStream(this.s3Client.getObject(this.bucket, this.path, offset)));
            this.fp = offset;
            this.mark = offset;
        }
        catch (S3ClientServiceException e) {
            throw new IOException(String.format("failed to load s3: %s\n\t%s", this.uri, this), e);
        }
    }

    public String toString() {
        boolean found = this.objectNotFound == null && (this.isBucket || this.stat != null);
        return String.format("server:%s port:%d bucket:%s path:%s found:%s", this.server, this.port, this.bucket, this.path, found);
    }

    class DelayedObjectNotFound
    extends IOException {
        DelayedObjectNotFound(S3Handle s3) {
            super(String.format("Object not found: [%s] %s", s3, s3.objectNotFound), s3.objectNotFound);
        }
    }
}

