# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.async_support.base.exchange import Exchange
import hashlib
import math
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import AuthenticationError
from ccxt.base.errors import PermissionDenied
from ccxt.base.errors import ArgumentsRequired
from ccxt.base.errors import BadRequest
from ccxt.base.errors import BadSymbol
from ccxt.base.errors import InsufficientFunds
from ccxt.base.errors import InvalidOrder
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import ExchangeNotAvailable
from ccxt.base.errors import RequestTimeout


class huobipro(Exchange):

    def describe(self):
        return self.deep_extend(super(huobipro, self).describe(), {
            'id': 'huobipro',
            'name': 'Huobi Pro',
            'countries': ['CN'],
            'rateLimit': 2000,
            'userAgent': self.userAgents['chrome39'],
            'version': 'v1',
            'accounts': None,
            'accountsById': None,
            'hostname': 'api.huobi.pro',
            'pro': True,
            'has': {
                'CORS': False,
                'fetchTickers': True,
                'fetchDepositAddress': False,
                'fetchOHLCV': True,
                'fetchOrder': True,
                'fetchOrders': True,
                'fetchOpenOrders': True,
                'fetchClosedOrders': True,
                'fetchTradingLimits': True,
                'fetchMyTrades': True,
                'withdraw': True,
                'fetchCurrencies': True,
                'fetchDeposits': True,
                'fetchWithdrawals': True,
            },
            'timeframes': {
                '1m': '1min',
                '5m': '5min',
                '15m': '15min',
                '30m': '30min',
                '1h': '60min',
                '4h': '4hour',
                '1d': '1day',
                '1w': '1week',
                '1M': '1mon',
                '1y': '1year',
            },
            'urls': {
                'logo': 'https://user-images.githubusercontent.com/1294454/76137448-22748a80-604e-11ea-8069-6e389271911d.jpg',
                'api': {
                    'market': 'https://{hostname}',
                    'public': 'https://{hostname}',
                    'private': 'https://{hostname}',
                    'zendesk': 'https://huobiglobal.zendesk.com/hc/en-us/articles',
                },
                'www': 'https://www.huobi.pro',
                'referral': 'https://www.huobi.co/en-us/topic/invited/?invite_code=rwrd3',
                'doc': 'https://huobiapi.github.io/docs/spot/v1/cn/',
                'fees': 'https://www.huobi.pro/about/fee/',
            },
            'api': {
                'zendesk': {
                    'get': [
                        '360000400491-Trade-Limits',
                    ],
                },
                'market': {
                    'get': [
                        'history/kline',  # 获取K线数据
                        'detail/merged',  # 获取聚合行情(Ticker)
                        'depth',  # 获取 Market Depth 数据
                        'trade',  # 获取 Trade Detail 数据
                        'history/trade',  # 批量获取最近的交易记录
                        'detail',  # 获取 Market Detail 24小时成交量数据
                        'tickers',
                    ],
                },
                'public': {
                    'get': [
                        'common/symbols',  # 查询系统支持的所有交易对
                        'common/currencys',  # 查询系统支持的所有币种
                        'common/timestamp',  # 查询系统当前时间
                        'common/exchange',  # order limits
                        'settings/currencys',  # ?language=en-US
                    ],
                },
                'private': {
                    # todo add v2 endpoints
                    # 'GET /v2/account/withdraw/quota
                    # 'GET /v2/reference/currencies
                    # 'GET /v2/account/deposit/address
                    'get': [
                        'account/accounts',  # 查询当前用户的所有账户(即account-id)
                        'account/accounts/{id}/balance',  # 查询指定账户的余额
                        'account/accounts/{sub-uid}',
                        'account/history',
                        'cross-margin/loan-info',
                        'fee/fee-rate/get',
                        'order/openOrders',
                        'order/orders',
                        'order/orders/{id}',  # 查询某个订单详情
                        'order/orders/{id}/matchresults',  # 查询某个订单的成交明细
                        'order/orders/getClientOrder',
                        'order/history',  # 查询当前委托、历史委托
                        'order/matchresults',  # 查询当前成交、历史成交
                        'dw/withdraw-virtual/addresses',  # 查询虚拟币提现地址
                        'query/deposit-withdraw',
                        'margin/loan-orders',  # 借贷订单
                        'margin/accounts/balance',  # 借贷账户详情
                        'points/actions',
                        'points/orders',
                        'subuser/aggregate-balance',
                        'stable-coin/exchange_rate',
                        'stable-coin/quote',
                    ],
                    # todo add v2 endpoints
                    # POST /v2/sub-user/management
                    'post': [
                        'futures/transfer',
                        'order/batch-orders',
                        'order/orders/place',  # 创建并执行一个新订单(一步下单， 推荐使用)
                        'order/orders/submitCancelClientOrder',
                        'order/orders/batchCancelOpenOrders',
                        'order/orders',  # 创建一个新的订单请求 （仅创建订单，不执行下单）
                        'order/orders/{id}/place',  # 执行一个订单 （仅执行已创建的订单）
                        'order/orders/{id}/submitcancel',  # 申请撤销一个订单请求
                        'order/orders/batchcancel',  # 批量撤销订单
                        'dw/balance/transfer',  # 资产划转
                        'dw/withdraw/api/create',  # 申请提现虚拟币
                        'dw/withdraw-virtual/create',  # 申请提现虚拟币
                        'dw/withdraw-virtual/{id}/place',  # 确认申请虚拟币提现
                        'dw/withdraw-virtual/{id}/cancel',  # 申请取消提现虚拟币
                        'dw/transfer-in/margin',  # 现货账户划入至借贷账户
                        'dw/transfer-out/margin',  # 借贷账户划出至现货账户
                        'margin/orders',  # 申请借贷
                        'margin/orders/{id}/repay',  # 归还借贷
                        'stable-coin/exchange',
                        'subuser/transfer',
                    ],
                },
            },
            'fees': {
                'trading': {
                    'tierBased': False,
                    'percentage': True,
                    'maker': 0.002,
                    'taker': 0.002,
                },
            },
            'exceptions': {
                'exact': {
                    # err-code
                    'api-not-support-temp-addr': PermissionDenied,  # {"status":"error","err-code":"api-not-support-temp-addr","err-msg":"API withdrawal does not support temporary addresses","data":null}
                    'timeout': RequestTimeout,  # {"ts":1571653730865,"status":"error","err-code":"timeout","err-msg":"Request Timeout"}
                    'gateway-internal-error': ExchangeNotAvailable,  # {"status":"error","err-code":"gateway-internal-error","err-msg":"Failed to load data. Try again later.","data":null}
                    'account-frozen-balance-insufficient-error': InsufficientFunds,  # {"status":"error","err-code":"account-frozen-balance-insufficient-error","err-msg":"trade account balance is not enough, left: `0.0027`","data":null}
                    'invalid-amount': InvalidOrder,  # eg "Paramemter `amount` is invalid."
                    'order-limitorder-amount-min-error': InvalidOrder,  # limit order amount error, min: `0.001`
                    'order-marketorder-amount-min-error': InvalidOrder,  # market order amount error, min: `0.01`
                    'order-limitorder-price-min-error': InvalidOrder,  # limit order price error
                    'order-limitorder-price-max-error': InvalidOrder,  # limit order price error
                    'order-orderstate-error': OrderNotFound,  # canceling an already canceled order
                    'order-queryorder-invalid': OrderNotFound,  # querying a non-existent order
                    'order-update-error': ExchangeNotAvailable,  # undocumented error
                    'api-signature-check-failed': AuthenticationError,
                    'api-signature-not-valid': AuthenticationError,  # {"status":"error","err-code":"api-signature-not-valid","err-msg":"Signature not valid: Incorrect Access key [Access key错误]","data":null}
                    'base-record-invalid': OrderNotFound,  # https://github.com/ccxt/ccxt/issues/5750
                    # err-msg
                    'invalid symbol': BadSymbol,  # {"ts":1568813334794,"status":"error","err-code":"invalid-parameter","err-msg":"invalid symbol"}
                    'invalid-parameter': BadRequest,  # {"ts":1576210479343,"status":"error","err-code":"invalid-parameter","err-msg":"symbol trade not open now"}
                    'base-symbol-trade-disabled': BadSymbol,  # {"status":"error","err-code":"base-symbol-trade-disabled","err-msg":"Trading is disabled for self symbol","data":null}
                },
            },
            'options': {
                # https://github.com/ccxt/ccxt/issues/5376
                'fetchOrdersByStatesMethod': 'private_get_order_orders',  # 'private_get_order_history'  # https://github.com/ccxt/ccxt/pull/5392
                'fetchOpenOrdersMethod': 'fetch_open_orders_v1',  # 'fetch_open_orders_v2'  # https://github.com/ccxt/ccxt/issues/5388
                'createMarketBuyOrderRequiresPrice': True,
                'fetchMarketsMethod': 'publicGetCommonSymbols',
                'fetchBalanceMethod': 'privateGetAccountAccountsIdBalance',
                'createOrderMethod': 'privatePostOrderOrdersPlace',
                'language': 'en-US',
            },
            'commonCurrencies': {
                # https://github.com/ccxt/ccxt/issues/6081
                # https://github.com/ccxt/ccxt/issues/3365
                # https://github.com/ccxt/ccxt/issues/2873
                'GET': 'Themis',  # conflict with GET(Guaranteed Entrance Token, GET Protocol)
                'HOT': 'Hydro Protocol',  # conflict with HOT(Holo) https://github.com/ccxt/ccxt/issues/4929
            },
        })

    async def fetch_trading_limits(self, symbols=None, params={}):
        # self method should not be called directly, use loadTradingLimits() instead
        #  by default it will try load withdrawal fees of all currencies(with separate requests)
        #  however if you define symbols = ['ETH/BTC', 'LTC/BTC'] in args it will only load those
        await self.load_markets()
        if symbols is None:
            symbols = self.symbols
        result = {}
        for i in range(0, len(symbols)):
            symbol = symbols[i]
            result[symbol] = await self.fetch_trading_limits_by_id(self.market_id(symbol), params)
        return result

    async def fetch_trading_limits_by_id(self, id, params={}):
        request = {
            'symbol': id,
        }
        response = await self.publicGetCommonExchange(self.extend(request, params))
        #
        #     {status:   "ok",
        #         data: {                                 symbol: "aidocbtc",
        #                              'buy-limit-must-less-than':  1.1,
        #                          'sell-limit-must-greater-than':  0.9,
        #                         'limit-order-must-greater-than':  1,
        #                            'limit-order-must-less-than':  5000000,
        #                    'market-buy-order-must-greater-than':  0.0001,
        #                       'market-buy-order-must-less-than':  100,
        #                   'market-sell-order-must-greater-than':  1,
        #                      'market-sell-order-must-less-than':  500000,
        #                       'circuit-break-when-greater-than':  10000,
        #                          'circuit-break-when-less-than':  10,
        #                 'market-sell-order-rate-must-less-than':  0.1,
        #                  'market-buy-order-rate-must-less-than':  0.1        }}
        #
        return self.parse_trading_limits(self.safe_value(response, 'data', {}))

    def parse_trading_limits(self, limits, symbol=None, params={}):
        #
        #   {                                 symbol: "aidocbtc",
        #                  'buy-limit-must-less-than':  1.1,
        #              'sell-limit-must-greater-than':  0.9,
        #             'limit-order-must-greater-than':  1,
        #                'limit-order-must-less-than':  5000000,
        #        'market-buy-order-must-greater-than':  0.0001,
        #           'market-buy-order-must-less-than':  100,
        #       'market-sell-order-must-greater-than':  1,
        #          'market-sell-order-must-less-than':  500000,
        #           'circuit-break-when-greater-than':  10000,
        #              'circuit-break-when-less-than':  10,
        #     'market-sell-order-rate-must-less-than':  0.1,
        #      'market-buy-order-rate-must-less-than':  0.1        }
        #
        return {
            'info': limits,
            'limits': {
                'amount': {
                    'min': self.safe_float(limits, 'limit-order-must-greater-than'),
                    'max': self.safe_float(limits, 'limit-order-must-less-than'),
                },
            },
        }

    async def fetch_markets(self, params={}):
        method = self.options['fetchMarketsMethod']
        response = await getattr(self, method)(params)
        markets = self.safe_value(response, 'data')
        numMarkets = len(markets)
        if numMarkets < 1:
            raise ExchangeError(self.id + ' publicGetCommonSymbols returned empty response: ' + self.json(markets))
        result = []
        for i in range(0, len(markets)):
            market = markets[i]
            baseId = self.safe_string(market, 'base-currency')
            quoteId = self.safe_string(market, 'quote-currency')
            id = baseId + quoteId
            base = self.safe_currency_code(baseId)
            quote = self.safe_currency_code(quoteId)
            symbol = base + '/' + quote
            precision = {
                'amount': market['amount-precision'],
                'price': market['price-precision'],
            }
            maker = 0 if (base == 'OMG') else 0.2 / 100
            taker = 0 if (base == 'OMG') else 0.2 / 100
            minAmount = self.safe_float(market, 'min-order-amt', math.pow(10, -precision['amount']))
            maxAmount = self.safe_float(market, 'max-order-amt')
            minCost = self.safe_float(market, 'min-order-value', 0)
            state = self.safe_string(market, 'state')
            active = (state == 'online')
            result.append({
                'id': id,
                'symbol': symbol,
                'base': base,
                'quote': quote,
                'baseId': baseId,
                'quoteId': quoteId,
                'active': active,
                'precision': precision,
                'taker': taker,
                'maker': maker,
                'limits': {
                    'amount': {
                        'min': minAmount,
                        'max': maxAmount,
                    },
                    'price': {
                        'min': math.pow(10, -precision['price']),
                        'max': None,
                    },
                    'cost': {
                        'min': minCost,
                        'max': None,
                    },
                },
                'info': market,
            })
        return result

    def parse_ticker(self, ticker, market=None):
        #
        #     {
        #         "amount": 26228.672978342216,
        #         "open": 9078.95,
        #         "close": 9146.86,
        #         "high": 9155.41,
        #         "id": 209988544334,
        #         "count": 265846,
        #         "low": 8988.0,
        #         "version": 209988544334,
        #         "ask": [9146.87, 0.156134],
        #         "vol": 2.3822168242201668E8,
        #         "bid": [9146.86, 0.080758],
        #     }
        #
        symbol = None
        if market is not None:
            symbol = market['symbol']
        timestamp = self.safe_integer(ticker, 'ts')
        bid = None
        ask = None
        bidVolume = None
        askVolume = None
        if 'bid' in ticker:
            if isinstance(ticker['bid'], list):
                bid = self.safe_float(ticker['bid'], 0)
                bidVolume = self.safe_float(ticker['bid'], 1)
        if 'ask' in ticker:
            if isinstance(ticker['ask'], list):
                ask = self.safe_float(ticker['ask'], 0)
                askVolume = self.safe_float(ticker['ask'], 1)
        open = self.safe_float(ticker, 'open')
        close = self.safe_float(ticker, 'close')
        change = None
        percentage = None
        average = None
        if (open is not None) and (close is not None):
            change = close - open
            average = self.sum(open, close) / 2
            if (close is not None) and (close > 0):
                percentage = (change / open) * 100
        baseVolume = self.safe_float(ticker, 'amount')
        quoteVolume = self.safe_float(ticker, 'vol')
        vwap = None
        if baseVolume is not None and quoteVolume is not None and baseVolume > 0:
            vwap = quoteVolume / baseVolume
        return {
            'symbol': symbol,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'high': self.safe_float(ticker, 'high'),
            'low': self.safe_float(ticker, 'low'),
            'bid': bid,
            'bidVolume': bidVolume,
            'ask': ask,
            'askVolume': askVolume,
            'vwap': vwap,
            'open': open,
            'close': close,
            'last': close,
            'previousClose': None,
            'change': change,
            'percentage': percentage,
            'average': average,
            'baseVolume': baseVolume,
            'quoteVolume': quoteVolume,
            'info': ticker,
        }

    async def fetch_order_book(self, symbol, limit=None, params={}):
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
            'type': 'step0',
        }
        response = await self.marketGetDepth(self.extend(request, params))
        #
        #     {
        #         "status": "ok",
        #         "ch": "market.btcusdt.depth.step0",
        #         "ts": 1583474832790,
        #         "tick": {
        #             "bids": [
        #                 [9100.290000000000000000, 0.200000000000000000],
        #                 [9099.820000000000000000, 0.200000000000000000],
        #                 [9099.610000000000000000, 0.205000000000000000],
        #             ],
        #             "asks": [
        #                 [9100.640000000000000000, 0.005904000000000000],
        #                 [9101.010000000000000000, 0.287311000000000000],
        #                 [9101.030000000000000000, 0.012121000000000000],
        #             ],
        #             "ts":1583474832008,
        #             "version":104999698780
        #         }
        #     }
        #
        if 'tick' in response:
            if not response['tick']:
                raise ExchangeError(self.id + ' fetchOrderBook() returned empty response: ' + self.json(response))
            tick = self.safe_value(response, 'tick')
            timestamp = self.safe_integer(tick, 'ts', self.safe_integer(response, 'ts'))
            result = self.parse_order_book(tick, timestamp)
            result['nonce'] = self.safe_integer(tick, 'version')
            return result
        raise ExchangeError(self.id + ' fetchOrderBook() returned unrecognized response: ' + self.json(response))

    async def fetch_ticker(self, symbol, params={}):
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        response = await self.marketGetDetailMerged(self.extend(request, params))
        #
        #     {
        #         "status": "ok",
        #         "ch": "market.btcusdt.detail.merged",
        #         "ts": 1583494336669,
        #         "tick": {
        #             "amount": 26228.672978342216,
        #             "open": 9078.95,
        #             "close": 9146.86,
        #             "high": 9155.41,
        #             "id": 209988544334,
        #             "count": 265846,
        #             "low": 8988.0,
        #             "version": 209988544334,
        #             "ask": [9146.87, 0.156134],
        #             "vol": 2.3822168242201668E8,
        #             "bid": [9146.86, 0.080758],
        #         }
        #     }
        #
        ticker = self.parse_ticker(response['tick'], market)
        timestamp = self.safe_value(response, 'ts')
        ticker['timestamp'] = timestamp
        ticker['datetime'] = self.iso8601(timestamp)
        return ticker

    async def fetch_tickers(self, symbols=None, params={}):
        await self.load_markets()
        response = await self.marketGetTickers(params)
        tickers = self.safe_value(response, 'data')
        timestamp = self.safe_integer(response, 'ts')
        result = {}
        for i in range(0, len(tickers)):
            marketId = self.safe_string(tickers[i], 'symbol')
            market = self.safe_value(self.markets_by_id, marketId)
            symbol = marketId
            if market is not None:
                symbol = market['symbol']
                ticker = self.parse_ticker(tickers[i], market)
                ticker['timestamp'] = timestamp
                ticker['datetime'] = self.iso8601(timestamp)
                result[symbol] = ticker
        return result

    def parse_trade(self, trade, market=None):
        #
        # fetchTrades(public)
        #
        #     {
        #         "amount": 0.010411000000000000,
        #         "trade-id": 102090736910,
        #         "ts": 1583497692182,
        #         "id": 10500517034273194594947,
        #         "price": 9096.050000000000000000,
        #         "direction": "sell"
        #     }
        #
        # fetchMyTrades(private)
        #
        symbol = None
        if market is None:
            marketId = self.safe_string(trade, 'symbol')
            if marketId in self.markets_by_id:
                market = self.markets_by_id[marketId]
        if market is not None:
            symbol = market['symbol']
        timestamp = self.safe_integer_2(trade, 'ts', 'created-at')
        order = self.safe_string(trade, 'order-id')
        side = self.safe_string(trade, 'direction')
        type = self.safe_string(trade, 'type')
        if type is not None:
            typeParts = type.split('-')
            side = typeParts[0]
            type = typeParts[1]
        takerOrMaker = self.safe_string(trade, 'role')
        price = self.safe_float(trade, 'price')
        amount = self.safe_float_2(trade, 'filled-amount', 'amount')
        cost = None
        if price is not None:
            if amount is not None:
                cost = amount * price
        fee = None
        feeCost = self.safe_float(trade, 'filled-fees')
        feeCurrency = None
        if market is not None:
            feeCurrency = market['base'] if (side == 'buy') else market['quote']
        filledPoints = self.safe_float(trade, 'filled-points')
        if filledPoints is not None:
            if (feeCost is None) or (feeCost == 0.0):
                feeCost = filledPoints
                feeCurrency = self.safe_currency_code(self.safe_string(trade, 'fee-deduct-currency'))
        if feeCost is not None:
            fee = {
                'cost': feeCost,
                'currency': feeCurrency,
            }
        tradeId = self.safe_string_2(trade, 'trade-id', 'tradeId')
        id = self.safe_string(trade, 'id', tradeId)
        return {
            'id': id,
            'info': trade,
            'order': order,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'symbol': symbol,
            'type': type,
            'side': side,
            'takerOrMaker': takerOrMaker,
            'price': price,
            'amount': amount,
            'cost': cost,
            'fee': fee,
        }

    async def fetch_my_trades(self, symbol=None, since=None, limit=None, params={}):
        await self.load_markets()
        market = None
        request = {}
        if symbol is not None:
            market = self.market(symbol)
            request['symbol'] = market['id']
        if limit is not None:
            request['size'] = limit  # 1-100 orders, default is 100
        if since is not None:
            request['start-date'] = self.ymd(since)  # maximum query window size is 2 days, query window shift should be within past 120 days
        response = await self.privateGetOrderMatchresults(self.extend(request, params))
        trades = self.parse_trades(response['data'], market, since, limit)
        return trades

    async def fetch_trades(self, symbol, since=None, limit=1000, params={}):
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
        }
        if limit is not None:
            request['size'] = limit
        response = await self.marketGetHistoryTrade(self.extend(request, params))
        #
        #     {
        #         "status": "ok",
        #         "ch": "market.btcusdt.trade.detail",
        #         "ts": 1583497692365,
        #         "data": [
        #             {
        #                 "id": 105005170342,
        #                 "ts": 1583497692182,
        #                 "data": [
        #                     {
        #                         "amount": 0.010411000000000000,
        #                         "trade-id": 102090736910,
        #                         "ts": 1583497692182,
        #                         "id": 10500517034273194594947,
        #                         "price": 9096.050000000000000000,
        #                         "direction": "sell"
        #                     }
        #                 ]
        #             },
        #             # ...
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data')
        result = []
        for i in range(0, len(data)):
            trades = self.safe_value(data[i], 'data', [])
            for j in range(0, len(trades)):
                trade = self.parse_trade(trades[j], market)
                result.append(trade)
        result = self.sort_by(result, 'timestamp')
        return self.filter_by_symbol_since_limit(result, symbol, since, limit)

    def parse_ohlcv(self, ohlcv, market=None, timeframe='1m', since=None, limit=None):
        return [
            self.safe_timestamp(ohlcv, 'id'),
            self.safe_float(ohlcv, 'open'),
            self.safe_float(ohlcv, 'high'),
            self.safe_float(ohlcv, 'low'),
            self.safe_float(ohlcv, 'close'),
            self.safe_float(ohlcv, 'amount'),
        ]

    async def fetch_ohlcv(self, symbol, timeframe='1m', since=None, limit=1000, params={}):
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'symbol': market['id'],
            'period': self.timeframes[timeframe],
        }
        if limit is not None:
            request['size'] = limit
        response = await self.marketGetHistoryKline(self.extend(request, params))
        return self.parse_ohlcvs(response['data'], market, timeframe, since, limit)

    async def fetch_accounts(self, params={}):
        await self.load_markets()
        response = await self.privateGetAccountAccounts(params)
        return response['data']

    async def fetch_currencies(self, params={}):
        request = {
            'language': self.options['language'],
        }
        response = await self.publicGetSettingsCurrencys(self.extend(request, params))
        currencies = self.safe_value(response, 'data')
        result = {}
        for i in range(0, len(currencies)):
            currency = currencies[i]
            #
            #  {                    name: "ctxc",
            #              'display-name': "CTXC",
            #        'withdraw-precision':  8,
            #             'currency-type': "eth",
            #        'currency-partition': "pro",
            #             'support-sites':  null,
            #                'otc-enable':  0,
            #        'deposit-min-amount': "2",
            #       'withdraw-min-amount': "4",
            #            'show-precision': "8",
            #                      weight: "2988",
            #                     visible:  True,
            #              'deposit-desc': "Please don’t deposit any other digital assets except CTXC t…",
            #             'withdraw-desc': "Minimum withdrawal amount: 4 CTXC. not >_<not For security reason…",
            #           'deposit-enabled':  True,
            #          'withdraw-enabled':  True,
            #    'currency-addr-with-tag':  False,
            #             'fast-confirms':  15,
            #             'safe-confirms':  30                                                             }
            #
            id = self.safe_value(currency, 'name')
            precision = self.safe_integer(currency, 'withdraw-precision')
            code = self.safe_currency_code(id)
            active = currency['visible'] and currency['deposit-enabled'] and currency['withdraw-enabled']
            name = self.safe_string(currency, 'display-name')
            result[code] = {
                'id': id,
                'code': code,
                'type': 'crypto',
                # 'payin': currency['deposit-enabled'],
                # 'payout': currency['withdraw-enabled'],
                # 'transfer': None,
                'name': name,
                'active': active,
                'fee': None,  # todo need to fetch from fee endpoint
                'precision': precision,
                'limits': {
                    'amount': {
                        'min': math.pow(10, -precision),
                        'max': math.pow(10, precision),
                    },
                    'price': {
                        'min': math.pow(10, -precision),
                        'max': math.pow(10, precision),
                    },
                    'cost': {
                        'min': None,
                        'max': None,
                    },
                    'deposit': {
                        'min': self.safe_float(currency, 'deposit-min-amount'),
                        'max': math.pow(10, precision),
                    },
                    'withdraw': {
                        'min': self.safe_float(currency, 'withdraw-min-amount'),
                        'max': math.pow(10, precision),
                    },
                },
                'info': currency,
            }
        return result

    async def fetch_balance(self, params={}):
        await self.load_markets()
        await self.load_accounts()
        method = self.options['fetchBalanceMethod']
        request = {
            'id': self.accounts[0]['id'],
        }
        response = await getattr(self, method)(self.extend(request, params))
        balances = self.safe_value(response['data'], 'list', [])
        result = {'info': response}
        for i in range(0, len(balances)):
            balance = balances[i]
            currencyId = self.safe_string(balance, 'currency')
            code = self.safe_currency_code(currencyId)
            account = None
            if code in result:
                account = result[code]
            else:
                account = self.account()
            if balance['type'] == 'trade':
                account['free'] = self.safe_float(balance, 'balance')
            if balance['type'] == 'frozen':
                account['used'] = self.safe_float(balance, 'balance')
            result[code] = account
        return self.parse_balance(result)

    async def fetch_orders_by_states(self, states, symbol=None, since=None, limit=None, params={}):
        await self.load_markets()
        request = {
            'states': states,
        }
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['symbol'] = market['id']
        method = self.safe_string(self.options, 'fetchOrdersByStatesMethod', 'private_get_order_orders')
        response = await getattr(self, method)(self.extend(request, params))
        #
        #     {status:   "ok",
        #         data: [{                 id:  13997833014,
        #                                symbol: "ethbtc",
        #                          'account-id':  3398321,
        #                                amount: "0.045000000000000000",
        #                                 price: "0.034014000000000000",
        #                          'created-at':  1545836976871,
        #                                  type: "sell-limit",
        #                        'field-amount': "0.045000000000000000",
        #                   'field-cash-amount': "0.001530630000000000",
        #                          'field-fees': "0.000003061260000000",
        #                         'finished-at':  1545837948214,
        #                                source: "spot-api",
        #                                 state: "filled",
        #                         'canceled-at':  0                      }  ]}
        #
        return self.parse_orders(response['data'], market, since, limit)

    async def fetch_order(self, id, symbol=None, params={}):
        await self.load_markets()
        request = {
            'id': id,
        }
        response = await self.privateGetOrderOrdersId(self.extend(request, params))
        order = self.safe_value(response, 'data')
        return self.parse_order(order)

    async def fetch_orders(self, symbol=None, since=None, limit=None, params={}):
        return await self.fetch_orders_by_states('pre-submitted,submitted,partial-filled,filled,partial-canceled,canceled', symbol, since, limit, params)

    async def fetch_open_orders(self, symbol=None, since=None, limit=None, params={}):
        method = self.safe_string(self.options, 'fetchOpenOrdersMethod', 'fetch_open_orders_v1')
        return await getattr(self, method)(symbol, since, limit, params)

    async def fetch_open_orders_v1(self, symbol=None, since=None, limit=None, params={}):
        if symbol is None:
            raise ArgumentsRequired(self.id + ' fetchOpenOrdersV1 requires a symbol argument')
        return await self.fetch_orders_by_states('pre-submitted,submitted,partial-filled', symbol, since, limit, params)

    async def fetch_closed_orders(self, symbol=None, since=None, limit=None, params={}):
        return await self.fetch_orders_by_states('filled,partial-canceled,canceled', symbol, since, limit, params)

    async def fetch_open_orders_v2(self, symbol=None, since=None, limit=None, params={}):
        await self.load_markets()
        if symbol is None:
            raise ArgumentsRequired(self.id + ' fetchOpenOrders requires a symbol argument')
        market = self.market(symbol)
        accountId = self.safe_string(params, 'account-id')
        if accountId is None:
            # pick the first account
            await self.load_accounts()
            for i in range(0, len(self.accounts)):
                account = self.accounts[i]
                if account['type'] == 'spot':
                    accountId = self.safe_string(account, 'id')
                    if accountId is not None:
                        break
        request = {
            'symbol': market['id'],
            'account-id': accountId,
        }
        if limit is not None:
            request['size'] = limit
        omitted = self.omit(params, 'account-id')
        response = await self.privateGetOrderOpenOrders(self.extend(request, omitted))
        #
        #     {
        #         "status":"ok",
        #         "data":[
        #             {
        #                 "symbol":"ethusdt",
        #                 "source":"api",
        #                 "amount":"0.010000000000000000",
        #                 "account-id":1528640,
        #                 "created-at":1561597491963,
        #                 "price":"400.000000000000000000",
        #                 "filled-amount":"0.0",
        #                 "filled-cash-amount":"0.0",
        #                 "filled-fees":"0.0",
        #                 "id":38477101630,
        #                 "state":"submitted",
        #                 "type":"sell-limit"
        #             }
        #         ]
        #     }
        #
        data = self.safe_value(response, 'data', [])
        return self.parse_orders(data, market, since, limit)

    def parse_order_status(self, status):
        statuses = {
            'partial-filled': 'open',
            'partial-canceled': 'canceled',
            'filled': 'closed',
            'canceled': 'canceled',
            'submitted': 'open',
        }
        return self.safe_string(statuses, status, status)

    def parse_order(self, order, market=None):
        #
        #     {                 id:  13997833014,
        #                    symbol: "ethbtc",
        #              'account-id':  3398321,
        #                    amount: "0.045000000000000000",
        #                     price: "0.034014000000000000",
        #              'created-at':  1545836976871,
        #                      type: "sell-limit",
        #            'field-amount': "0.045000000000000000",  # they have fixed it for filled-amount
        #       'field-cash-amount': "0.001530630000000000",  # they have fixed it for filled-cash-amount
        #              'field-fees': "0.000003061260000000",  # they have fixed it for filled-fees
        #             'finished-at':  1545837948214,
        #                    source: "spot-api",
        #                     state: "filled",
        #             'canceled-at':  0                      }
        #
        #     {                 id:  20395337822,
        #                    symbol: "ethbtc",
        #              'account-id':  5685075,
        #                    amount: "0.001000000000000000",
        #                     price: "0.0",
        #              'created-at':  1545831584023,
        #                      type: "buy-market",
        #            'field-amount': "0.029100000000000000",  # they have fixed it for filled-amount
        #       'field-cash-amount': "0.000999788700000000",  # they have fixed it for filled-cash-amount
        #              'field-fees': "0.000058200000000000",  # they have fixed it for filled-fees
        #             'finished-at':  1545831584181,
        #                    source: "spot-api",
        #                     state: "filled",
        #             'canceled-at':  0                      }
        #
        id = self.safe_string(order, 'id')
        side = None
        type = None
        status = None
        if 'type' in order:
            orderType = order['type'].split('-')
            side = orderType[0]
            type = orderType[1]
            status = self.parse_order_status(self.safe_string(order, 'state'))
        symbol = None
        if market is None:
            if 'symbol' in order:
                if order['symbol'] in self.markets_by_id:
                    marketId = order['symbol']
                    market = self.markets_by_id[marketId]
        if market is not None:
            symbol = market['symbol']
        timestamp = self.safe_integer(order, 'created-at')
        amount = self.safe_float(order, 'amount')
        filled = self.safe_float_2(order, 'filled-amount', 'field-amount')  # typo in their API, filled amount
        if (type == 'market') and (side == 'buy'):
            amount = filled if (status == 'closed') else None
        price = self.safe_float(order, 'price')
        if price == 0.0:
            price = None
        cost = self.safe_float_2(order, 'filled-cash-amount', 'field-cash-amount')  # same typo
        remaining = None
        average = None
        if filled is not None:
            if amount is not None:
                remaining = amount - filled
            # if cost is defined and filled is not zero
            if (cost is not None) and (filled > 0):
                average = cost / filled
        feeCost = self.safe_float_2(order, 'filled-fees', 'field-fees')  # typo in their API, filled fees
        fee = None
        if feeCost is not None:
            feeCurrency = None
            if market is not None:
                feeCurrency = market['quote'] if (side == 'sell') else market['base']
            fee = {
                'cost': feeCost,
                'currency': feeCurrency,
            }
        return {
            'info': order,
            'id': id,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'lastTradeTimestamp': None,
            'symbol': symbol,
            'type': type,
            'side': side,
            'price': price,
            'average': average,
            'cost': cost,
            'amount': amount,
            'filled': filled,
            'remaining': remaining,
            'status': status,
            'fee': fee,
        }

    async def create_order(self, symbol, type, side, amount, price=None, params={}):
        await self.load_markets()
        await self.load_accounts()
        market = self.market(symbol)
        request = {
            'account-id': self.accounts[0]['id'],
            'symbol': market['id'],
            'type': side + '-' + type,
        }
        if (type == 'market') and (side == 'buy'):
            if self.options['createMarketBuyOrderRequiresPrice']:
                if price is None:
                    raise InvalidOrder(self.id + " market buy order requires price argument to calculate cost(total amount of quote currency to spend for buying, amount * price). To switch off self warning exception and specify cost in the amount argument, set .options['createMarketBuyOrderRequiresPrice'] = False. Make sure you know what you're doing.")
                else:
                    # despite that cost = amount * price is in quote currency and should have quote precision
                    # the exchange API requires the cost supplied in 'amount' to be of base precision
                    # more about it here: https://github.com/ccxt/ccxt/pull/4395
                    # we use priceToPrecision instead of amountToPrecision here
                    # because in self case the amount is in the quote currency
                    request['amount'] = self.cost_to_precision(symbol, float(amount) * float(price))
            else:
                request['amount'] = self.cost_to_precision(symbol, amount)
        else:
            request['amount'] = self.amount_to_precision(symbol, amount)
        if type == 'limit' or type == 'ioc' or type == 'limit-maker':
            request['price'] = self.price_to_precision(symbol, price)
        method = self.options['createOrderMethod']
        response = await getattr(self, method)(self.extend(request, params))
        timestamp = self.milliseconds()
        id = self.safe_string(response, 'data')
        return {
            'info': response,
            'id': id,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'lastTradeTimestamp': None,
            'status': None,
            'symbol': symbol,
            'type': type,
            'side': side,
            'price': price,
            'amount': amount,
            'filled': None,
            'remaining': None,
            'cost': None,
            'trades': None,
            'fee': None,
        }

    async def cancel_order(self, id, symbol=None, params={}):
        response = await self.privatePostOrderOrdersIdSubmitcancel({'id': id})
        #
        #     response = {
        #         'status': 'ok',
        #         'data': '10138899000',
        #     }
        #
        return self.extend(self.parse_order(response), {
            'id': id,
            'status': 'canceled',
        })

    def currency_to_precision(self, currency, fee):
        return self.decimal_to_precision(fee, 0, self.currencies[currency]['precision'])

    def calculate_fee(self, symbol, type, side, amount, price, takerOrMaker='taker', params={}):
        market = self.markets[symbol]
        rate = market[takerOrMaker]
        cost = amount * rate
        key = 'quote'
        if side == 'sell':
            cost *= price
        else:
            key = 'base'
        return {
            'type': takerOrMaker,
            'currency': market[key],
            'rate': rate,
            'cost': float(self.currency_to_precision(market[key], cost)),
        }

    async def withdraw(self, code, amount, address, tag=None, params={}):
        await self.load_markets()
        self.check_address(address)
        currency = self.currency(code)
        request = {
            'address': address,  # only supports existing addresses in your withdraw address list
            'amount': amount,
            'currency': currency['id'].lower(),
        }
        if tag is not None:
            request['addr-tag'] = tag  # only for XRP?
        response = await self.privatePostDwWithdrawApiCreate(self.extend(request, params))
        id = self.safe_string(response, 'data')
        return {
            'info': response,
            'id': id,
        }

    def sign(self, path, api='public', method='GET', params={}, headers=None, body=None):
        url = '/'
        if api == 'market':
            url += api
        elif (api == 'public') or (api == 'private'):
            url += self.version
        url += '/' + self.implode_params(path, params)
        query = self.omit(params, self.extract_params(path))
        if api == 'private':
            self.check_required_credentials()
            timestamp = self.ymdhms(self.milliseconds(), 'T')
            request = {
                'SignatureMethod': 'HmacSHA256',
                'SignatureVersion': '2',
                'AccessKeyId': self.apiKey,
                'Timestamp': timestamp,
            }
            if method != 'POST':
                request = self.extend(request, query)
            request = self.keysort(request)
            auth = self.urlencode(request)
            # unfortunately, PHP demands double quotes for the escaped newline symbol
            # eslint-disable-next-line quotes
            payload = "\n".join([method, self.hostname, url, auth])
            signature = self.hmac(self.encode(payload), self.encode(self.secret), hashlib.sha256, 'base64')
            auth += '&' + self.urlencode({'Signature': signature})
            url += '?' + auth
            if method == 'POST':
                body = self.json(query)
                headers = {
                    'Content-Type': 'application/json',
                }
            else:
                headers = {
                    'Content-Type': 'application/x-www-form-urlencoded',
                }
        else:
            if params:
                url += '?' + self.urlencode(params)
        url = self.implode_params(self.urls['api'][api], {
            'hostname': self.hostname,
        }) + url
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def handle_errors(self, httpCode, reason, url, method, headers, body, response, requestHeaders, requestBody):
        if response is None:
            return  # fallback to default error handler
        if 'status' in response:
            #
            #     {"status":"error","err-code":"order-limitorder-amount-min-error","err-msg":"limit order amount error, min: `0.001`","data":null}
            #
            status = self.safe_string(response, 'status')
            if status == 'error':
                code = self.safe_string(response, 'err-code')
                feedback = self.id + ' ' + body
                self.throw_exactly_matched_exception(self.exceptions['exact'], code, feedback)
                message = self.safe_string(response, 'err-msg')
                self.throw_exactly_matched_exception(self.exceptions['exact'], message, feedback)
                raise ExchangeError(feedback)

    async def fetch_deposits(self, code=None, since=None, limit=None, params={}):
        if limit is None or limit > 100:
            limit = 100
        await self.load_markets()
        currency = None
        if code is not None:
            currency = self.currency(code)
        request = {
            'type': 'deposit',
            'from': 0,  # From 'id' ... if you want to get results after a particular transaction id, pass the id in params.from
        }
        if currency is not None:
            request['currency'] = currency['id']
        if limit is not None:
            request['size'] = limit  # max 100
        response = await self.privateGetQueryDepositWithdraw(self.extend(request, params))
        # return response
        return self.parse_transactions(response['data'], currency, since, limit)

    async def fetch_withdrawals(self, code=None, since=None, limit=None, params={}):
        if limit is None or limit > 100:
            limit = 100
        await self.load_markets()
        currency = None
        if code is not None:
            currency = self.currency(code)
        request = {
            'type': 'withdraw',
            'from': 0,  # From 'id' ... if you want to get results after a particular transaction id, pass the id in params.from
        }
        if currency is not None:
            request['currency'] = currency['id']
        if limit is not None:
            request['size'] = limit  # max 100
        response = await self.privateGetQueryDepositWithdraw(self.extend(request, params))
        # return response
        return self.parse_transactions(response['data'], currency, since, limit)

    def parse_transaction(self, transaction, currency=None):
        #
        # fetchDeposits
        #
        #     {
        #         'id': 8211029,
        #         'type': 'deposit',
        #         'currency': 'eth',
        #         'chain': 'eth',
        #         'tx-hash': 'bd315....',
        #         'amount': 0.81162421,
        #         'address': '4b8b....',
        #         'address-tag': '',
        #         'fee': 0,
        #         'state': 'safe',
        #         'created-at': 1542180380965,
        #         'updated-at': 1542180788077
        #     }
        #
        # fetchWithdrawals
        #
        #     {
        #         'id': 6908275,
        #         'type': 'withdraw',
        #         'currency': 'btc',
        #         'chain': 'btc',
        #         'tx-hash': 'c1a1a....',
        #         'amount': 0.80257005,
        #         'address': '1QR....',
        #         'address-tag': '',
        #         'fee': 0.0005,
        #         'state': 'confirmed',
        #         'created-at': 1552107295685,
        #         'updated-at': 1552108032859
        #     }
        #
        timestamp = self.safe_integer(transaction, 'created-at')
        updated = self.safe_integer(transaction, 'updated-at')
        code = self.safe_currency_code(self.safe_string(transaction, 'currency'))
        type = self.safe_string(transaction, 'type')
        if type == 'withdraw':
            type = 'withdrawal'
        status = self.parse_transaction_status(self.safe_string(transaction, 'state'))
        tag = self.safe_string(transaction, 'address-tag')
        feeCost = self.safe_float(transaction, 'fee')
        if feeCost is not None:
            feeCost = abs(feeCost)
        return {
            'info': transaction,
            'id': self.safe_string(transaction, 'id'),
            'txid': self.safe_string(transaction, 'tx-hash'),
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'address': self.safe_string(transaction, 'address'),
            'tag': tag,
            'type': type,
            'amount': self.safe_float(transaction, 'amount'),
            'currency': code,
            'status': status,
            'updated': updated,
            'fee': {
                'currency': code,
                'cost': feeCost,
                'rate': None,
            },
        }

    def parse_transaction_status(self, status):
        statuses = {
            # deposit statuses
            'unknown': 'failed',
            'confirming': 'pending',
            'confirmed': 'ok',
            'safe': 'ok',
            'orphan': 'failed',
            # withdrawal statuses
            'submitted': 'pending',
            'canceled': 'canceled',
            'reexamine': 'pending',
            'reject': 'failed',
            'pass': 'pending',
            'wallet-reject': 'failed',
            # 'confirmed': 'ok',  # present in deposit statuses
            'confirm-error': 'failed',
            'repealed': 'failed',
            'wallet-transfer': 'pending',
            'pre-transfer': 'pending',
        }
        return self.safe_string(statuses, status, status)
