import { IResource, Resource } from '@aws-cdk/core';
import * as constructs from 'constructs';
import { INotificationRuleSource } from './notification-rule-source';
import { INotificationRuleTarget } from './notification-rule-target';
/**
 * The level of detail to include in the notifications for this resource.
 *
 * @stability stable
 */
export declare enum DetailType {
    /**
     * BASIC will include only the contents of the event as it would appear in AWS CloudWatch.
     *
     * @stability stable
     */
    BASIC = "BASIC",
    /**
     * FULL will include any supplemental information provided by AWS CodeStar Notifications and/or the service for the resource for which the notification is created.
     *
     * @stability stable
     */
    FULL = "FULL"
}
/**
 * Standard set of options for `notifyOnXxx` codestar notification handler on construct.
 *
 * @stability stable
 */
export interface NotificationRuleOptions {
    /**
     * The name for the notification rule.
     *
     * Notification rule names must be unique in your AWS account.
     *
     * @default - generated from the `id`
     * @stability stable
     */
    readonly notificationRuleName?: string;
    /**
     * The status of the notification rule.
     *
     * If the enabled is set to DISABLED, notifications aren't sent for the notification rule.
     *
     * @default true
     * @stability stable
     */
    readonly enabled?: boolean;
    /**
     * The level of detail to include in the notifications for this resource.
     *
     * BASIC will include only the contents of the event as it would appear in AWS CloudWatch.
     * FULL will include any supplemental information provided by AWS CodeStar Notifications and/or the service for the resource for which the notification is created.
     *
     * @default DetailType.FULL
     * @stability stable
     */
    readonly detailType?: DetailType;
}
/**
 * Properties for a new notification rule.
 *
 * @stability stable
 */
export interface NotificationRuleProps extends NotificationRuleOptions {
    /**
     * A list of event types associated with this notification rule.
     *
     * For a complete list of event types and IDs, see Notification concepts in the Developer Tools Console User Guide.
     *
     * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#concepts-api
     * @stability stable
     */
    readonly events: string[];
    /**
     * The Amazon Resource Name (ARN) of the resource to associate with the notification rule.
     *
     * Currently, Supported sources include pipelines in AWS CodePipeline and build projects in AWS CodeBuild in this L2 constructor.
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-codestarnotifications-notificationrule.html#cfn-codestarnotifications-notificationrule-resource
     * @stability stable
     */
    readonly source: INotificationRuleSource;
    /**
     * The targets to register for the notification destination.
     *
     * @default - No targets are added to the rule. Use `addTarget()` to add a target.
     * @stability stable
     */
    readonly targets?: INotificationRuleTarget[];
}
/**
 * Represents a notification rule.
 *
 * @stability stable
 */
export interface INotificationRule extends IResource {
    /**
     * The ARN of the notification rule (i.e. arn:aws:codestar-notifications:::notificationrule/01234abcde).
     *
     * @stability stable
     * @attribute true
     */
    readonly notificationRuleArn: string;
    /**
     * Adds target to notification rule.
     *
     * @param target The SNS topic or AWS Chatbot Slack target.
     * @returns boolean - return true if it had any effect
     * @stability stable
     */
    addTarget(target: INotificationRuleTarget): boolean;
}
/**
 * A new notification rule.
 *
 * @stability stable
 * @resource AWS::CodeStarNotifications::NotificationRule
 */
export declare class NotificationRule extends Resource implements INotificationRule {
    /**
     * Import an existing notification rule provided an ARN.
     *
     * @param scope The parent creating construct.
     * @param id The construct's name.
     * @param notificationRuleArn Notification rule ARN (i.e. arn:aws:codestar-notifications:::notificationrule/01234abcde).
     * @stability stable
     */
    static fromNotificationRuleArn(scope: constructs.Construct, id: string, notificationRuleArn: string): INotificationRule;
    /**
     * The ARN of the notification rule (i.e. arn:aws:codestar-notifications:::notificationrule/01234abcde).
     *
     * @stability stable
     * @attribute true
     */
    readonly notificationRuleArn: string;
    private readonly targets;
    private readonly events;
    /**
     * @stability stable
     */
    constructor(scope: constructs.Construct, id: string, props: NotificationRuleProps);
    /**
     * Adds target to notification rule.
     *
     * @param target The SNS topic or AWS Chatbot Slack target.
     * @stability stable
     */
    addTarget(target: INotificationRuleTarget): boolean;
    /**
     * Adds events to notification rule
     *
     * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-pipeline
     * @see https://docs.aws.amazon.com/dtconsole/latest/userguide/concepts.html#events-ref-buildproject
     * @param events The list of event types for AWS Codebuild and AWS CodePipeline
     */
    private addEvents;
}
