# pyimpspec is licensed under the GPLv3 or later (https://www.gnu.org/licenses/gpl-3.0.html).
# Copyright 2023 pyimpspec developers
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# The licenses of pyimpspec's dependencies and/or sources of portions of code are included in
# the LICENSES folder.

from typing import Optional
from numpy import (
    array,
    logspace,
    ndarray,
)
from pyimpspec.data import DataSet
from .base import Connection
from .circuit import Circuit
from .parser import Parser
from pyimpspec.typing import (
    ComplexImpedances,
    Frequencies,
    Frequency,
)
from .diagrams import (
    to_circuitikz as _to_circuitikz,
    to_drawing as _to_drawing,
)


# Replace the to_circuitikz and to_drawing methods of the Circuit and Connection classes
Circuit.to_drawing = _to_drawing
Circuit.to_circuitikz = _to_circuitikz
Connection.to_drawing = _to_drawing
Connection.to_circuitikz = _to_circuitikz


def parse_cdc(cdc: str) -> Circuit:
    """
    Generate a Circuit instance from a string that contains a circuit description code (CDC).

    Parameters
    ----------
    cdc: str
        A circuit description code (CDC) corresponding to an equivalent circuit.

    Returns
    -------
    Circuit
    """
    assert isinstance(cdc, str), cdc
    return Parser().process(cdc)


def simulate_spectrum(
    circuit: Circuit,
    frequencies: Optional[Frequencies] = None,
    label: str = "",
) -> DataSet:
    """
    Simulate the impedance spectrum generated by a circuit in a certain frequency range.

    Parameters
    ----------
    circuit: Circuit
        The circuit to use when calculating impedances at various frequencies.

    frequencies: Optional[Frequencies], optional
        Excitation frequencies in hertz.
        If no frequencies are provided, then a frequency range of 10 mHz to 100 kHz with 10 points per decade will be used.

    label: str, optional
        The label for the DataSet that is returned.

    Returns
    -------
    DataSet
    """
    assert isinstance(circuit, Circuit), circuit
    assert isinstance(label, str), label
    if frequencies is None or len(frequencies) == 0:
        frequencies = logspace(5, -2, 71)
    elif not isinstance(frequencies, ndarray):
        frequencies = array(frequencies, dtype=Frequency)
    Z: ComplexImpedances = circuit.get_impedances(frequencies)
    return DataSet(frequencies=frequencies, impedances=Z, label=label)
