"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ToolkitCleaner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_ecr_assets_1 = require("aws-cdk-lib/aws-ecr-assets");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_s3_assets_1 = require("aws-cdk-lib/aws-s3-assets");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
const constructs_1 = require("constructs");
const clean_images_function_1 = require("./clean-images-function");
const clean_objects_function_1 = require("./clean-objects-function");
const get_stack_names_function_1 = require("./get-stack-names-function");
/**
 * Clean unused S3 and ECR assets from your CDK Toolkit.
 *
 * @stability stable
 */
class ToolkitCleaner extends constructs_1.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props = {}) {
        var _b, _c;
        super(scope, id);
        // Dummy assets to reference S3 bucket and ECR repository
        const fileAsset = new aws_s3_assets_1.Asset(this, 'FileAsset', {
            path: path.join(__dirname, '..', '..', 'assets', 'toolkit-cleaner', 'docker', 'dummy.txt'),
        });
        const dockerImageAsset = new aws_ecr_assets_1.DockerImageAsset(this, 'DockerImageAsset', {
            directory: path.join(__dirname, '..', '..', 'assets', 'toolkit-cleaner', 'docker'),
        });
        const getStackNamesFunction = new get_stack_names_function_1.GetStackNamesFunction(this, 'GetStackNamesFunction');
        getStackNamesFunction.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ['cloudformation:DescribeStacks'],
            resources: ['*'],
        }));
        const getStackNames = new tasks.LambdaInvoke(this, 'GetStackNames', {
            lambdaFunction: getStackNamesFunction,
            payloadResponseOnly: true,
        });
        const stacksMap = new sfn.Map(this, 'StacksMap', {
            maxConcurrency: 1,
            resultSelector: {
                AssetHashes: sfn.JsonPath.stringAt('$'),
            },
        });
        const getTemplate = new tasks.CallAwsService(this, 'GetTemplate', {
            service: 'cloudformation',
            action: 'getTemplate',
            parameters: {
                StackName: sfn.JsonPath.stringAt('$'),
            },
            iamResources: ['*'],
        }).addRetry({
            errors: ['CloudFormation.CloudFormationException'],
        });
        const extractHashes = new tasks.EvaluateExpression(this, 'ExtractHashes', {
            expression: '[...new Set(($.TemplateBody).match(/[a-f0-9]{64}/g))]',
        });
        const flattenHashes = new tasks.EvaluateExpression(this, 'FlattenHashes', {
            expression: '[...new Set(($.AssetHashes).flat())]',
        });
        const cleanObjectsFunction = new clean_objects_function_1.CleanObjectsFunction(this, 'CleanObjectsFunction', {
            timeout: aws_cdk_lib_1.Duration.minutes(5),
        });
        cleanObjectsFunction.addEnvironment('BUCKET_NAME', fileAsset.bucket.bucketName);
        fileAsset.bucket.grantRead(cleanObjectsFunction);
        fileAsset.bucket.grantDelete(cleanObjectsFunction);
        const cleanObjects = new tasks.LambdaInvoke(this, 'CleanObjects', {
            lambdaFunction: cleanObjectsFunction,
            payloadResponseOnly: true,
        });
        const cleanImagesFunction = new clean_images_function_1.CleanImagesFunction(this, 'CleanImagesFunction', {
            timeout: aws_cdk_lib_1.Duration.minutes(5),
        });
        cleanImagesFunction.addEnvironment('REPOSITORY_NAME', dockerImageAsset.repository.repositoryName);
        dockerImageAsset.repository.grant(cleanImagesFunction, 'ecr:DescribeImages', 'ecr:BatchDeleteImage');
        const cleanImages = new tasks.LambdaInvoke(this, 'CleanImages', {
            lambdaFunction: cleanImagesFunction,
            payloadResponseOnly: true,
        });
        if (!props.dryRun) {
            cleanObjectsFunction.addEnvironment('RUN', 'true');
            cleanImagesFunction.addEnvironment('RUN', 'true');
        }
        if (props.retainAssetsNewerThan) {
            const retainMilliseconds = props.retainAssetsNewerThan.toMilliseconds().toString();
            cleanObjectsFunction.addEnvironment('RETAIN_MILLISECONDS', retainMilliseconds);
            cleanImagesFunction.addEnvironment('RETAIN_MILLISECONDS', retainMilliseconds);
        }
        const sumReclaimed = new tasks.EvaluateExpression(this, 'SumReclaimed', {
            expression: '({ Deleted: $[0].Deleted + $[1].Deleted, Reclaimed: $[0].Reclaimed + $[1].Reclaimed })',
        });
        const stateMachine = new sfn.StateMachine(this, 'Resource', {
            definition: getStackNames
                .next(stacksMap.iterator(getTemplate.next(extractHashes)))
                .next(flattenHashes)
                .next(new sfn.Parallel(this, 'Clean')
                .branch(cleanObjects)
                .branch(cleanImages))
                .next(sumReclaimed),
        });
        const rule = new aws_events_1.Rule(this, 'Rule', {
            enabled: (_b = props.scheduleEnabled) !== null && _b !== void 0 ? _b : true,
            schedule: (_c = props.schedule) !== null && _c !== void 0 ? _c : aws_events_1.Schedule.rate(aws_cdk_lib_1.Duration.days(1)),
        });
        rule.addTarget(new aws_events_targets_1.SfnStateMachine(stateMachine));
    }
}
exports.ToolkitCleaner = ToolkitCleaner;
_a = JSII_RTTI_SYMBOL_1;
ToolkitCleaner[_a] = { fqn: "cloudstructs.ToolkitCleaner", version: "0.4.16" };
//# sourceMappingURL=data:application/json;base64,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