import * as ec2 from 'aws-cdk-lib/aws-ec2';
import * as ecs from 'aws-cdk-lib/aws-ecs';
import * as events from 'aws-cdk-lib/aws-events';
import { Construct } from 'constructs';
/**
 * Properties for a CodeCommitMirror.
 *
 * @stability stable
 */
export interface CodeCommitMirrorProps {
    /**
     * The source repository.
     *
     * @stability stable
     */
    readonly repository: CodeCommitMirrorSourceRepository;
    /**
     * The ECS cluster where to run the mirroring operation.
     *
     * @stability stable
     */
    readonly cluster: ecs.ICluster;
    /**
     * The schedule for the mirroring operation.
     *
     * @default - everyday at midnight
     * @stability stable
     */
    readonly schedule?: events.Schedule;
    /**
     * Where to run the mirroring Fargate tasks.
     *
     * @default - public subnets
     * @stability stable
     */
    readonly subnetSelection?: ec2.SubnetSelection;
}
/**
 * A source repository for AWS CodeCommit mirroring.
 *
 * @stability stable
 */
export declare abstract class CodeCommitMirrorSourceRepository {
    /**
     * Public GitHub repository.
     *
     * @stability stable
     */
    static gitHub(owner: string, name: string): CodeCommitMirrorSourceRepository;
    /**
     * Private repository with HTTPS clone URL stored in a AWS Secrets Manager secret or a AWS Systems Manager secure string parameter.
     *
     * @param name the repository name.
     * @param url the secret containing the HTTPS clone URL.
     * @stability stable
     */
    static private(name: string, url: ecs.Secret): CodeCommitMirrorSourceRepository;
    /**
     * The name of the repository.
     *
     * @stability stable
     */
    abstract readonly name: string;
    /**
     * The HTTPS clone URL in plain text, used for a public repository.
     *
     * @stability stable
     */
    abstract readonly plainTextUrl?: string;
    /**
     * The HTTPS clone URL if the repository is private.
     *
     * The secret should contain the username and/or token.
     *
     * @stability stable
     * @example
     *
     * `https://TOKEN@github.com/owner/name`
     * `https://USERNAME:TOKEN@bitbucket.org/owner/name.git`
     */
    abstract readonly secretUrl?: ecs.Secret;
}
/**
 * Mirror a repository to AWS CodeCommit on schedule.
 *
 * @stability stable
 */
export declare class CodeCommitMirror extends Construct {
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props: CodeCommitMirrorProps);
}
