# -*- coding: utf-8 -*-
from setuptools import setup

modules = \
['cmdy']
install_requires = \
['modkit', 'python-simpleconf']

setup_kwargs = {
    'name': 'cmdy',
    'version': '0.2.1',
    'description': 'A handy package to run command from python',
    'long_description': '# cmdy\nA handy package to run command from python\n\n[![pypi][1]][2] [![tag][3]][4] [![travis][5]][6] [![codacy quality][7]][8] [![codacy quality][9]][8] ![pyver][10]\n\n## Installation\n```shell\n# latest version\npip install git+https://github.com/pwwang/cmdy\n# released version\npip install cmdy\n```\n\n## Why another one beyond `sh`?\n- `oncotator` not running with `sh`, no matter what I tried.\n- Unable to replace arguments with baked command, see example below:\n  ```python\n  from sh import ls\n  ls = ls.bake(l = True)\n  print(ls()) # ls -l\n  # but now I somehow want to run `ls` (without `-l`) command with `ls()`\n  ls(l = False) # not work\n  ```\n- Unable to save configurations for commands, since commands have their solid preferences.\n- Unable to specify full path of an executable.\n- No pipe/redirection notations.\n\n## Basic usage\n```python\nfrom cmdy import ls\nprint(ls())\n```\n### With arguments\nLike `sh`, `cmdy` can have non-keyword arguments, but keyword arguments preferred.\n```python\nfrom cmdy import tar\ntar("cvf", "/tmp/test.tar", "/my/home/directory/")\n```\n\n### With keyword arguments\n```python\ncurl("http://duckduckgo.com/", o="page.html", silent=True)\n# curl http://duckduckgo.com/ -o page.html --silent\n```\n__Order keyword arguments:__\n```python\ncurl("http://duckduckgo.com/", "-o", "page.html", "--silent")\n# or\nfrom collections import OrderedDict\ncurl("http://duckduckgo.com/", OrderedDict([(\'o\', \'page.html\'), (\'silent\', True)]))\n```\n\n__Prefix and separator for keyword arguments:__\n```python\nfrom cmdy import bedtools, bcftools, ls\nbedtools.intersect(wa = True, wb = True, a = \'query.bed\', b = [\'d1.bed\', \'d2.bed\', \'d3.bed\'], names = [\'d1\', \'d2\', \'d3\'], sorted = True, _prefix = \'-\')\n#bedtools intersect -wa -wb -a query.bed -b d1.bed d2.bed d3.bed -names d1 d2 d3 -sorted\n\nbcftools.query(_ = [\'a.vcf\', \'b.vcf\'], H = True, format = \'%CHROM\\t%POS\\t%REF\\t%ALT\\n\')\n# bcftools query --format \'%CHROM\\t%POS\\t%REF\\t%ALT\\n\' -H a.vcf b.vcf\n# _prefix defaults to \'auto\' (\'-\' for short options, \'--\' for long)\n# You may also define arbitrary prefix:\n# command(a = 1, bc = 2, _prefix = \'---\')\n# command ---a 1 ---bc 2\n\nls(l = True, block_size = \'KB\', _sep = \'auto\')\n# ls -l --block-size=KB\n# _sep defaults to \' \'. \'auto\' means \' \' for short options, \'=\' for long\n```\n\n__Different combinations of prefices and separators in one command:__\n```python\nfrom cmdy import java\n# Note this is just an example for old verion picard. Picard is changing it\'s style\njava({\'jar\': \'picard.jar\', \'_prefix\': \'-\', \'_sep\': \' \'}, \'SortSam\', I = \'input.bam\', O = \'sorted.bam\', SORTED_ORDER = \'coordinate\', _raw = True, _prefix = \'\', _sep = \'=\')\n# java -jar picard.jar SortSam I=input.bam O=sorted.bam SORT_ORDER=coordinate\n# The first dictionary usees _prefix and _sep in itself if specified, otherwise uses the ones specified with kwargs.\n# _raw = True keeps SORTED_ORDER as it is, otherwise, it\'ll be transformed into SORTED-ORDER\n# This is useful when some command has option like \'--block-size\', but you can still use \'block_size\' as keyword argument.\n```\n\n__Duplicated keys for list arguments:__\n```python\nfrom cmdy import sort\nsort(k = [\'1,1\', \'2,2n\'], _ = \'a.bed\', _dupkey = True)\n# sort -k 1,1 -k 2,2n a.bed\n```\n\n## Return codes and exceptions\n```python\nfrom cmdy import x\nx()\n```\n\n```shell\n    ... ...\n    raise CmdyReturnCodeException(self)\n          │                       └\n          └ <class \'cmdy.CmdyReturnCodeException\'>\ncmdy.CmdyReturnCodeException: Unexpected RETURN CODE 127, expecting: [0]\n\n  [PID]    38254\n\n  [CMD]    x\n\n  [STDERR] /bin/sh: x: command not found\n\n```\nYou can use try/catch to catch it:\n```python\nfrom cmdy import x, CmdyReturnCodeException\ntry:\n  x()\nexcept CmdyReturnCodeException as ex\n  print(\'Command returned %s\' % ex.cmdy.rc)\n```\n\nYou can allow multiple return codes by: `x(_okcode = [0, 127])` or `x(_okcode = \'0,127\')`\n\n## Redirection\n```python\nfrom cmdy import ifconfig\nifconfig(_out="/tmp/interfaces")\n# or you can use shell redirection notation\nifconfig(_out = ">") > "/tmp/interfaces"\n# or\n# ifconfig(_out = "/tmp/interfaces")\n\n## append\nifconfig(_out = ">") >> "/tmp/interfaces"\n# or\n# ifconfig(_out_ = "/tmp/interfaces")\n\n# redirect stderr\nifconfig(_err = ">") > "/tmp/ifconfig.errors"\n# or ifconfig(_err = "/tmp/ifconfig.errors")\n```\n\n## Iteration on output\n```python\nfrom cmdy import tail\nfor line in tail(_ = \'test.txt\', _iter = True):\n  print(line)\n```\n\n## Background processes\nFor command you want to run it in non-block mode, you probably would like to use `_bg` keyword:\n```python\nfor line in tail(_ = \'test.txt\', _bg = True, _iter = True):\n  print(line)\n```\n\n```python\n# blocks\nsleep(3)\nprint("...3 seconds later")\n\n# doesn\'t block\np = sleep(3, _bg=True)\nprint("prints immediately!")\np.wait()\nprint("...and 3 seconds later")\n```\n\nCallbacks for background processes:\n```python\nimport time\nfrom cmdy import sleep\ndef callback(cmdy):\n  print(cmdy.rc)\np = sleep(3, _bg = callback)\np.wait()\n# prints 0\n```\n\n## Baking\n```python\nfrom cmdy import java\npicard = java.bake(dict(jar = \'picard.jar\', _sep = \' \', _prefix = \'-\'))\n#picard.SortSam(...)\n```\n\nUnlike `sh`, `cmdy` holds the keyword arguments, and updates them while baked into a new command. This enables it to replace some arguments with the new baked command.\n```python\nfrom cmdy import ls\n\nls = ls.bake(l = True)\n# or ls = ls(l = True, _bake = True)\nls() # ls -l\n\n# I don\'t want -l anymore\nls(l = False)\n```\n\nNote that non-keyword arguments is not updatable.\n```python\nls = ls.bake(\'-l\')\nls() # ls -l\n\n# Not work, still ls -l\nls(l = False)\n```\n\nBake the whole module:\n```python\nimport cmdy\ncmdy = cmdy(_fg = True)\n# all commands under new module is running at foreground (stdout = sys.stdout, stderr = stderr)\nfrom cmdy import ls\nls()\n```\n\n## Piping\n```python\n# get the permission column\nls(l = True, _pipe = True) | cut(f = 1, _fg = True)\n```\n\n## Sub-commands\n```python\nfrom cmdy import git\nprint(git.branch(v = True))\n```\n\n## Aliasing/Specifying full path of executables for commands\n```python\nfrom cmdy import fc_cache, python\nfc_cache(_exe = \'fc-cache\', vf = \'~/.fonts/\', _prefix = \'-\')\n# fc-cache -vf ~/.fonts/\n\npython(_exe = \'/home/user/miniconda3/bin/python3\', version = True)\n# /home/user/miniconda3/bin/python3 --version\n```\nAlways alias `fc_cache` to `fc-cache` and point `python` to `/home/user/miniconda3/bin/python3`, add the following to your `~/.cmdy.ini`\n```ini\n[fc_cache]\n_exe = fc-cache\n\n[python]\n_exe = /home/user/miniconda3/bin/python3\n```\n\nThen you don\'t need to care about the paths any more:\n```python\nfrom cmdy import fc_cache, python\nfc_cache(vf = \'~/.fonts/\', _prefix = \'-\')\n# fc-cache -vf ~/.fonts/\n\npython(version = True)\n# /home/user/miniconda3/bin/python3 --version\n```\n\n## Configuration\n`cmdy` will first load default arguments:\n```python\n{\n  \'_okcode\'  : 0,\n  \'_exe\'     : None,\n  \'_sep\'     : \' \',\n  \'_prefix\'  : \'auto\',\n  \'_dupkey\'  : False,\n  \'_bake\'    : False,\n  \'_iter\'    : False,\n  \'_pipe\'    : False,\n  \'_raw\'     : False,\n  \'_timeout\' : 0,\n  \'_encoding\': \'utf-8\',\n  \'_bg\'      : False,\n  \'_fg\'      : False,\n  \'_out\'     : None,\n  \'_out_\'    : None,\n  \'_err\'     : None,\n  \'_err_\'    : None\n}\n```\nAnd then try to load `$HOME/.cmdy.ini`, `./.cmdy.ini` and environment variables starting with `CMDY_`, so you can overwrite the configurations with temporary environment variables.\n\nFor example, I want to always use raw keyword option:\n`~/.cmdy.ini`\n```ini\n[default]\n_raw: True\n```\n\n`ontotator.py`:\n```python\nfrom cmdy import oncotator\noncotator(log_name = \'/path/to/log\', ...)\n# oncotator --log_name LOG_NAME ...\n# you don\'t have to specify _raw = True any more.\n```\n\n__Override the settings with environment variable:__\n```bash\nexport CMDY_oncotator__raw=False\npython oncotator.py\n# will run:\n# oncotator --log-name LOG_NAME ...\n#                ^\n```\n\n[1]: https://img.shields.io/pypi/v/pyppl-report?style=flat-square\n[2]: https://pypi.org/project/pyppl-report/\n[3]: https://img.shields.io/github/tag/pwwang/pyppl-report?style=flat-square\n[4]: https://github.com/pwwang/cmdy\n[5]: https://img.shields.io/travis/pwwang/cmdy?style=flat-square\n[6]: https://travis-ci.org/pwwang/cmdy\n[7]: https://img.shields.io/codacy/grade/c82a7081cfc94f089199dafed484e5c3?style=flat-square\n[8]: https://app.codacy.com/project/pwwang/cmdy/dashboard\n[9]: https://img.shields.io/codacy/coverage/c82a7081cfc94f089199dafed484e5c3?style=flat-square\n[10]: https://img.shields.io/pypi/pyversions/cmdy?style=flat-square\n',
    'author': 'pwwang',
    'author_email': 'pwwang@pwwang.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/pwwang/cmdy',
    'py_modules': modules,
    'install_requires': install_requires,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
