# AUTOGENERATED! DO NOT EDIT! File to edit: 01_prophet_sensibility.ipynb (unless otherwise specified).

__all__ = ['prophet_sensibility']

# Cell
import simple_interpolation.simple_interpolation as si

import numpy as np
import pandas as pd
import matplotlib.pyplot as plt
import datetime
from pandas.api.types import is_datetime64_any_dtype as is_datetime

# Cell
# Facebook's Prophet init
import os.path
import json
# import logging
import fbprophet
from fbprophet.serialize import model_to_json, model_from_json
from fbprophet.plot import add_changepoints_to_plot
from fbprophet import Prophet
from fbprophet.plot import plot_forecast_component, plot_yearly, plot_weekly, plot_seasonality

# Internal Cell
def compare_models(models):
    """
    Helper function called at the end of
    prophet_sensibility() to plot all
    different seasonalities from the models.

    Prophet demands this tinkering with its
    methods to plot reproduce plot_components()
    nicely, check:
    https://github.com/facebook/prophet/issues/993
    """

    # Create fig
    fig = plt.figure(figsize=(25, 10))#, constrained_layout=True)
    fig.patch.set_facecolor('white')

    # Create axes
    # subplots = (rows = models, columns = seasonalities, index)
    seasonalities = ['trend', 'yearly', 'weekly', 'daily']
    column_number = len(models)
    row_number = len(seasonalities)
    total_plots = column_number * row_number
    axes= []
    for i in range(1, total_plots + 1):
        ax = fig.add_subplot(row_number, column_number, i)
        axes.append(ax)

    # Plotting loop
    i=0
    for m in models:
        # Model name
        model_name = m
        m = models[model_name]

        # Forecast
        print(f"Extending {model_name} dataframe to the future...")
        future = m.make_future_dataframe(10, freq = 'min', include_history = True)
        print(f"Forecasting with {model_name}...")
        forecast = m.predict(future)

        # Plots
        plot_forecast_component(m=m, fcst=forecast, name='trend', ax=axes[i])#, figsize=(10,2))
        axes[i].set_title(f'{model_name}')

        plot_yearly(m=m, ax=axes[i+column_number])
        axes[i+column_number].set_title(f'{model_name}')

        plot_weekly(m=m, ax=axes[i+column_number*2])
        axes[i+column_number*2].set_title(f'{model_name}')

        plot_seasonality(m=m, name='daily', ax=axes[i+column_number*3])
        axes[i+column_number*3].set_title(f'{model_name}')
        i += 1

    # Adjust layout, title
    fig.suptitle("Seasonalities of the original data, and its available transformations on the library (original, interpolation, adding gaps, and gaps and then interpolation)")
    fig.tight_layout(rect=[0, 0.03, 1, 0.95])

# Cell
def prophet_sensibility ( df, slicing_threshold = 100_000 ):
    """
    Shows how interpolations or adding gaps affect
    Prophet models, by fitting 4 different models
    from the same dataset and plotting its different
    seasonalities.

        Parameters:
            df (DataFrame): DataFrame that is
                used to generate another 3 versions
                of it (interpolated, emptied,
                emptied-then-interpolated)
            slicing_threshold (int): number of elements
                of the input df that will be kept on the
                model fitting (default 100_000). Otherwise
                Prophet takes too long to fit (several hours)

        Returns:
            models (dict): dictionary of already fit Prophet
                models, one for every version of the input df
                (original, interpolated, emptied,
                emptied-then-interpolated)
    """

    # For DEV
#     slicing_threshold = 5_000
    print(f"Will only use the first {slicing_threshold} elements of input df. Change passing slice_threshold = value")

    # In the order of 1M elements fitting Prophet would take too long
    if slicing_threshold != None:
        if len(df) > slicing_threshold:
            print(f"df is too large (over {slicing_threshold} elements): I will keep only the first {slicing_threshold}. Pass argument slicing_threshold = None if you want otherwise.")
            df_original = df[:slicing_threshold]
        else:
            df_original = df.copy()

    # Generate 4 df alternatives, from more gaps to less:
    # - gaps added,
    # - original data (only gaps already present),
    # - gaps added interpolated
    # - original interpolated
    # Unpack parameters
    _, params_deduced = si.check_dataframe(df_original)
    x_column = params_deduced['x_column']
    y_column = params_deduced['y_column']
    datetime_present = params_deduced['datetime_flag']
    df_emptied = add_artificial_gaps_to_data( df_original )
    df_emptied_i = interpolate_gaps( df_emptied[[x_column, y_column]] )
    df_i = interpolate_gaps ( df_original )
    dataframes = [df_original, df_i, df_emptied, df_emptied_i]

    # Generate models on Prophet
    models = {'df_original': None,
              'df_interpolated': None,
              'df_emptied': None,
              'df_emptied_and_interpolated': None}

    id = 0
    i = 0
    for dataframe in dataframes:
#         print(i)
        # # Fit a new model and save it (hours to fit! Except volume, ~ 12m)
        # (not overwritting others, by checking if same filename already exists)
        model_export_path = f'serialized_model_{id}.json'
        while os.path.isfile(model_export_path):
            id += 1
            model_export_path = f'serialized_model_{id}.json'

        # Parsing dataframe to Prophet
        x_column = dataframe.columns[0]
        y_column = dataframe.columns[1]
        df_for_prophet =  dataframe[[x_column, y_column]]
        df_for_prophet = df_for_prophet.rename(columns={x_column:'ds', y_column:'y'})

        # Generic model
        m = Prophet(yearly_seasonality=True, weekly_seasonality=True)#growth = 'linear', seasonality_mode = 'multiplicative')#, mcmc_samples = 0)
        m.fit(df_for_prophet)
        models[list(models.keys())[i]] = m

        # Save model securely
        print(f"Saving fit model {model_export_path}")
        with open(model_export_path, 'w') as fout:
            json.dump(model_to_json(m), fout)

        i += 1

    compare_models(models)

    return models