# Development and Testing

For development and testing purposes, it is essential to use a virtual environment.

## Setup Python and tox

`mbed-tools` is compatible with Python 3.6 or later.

If you are on a Linux distribution, or MacOS, you will find that Python comes pre-installed on your system. **Do not use the pre-installed versions of Python for development.**

Below are links to guides for correctly setting up a development ready version of Python 3 on common platforms:

* [MacOS](https://docs.python-guide.org/starting/install3/osx/#doing-it-right)
* [Ubuntu](https://docs.python-guide.org/starting/install3/linux/)
* [Windows](https://docs.python-guide.org/starting/install3/win/)

On Windows we have also found the [Python Launcher for Windows](https://docs.python.org/3/using/windows.html) to be a reasonable alternative.

After you have set up your Python 3 installation, install tox (the `--user` flag is important **never use `sudo` to install Python packages!**)

```bash
python(3) -m pip install --user tox
```

Check that tox is in the binary path

```bash
tox --version
```

If not, find the user base binary directory

```bash
python -m site --user-base
#~ /Users/<username>/Library/Python/3.7
```

Append `bin` to the directory returned and add this to your path by updating `~/.profile`. For example you might add the following:

```bash
export PATH=~/Library/Python/3.7/bin/:$PATH
```

## Setup Development Environment

Clone the `mbed-tools` GitHub repository

```bash
git clone git@github.com:ARMmbed/mbed-tools.git
```

Set up the development environment using tox (tox will create a development environment at mbed-tools/.venv):

```bash
cd mbed-tools/
tox -e dev
source .venv/bin/activate
```

## Unit Tests, Code Formatting and Static Analysis

After you have activated your development environment, run `pre-commit` to run unit tests and static code analysis checks:

```bash
pre-commit run --all-files
```

This will run `black`, `flake8`, `mypy` and `pytest`. If you would like to run these tools individually, see below:

Run unit tests:

```bash
pytest
```

Run code formatter (it will format files in place):

```bash
black .
```

Run static analysis (note that no output means all is well):

```bash
flake8
```

Perform static type check:

```bash
mypy src/mbed_tools
```

## Documenting code

Inclusion of docstrings is needed in all areas of the code for Flake8 checks in the CI to pass.

We use [google-style](http://google.github.io/styleguide/pyguide.html#381-docstrings) docstrings.

To set up google-style docstring prompts in Pycharm:

* in the menu navigate to Preferences > Tools > Python Integrated Tools
* in the dropdown for docstring format select 'Google'

For longer explanations, you can also include markdown. Markdown can also be kept in separate files in the
`docs/user_docs` folder and included in a docstring in the relevant place using the [reST include](https://docutils.sourceforge.io/docs/ref/rst/directives.html#including-an-external-document-fragment) as follows:

```python
    .. include:: ../docs/user_docs/documentation.md
```

### Building docs locally

You can do a preview build of the documentation locally by running:

```bash
generate-docs
```

This will generate the docs and output them to `local_docs`.
This should only be a preview. Since documentation is automatically generated by the CI you shouldn't commit any docs html files manually.

### Viewing docs generated by the CI

If you want to preview the docs generated by the CI you can view them in the Azure pipeline artifacts directory for the build.

Documentation only gets committed back to this repo to the `docs`
directory during a release and this is what gets published to Github pages.
Don't modify any of the files in this directory by hand.

## Type hints

Type hints should be used in the code wherever possible. Since the
documentation shows the function signatures with the type hints
there is no need to include additional type information in the docstrings.

## Code Climate

Code Climate is integrated with our GitHub flow. Failing the configured rules will yield a pull request not mergeable.

If you prefer to view the Code Climate report on your machine, prior to sending a pull request, you can use the [cli provided by Code Climate](https://docs.codeclimate.com/docs/command-line-interface).

Plugins for various tools are also available:
  - [Atom](https://docs.codeclimate.com/docs/code-climate-atom-package)
  - [PyCharm](https://plugins.jetbrains.com/plugin/13306-code-cleaner-with-code-climate-cli)
  - [Vim](https://docs.codeclimate.com/docs/vim-plugin)
