"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const sqs = require("@aws-cdk/aws-sqs");
const defaults = require("../");
require("@aws-cdk/assert/jest");
const sqs_helper_1 = require("../lib/sqs-helper");
// --------------------------------------------------------------
// Test deployment w/ imported encryption key
// --------------------------------------------------------------
test('Test deployment w/ imported encryption key', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        },
        enableEncryptionWithCustomerManagedKey: true,
        encryptionKey: defaults.buildEncryptionKey(stack)
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue"
    });
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
});
// --------------------------------------------------------------
// Test deployment without imported encryption key
// --------------------------------------------------------------
test('Test deployment without imported encryption key', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        }
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue",
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
// --------------------------------------------------------------
// Test deployment w/ construct created encryption key
// --------------------------------------------------------------
test('Test deployment w/ construct created encryption key', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const [queue, key] = defaults.buildQueue(stack, 'existing-queue', {
        queueProps: {
            queueName: 'existing-queue'
        },
        enableEncryptionWithCustomerManagedKey: true,
    });
    expect(stack).toHaveResource("AWS::SQS::Queue", {
        QueueName: "existing-queue"
    });
    expect(stack).toHaveResource("AWS::KMS::Key", {
        EnableKeyRotation: true
    });
    expect(queue).toBeDefined();
    expect(key).toBeDefined();
});
test('Test DLQ when existing Queue Provided', () => {
    const stack = new core_1.Stack();
    const existingQueue = new sqs.Queue(stack, 'test-queue');
    const buildDlqProps = {
        existingQueueObj: existingQueue,
    };
    const returnedQueueu = defaults.buildDeadLetterQueue(stack, buildDlqProps);
    expect(returnedQueueu).toBeUndefined();
    expect(stack).toCountResources("AWS::SQS::Queue", 1);
});
test('Test DLQ with all defaults', () => {
    const stack = new core_1.Stack();
    sqs_helper_1.buildDeadLetterQueue(stack, {});
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        KmsMasterKeyId: "alias/aws/sqs"
    });
});
test("Test DLQ with a provided properties", () => {
    const stack = new core_1.Stack();
    const testQueueName = "test-unique252";
    const returnedQueue = sqs_helper_1.buildDeadLetterQueue(stack, {
        deadLetterQueueProps: {
            queueName: testQueueName,
        },
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        QueueName: testQueueName,
    });
    expect(returnedQueue).toBeDefined();
});
test('Test DLQ with a provided maxReceiveCount', () => {
    const stack = new core_1.Stack();
    const testMaxReceiveCount = 31;
    const dlqInterface = sqs_helper_1.buildDeadLetterQueue(stack, {
        maxReceiveCount: testMaxReceiveCount
    });
    expect(dlqInterface === null || dlqInterface === void 0 ? void 0 : dlqInterface.maxReceiveCount).toEqual(testMaxReceiveCount);
});
test('Test returning an existing Queue', () => {
    const stack = new core_1.Stack();
    const testQueueName = 'existing-queue';
    const existingQueue = new sqs.Queue(stack, 'test-queue', {
        queueName: testQueueName
    });
    const [returnedQueue] = defaults.buildQueue(stack, 'newQueue', {
        existingQueueObj: existingQueue
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        QueueName: testQueueName,
    });
    expect(existingQueue.queueName).toEqual(returnedQueue.queueName);
});
test('Test creating a queue with a DLQ', () => {
    const stack = new core_1.Stack();
    const dlqInterface = sqs_helper_1.buildDeadLetterQueue(stack, {});
    const [newQueue] = sqs_helper_1.buildQueue(stack, 'new-queue', {
        deadLetterQueue: dlqInterface
    });
    expect(stack).toCountResources("AWS::SQS::Queue", 2);
    expect(newQueue).toBeDefined();
    expect(newQueue.deadLetterQueue).toBeDefined();
});
test('Test creating a FIFO queue', () => {
    const stack = new core_1.Stack();
    const [newFifoQueue] = sqs_helper_1.buildQueue(stack, 'new-queue', {
        queueProps: {
            fifo: true
        }
    });
    expect(stack).toHaveResourceLike("AWS::SQS::Queue", {
        FifoQueue: true
    });
    expect(newFifoQueue.fifo).toBe(true);
});
//# sourceMappingURL=data:application/json;base64,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