"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const cdk = require("@aws-cdk/core");
const kinesisanalytics = require("@aws-cdk/aws-kinesisanalytics");
const kinesisFirehose = require("@aws-cdk/aws-kinesisfirehose");
const iam = require("@aws-cdk/aws-iam");
const kms = require("@aws-cdk/aws-kms");
const defaults = require("../index");
const utils_1 = require("../lib/utils");
require("@aws-cdk/assert/jest");
test("test kinesisanalytics override inputProperty", () => {
    const stack = new core_1.Stack();
    const inputProperty = {
        inputSchema: {
            recordColumns: [{ name: "x", sqlType: "y" }],
            recordFormat: { recordFormatType: "csv" },
        },
        namePrefix: "zzz",
    };
    const defaultProps = defaults.DefaultCfnApplicationProps;
    const inProps = {
        inputs: [inputProperty],
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new kinesisanalytics.CfnApplication(stack, "KinesisAnalytics", outProps);
    expect(stack).toHaveResource("AWS::KinesisAnalytics::Application", {
        Inputs: [
            {
                InputSchema: {
                    RecordColumns: [
                        {
                            Name: "x",
                            SqlType: "y",
                        },
                    ],
                    RecordFormat: {
                        RecordFormatType: "csv",
                    },
                },
                NamePrefix: "zzz",
            },
        ],
    });
});
test("Test default implementation", () => {
    const stack = new core_1.Stack();
    const newFirehose = CreateFirehose(stack);
    const kinesisProps = {
        kinesisFirehose: newFirehose,
        kinesisAnalyticsProps: {
            inputs: [{
                    inputSchema: {
                        recordColumns: [{
                                name: 'ts',
                                sqlType: 'TIMESTAMP',
                                mapping: '$.timestamp'
                            }, {
                                name: 'trip_id',
                                sqlType: 'VARCHAR(64)',
                                mapping: '$.trip_id'
                            }],
                        recordFormat: {
                            recordFormatType: 'JSON'
                        },
                        recordEncoding: 'UTF-8'
                    },
                    namePrefix: 'SOURCE_SQL_STREAM'
                }]
        },
    };
    defaults.buildKinesisAnalyticsApp(stack, kinesisProps);
    expect(stack).toHaveResourceLike("AWS::KinesisAnalytics::Application", {
        Inputs: [{
                InputSchema: {
                    RecordColumns: [{
                            Name: 'ts',
                            SqlType: 'TIMESTAMP',
                            Mapping: '$.timestamp'
                        }, {
                            Name: 'trip_id',
                            SqlType: 'VARCHAR(64)',
                            Mapping: '$.trip_id'
                        }],
                    RecordFormat: {
                        RecordFormatType: 'JSON'
                    },
                    RecordEncoding: 'UTF-8'
                },
                NamePrefix: 'SOURCE_SQL_STREAM'
            }]
    });
});
// test('Test for customer overrides', {
// test('Check policy created', {
function CreateFirehose(stack) {
    // Creating the Firehose is kind of a big deal. FirehoseToS3 is not readily available here in core,
    // so this routine pretty much replicates it. If this function ceases to work correctly, look at
    // FirehoseToS3 and see if that changed.
    const destinationBucket = defaults.CreateScrapBucket(stack, {
        removalPolicy: core_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true,
    });
    const kinesisFirehoseLogGroup = defaults.buildLogGroup(stack, "firehose-log-group", {});
    const cwLogStream = kinesisFirehoseLogGroup.addStream("firehose-log-stream");
    const firehoseRole = new iam.Role(stack, "test-role", {
        assumedBy: new iam.ServicePrincipal("firehose.amazonaws.com"),
    });
    // Setup the IAM policy for Kinesis Firehose
    const firehosePolicy = new iam.Policy(stack, "KinesisFirehosePolicy", {
        statements: [
            new iam.PolicyStatement({
                actions: [
                    "s3:AbortMultipartUpload",
                    "s3:GetBucketLocation",
                    "s3:GetObject",
                    "s3:ListBucket",
                    "s3:ListBucketMultipartUploads",
                    "s3:PutObject",
                ],
                resources: [
                    `${destinationBucket.bucketArn}`,
                    `${destinationBucket.bucketArn}/*`,
                ],
            }),
            new iam.PolicyStatement({
                actions: ["logs:PutLogEvents"],
                resources: [
                    `arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:log-group:${kinesisFirehoseLogGroup.logGroupName}:log-stream:${cwLogStream.logStreamName}`,
                ],
            }),
        ],
    });
    // Attach policy to role
    firehosePolicy.attachToRole(firehoseRole);
    const awsManagedKey = kms.Alias.fromAliasName(stack, "aws-managed-key", "alias/aws/s3");
    const defaultKinesisFirehoseProps = defaults.DefaultCfnDeliveryStreamProps(destinationBucket.bucketArn, firehoseRole.roleArn, kinesisFirehoseLogGroup.logGroupName, cwLogStream.logStreamName, awsManagedKey);
    destinationBucket.grantPut(firehoseRole);
    const firehose = new kinesisFirehose.CfnDeliveryStream(stack, "KinesisFirehose", defaultKinesisFirehoseProps);
    return firehose;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia2luZXNpcy1hbmFseXRpY3MudGVzdC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImtpbmVzaXMtYW5hbHl0aWNzLnRlc3QudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOztBQUVILHdDQUFxRDtBQUNyRCxxQ0FBcUM7QUFDckMsa0VBQWtFO0FBQ2xFLGdFQUFnRTtBQUNoRSx3Q0FBd0M7QUFDeEMsd0NBQXdDO0FBRXhDLHFDQUFxQztBQUNyQyx3Q0FBNkM7QUFDN0MsZ0NBQThCO0FBRTlCLElBQUksQ0FBQyw4Q0FBOEMsRUFBRSxHQUFHLEVBQUU7SUFDeEQsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUUxQixNQUFNLGFBQWEsR0FBa0Q7UUFDbkUsV0FBVyxFQUFFO1lBQ1gsYUFBYSxFQUFFLENBQUMsRUFBRSxJQUFJLEVBQUUsR0FBRyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQztZQUM1QyxZQUFZLEVBQUUsRUFBRSxnQkFBZ0IsRUFBRSxLQUFLLEVBQUU7U0FDMUM7UUFDRCxVQUFVLEVBQUUsS0FBSztLQUNsQixDQUFDO0lBRUYsTUFBTSxZQUFZLEdBQ2hCLFFBQVEsQ0FBQywwQkFBMEIsQ0FBQztJQUV0QyxNQUFNLE9BQU8sR0FBeUM7UUFDcEQsTUFBTSxFQUFFLENBQUMsYUFBYSxDQUFDO0tBQ3hCLENBQUM7SUFFRixNQUFNLFFBQVEsR0FBRyxxQkFBYSxDQUFDLFlBQVksRUFBRSxPQUFPLENBQUMsQ0FBQztJQUV0RCxJQUFJLGdCQUFnQixDQUFDLGNBQWMsQ0FBQyxLQUFLLEVBQUUsa0JBQWtCLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFFekUsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDLGNBQWMsQ0FBQyxvQ0FBb0MsRUFBRTtRQUNqRSxNQUFNLEVBQUU7WUFDTjtnQkFDRSxXQUFXLEVBQUU7b0JBQ1gsYUFBYSxFQUFFO3dCQUNiOzRCQUNFLElBQUksRUFBRSxHQUFHOzRCQUNULE9BQU8sRUFBRSxHQUFHO3lCQUNiO3FCQUNGO29CQUNELFlBQVksRUFBRTt3QkFDWixnQkFBZ0IsRUFBRSxLQUFLO3FCQUN4QjtpQkFDRjtnQkFDRCxVQUFVLEVBQUUsS0FBSzthQUNsQjtTQUNGO0tBQ0YsQ0FBQyxDQUFDO0FBQ0wsQ0FBQyxDQUFDLENBQUM7QUFFSCxJQUFJLENBQUMsNkJBQTZCLEVBQUUsR0FBRyxFQUFFO0lBQ3ZDLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7SUFFMUIsTUFBTSxXQUFXLEdBQUcsY0FBYyxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQzFDLE1BQU0sWUFBWSxHQUEyQztRQUMzRCxlQUFlLEVBQUUsV0FBVztRQUM1QixxQkFBcUIsRUFBRTtZQUNyQixNQUFNLEVBQUUsQ0FBQztvQkFDUCxXQUFXLEVBQUU7d0JBQ1gsYUFBYSxFQUFFLENBQUM7Z0NBQ2QsSUFBSSxFQUFFLElBQUk7Z0NBQ1YsT0FBTyxFQUFFLFdBQVc7Z0NBQ3BCLE9BQU8sRUFBRSxhQUFhOzZCQUN2QixFQUFFO2dDQUNELElBQUksRUFBRSxTQUFTO2dDQUNmLE9BQU8sRUFBRSxhQUFhO2dDQUN0QixPQUFPLEVBQUUsV0FBVzs2QkFDckIsQ0FBQzt3QkFDRixZQUFZLEVBQUU7NEJBQ1osZ0JBQWdCLEVBQUUsTUFBTTt5QkFDekI7d0JBQ0QsY0FBYyxFQUFFLE9BQU87cUJBQ3hCO29CQUNELFVBQVUsRUFBRSxtQkFBbUI7aUJBQ2hDLENBQUM7U0FDSDtLQUNGLENBQUM7SUFFRixRQUFRLENBQUMsd0JBQXdCLENBQUMsS0FBSyxFQUFFLFlBQVksQ0FBQyxDQUFDO0lBRXZELE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxvQ0FBb0MsRUFBRTtRQUNyRSxNQUFNLEVBQUUsQ0FBQztnQkFDUCxXQUFXLEVBQUU7b0JBQ1gsYUFBYSxFQUFFLENBQUM7NEJBQ2QsSUFBSSxFQUFFLElBQUk7NEJBQ1YsT0FBTyxFQUFFLFdBQVc7NEJBQ3BCLE9BQU8sRUFBRSxhQUFhO3lCQUN2QixFQUFFOzRCQUNELElBQUksRUFBRSxTQUFTOzRCQUNmLE9BQU8sRUFBRSxhQUFhOzRCQUN0QixPQUFPLEVBQUUsV0FBVzt5QkFDckIsQ0FBQztvQkFDRixZQUFZLEVBQUU7d0JBQ1osZ0JBQWdCLEVBQUUsTUFBTTtxQkFDekI7b0JBQ0QsY0FBYyxFQUFFLE9BQU87aUJBQ3hCO2dCQUNELFVBQVUsRUFBRSxtQkFBbUI7YUFDaEMsQ0FBQztLQUNILENBQUMsQ0FBQztBQUNMLENBQUMsQ0FBQyxDQUFDO0FBRUgsd0NBQXdDO0FBQ3hDLGlDQUFpQztBQUVqQyxTQUFTLGNBQWMsQ0FBQyxLQUFZO0lBQ2xDLG1HQUFtRztJQUNuRyxnR0FBZ0c7SUFDaEcsd0NBQXdDO0lBQ3hDLE1BQU0saUJBQWlCLEdBQUcsUUFBUSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRTtRQUMxRCxhQUFhLEVBQUUsb0JBQWEsQ0FBQyxPQUFPO1FBQ3BDLGlCQUFpQixFQUFFLElBQUk7S0FDeEIsQ0FBQyxDQUFDO0lBRUgsTUFBTSx1QkFBdUIsR0FBRyxRQUFRLENBQUMsYUFBYSxDQUNwRCxLQUFLLEVBQ0wsb0JBQW9CLEVBQ3BCLEVBQUUsQ0FDSCxDQUFDO0lBRUYsTUFBTSxXQUFXLEdBQW1CLHVCQUF1QixDQUFDLFNBQVMsQ0FDbkUscUJBQXFCLENBQ3RCLENBQUM7SUFFRixNQUFNLFlBQVksR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLFdBQVcsRUFBRTtRQUNwRCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsZ0JBQWdCLENBQUMsd0JBQXdCLENBQUM7S0FDOUQsQ0FBQyxDQUFDO0lBRUgsNENBQTRDO0lBQzVDLE1BQU0sY0FBYyxHQUFHLElBQUksR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsdUJBQXVCLEVBQUU7UUFDcEUsVUFBVSxFQUFFO1lBQ1YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixPQUFPLEVBQUU7b0JBQ1AseUJBQXlCO29CQUN6QixzQkFBc0I7b0JBQ3RCLGNBQWM7b0JBQ2QsZUFBZTtvQkFDZiwrQkFBK0I7b0JBQy9CLGNBQWM7aUJBQ2Y7Z0JBQ0QsU0FBUyxFQUFFO29CQUNULEdBQUcsaUJBQWlCLENBQUMsU0FBUyxFQUFFO29CQUNoQyxHQUFHLGlCQUFpQixDQUFDLFNBQVMsSUFBSTtpQkFDbkM7YUFDRixDQUFDO1lBQ0YsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUN0QixPQUFPLEVBQUUsQ0FBQyxtQkFBbUIsQ0FBQztnQkFDOUIsU0FBUyxFQUFFO29CQUNULE9BQU8sR0FBRyxDQUFDLEdBQUcsQ0FBQyxTQUFTLFNBQVMsR0FBRyxDQUFDLEdBQUcsQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLEdBQUcsQ0FBQyxVQUFVLGNBQWMsdUJBQXVCLENBQUMsWUFBWSxlQUFlLFdBQVcsQ0FBQyxhQUFhLEVBQUU7aUJBQ2xLO2FBQ0YsQ0FBQztTQUNIO0tBQ0YsQ0FBQyxDQUFDO0lBRUgsd0JBQXdCO0lBQ3hCLGNBQWMsQ0FBQyxZQUFZLENBQUMsWUFBWSxDQUFDLENBQUM7SUFFMUMsTUFBTSxhQUFhLEdBQWEsR0FBRyxDQUFDLEtBQUssQ0FBQyxhQUFhLENBQ3JELEtBQUssRUFDTCxpQkFBaUIsRUFDakIsY0FBYyxDQUNmLENBQUM7SUFFRixNQUFNLDJCQUEyQixHQUEyQyxRQUFRLENBQUMsNkJBQTZCLENBQ2hILGlCQUFpQixDQUFDLFNBQVMsRUFDM0IsWUFBWSxDQUFDLE9BQU8sRUFDcEIsdUJBQXVCLENBQUMsWUFBWSxFQUNwQyxXQUFXLENBQUMsYUFBYSxFQUN6QixhQUFhLENBQ2QsQ0FBQztJQUVGLGlCQUFpQixDQUFDLFFBQVEsQ0FBQyxZQUFZLENBQUMsQ0FBQztJQUV6QyxNQUFNLFFBQVEsR0FBRyxJQUFJLGVBQWUsQ0FBQyxpQkFBaUIsQ0FDcEQsS0FBSyxFQUNMLGlCQUFpQixFQUNqQiwyQkFBMkIsQ0FDNUIsQ0FBQztJQUNGLE9BQU8sUUFBUSxDQUFDO0FBQ2xCLENBQUMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqICBDb3B5cmlnaHQgMjAyMiBBbWF6b24uY29tLCBJbmMuIG9yIGl0cyBhZmZpbGlhdGVzLiBBbGwgUmlnaHRzIFJlc2VydmVkLlxuICpcbiAqICBMaWNlbnNlZCB1bmRlciB0aGUgQXBhY2hlIExpY2Vuc2UsIFZlcnNpb24gMi4wICh0aGUgXCJMaWNlbnNlXCIpLiBZb3UgbWF5IG5vdCB1c2UgdGhpcyBmaWxlIGV4Y2VwdCBpbiBjb21wbGlhbmNlXG4gKiAgd2l0aCB0aGUgTGljZW5zZS4gQSBjb3B5IG9mIHRoZSBMaWNlbnNlIGlzIGxvY2F0ZWQgYXRcbiAqXG4gKiAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICBvciBpbiB0aGUgJ2xpY2Vuc2UnIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkIG9uIGFuICdBUyBJUycgQkFTSVMsIFdJVEhPVVQgV0FSUkFOVElFU1xuICogIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGV4cHJlc3Mgb3IgaW1wbGllZC4gU2VlIHRoZSBMaWNlbnNlIGZvciB0aGUgc3BlY2lmaWMgbGFuZ3VhZ2UgZ292ZXJuaW5nIHBlcm1pc3Npb25zXG4gKiAgYW5kIGxpbWl0YXRpb25zIHVuZGVyIHRoZSBMaWNlbnNlLlxuICovXG5cbmltcG9ydCB7IFN0YWNrLCBSZW1vdmFsUG9saWN5IH0gZnJvbSBcIkBhd3MtY2RrL2NvcmVcIjtcbmltcG9ydCAqIGFzIGNkayBmcm9tIFwiQGF3cy1jZGsvY29yZVwiO1xuaW1wb3J0ICogYXMga2luZXNpc2FuYWx5dGljcyBmcm9tIFwiQGF3cy1jZGsvYXdzLWtpbmVzaXNhbmFseXRpY3NcIjtcbmltcG9ydCAqIGFzIGtpbmVzaXNGaXJlaG9zZSBmcm9tIFwiQGF3cy1jZGsvYXdzLWtpbmVzaXNmaXJlaG9zZVwiO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gXCJAYXdzLWNkay9hd3MtaWFtXCI7XG5pbXBvcnQgKiBhcyBrbXMgZnJvbSBcIkBhd3MtY2RrL2F3cy1rbXNcIjtcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSBcIkBhd3MtY2RrL2F3cy1sb2dzXCI7XG5pbXBvcnQgKiBhcyBkZWZhdWx0cyBmcm9tIFwiLi4vaW5kZXhcIjtcbmltcG9ydCB7IG92ZXJyaWRlUHJvcHMgfSBmcm9tIFwiLi4vbGliL3V0aWxzXCI7XG5pbXBvcnQgXCJAYXdzLWNkay9hc3NlcnQvamVzdFwiO1xuXG50ZXN0KFwidGVzdCBraW5lc2lzYW5hbHl0aWNzIG92ZXJyaWRlIGlucHV0UHJvcGVydHlcIiwgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gIGNvbnN0IGlucHV0UHJvcGVydHk6IGtpbmVzaXNhbmFseXRpY3MuQ2ZuQXBwbGljYXRpb24uSW5wdXRQcm9wZXJ0eSA9IHtcbiAgICBpbnB1dFNjaGVtYToge1xuICAgICAgcmVjb3JkQ29sdW1uczogW3sgbmFtZTogXCJ4XCIsIHNxbFR5cGU6IFwieVwiIH1dLFxuICAgICAgcmVjb3JkRm9ybWF0OiB7IHJlY29yZEZvcm1hdFR5cGU6IFwiY3N2XCIgfSxcbiAgICB9LFxuICAgIG5hbWVQcmVmaXg6IFwienp6XCIsXG4gIH07XG5cbiAgY29uc3QgZGVmYXVsdFByb3BzOiBraW5lc2lzYW5hbHl0aWNzLkNmbkFwcGxpY2F0aW9uUHJvcHMgPVxuICAgIGRlZmF1bHRzLkRlZmF1bHRDZm5BcHBsaWNhdGlvblByb3BzO1xuXG4gIGNvbnN0IGluUHJvcHM6IGtpbmVzaXNhbmFseXRpY3MuQ2ZuQXBwbGljYXRpb25Qcm9wcyA9IHtcbiAgICBpbnB1dHM6IFtpbnB1dFByb3BlcnR5XSxcbiAgfTtcblxuICBjb25zdCBvdXRQcm9wcyA9IG92ZXJyaWRlUHJvcHMoZGVmYXVsdFByb3BzLCBpblByb3BzKTtcblxuICBuZXcga2luZXNpc2FuYWx5dGljcy5DZm5BcHBsaWNhdGlvbihzdGFjaywgXCJLaW5lc2lzQW5hbHl0aWNzXCIsIG91dFByb3BzKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlKFwiQVdTOjpLaW5lc2lzQW5hbHl0aWNzOjpBcHBsaWNhdGlvblwiLCB7XG4gICAgSW5wdXRzOiBbXG4gICAgICB7XG4gICAgICAgIElucHV0U2NoZW1hOiB7XG4gICAgICAgICAgUmVjb3JkQ29sdW1uczogW1xuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBOYW1lOiBcInhcIixcbiAgICAgICAgICAgICAgU3FsVHlwZTogXCJ5XCIsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgIF0sXG4gICAgICAgICAgUmVjb3JkRm9ybWF0OiB7XG4gICAgICAgICAgICBSZWNvcmRGb3JtYXRUeXBlOiBcImNzdlwiLFxuICAgICAgICAgIH0sXG4gICAgICAgIH0sXG4gICAgICAgIE5hbWVQcmVmaXg6IFwienp6XCIsXG4gICAgICB9LFxuICAgIF0sXG4gIH0pO1xufSk7XG5cbnRlc3QoXCJUZXN0IGRlZmF1bHQgaW1wbGVtZW50YXRpb25cIiwgKCkgPT4ge1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuXG4gIGNvbnN0IG5ld0ZpcmVob3NlID0gQ3JlYXRlRmlyZWhvc2Uoc3RhY2spO1xuICBjb25zdCBraW5lc2lzUHJvcHM6IGRlZmF1bHRzLkJ1aWxkS2luZXNpc0FuYWx5dGljc0FwcFByb3BzID0ge1xuICAgIGtpbmVzaXNGaXJlaG9zZTogbmV3RmlyZWhvc2UsXG4gICAga2luZXNpc0FuYWx5dGljc1Byb3BzOiB7XG4gICAgICBpbnB1dHM6IFt7XG4gICAgICAgIGlucHV0U2NoZW1hOiB7XG4gICAgICAgICAgcmVjb3JkQ29sdW1uczogW3tcbiAgICAgICAgICAgIG5hbWU6ICd0cycsXG4gICAgICAgICAgICBzcWxUeXBlOiAnVElNRVNUQU1QJyxcbiAgICAgICAgICAgIG1hcHBpbmc6ICckLnRpbWVzdGFtcCdcbiAgICAgICAgICB9LCB7XG4gICAgICAgICAgICBuYW1lOiAndHJpcF9pZCcsXG4gICAgICAgICAgICBzcWxUeXBlOiAnVkFSQ0hBUig2NCknLFxuICAgICAgICAgICAgbWFwcGluZzogJyQudHJpcF9pZCdcbiAgICAgICAgICB9XSxcbiAgICAgICAgICByZWNvcmRGb3JtYXQ6IHtcbiAgICAgICAgICAgIHJlY29yZEZvcm1hdFR5cGU6ICdKU09OJ1xuICAgICAgICAgIH0sXG4gICAgICAgICAgcmVjb3JkRW5jb2Rpbmc6ICdVVEYtOCdcbiAgICAgICAgfSxcbiAgICAgICAgbmFtZVByZWZpeDogJ1NPVVJDRV9TUUxfU1RSRUFNJ1xuICAgICAgfV1cbiAgICB9LFxuICB9O1xuXG4gIGRlZmF1bHRzLmJ1aWxkS2luZXNpc0FuYWx5dGljc0FwcChzdGFjaywga2luZXNpc1Byb3BzKTtcblxuICBleHBlY3Qoc3RhY2spLnRvSGF2ZVJlc291cmNlTGlrZShcIkFXUzo6S2luZXNpc0FuYWx5dGljczo6QXBwbGljYXRpb25cIiwge1xuICAgIElucHV0czogW3tcbiAgICAgIElucHV0U2NoZW1hOiB7XG4gICAgICAgIFJlY29yZENvbHVtbnM6IFt7XG4gICAgICAgICAgTmFtZTogJ3RzJyxcbiAgICAgICAgICBTcWxUeXBlOiAnVElNRVNUQU1QJyxcbiAgICAgICAgICBNYXBwaW5nOiAnJC50aW1lc3RhbXAnXG4gICAgICAgIH0sIHtcbiAgICAgICAgICBOYW1lOiAndHJpcF9pZCcsXG4gICAgICAgICAgU3FsVHlwZTogJ1ZBUkNIQVIoNjQpJyxcbiAgICAgICAgICBNYXBwaW5nOiAnJC50cmlwX2lkJ1xuICAgICAgICB9XSxcbiAgICAgICAgUmVjb3JkRm9ybWF0OiB7XG4gICAgICAgICAgUmVjb3JkRm9ybWF0VHlwZTogJ0pTT04nXG4gICAgICAgIH0sXG4gICAgICAgIFJlY29yZEVuY29kaW5nOiAnVVRGLTgnXG4gICAgICB9LFxuICAgICAgTmFtZVByZWZpeDogJ1NPVVJDRV9TUUxfU1RSRUFNJ1xuICAgIH1dXG4gIH0pO1xufSk7XG5cbi8vIHRlc3QoJ1Rlc3QgZm9yIGN1c3RvbWVyIG92ZXJyaWRlcycsIHtcbi8vIHRlc3QoJ0NoZWNrIHBvbGljeSBjcmVhdGVkJywge1xuXG5mdW5jdGlvbiBDcmVhdGVGaXJlaG9zZShzdGFjazogU3RhY2spOiBraW5lc2lzRmlyZWhvc2UuQ2ZuRGVsaXZlcnlTdHJlYW0ge1xuICAvLyBDcmVhdGluZyB0aGUgRmlyZWhvc2UgaXMga2luZCBvZiBhIGJpZyBkZWFsLiBGaXJlaG9zZVRvUzMgaXMgbm90IHJlYWRpbHkgYXZhaWxhYmxlIGhlcmUgaW4gY29yZSxcbiAgLy8gc28gdGhpcyByb3V0aW5lIHByZXR0eSBtdWNoIHJlcGxpY2F0ZXMgaXQuIElmIHRoaXMgZnVuY3Rpb24gY2Vhc2VzIHRvIHdvcmsgY29ycmVjdGx5LCBsb29rIGF0XG4gIC8vIEZpcmVob3NlVG9TMyBhbmQgc2VlIGlmIHRoYXQgY2hhbmdlZC5cbiAgY29uc3QgZGVzdGluYXRpb25CdWNrZXQgPSBkZWZhdWx0cy5DcmVhdGVTY3JhcEJ1Y2tldChzdGFjaywge1xuICAgIHJlbW92YWxQb2xpY3k6IFJlbW92YWxQb2xpY3kuREVTVFJPWSxcbiAgICBhdXRvRGVsZXRlT2JqZWN0czogdHJ1ZSxcbiAgfSk7XG5cbiAgY29uc3Qga2luZXNpc0ZpcmVob3NlTG9nR3JvdXAgPSBkZWZhdWx0cy5idWlsZExvZ0dyb3VwKFxuICAgIHN0YWNrLFxuICAgIFwiZmlyZWhvc2UtbG9nLWdyb3VwXCIsXG4gICAge31cbiAgKTtcblxuICBjb25zdCBjd0xvZ1N0cmVhbTogbG9ncy5Mb2dTdHJlYW0gPSBraW5lc2lzRmlyZWhvc2VMb2dHcm91cC5hZGRTdHJlYW0oXG4gICAgXCJmaXJlaG9zZS1sb2ctc3RyZWFtXCJcbiAgKTtcblxuICBjb25zdCBmaXJlaG9zZVJvbGUgPSBuZXcgaWFtLlJvbGUoc3RhY2ssIFwidGVzdC1yb2xlXCIsIHtcbiAgICBhc3N1bWVkQnk6IG5ldyBpYW0uU2VydmljZVByaW5jaXBhbChcImZpcmVob3NlLmFtYXpvbmF3cy5jb21cIiksXG4gIH0pO1xuXG4gIC8vIFNldHVwIHRoZSBJQU0gcG9saWN5IGZvciBLaW5lc2lzIEZpcmVob3NlXG4gIGNvbnN0IGZpcmVob3NlUG9saWN5ID0gbmV3IGlhbS5Qb2xpY3koc3RhY2ssIFwiS2luZXNpc0ZpcmVob3NlUG9saWN5XCIsIHtcbiAgICBzdGF0ZW1lbnRzOiBbXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICBcInMzOkFib3J0TXVsdGlwYXJ0VXBsb2FkXCIsXG4gICAgICAgICAgXCJzMzpHZXRCdWNrZXRMb2NhdGlvblwiLFxuICAgICAgICAgIFwiczM6R2V0T2JqZWN0XCIsXG4gICAgICAgICAgXCJzMzpMaXN0QnVja2V0XCIsXG4gICAgICAgICAgXCJzMzpMaXN0QnVja2V0TXVsdGlwYXJ0VXBsb2Fkc1wiLFxuICAgICAgICAgIFwiczM6UHV0T2JqZWN0XCIsXG4gICAgICAgIF0sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIGAke2Rlc3RpbmF0aW9uQnVja2V0LmJ1Y2tldEFybn1gLFxuICAgICAgICAgIGAke2Rlc3RpbmF0aW9uQnVja2V0LmJ1Y2tldEFybn0vKmAsXG4gICAgICAgIF0sXG4gICAgICB9KSxcbiAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogW1wibG9nczpQdXRMb2dFdmVudHNcIl0sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIGBhcm46JHtjZGsuQXdzLlBBUlRJVElPTn06bG9nczoke2Nkay5Bd3MuUkVHSU9OfToke2Nkay5Bd3MuQUNDT1VOVF9JRH06bG9nLWdyb3VwOiR7a2luZXNpc0ZpcmVob3NlTG9nR3JvdXAubG9nR3JvdXBOYW1lfTpsb2ctc3RyZWFtOiR7Y3dMb2dTdHJlYW0ubG9nU3RyZWFtTmFtZX1gLFxuICAgICAgICBdLFxuICAgICAgfSksXG4gICAgXSxcbiAgfSk7XG5cbiAgLy8gQXR0YWNoIHBvbGljeSB0byByb2xlXG4gIGZpcmVob3NlUG9saWN5LmF0dGFjaFRvUm9sZShmaXJlaG9zZVJvbGUpO1xuXG4gIGNvbnN0IGF3c01hbmFnZWRLZXk6IGttcy5JS2V5ID0ga21zLkFsaWFzLmZyb21BbGlhc05hbWUoXG4gICAgc3RhY2ssXG4gICAgXCJhd3MtbWFuYWdlZC1rZXlcIixcbiAgICBcImFsaWFzL2F3cy9zM1wiXG4gICk7XG5cbiAgY29uc3QgZGVmYXVsdEtpbmVzaXNGaXJlaG9zZVByb3BzOiBraW5lc2lzRmlyZWhvc2UuQ2ZuRGVsaXZlcnlTdHJlYW1Qcm9wcyA9IGRlZmF1bHRzLkRlZmF1bHRDZm5EZWxpdmVyeVN0cmVhbVByb3BzKFxuICAgIGRlc3RpbmF0aW9uQnVja2V0LmJ1Y2tldEFybixcbiAgICBmaXJlaG9zZVJvbGUucm9sZUFybixcbiAgICBraW5lc2lzRmlyZWhvc2VMb2dHcm91cC5sb2dHcm91cE5hbWUsXG4gICAgY3dMb2dTdHJlYW0ubG9nU3RyZWFtTmFtZSxcbiAgICBhd3NNYW5hZ2VkS2V5XG4gICk7XG5cbiAgZGVzdGluYXRpb25CdWNrZXQuZ3JhbnRQdXQoZmlyZWhvc2VSb2xlKTtcblxuICBjb25zdCBmaXJlaG9zZSA9IG5ldyBraW5lc2lzRmlyZWhvc2UuQ2ZuRGVsaXZlcnlTdHJlYW0oXG4gICAgc3RhY2ssXG4gICAgXCJLaW5lc2lzRmlyZWhvc2VcIixcbiAgICBkZWZhdWx0S2luZXNpc0ZpcmVob3NlUHJvcHNcbiAgKTtcbiAgcmV0dXJuIGZpcmVob3NlO1xufVxuIl19