"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const aws_glue_1 = require("@aws-cdk/aws-glue");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("..");
// --------------------------------------------------------------
// Test deployment with role creation
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    var _a, _b;
    // Stack
    const stack = new core_1.Stack();
    const _jobRole = new aws_iam_1.Role(stack, 'CustomETLJobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com')
    });
    const cfnJobProps = defaults.DefaultGlueJobProps(_jobRole, {
        command: {
            name: 'glueetl',
            pythonVersion: '3',
            scriptLocation: 's3://fakescriptlocation/fakebucket',
        },
        role: _jobRole.roleArn
    }, 'testETLJob', {});
    const _database = defaults.createGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
    const _glueJob = defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect((_a = _glueJob[2]) === null || _a === void 0 ? void 0 : _a[0]).toBeDefined();
    expect((_b = _glueJob[2]) === null || _b === void 0 ? void 0 : _b[0]).toBeInstanceOf(aws_s3_1.Bucket);
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Type: "AWS::Glue::Job",
        Properties: {
            Command: {
                Name: "glueetl",
                PythonVersion: "3",
                ScriptLocation: "s3://fakescriptlocation/fakebucket"
            },
            Role: {
                "Fn::GetAtt": [
                    "CustomETLJobRole90A83A66",
                    "Arn"
                ]
            },
            GlueVersion: "2.0",
            NumberOfWorkers: 2,
            SecurityConfiguration: "testETLJob",
            WorkerType: "G.1X"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Pass an existing Glue Job
// --------------------------------------------------------------
test('Create a Glue Job outside the construct', () => {
    // Stack
    const stack = new core_1.Stack();
    const _existingCfnJob = new aws_glue_1.CfnJob(stack, 'ExistingJob', {
        command: {
            name: 'pythonshell',
            pythonVersion: '2',
            scriptLocation: 's3://existingfakelocation/existingScript'
        },
        role: new aws_iam_1.Role(stack, 'ExistingJobRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazon.com'),
            description: 'Existing role'
        }).roleArn,
        glueVersion: '1',
        allocatedCapacity: 2,
        maxCapacity: 4,
        numberOfWorkers: 2,
        workerType: 'Standard'
    });
    const _database = defaults.createGlueDatabase(stack, defaults.DefaultGlueDatabaseProps());
    const _glueJob = defaults.buildGlueJob(stack, {
        existingCfnJob: _existingCfnJob,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(_glueJob[2]).not.toBeDefined();
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Type: "AWS::Glue::Job",
        Properties: {
            AllocatedCapacity: 2,
            Command: {
                Name: "pythonshell",
                PythonVersion: "2",
                ScriptLocation: "s3://existingfakelocation/existingScript",
            },
            GlueVersion: "1",
            MaxCapacity: 4,
            NumberOfWorkers: 2,
            Role: {
                "Fn::GetAtt": [
                    "ExistingJobRole8F750976",
                    "Arn",
                ],
            },
            WorkerType: "Standard",
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Provide additional parameters other than default ones
// --------------------------------------------------------------
test('Test custom deployment properties', () => {
    // Stack
    const stack = new core_1.Stack();
    const _commandName = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _commandName,
            pythonVersion: '3',
            scriptLocation: 's3://existingfakelocation/existingScript'
        },
        role: new aws_iam_1.Role(stack, 'ExistingJobRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('glue.amazon.com'),
            description: 'Existing role'
        }).roleArn,
        glueVersion: '1',
        numberOfWorkers: 2,
        workerType: 'Standard'
    };
    const _database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    // check if Glue Job Resource was created correctly
    expect(stack).toHaveResourceLike('AWS::Glue::Job', {
        Properties: {
            Command: {
                Name: "glueetl",
                PythonVersion: "3",
                ScriptLocation: "s3://existingfakelocation/existingScript",
            },
            GlueVersion: "1",
            NumberOfWorkers: 2,
            Role: {
                "Fn::GetAtt": [
                    "ExistingJobRole8F750976",
                    "Arn",
                ],
            },
            SecurityConfiguration: "ETLJobSecurityConfig",
            WorkerType: "Standard",
        },
        Type: "AWS::Glue::Job"
    }, assert_1.ResourcePart.CompleteDefinition);
    // check if the role is created
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Type: "AWS::IAM::Role",
        Properties: {
            AssumeRolePolicyDocument: {
                Statement: [
                    {
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "glue.amazon.com"
                        }
                    }
                ],
                Version: "2012-10-17"
            },
            Description: "Existing role"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
    // check if the security config is created
    expect(stack).toHaveResourceLike('AWS::Glue::SecurityConfiguration', {
        Properties: {
            EncryptionConfiguration: {
                JobBookmarksEncryption: {
                    JobBookmarksEncryptionMode: "CSE-KMS",
                    KmsKeyArn: {
                        "Fn::Join": [
                            "", [
                                "arn:", {
                                    Ref: "AWS::Partition",
                                },
                                ":kms:", {
                                    Ref: "AWS::Region",
                                },
                                ":", {
                                    Ref: "AWS::AccountId",
                                },
                                ":alias/aws/glue",
                            ],
                        ],
                    },
                },
                S3Encryptions: [{
                        S3EncryptionMode: "SSE-S3",
                    }],
            },
            Name: "ETLJobSecurityConfig",
        },
        Type: "AWS::Glue::SecurityConfiguration",
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Do not supply parameters and error out
// --------------------------------------------------------------
test('Do no supply glueJobProps or existingCfnJob and error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' }))
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Allow the construct to create the job role required
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _jobID,
            pythonVersion: '3',
            scriptLocation: 's3://fakelocation/script'
        }
    };
    const _database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(stack).toHaveResourceLike('AWS::IAM::Role', {
        Type: "AWS::IAM::Role",
        Properties: {
            AssumeRolePolicyDocument: {
                Statement: [{
                        Action: "sts:AssumeRole",
                        Effect: "Allow",
                        Principal: {
                            Service: "glue.amazonaws.com"
                        }
                    }],
                Version: "2012-10-17"
            },
            Description: "Service role that Glue custom ETL jobs will assume for exeuction"
        }
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test deployment when output location is provided
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _jobID,
            pythonVersion: '3',
            scriptLocation: 's3://fakelocation/script'
        }
    };
    const _database = defaults.createGlueDatabase(stack);
    defaults.buildGlueJob(stack, {
        glueJobProps: cfnJobProps,
        outputDataStore: {
            datastoreType: defaults.SinkStoreType.S3,
            existingS3OutputBucket: new aws_s3_1.Bucket(stack, 'OutputBucket', {
                versioned: false,
                bucketName: 'outputbucket',
                encryption: aws_s3_1.BucketEncryption.S3_MANAGED,
                removalPolicy: core_1.RemovalPolicy.DESTROY
            })
        },
        database: _database,
        table: defaults.createGlueTable(stack, _database, undefined, [{
                name: "id",
                type: "int",
                comment: ""
            }], 'kinesis', { STREAM_NAME: 'testStream' })
    });
    expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
        Type: 'AWS::S3::Bucket',
        Properties: {
            BucketEncryption: {
                ServerSideEncryptionConfiguration: [{
                        ServerSideEncryptionByDefault: {
                            SSEAlgorithm: "AES256"
                        }
                    }]
            },
            BucketName: "outputbucket"
        },
        UpdateReplacePolicy: "Delete",
        DeletionPolicy: "Delete"
    }, assert_1.ResourcePart.CompleteDefinition);
});
// --------------------------------------------------------------
// Test deployment when script location not provided - throw error
// --------------------------------------------------------------
test('Test deployment with role creation', () => {
    // Stack
    const stack = new core_1.Stack();
    const _jobID = 'glueetl';
    const cfnJobProps = {
        command: {
            name: _jobID,
            pythonVersion: '3'
        }
    };
    const _database = defaults.createGlueDatabase(stack);
    try {
        defaults.buildGlueJob(stack, {
            glueJobProps: cfnJobProps,
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3,
                existingS3OutputBucket: new aws_s3_1.Bucket(stack, 'OutputBucket', {
                    versioned: false
                })
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
        expect(error.message).toEqual('Either one of CfnJob.JobCommandProperty.scriptLocation or KinesisstreamsToGluejobProps.etlCodeAsset ' +
            'has to be provided. If the ETL Job code file exists in a local filesystem, please set KinesisstreamsToGluejobProps.etlCodeAsset. ' +
            'If the ETL Job is available in an S3 bucket, set the CfnJob.JobCommandProperty.scriptLocation property');
    }
});
// --------------------------------------------------------------
// Dont pass Job Command attributes and it should throw an error
// --------------------------------------------------------------
test('Test for incorrect Job Command property', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            glueJobProps: {},
            database: _database,
            table: defaults.createGlueTable(stack, _database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Check for CfnJob.JobCommandProperty type
// --------------------------------------------------------------
test('check for JobCommandProperty type', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            glueJobProps: {
                command: {
                    fakekey: 'fakevalue'
                }
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, undefined, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Supply maxCapacity with GlueVersion 2.0 and error out
// --------------------------------------------------------------
test('GlueJob configuration with glueVersion 2.0 should not support maxCapacity and error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                glueVersion: '2.0',
                maxCapacity: '2'
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
// --------------------------------------------------------------
// Fail if setting maxCapacity and WorkerType/ NumberOfWorkers
// --------------------------------------------------------------
test('Cannot use maxCapacity and WorkerType, so error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                command: {
                    name: "gluejob1.0",
                    pythonVersion: '3'
                },
                glueVersion: '1.0',
                maxCapacity: '2',
                workerType: 'Standard'
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
test('Cannot use maxCapacity and WorkerType, so error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                command: {
                    name: "gluejob1.0",
                    pythonVersion: '3'
                },
                glueVersion: '1.0',
                maxCapacity: '2',
                numberOfWorkers: 2
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
test('Cannot use maxCapacity and WorkerType, so error out', () => {
    const stack = new core_1.Stack();
    try {
        const _database = defaults.createGlueDatabase(stack);
        defaults.buildGlueJob(stack, {
            outputDataStore: {
                datastoreType: defaults.SinkStoreType.S3
            },
            database: _database,
            table: defaults.createGlueTable(stack, _database, defaults.DefaultGlueTableProps(_database, [{
                    name: "id",
                    type: "int",
                    comment: ""
                }], 'kinesis', { STREAM_NAME: 'testStream' })),
            glueJobProps: {
                command: {
                    name: "gluejob1.0",
                    pythonVersion: '3'
                },
                glueVersion: '1.0',
                maxCapacity: '2',
                numberOfWorkers: 2,
                workerType: 'G1.X'
            }
        });
    }
    catch (error) {
        expect(error).toBeInstanceOf(Error);
    }
});
//# sourceMappingURL=data:application/json;base64,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