"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const dynamodb = require("@aws-cdk/aws-dynamodb");
const defaults = require("../index");
const utils_1 = require("../lib/utils");
require("@aws-cdk/assert/jest");
const dynamodb_table_helper_1 = require("../lib/dynamodb-table-helper");
test('test TableProps change billing mode', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultTableProps;
    const inProps = {
        billingMode: dynamodb.BillingMode.PROVISIONED,
        readCapacity: 3,
        writeCapacity: 3,
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        }
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new dynamodb.Table(stack, 'test-dynamo-override', outProps);
    expect(stack).toHaveResource("AWS::DynamoDB::Table", {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            }
        ],
        ProvisionedThroughput: {
            ReadCapacityUnits: 3,
            WriteCapacityUnits: 3
        },
        SSESpecification: {
            SSEEnabled: true
        }
    });
});
test('test TableProps override add sort key', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultTableProps;
    const inProps = {
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        },
        sortKey: {
            name: 'sort_key',
            type: dynamodb.AttributeType.STRING
        }
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new dynamodb.Table(stack, 'test-dynamo-override', outProps);
    expect(stack).toHaveResource("AWS::DynamoDB::Table", {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            },
            {
                AttributeName: "sort_key",
                KeyType: "RANGE"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            },
            {
                AttributeName: "sort_key",
                AttributeType: "S"
            }
        ],
        BillingMode: "PAY_PER_REQUEST",
        SSESpecification: {
            SSEEnabled: true
        }
    });
});
test('test TableWithStreamProps override stream view type', () => {
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultTableWithStreamProps;
    const inProps = {
        partitionKey: {
            name: 'id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    new dynamodb.Table(stack, 'test-dynamo-override', outProps);
    expect(stack).toHaveResource("AWS::DynamoDB::Table", {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ],
        AttributeDefinitions: [
            {
                AttributeName: "id",
                AttributeType: "S"
            }
        ],
        BillingMode: "PAY_PER_REQUEST",
        SSESpecification: {
            SSEEnabled: true
        },
        StreamSpecification: {
            StreamViewType: "NEW_IMAGE"
        }
    });
});
test('test buildDynamoDBTable with existingTableObj', () => {
    const stack = new core_1.Stack();
    const tableProps = {
        billingMode: dynamodb.BillingMode.PROVISIONED,
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        }
    };
    const existingTableObj = new dynamodb.Table(stack, 'DynamoTable', tableProps);
    defaults.buildDynamoDBTable(stack, {
        existingTableObj
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        ProvisionedThroughput: {
            ReadCapacityUnits: 5,
            WriteCapacityUnits: 5
        }
    }));
});
test('test buildDynamoDBTable without any arguments', () => {
    const stack = new core_1.Stack();
    defaults.buildDynamoDBTable(stack, {});
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        BillingMode: "PAY_PER_REQUEST"
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        SSESpecification: {
            SSEEnabled: true
        }
    }));
});
test('test buildDynamoDBTable with TableProps', () => {
    const stack = new core_1.Stack();
    const dynamoTableProps = {
        billingMode: dynamodb.BillingMode.PROVISIONED,
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        }
    };
    defaults.buildDynamoDBTable(stack, {
        dynamoTableProps
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        ProvisionedThroughput: {
            ReadCapacityUnits: 5,
            WriteCapacityUnits: 5
        }
    }));
});
test('test buildDynamoDBTableWithStream with TableProps', () => {
    const stack = new core_1.Stack();
    const dynamoTableProps = {
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    defaults.buildDynamoDBTableWithStream(stack, {
        dynamoTableProps
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        StreamSpecification: {
            StreamViewType: "NEW_IMAGE"
        }
    }));
});
test('test buildDynamoDBTableWithStream without any arguments', () => {
    const stack = new core_1.Stack();
    defaults.buildDynamoDBTableWithStream(stack, {});
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        BillingMode: "PAY_PER_REQUEST"
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        SSESpecification: {
            SSEEnabled: true
        }
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        StreamSpecification: {
            StreamViewType: "NEW_AND_OLD_IMAGES"
        }
    }));
});
test('test buildDynamoDBTableWithStream with existingTableObj', () => {
    const stack = new core_1.Stack();
    const tableProps = {
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    const existingTableInterface = new dynamodb.Table(stack, 'DynamoTable', tableProps);
    defaults.buildDynamoDBTableWithStream(stack, {
        existingTableInterface
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        StreamSpecification: {
            StreamViewType: "NEW_IMAGE"
        }
    }));
});
test('test buildDynamoDBTable with existingTableInterface', () => {
    const stack = new core_1.Stack();
    const tableProps = {
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    const existingTableInterface = new dynamodb.Table(stack, 'DynamoTable', tableProps);
    defaults.buildDynamoDBTable(stack, {
        existingTableInterface
    });
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        KeySchema: [
            {
                AttributeName: "table_id",
                KeyType: "HASH"
            }
        ]
    }));
    assert_1.expect(stack).to(assert_1.haveResource('AWS::DynamoDB::Table', {
        StreamSpecification: {
            StreamViewType: "NEW_IMAGE"
        }
    }));
});
test('test getPartitionKeyNameFromTable()', () => {
    const partitionKeyName = 'testPartitionKey';
    const stack = new core_1.Stack();
    const defaultProps = defaults.DefaultTableProps;
    const inProps = {
        partitionKey: {
            name: partitionKeyName,
            type: dynamodb.AttributeType.STRING
        },
        sortKey: {
            name: 'sort_key',
            type: dynamodb.AttributeType.STRING
        }
    };
    const outProps = utils_1.overrideProps(defaultProps, inProps);
    const newTable = new dynamodb.Table(stack, 'test-dynamo-override', outProps);
    const testKeyName = dynamodb_table_helper_1.getPartitionKeyNameFromTable(newTable);
    expect(testKeyName).toEqual(partitionKeyName);
});
test('Test providing both existingTableInterface and existingTableObj', () => {
    const stack = new core_1.Stack();
    const tableProps = {
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    const existingTableInterface = new dynamodb.Table(stack, 'DynamoTable', tableProps);
    const newProps = {
        existingTableInterface,
        existingTableObj: existingTableInterface
    };
    const app = () => {
        defaults.buildDynamoDBTable(stack, newProps);
    };
    expect(app).toThrowError('Error - Either provide existingTableInterface or existingTableObj, but not both.\n');
});
test('Test providing both existingTableInterface and dynamoTableProps', () => {
    const stack = new core_1.Stack();
    const dynamoTableProps = {
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    const existingTableInterface = new dynamodb.Table(stack, 'DynamoTable', dynamoTableProps);
    const newProps = {
        existingTableInterface,
        dynamoTableProps
    };
    const app = () => {
        defaults.buildDynamoDBTable(stack, newProps);
    };
    expect(app).toThrowError('Error - Either provide existingTableInterface or dynamoTableProps, but not both.\n');
});
test('Test providing both existingTableObj and dynamoTableProps', () => {
    const stack = new core_1.Stack();
    const dynamoTableProps = {
        partitionKey: {
            name: 'table_id',
            type: dynamodb.AttributeType.STRING
        },
        stream: dynamodb.StreamViewType.NEW_IMAGE
    };
    const existingTableObj = new dynamodb.Table(stack, 'DynamoTable', dynamoTableProps);
    const newProps = {
        existingTableObj,
        dynamoTableProps
    };
    const app = () => {
        defaults.buildDynamoDBTable(stack, newProps);
    };
    expect(app).toThrowError('Error - Either provide existingTableObj or dynamoTableProps, but not both.\n');
});
//# sourceMappingURL=data:application/json;base64,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