"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@aws-cdk/core");
const elb = require("@aws-cdk/aws-elasticloadbalancingv2");
const lambda = require("@aws-cdk/aws-lambda");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
test('Test ObtainAlb with existing ALB', () => {
    const stack = new core_1.Stack();
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const existingLoadBalancer = new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
    defaults.ObtainAlb(stack, 'test', vpc, true, existingLoadBalancer);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: "unique-name",
    });
});
test('Test ObtainAlb for new ALB with provided props', () => {
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    defaults.ObtainAlb(stack, 'test', vpc, true, undefined, {
        loadBalancerName: 'new-loadbalancer',
        vpc,
        internetFacing: true
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Name: "new-loadbalancer",
        Scheme: "internet-facing",
    });
});
test('Test ObtainAlb for new ALB with default props', () => {
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    defaults.ObtainAlb(stack, 'test', vpc, false);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::LoadBalancer', {
        Scheme: "internal",
    });
});
test('Test with custom logging bucket props', () => {
    // Creating ALB logging requires a region and account (but
    // these can be fake in unit tests)
    const stack = new core_1.Stack(undefined, undefined, {
        env: { account: "123456789012", region: 'us-east-1' },
    });
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    const testName = 'test-name';
    defaults.ObtainAlb(stack, 'test', vpc, false, undefined, undefined, true, { bucketName: testName });
    expect(stack).toHaveResourceLike('AWS::S3::Bucket', {
        BucketName: testName
    });
});
test('Test with no logging', () => {
    const stack = new core_1.Stack();
    // Build VPC
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultPublicPrivateVpcProps(),
    });
    defaults.ObtainAlb(stack, 'test', vpc, false, undefined, undefined, false);
    expect(stack).not.toHaveResourceLike('AWS::S3::Bucket', {});
});
test('Test add single lambda target group with no customization', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testFunction = CreateTestFunction(stack, 'test-function');
    const testVpc = defaults.getTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testListener = CreateTestListener(stack, 'test-listener', testAlb);
    // This is the code we're testing
    defaults.AddLambdaTarget(stack, 'test-lambda-target', testListener, testFunction);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        DefaultActions: [
            {
                TargetGroupArn: {
                    Ref: "testlambdatargettg7E5C32F4"
                },
                Type: "forward"
            }
        ],
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: "lambda",
    });
});
test('Test add single lambda target group with target group props', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testFunction = CreateTestFunction(stack, 'test-function');
    const testVpc = defaults.getTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testListener = CreateTestListener(stack, 'test-listener', testAlb);
    const targetGroupName = 'test-group';
    // This is the code we're testing
    defaults.AddLambdaTarget(stack, 'test-lambda-target', testListener, testFunction, undefined, { targetGroupName });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: "lambda",
        Name: targetGroupName,
    });
});
test('Test add rule props for second lambda target group', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testFunction = CreateTestFunction(stack, 'test-function');
    const testVpc = defaults.getTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testListener = CreateTestListener(stack, 'test-listener', testAlb);
    const targetGroupName = 'test-group';
    const pathPattern = '*admin*';
    defaults.AddLambdaTarget(stack, 'test-lambda-target', testListener, testFunction, undefined, { targetGroupName });
    // This is the code we're testing
    const ruleProps = {
        conditions: [elb.ListenerCondition.pathPatterns([pathPattern])],
        priority: 10
    };
    defaults.AddLambdaTarget(stack, 'test-second-lambda-target', testListener, testFunction, ruleProps, { targetGroupName });
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::TargetGroup', 2);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        pathPattern
                    ]
                }
            }
        ]
    });
});
test('Test add single fargate target with no customization', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = defaults.getTestVpc(stack);
    const testService = CreateTestFargateService(stack, 'test-service', testVpc);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testListener = CreateTestListener(stack, 'test-listener', testAlb);
    // This is the code we're testing
    defaults.AddFargateTarget(stack, 'test-fargate-target', testListener, testService, undefined, {
        vpc: testVpc,
        protocol: elb.ApplicationProtocol.HTTP
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        DefaultActions: [
            {
                TargetGroupArn: {
                    Ref: "testfargatetargettg01FF5AA3"
                },
                Type: "forward"
            }
        ],
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::TargetGroup', {
        TargetType: "ip",
    });
});
test('Test add two fargate targets with rules', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = defaults.getTestVpc(stack);
    const testService = CreateTestFargateService(stack, 'test-service', testVpc);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testListener = CreateTestListener(stack, 'test-listener', testAlb);
    const pathPattern = '*admin*';
    defaults.AddFargateTarget(stack, 'test-fargate-target', testListener, testService, undefined, {
        vpc: testVpc,
        protocol: elb.ApplicationProtocol.HTTP
    });
    // This is the code we're testing
    const ruleProps = {
        conditions: [elb.ListenerCondition.pathPatterns([pathPattern])],
        priority: 10
    };
    defaults.AddFargateTarget(stack, 'test-second-fargate-target', testListener, testService, ruleProps, {
        vpc: testVpc,
        protocol: elb.ApplicationProtocol.HTTP
    });
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::TargetGroup', 2);
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::ListenerRule', {
        Conditions: [
            {
                Field: "path-pattern",
                PathPatternConfig: {
                    Values: [
                        pathPattern
                    ]
                }
            }
        ]
    });
});
test('Test adding a listener with defaults', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = defaults.getTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testCert = defaults.getFakeCertificate(stack, 'not-really-a-cert');
    const listener = defaults.AddListener(stack, 'test', testAlb, { certificates: [testCert] });
    //  Need to add a target because a listener is not allowed to exist without a target or action
    defaults.AddLambdaTarget(stack, 'dummy-target', listener, CreateTestFunction(stack, 'dummy-function'));
    // This should create 2 listeners, HTTPS plus redirect of HTTP
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTPS',
    });
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP',
    });
});
test('Test adding an HTTPS listener with no cert (error)', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = defaults.getTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const app = () => {
        defaults.AddListener(stack, 'test', testAlb, {});
    };
    expect(app).toThrowError('A listener using HTTPS protocol requires a certificate');
});
test('Test adding an HTTP listener with a cert (error)', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = defaults.getTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testCert = defaults.getFakeCertificate(stack, 'not-really-a-cert');
    const app = () => {
        defaults.AddListener(stack, 'test', testAlb, { protocol: 'HTTP', certificates: [testCert] });
    };
    expect(app).toThrowError('HTTP listeners cannot use a certificate');
});
test('Test adding a HTTP listener', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = defaults.getTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const listener = defaults.AddListener(stack, 'test', testAlb, { protocol: 'HTTP' });
    //  Need to add a target because a listener is not allowed to exist without a target or action
    defaults.AddLambdaTarget(stack, 'dummy-target', listener, CreateTestFunction(stack, 'dummy-function'));
    expect(stack).toHaveResourceLike('AWS::ElasticLoadBalancingV2::Listener', {
        Protocol: 'HTTP',
    });
    expect(stack).toCountResources('AWS::ElasticLoadBalancingV2::Listener', 1);
});
test('Test sending custom logging bucket props', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = defaults.getTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const listener = defaults.AddListener(stack, 'test', testAlb, { protocol: 'HTTP' });
    //  Need to add a target because a listener is not allowed to exist without a target or action
    defaults.AddLambdaTarget(stack, 'dummy-target', listener, CreateTestFunction(stack, 'dummy-function'));
});
test('Test GetActiveListener with 0 listeners', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = defaults.getTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const app = () => {
        defaults.GetActiveListener(testAlb.listeners);
    };
    expect(app).toThrowError('There are no listeners in the ALB');
});
test('Test GetActiveListener with 1 listener', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = defaults.getTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    defaults.AddListener(stack, 'test', testAlb, { protocol: 'HTTP' });
    const listener = defaults.GetActiveListener(testAlb.listeners);
    expect(listener.node.defaultChild.protocol).toBe('HTTP');
});
test('Test GetActiveListener with 2 listeners', () => {
    const stack = new core_1.Stack();
    // Set up test framework independent of our code for unit testing
    const testVpc = defaults.getTestVpc(stack);
    const testAlb = CreateTestLoadBalancer(stack, testVpc);
    const testCert = defaults.getFakeCertificate(stack, 'not-really-a-cert');
    defaults.AddListener(stack, 'test', testAlb, { certificates: [testCert] });
    const listener = defaults.GetActiveListener(testAlb.listeners);
    expect(listener.node.defaultChild.protocol).toBe('HTTPS');
});
test('Test use of certificateArns error', () => {
    const props = {
        listenerProps: {
            certificateArns: ['arn1'],
        }
    };
    const app = () => {
        defaults.CheckAlbProps(props);
    };
    expect(app).toThrowError("certificateArns is deprecated. Please supply certificates using props.listenerProps.certificates\n");
});
test('Test bad first listener error', () => {
    const props = {
        existingLoadBalancerObj: {
            listeners: [],
        }
    };
    const app = () => {
        defaults.CheckAlbProps(props);
    };
    expect(app).toThrowError("When adding the first listener and target to a load balancer, listenerProps must be specified and include at least a certificate or protocol: HTTP\n");
    const app2 = () => {
        defaults.CheckAlbProps({});
    };
    expect(app2).toThrowError("When adding the first listener and target to a load balancer, listenerProps must be specified and include at least a certificate or protocol: HTTP\n");
});
test('Test second target with no rules error', () => {
    const props = {
        existingLoadBalancerObj: {
            listeners: ['fake listener'],
        },
        existingVpc: { fake: 'vpc' }
    };
    const app = () => {
        defaults.CheckAlbProps(props);
    };
    expect(app).toThrowError("When adding a second target to an existing listener, there must be rules provided\n");
});
test('Test existing Load Balancer with no VPC provided error', () => {
    const props = {
        existingLoadBalancerObj: {
            name: 'placeholder',
            listeners: []
        }
    };
    const app = () => {
        defaults.CheckAlbProps(props);
    };
    expect(app).toThrowError("An existing ALB is already in a VPC, that VPC must be provided in props.existingVpc for the rest of the construct to use.\n");
});
test('Test sending listenerProps to existingListener error', () => {
    const props = {
        existingLoadBalancerObj: {
            listeners: ['placeholder']
        },
        listenerProps: { val: 'placeholder' }
    };
    const app = () => {
        defaults.CheckAlbProps(props);
    };
    expect(app).toThrowError("This load balancer already has a listener, listenerProps may not be specified\n");
});
test('Test sending VPC in loadBalancerProps error', () => {
    const props = {
        loadBalancerProps: {
            vpc: { val: 'placeholder' }
        }
    };
    const app = () => {
        defaults.CheckAlbProps(props);
    };
    expect(app).toThrowError('Specify any existing VPC at the construct level, not within loadBalancerProps.\n');
});
function CreateTestLoadBalancer(stack, vpc) {
    return new elb.ApplicationLoadBalancer(stack, 'load-balancer', {
        vpc,
        internetFacing: true,
        loadBalancerName: 'unique-name'
    });
}
function CreateTestFunction(stack, id) {
    return new lambda.Function(stack, id, {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: lambda.Runtime.NODEJS_14_X,
        handler: "index.handler",
    });
}
function CreateTestFargateService(stack, id, vpc) {
    const [svc] = defaults.CreateFargateService(stack, `${id}-fg-svc`, vpc, undefined, 'arn:aws:ecr:us-east-1:123456789012:repository/fake-repo', 'latest');
    return svc;
}
function CreateTestListener(stack, id, alb) {
    return new elb.ApplicationListener(stack, id, {
        loadBalancer: alb,
        protocol: elb.ApplicationProtocol.HTTP
    });
}
//# sourceMappingURL=data:application/json;base64,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