"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.AddAwsServiceEndpoint = exports.ServiceEndpointTypes = exports.buildVpc = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const security_group_helper_1 = require("./security-group-helper");
const utils_1 = require("./utils");
function buildVpc(scope, props) {
    if (props === null || props === void 0 ? void 0 : props.existingVpc) {
        return props === null || props === void 0 ? void 0 : props.existingVpc;
    }
    let cumulativeProps = props === null || props === void 0 ? void 0 : props.defaultVpcProps;
    cumulativeProps = utils_1.consolidateProps(cumulativeProps, props === null || props === void 0 ? void 0 : props.userVpcProps, props === null || props === void 0 ? void 0 : props.constructVpcProps);
    const vpc = new ec2.Vpc(scope, "Vpc", cumulativeProps);
    // Add VPC FlowLogs with the default setting of trafficType:ALL and destination: CloudWatch Logs
    const flowLog = vpc.addFlowLog("FlowLog");
    SuppressMapPublicIpWarnings(vpc);
    SuppressEncryptedLogWarnings(flowLog);
    return vpc;
}
exports.buildVpc = buildVpc;
var ServiceEndpointTypes;
(function (ServiceEndpointTypes) {
    ServiceEndpointTypes["DYNAMODB"] = "DDB";
    ServiceEndpointTypes["SNS"] = "SNS";
    ServiceEndpointTypes["SQS"] = "SQS";
    ServiceEndpointTypes["S3"] = "S3";
    ServiceEndpointTypes["STEP_FUNCTIONS"] = "STEP_FUNCTIONS";
    ServiceEndpointTypes["SAGEMAKER_RUNTIME"] = "SAGEMAKER_RUNTIME";
    ServiceEndpointTypes["SECRETS_MANAGER"] = "SECRETS_MANAGER";
    ServiceEndpointTypes["SSM"] = "SSM";
    ServiceEndpointTypes["ECR_API"] = "ECR_API";
    ServiceEndpointTypes["ECR_DKR"] = "ECR_DKR";
    ServiceEndpointTypes["EVENTS"] = "CLOUDWATCH_EVENTS";
})(ServiceEndpointTypes = exports.ServiceEndpointTypes || (exports.ServiceEndpointTypes = {}));
var EndpointTypes;
(function (EndpointTypes) {
    EndpointTypes["GATEWAY"] = "Gateway";
    EndpointTypes["INTERFACE"] = "Interface";
})(EndpointTypes || (EndpointTypes = {}));
const endpointSettings = [
    {
        endpointName: ServiceEndpointTypes.DYNAMODB,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.DYNAMODB,
    },
    {
        endpointName: ServiceEndpointTypes.S3,
        endpointType: EndpointTypes.GATEWAY,
        endpointGatewayService: ec2.GatewayVpcEndpointAwsService.S3,
    },
    {
        endpointName: ServiceEndpointTypes.STEP_FUNCTIONS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.STEP_FUNCTIONS,
    },
    {
        endpointName: ServiceEndpointTypes.SNS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SNS,
    },
    {
        endpointName: ServiceEndpointTypes.SQS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SQS,
    },
    {
        endpointName: ServiceEndpointTypes.SAGEMAKER_RUNTIME,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SAGEMAKER_RUNTIME,
    },
    {
        endpointName: ServiceEndpointTypes.SECRETS_MANAGER,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SECRETS_MANAGER,
    },
    {
        endpointName: ServiceEndpointTypes.SSM,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.SSM,
    },
    {
        endpointName: ServiceEndpointTypes.ECR_API,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.ECR
    },
    {
        endpointName: ServiceEndpointTypes.ECR_DKR,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.ECR_DOCKER
    },
    {
        endpointName: ServiceEndpointTypes.EVENTS,
        endpointType: EndpointTypes.INTERFACE,
        endpointInterfaceService: ec2.InterfaceVpcEndpointAwsService.CLOUDWATCH_EVENTS
    }
];
function AddAwsServiceEndpoint(scope, vpc, interfaceTag) {
    if (CheckIfEndpointAlreadyExists(vpc, interfaceTag)) {
        return;
    }
    const service = endpointSettings.find((endpoint) => endpoint.endpointName === interfaceTag);
    if (!service) {
        throw new Error("Unsupported Service sent to AddServiceEndpoint");
    }
    if (service.endpointType === EndpointTypes.GATEWAY) {
        AddGatewayEndpoint(vpc, service, interfaceTag);
    }
    if (service.endpointType === EndpointTypes.INTERFACE) {
        AddInterfaceEndpoint(scope, vpc, service, interfaceTag);
    }
    return;
}
exports.AddAwsServiceEndpoint = AddAwsServiceEndpoint;
function CheckIfEndpointAlreadyExists(vpc, interfaceTag) {
    return vpc.node.children.some((child) => child.node.id === interfaceTag);
}
function SuppressMapPublicIpWarnings(vpc) {
    // Add Cfn Nag suppression for PUBLIC subnets to suppress WARN W33: EC2 Subnet should not have MapPublicIpOnLaunch set to true
    vpc.publicSubnets.forEach((subnet) => {
        const cfnSubnet = subnet.node.defaultChild;
        utils_1.addCfnSuppressRules(cfnSubnet, [
            {
                id: 'W33',
                reason: 'Allow Public Subnets to have MapPublicIpOnLaunch set to true'
            }
        ]);
    });
}
function SuppressEncryptedLogWarnings(flowLog) {
    var _a;
    // Add Cfn Nag suppression for CloudWatchLogs LogGroups data is encrypted
    const cfnLogGroup = (_a = flowLog.logGroup) === null || _a === void 0 ? void 0 : _a.node.defaultChild;
    utils_1.addCfnSuppressRules(cfnLogGroup, [
        {
            id: 'W84',
            reason: 'By default CloudWatchLogs LogGroups data is encrypted using the CloudWatch server-side encryption keys (AWS Managed Keys)'
        }
    ]);
}
function AddInterfaceEndpoint(scope, vpc, service, interfaceTag) {
    const endpointDefaultSecurityGroup = security_group_helper_1.buildSecurityGroup(scope, `${scope.node.id}-${service.endpointName}`, {
        vpc,
        allowAllOutbound: true,
    }, [{ peer: ec2.Peer.ipv4(vpc.vpcCidrBlock), connection: ec2.Port.tcp(443) }], []);
    vpc.addInterfaceEndpoint(interfaceTag, {
        service: service.endpointInterfaceService,
        securityGroups: [endpointDefaultSecurityGroup],
    });
}
function AddGatewayEndpoint(vpc, service, interfaceTag) {
    vpc.addGatewayEndpoint(interfaceTag, {
        service: service.endpointGatewayService,
    });
}
//# sourceMappingURL=data:application/json;base64,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