"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildStepFunctionCWAlarms = exports.buildStateMachine = void 0;
const cdk = require("@aws-cdk/core");
const smDefaults = require("./step-function-defaults");
const sfn = require("@aws-cdk/aws-stepfunctions");
const utils_1 = require("./utils");
const iam = require("@aws-cdk/aws-iam");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
/**
 * Builds and returns a StateMachine.
 * @param scope - the construct to which the StateMachine should be attached to.
 * @param stateMachineProps - user-specified properties to override the default properties.
 */
function buildStateMachine(scope, stateMachineProps, logGroupProps) {
    var _a, _b;
    let logGroup;
    let _smProps;
    // If they sent a logGroup in stateMachineProps
    if ((_a = stateMachineProps.logs) === null || _a === void 0 ? void 0 : _a.destination) {
        logGroup = (_b = stateMachineProps.logs) === null || _b === void 0 ? void 0 : _b.destination;
        _smProps = stateMachineProps;
    }
    else {
        // Three possibilities
        // 1) logGroupProps not provided - create logGroupProps with just logGroupName
        // 2) logGroupProps provided with no logGroupName - override logGroupProps.logGroupName
        // 3) logGroupProps provided with logGroupName - pass unaltered logGroupProps
        let consolidatedLogGroupProps = logGroupProps;
        if (!consolidatedLogGroupProps) {
            consolidatedLogGroupProps = {};
        }
        if (!(consolidatedLogGroupProps === null || consolidatedLogGroupProps === void 0 ? void 0 : consolidatedLogGroupProps.logGroupName)) {
            const logGroupPrefix = '/aws/vendedlogs/states/';
            const maxResourceNameLength = 255 - logGroupPrefix.length;
            const nameParts = [
                cdk.Stack.of(scope).stackName,
                scope.node.id,
                'StateMachineLog' // Literal string for log group name portion
            ];
            const logGroupName = logGroupPrefix + utils_1.generateResourceName(nameParts, maxResourceNameLength);
            consolidatedLogGroupProps = utils_1.overrideProps(consolidatedLogGroupProps, { logGroupName });
        }
        // Create new Cloudwatch log group for Step function State Machine
        logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'StateMachineLogGroup', consolidatedLogGroupProps);
        // Override the defaults with the user provided props
        _smProps = utils_1.overrideProps(smDefaults.DefaultStateMachineProps(logGroup), stateMachineProps);
    }
    // Override the Cloudwatch permissions to make it more fine grained
    const _sm = new sfn.StateMachine(scope, 'StateMachine', _smProps);
    const role = _sm.node.findChild('Role');
    const cfnDefaultPolicy = role.node.findChild('DefaultPolicy').node.defaultChild;
    // Reduce the scope of actions for the existing DefaultPolicy
    cfnDefaultPolicy.addPropertyOverride('PolicyDocument.Statement.0.Action', [
        "logs:CreateLogDelivery",
        'logs:GetLogDelivery',
        'logs:UpdateLogDelivery',
        'logs:DeleteLogDelivery',
        'logs:ListLogDeliveries'
    ]);
    // Override Cfn Nag warning W12: IAM policy should not allow * resource
    utils_1.addCfnSuppressRules(cfnDefaultPolicy, [
        {
            id: 'W12',
            reason: `The 'LogDelivery' actions do not support resource-level authorizations`
        }
    ]);
    // Add a new policy with logging permissions for the given cloudwatch log group
    _sm.addToRolePolicy(new iam.PolicyStatement({
        actions: [
            'logs:PutResourcePolicy',
            'logs:DescribeResourcePolicies',
            'logs:DescribeLogGroups'
        ],
        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
    }));
    return [_sm, logGroup];
}
exports.buildStateMachine = buildStateMachine;
function buildStepFunctionCWAlarms(scope, sm) {
    // Setup CW Alarms for State Machine
    const alarms = new Array();
    // Sum of number of executions that failed is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionFailedAlarm', {
        metric: sm.metricFailed({
            statistic: 'Sum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that failed exceeded the threshold of 1. '
    }));
    // Sum of number of executions that failed maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionThrottledAlarm', {
        metric: sm.metricThrottled({
            statistic: 'Sum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that throttled exceeded the threshold of 1. '
    }));
    // Number of executions that aborted maximum is >= 1 for 5 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'ExecutionAbortedAlarm', {
        metric: sm.metricAborted({
            statistic: 'Maximum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Alarm for the number of executions that aborted exceeded the threshold of 1. '
    }));
    return alarms;
}
exports.buildStepFunctionCWAlarms = buildStepFunctionCWAlarms;
//# sourceMappingURL=data:application/json;base64,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