"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildTopic = void 0;
// Imports
const sns = require("@aws-cdk/aws-sns");
const kms = require("@aws-cdk/aws-kms");
const sns_defaults_1 = require("./sns-defaults");
const kms_helper_1 = require("./kms-helper");
const utils_1 = require("./utils");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
function applySecureTopicPolicy(topic) {
    // Apply topic policy to enforce only the topic owner can publish and subscribe to this topic
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'TopicOwnerOnlyAccess',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AccountPrincipal(core_1.Stack.of(topic).account)],
        effect: aws_iam_1.Effect.ALLOW,
        conditions: {
            StringEquals: {
                "AWS:SourceOwner": core_1.Stack.of(topic).account
            }
        }
    }));
    // Apply Topic policy to enforce encryption of data in transit
    topic.addToResourcePolicy(new aws_iam_1.PolicyStatement({
        sid: 'HttpsOnly',
        resources: [
            `${topic.topicArn}`
        ],
        actions: [
            "SNS:Publish",
            "SNS:RemovePermission",
            "SNS:SetTopicAttributes",
            "SNS:DeleteTopic",
            "SNS:ListSubscriptionsByTopic",
            "SNS:GetTopicAttributes",
            "SNS:Receive",
            "SNS:AddPermission",
            "SNS:Subscribe"
        ],
        principals: [new aws_iam_1.AnyPrincipal()],
        effect: aws_iam_1.Effect.DENY,
        conditions: {
            Bool: {
                'aws:SecureTransport': 'false'
            }
        }
    }));
}
function buildTopic(scope, props) {
    if (!props.existingTopicObj) {
        // Setup the topic properties
        const snsTopicProps = utils_1.consolidateProps(sns_defaults_1.DefaultSnsTopicProps, props.topicProps);
        // Set encryption properties
        if (props.enableEncryptionWithCustomerManagedKey === undefined || props.enableEncryptionWithCustomerManagedKey === false) {
            // Retrieve SNS managed key to encrypt the SNS Topic
            const awsManagedKey = kms.Alias.fromAliasName(scope, 'aws-managed-key', 'alias/aws/sns');
            snsTopicProps.masterKey = awsManagedKey;
        }
        else {
            // Use the imported Customer Managed KMS key
            if (props.encryptionKey) {
                snsTopicProps.masterKey = props.encryptionKey;
            }
            else {
                // Create a new Customer Managed KMS key
                snsTopicProps.masterKey = kms_helper_1.buildEncryptionKey(scope, props.encryptionKeyProps);
            }
        }
        // Create the SNS Topic
        const topic = new sns.Topic(scope, 'SnsTopic', snsTopicProps);
        applySecureTopicPolicy(topic);
        return [topic, snsTopicProps.masterKey];
    }
    else {
        return [props.existingTopicObj];
    }
}
exports.buildTopic = buildTopic;
//# sourceMappingURL=data:application/json;base64,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