"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CheckFargateProps = exports.CreateFargateService = void 0;
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const ecr = require("@aws-cdk/aws-ecr");
const defaults = require("..");
const __1 = require("..");
function CreateFargateService(scope, id, constructVpc, clientClusterProps, ecrRepositoryArn, ecrImageVersion, clientFargateTaskDefinitionProps, clientContainerDefinitionProps, clientFargateServiceProps) {
    defaults.AddAwsServiceEndpoint(scope, constructVpc, defaults.ServiceEndpointTypes.ECR_API);
    defaults.AddAwsServiceEndpoint(scope, constructVpc, defaults.ServiceEndpointTypes.ECR_DKR);
    defaults.AddAwsServiceEndpoint(scope, constructVpc, defaults.ServiceEndpointTypes.S3);
    const constructContainerDefintionProps = {};
    const constructFargateServiceDefinitionProps = {};
    if (!(clientFargateServiceProps === null || clientFargateServiceProps === void 0 ? void 0 : clientFargateServiceProps.cluster)) {
        // Construct Fargate Service
        constructFargateServiceDefinitionProps.cluster = CreateCluster(scope, `${id}-cluster`, constructVpc, clientClusterProps);
    }
    // Set up the Fargate service
    if (!(clientContainerDefinitionProps === null || clientContainerDefinitionProps === void 0 ? void 0 : clientContainerDefinitionProps.image)) {
        constructContainerDefintionProps.image = CreateImage(scope, id, ecrRepositoryArn, ecrImageVersion);
    }
    // Create the Fargate Service
    let newContainerDefinition;
    [constructFargateServiceDefinitionProps.taskDefinition, newContainerDefinition] = CreateTaskDefinition(scope, id, clientFargateTaskDefinitionProps, clientContainerDefinitionProps, constructContainerDefintionProps);
    if (!(clientFargateServiceProps === null || clientFargateServiceProps === void 0 ? void 0 : clientFargateServiceProps.vpcSubnets)) {
        if (constructVpc.isolatedSubnets.length) {
            constructFargateServiceDefinitionProps.vpcSubnets = {
                subnets: constructVpc.isolatedSubnets,
            };
        }
        else {
            constructFargateServiceDefinitionProps.vpcSubnets = {
                subnets: constructVpc.privateSubnets,
            };
        }
    }
    let defaultFargateServiceProps;
    if (!(clientFargateServiceProps === null || clientFargateServiceProps === void 0 ? void 0 : clientFargateServiceProps.securityGroups)) {
        const serviceSecurityGroup = new ec2.SecurityGroup(scope, `${id}-sg`, {
            allowAllOutbound: true,
            disableInlineRules: false,
            vpc: constructVpc,
            // We add a description here so that this SG can be easily identified in tests
            description: 'Construct created security group'
        });
        defaultFargateServiceProps = __1.overrideProps(defaults.DefaultFargateServiceProps(), { securityGroups: [serviceSecurityGroup] });
        defaults.addCfnSuppressRules(serviceSecurityGroup, [
            {
                id: 'W5',
                reason: 'Egress of 0.0.0.0/0 is default and generally considered OK',
            },
            {
                id: 'W40',
                reason: 'Egress IPProtocol of -1 is default and generally considered OK',
            }
        ]);
    }
    else {
        defaultFargateServiceProps = defaults.DefaultFargateServiceProps();
    }
    const fargateServiceProps = defaults.consolidateProps(defaultFargateServiceProps, clientFargateServiceProps, constructFargateServiceDefinitionProps);
    const newService = new ecs.FargateService(scope, `${id}-service`, fargateServiceProps);
    return [newService, newContainerDefinition];
}
exports.CreateFargateService = CreateFargateService;
function CreateCluster(scope, id, constructVpc, clientClusterProps) {
    const constructProps = { vpc: constructVpc };
    return new ecs.Cluster(scope, id, defaults.consolidateProps(defaults.DefaultClusterProps(), clientClusterProps, constructProps));
}
function CreateImage(scope, id, repositoryArn, imageVersion) {
    if (repositoryArn) {
        return ecs.ContainerImage.fromEcrRepository(ecr.Repository.fromRepositoryArn(scope, `${id}-container`, repositoryArn), imageVersion || "latest");
    }
    else {
        throw new Error("Not Implemented - image without repo name and version");
    }
}
function CreateTaskDefinition(scope, id, clientFargateTaskDefinitionProps, clientContainerDefinitionProps, constructContainerDefinitionProps) {
    const taskDefinitionProps = defaults.consolidateProps(defaults.DefaultFargateTaskDefinitionProps(), clientFargateTaskDefinitionProps);
    const taskDefinition = new ecs.FargateTaskDefinition(scope, `${id}-taskdef`, taskDefinitionProps);
    const defaultContainerDefinitionProps = defaults.consolidateProps(defaults.DefaultContainerDefinitionProps(), {
        containerName: `${id}-container`,
    });
    const containerDefinitionProps = defaults.consolidateProps(defaultContainerDefinitionProps, clientContainerDefinitionProps, constructContainerDefinitionProps);
    const containerDefinition = taskDefinition.addContainer(`${id}-container`, containerDefinitionProps);
    return [taskDefinition, containerDefinition];
}
function CheckFargateProps(props) {
    var _a, _b, _c, _d;
    let errorMessages = "";
    let errorFound = false;
    if (props.existingFargateServiceObject &&
        (props.existingImageObject ||
            props.ecrImageVersion ||
            props.containerDefinitionProps ||
            props.fargateTaskDefinitionProps ||
            props.ecrRepositoryArn ||
            props.fargateServiceProps ||
            props.clusterProps)) {
        errorFound = true;
        errorMessages +=
            "If you provide an existingFargateServiceObject, you cannot provide any props defining a new service\n";
    }
    if (props.existingImageObject &&
        (props.ecrImageVersion || props.ecrRepositoryArn)) {
        errorFound = true;
        errorMessages +=
            "If you provide an existingImageObject then you cannot provide an ecrRepositoryArn nor ecrImageVersion\n";
    }
    // Confirm no network information in Target Group Props
    if ((_a = props.targetGroupProps) === null || _a === void 0 ? void 0 : _a.vpc) {
        errorFound = true;
        errorMessages +=
            "Provide all VPC info at Construct level, not within targetGroupProps\n";
    }
    if ((_b = props.fargateServiceProps) === null || _b === void 0 ? void 0 : _b.taskDefinition) {
        errorFound = true;
        errorMessages +=
            "The construct cannot accept an existing task definition in fargateServiceProps\n";
    }
    if (((_c = props.fargateServiceProps) === null || _c === void 0 ? void 0 : _c.cluster) && props.clusterProps) {
        errorFound = true;
        errorMessages +=
            "If you provide a cluster in fargateServiceProps then you cannot provide clusterProps\n";
    }
    if ((_d = props.clusterProps) === null || _d === void 0 ? void 0 : _d.vpc) {
        errorFound = true;
        errorMessages +=
            "All services in the construct use the construct VPC, you cannot specify a VPC in clusterProps\n";
    }
    if ((props.existingFargateServiceObject ||
        props.existingContainerDefinitionObject) &&
        (!props.existingFargateServiceObject ||
            !props.existingContainerDefinitionObject ||
            !props.existingVpc)) {
        errorFound = true;
        errorMessages +=
            "If an existing Service is indicated by supplying either existingFargateServiceObject or existingContainerDefinitionObject, then existingFargateServiceObject, existingContainerDefinitionObject, and existingVpc must all be provided\n";
    }
    if (errorFound) {
        throw new Error(errorMessages);
    }
}
exports.CheckFargateProps = CheckFargateProps;
//# sourceMappingURL=data:application/json;base64,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