"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildElasticSearchCWAlarms = exports.buildElasticSearch = void 0;
const elasticsearch = require("@aws-cdk/aws-elasticsearch");
const elasticsearch_defaults_1 = require("./elasticsearch-defaults");
const utils_1 = require("./utils");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
function buildElasticSearch(scope, domainName, options, cfnDomainProps) {
    // Setup the IAM Role & policy for ES to configure Cognito User pool and Identity pool
    const cognitoKibanaConfigureRole = new iam.Role(scope, 'CognitoKibanaConfigureRole', {
        assumedBy: new iam.ServicePrincipal('es.amazonaws.com')
    });
    const cognitoKibanaConfigureRolePolicy = new iam.Policy(scope, 'CognitoKibanaConfigureRolePolicy', {
        statements: [
            new iam.PolicyStatement({
                actions: [
                    "cognito-idp:DescribeUserPool",
                    "cognito-idp:CreateUserPoolClient",
                    "cognito-idp:DeleteUserPoolClient",
                    "cognito-idp:DescribeUserPoolClient",
                    "cognito-idp:AdminInitiateAuth",
                    "cognito-idp:AdminUserGlobalSignOut",
                    "cognito-idp:ListUserPoolClients",
                    "cognito-identity:DescribeIdentityPool",
                    "cognito-identity:UpdateIdentityPool",
                    "cognito-identity:SetIdentityPoolRoles",
                    "cognito-identity:GetIdentityPoolRoles",
                    "es:UpdateElasticsearchDomainConfig"
                ],
                resources: [
                    options.userpool.userPoolArn,
                    `arn:aws:cognito-identity:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:identitypool/${options.identitypool.ref}`,
                    `arn:aws:es:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:domain/${domainName}`
                ]
            }),
            new iam.PolicyStatement({
                actions: [
                    "iam:PassRole"
                ],
                conditions: {
                    StringLike: { 'iam:PassedToService': 'cognito-identity.amazonaws.com' }
                },
                resources: [
                    cognitoKibanaConfigureRole.roleArn
                ]
            })
        ]
    });
    cognitoKibanaConfigureRolePolicy.attachToRole(cognitoKibanaConfigureRole);
    let _cfnDomainProps = elasticsearch_defaults_1.DefaultCfnDomainProps(domainName, cognitoKibanaConfigureRole, options);
    _cfnDomainProps = utils_1.consolidateProps(_cfnDomainProps, cfnDomainProps);
    const esDomain = new elasticsearch.CfnDomain(scope, "ElasticsearchDomain", _cfnDomainProps);
    utils_1.addCfnSuppressRules(esDomain, [
        {
            id: "W28",
            reason: `The ES Domain is passed dynamically as as parameter and explicitly specified to ensure that IAM policies are configured to lockdown access to this specific ES instance only`,
        },
        {
            id: "W90",
            reason: `This is not a rule for the general case, just for specific use cases/industries`,
        },
    ]);
    return [esDomain, cognitoKibanaConfigureRole];
}
exports.buildElasticSearch = buildElasticSearch;
function buildElasticSearchCWAlarms(scope) {
    // Setup CW Alarms for ES
    const alarms = new Array();
    // ClusterStatus.red maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusRedAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.red',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one primary shard and its replicas are not allocated to a node. '
    }));
    // ClusterStatus.yellow maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'StatusYellowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterStatus.yellow',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'At least one replica shard is not allocated to a node.'
    }));
    // FreeStorageSpace minimum is <= 20480 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'FreeStorageSpaceTooLowAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'FreeStorageSpace',
            statistic: 'Minimum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 20000,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'A node in your cluster is down to 20 GiB of free storage space.'
    }));
    // ClusterIndexWritesBlocked is >= 1 for 5 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'IndexWritesBlockedTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'ClusterIndexWritesBlocked',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(300),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Your cluster is blocking write requests.'
    }));
    // AutomatedSnapshotFailure maximum is >= 1 for 1 minute, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'AutomatedSnapshotFailureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'AutomatedSnapshotFailure',
            statistic: 'Maximum',
            period: cdk.Duration.seconds(60),
        }),
        threshold: 1,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'An automated snapshot failed. This failure is often the result of a red cluster health status.'
    }));
    // CPUUtilization maximum is >= 80% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'CPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'CPUUtilization',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 80,
        evaluationPeriods: 3,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: '100% CPU utilization is not uncommon, but sustained high usage is problematic. Consider using larger instance types or adding instances.'
    }));
    // JVMMemoryPressure maximum is >= 80% for 5 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'JVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'JVMMemoryPressure',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 80,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    // MasterCPUUtilization maximum is >= 50% for 15 minutes, 3 consecutive times
    alarms.push(new cloudwatch.Alarm(scope, 'MasterCPUUtilizationTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterCPUUtilization',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 50,
        evaluationPeriods: 3,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average CPU utilization over last 45 minutes too high. Consider using larger instance types for your dedicated master nodes.'
    }));
    // MasterJVMMemoryPressure maximum is >= 80% for 15 minutes, 1 consecutive time
    alarms.push(new cloudwatch.Alarm(scope, 'MasterJVMMemoryPressureTooHighAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/ES',
            metricName: 'MasterJVMMemoryPressure',
            statistic: 'Average',
            period: cdk.Duration.seconds(900),
        }),
        threshold: 50,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Average JVM memory pressure over last 15 minutes too high. Consider scaling vertically.'
    }));
    return alarms;
}
exports.buildElasticSearchCWAlarms = buildElasticSearchCWAlarms;
//# sourceMappingURL=data:application/json;base64,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