"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.getCachePort = exports.createCacheSubnetGroup = exports.obtainMemcachedCluster = void 0;
const cache = require("@aws-cdk/aws-elasticache");
const elasticache_defaults_1 = require("./elasticache-defaults");
const utils_1 = require("./utils");
function obtainMemcachedCluster(scope, id, props) {
    var _a;
    if (props.existingCache) {
        (_a = props.existingCache.vpcSecurityGroupIds) === null || _a === void 0 ? void 0 : _a.push(props.cacheSecurityGroupId);
        return props.existingCache;
    }
    else {
        if (!props.cachePort) {
            throw Error('props.cachePort required for new caches');
        }
        // Create the subnet group from all the isolated subnets in the VPC
        const subnetGroup = createCacheSubnetGroup(scope, props.vpc, id);
        const defaultProps = elasticache_defaults_1.GetMemcachedDefaults(id, props.cachePort);
        const requiredConstructProps = {
            vpcSecurityGroupIds: [props.cacheSecurityGroupId],
            cacheSubnetGroupName: subnetGroup.cacheSubnetGroupName,
        };
        const consolidatedProps = utils_1.consolidateProps(defaultProps, props.cacheProps, requiredConstructProps, true);
        const newCache = new cache.CfnCacheCluster(scope, `${id}-cluster`, consolidatedProps);
        newCache.addDependsOn(subnetGroup);
        return newCache;
    }
}
exports.obtainMemcachedCluster = obtainMemcachedCluster;
function createCacheSubnetGroup(construct, vpc, id) {
    // Memcached has no auth, all access control is
    // network based, so, at least initially, we will
    // only launch it in isolated subnets.
    const subnetIds = [];
    vpc.isolatedSubnets.forEach((subnet) => {
        subnetIds.push(subnet.subnetId);
    });
    return new cache.CfnSubnetGroup(construct, `ec-subnetgroup-${id}`, {
        description: "Solutions Constructs generated Cache Subnet Group",
        subnetIds,
        cacheSubnetGroupName: `${id}-subnet-group`,
    });
}
exports.createCacheSubnetGroup = createCacheSubnetGroup;
function getCachePort(clientCacheProps, existingCache) {
    if (existingCache) {
        return existingCache.attrConfigurationEndpointPort;
    }
    else if (clientCacheProps === null || clientCacheProps === void 0 ? void 0 : clientCacheProps.port) {
        return clientCacheProps.port;
    }
    else {
        return elasticache_defaults_1.GetDefaultCachePort();
    }
}
exports.getCachePort = getCachePort;
//# sourceMappingURL=data:application/json;base64,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