"use strict";
/**
 *  Copyright 2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addProxyMethodToApiResource = exports.RegionalRestApi = exports.GlobalRestApi = exports.RegionalLambdaRestApi = exports.GlobalLambdaRestApi = void 0;
const cdk = require("@aws-cdk/core");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const apiDefaults = require("./apigateway-defaults");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
const utils_1 = require("./utils");
/**
 * Create and configures access logging for API Gateway resources.
 * @param scope - the construct to which the access logging capabilities should be attached to.
 * @param _api - an existing api.RestApi or api.LambdaRestApi.
 */
function configureCloudwatchRoleForApi(scope, _api) {
    var _a;
    // Setup the IAM Role for API Gateway CloudWatch access
    const restApiCloudwatchRole = new iam.Role(scope, 'LambdaRestApiCloudWatchRole', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        inlinePolicies: {
            LambdaRestApiCloudWatchRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:DescribeLogGroups',
                            'logs:DescribeLogStreams',
                            'logs:PutLogEvents',
                            'logs:GetLogEvents',
                            'logs:FilterLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
                    })]
            })
        }
    });
    // Create and configure AWS::ApiGateway::Account with CloudWatch Role for ApiGateway
    const CfnApi = _api.node.findChild('Resource');
    const cfnAccount = new api.CfnAccount(scope, 'LambdaRestApiAccount', {
        cloudWatchRoleArn: restApiCloudwatchRole.roleArn
    });
    cfnAccount.addDependsOn(CfnApi);
    // Suppress Cfn Nag warning for APIG
    const deployment = (_a = _api.latestDeployment) === null || _a === void 0 ? void 0 : _a.node.findChild('Resource');
    utils_1.addCfnSuppressRules(deployment, [
        {
            id: 'W45',
            reason: `ApiGateway has AccessLogging enabled in AWS::ApiGateway::Stage resource, but cfn_nag checkes for it in AWS::ApiGateway::Deployment resource`
        }
    ]);
    // Return the CW Role
    return restApiCloudwatchRole;
}
/**
 * Creates and configures an api.LambdaRestApi.
 * @param scope - the construct to which the LambdaRestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the LambdaRestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureLambdaRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API object
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.consolidateProps(defaultApiGatewayProps, apiGatewayProps, { cloudWatchRole: false });
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    let cwRole;
    if ((apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.cloudWatchRole) !== false) {
        cwRole = configureCloudwatchRoleForApi(scope, _api);
    }
    // Configure Usage Plan
    const usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    const plan = _api.addUsagePlan('UsagePlan', usagePlanProps);
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        // Configure Usage Plan with API Key
        const key = _api.addApiKey('ApiKey');
        plan.addApiKey(key);
    }
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Creates and configures an api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the RestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API
    let _api;
    const _apiGatewayProps = utils_1.consolidateProps(defaultApiGatewayProps, apiGatewayProps, { cloudWatchRole: false });
    _api = new api.RestApi(scope, 'RestApi', _apiGatewayProps);
    let cwRole;
    // Configure API access logging
    if ((apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.cloudWatchRole) !== false) {
        cwRole = configureCloudwatchRoleForApi(scope, _api);
    }
    // Configure Usage Plan
    const usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    const plan = _api.addUsagePlan('UsagePlan', usagePlanProps);
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        // Configure Usage Plan with API Key
        const key = _api.addApiKey('ApiKey');
        plan.addApiKey(key);
    }
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Builds and returns a global api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalLambdaRestApi = GlobalLambdaRestApi;
/**
 * Builds and returns a regional api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalLambdaRestApi = RegionalLambdaRestApi;
/**
 * Builds and returns a standard api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalRestApi = GlobalRestApi;
/**
 * Builds and returns a Regional api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalRestApi = RegionalRestApi;
function addProxyMethodToApiResource(params) {
    let baseProps = {
        service: params.service,
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            credentialsRole: params.apiGatewayRole,
            requestParameters: {
                "integration.request.header.Content-Type": params.contentType ? params.contentType : "'application/json'"
            },
            requestTemplates: {
                "application/json": params.requestTemplate
            },
            integrationResponses: [
                {
                    statusCode: "200"
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        }
    };
    let extraProps;
    if (params.action) {
        extraProps = {
            action: params.action
        };
    }
    else if (params.path) {
        extraProps = {
            path: params.path
        };
    }
    else {
        throw Error('Either action or path is required');
    }
    // Setup the API Gateway AWS Integration
    baseProps = Object.assign(baseProps, extraProps);
    let apiGatewayIntegration;
    const newProps = utils_1.consolidateProps(baseProps, params.awsIntegrationProps);
    apiGatewayIntegration = new api.AwsIntegration(newProps);
    const defaultMethodOptions = {
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ],
        requestValidator: params.requestValidator,
        requestModels: params.requestModel
    };
    let apiMethod;
    // Setup the API Gateway method
    const overridenProps = utils_1.consolidateProps(defaultMethodOptions, params.methodOptions);
    apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, overridenProps);
    return apiMethod;
}
exports.addProxyMethodToApiResource = addProxyMethodToApiResource;
//# sourceMappingURL=data:application/json;base64,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