"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const s3 = require("../../s3");
const constants = require("../shared/constants");
const language_1 = require("../shared/language");
const transliterator_1 = require("./transliterator");
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e;
        super(scope, id);
        const repository = props.vpcEndpoints
            ? (_a = props.codeArtifact) === null || _a === void 0 ? void 0 : _a.throughVpcEndpoint(props.vpcEndpoints.codeArtifactApi, props.vpcEndpoints.codeArtifact) : props.codeArtifact;
        const bucket = props.vpcEndpoints
            ? s3.throughVpcEndpoint(props.bucket, props.vpcEndpoints.s3)
            : props.bucket;
        const environment = {
            // temporaty hack to generate construct-hub compliant markdown.
            // see https://github.com/cdklabs/jsii-docgen/blob/master/src/docgen/render/markdown.ts#L172
            HEADER_SPAN: 'true',
            // Set embedded metrics format environment to "Local", to have a consistent experience.
            AWS_EMF_ENVIRONMENT: 'Local',
        };
        if (props.vpcEndpoints) {
            // Those are returned as an array of HOSTED_ZONE_ID:DNS_NAME... We care
            // only about the DNS_NAME of the first entry in that array (which is
            // the AZ-agnostic DNS name).
            environment.CODE_ARTIFACT_API_ENDPOINT = core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, props.vpcEndpoints.codeArtifactApi.vpcEndpointDnsEntries)));
        }
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.repositoryNpmEndpoint;
        }
        this.logGroup = new aws_logs_1.LogGroup(this, 'LogGroup', { retention: props.logRetention });
        this.containerDefinition = new transliterator_1.Transliterator(this, 'Resource', {
            environment,
            logging: aws_ecs_1.LogDrivers.awsLogs({ logGroup: this.logGroup, streamPrefix: 'transliterator' }),
            taskDefinition: new aws_ecs_1.FargateTaskDefinition(this, 'TaskDefinition', {
                cpu: 4096,
                memoryLimitMiB: 8192,
            }),
        });
        repository === null || repository === void 0 ? void 0 : repository.grantReadFromRepository(this.taskDefinition.taskRole);
        // The task handler reads & writes to this bucket.
        bucket.grantRead(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.ASSEMBLY_KEY_SUFFIX}`);
        bucket.grantRead(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.PACKAGE_KEY_SUFFIX}`);
        for (const language of language_1.DocumentationLanguage.ALL) {
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}${constants.NOT_SUPPORTED_SUFFIX}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}${constants.NOT_SUPPORTED_SUFFIX}`);
        }
        const executionRole = this.taskDefinition.obtainExecutionRole();
        (_b = props.vpcEndpoints) === null || _b === void 0 ? void 0 : _b.ecrApi.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ecr:GetAuthorizationToken',
            ],
            resources: ['*'],
            principals: [executionRole],
            sid: 'Allow-ECR-ReadOnly',
        }));
        (_c = props.vpcEndpoints) === null || _c === void 0 ? void 0 : _c.ecr.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ecr:BatchCheckLayerAvailability',
                'ecr:GetDownloadUrlForLayer',
                'ecr:BatchGetImage',
            ],
            // We cannot get the ECR repository info from an asset... So scoping down to same-account repositories instead...
            resources: [core_1.Stack.of(this).formatArn({ service: 'ecr', resource: 'repository', arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME, resourceName: '*' })],
            principals: [executionRole],
            sid: 'Allow-ECR-ReadOnly',
        }));
        (_d = props.vpcEndpoints) === null || _d === void 0 ? void 0 : _d.cloudWatchLogs.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [
                core_1.Stack.of(this).formatArn({ service: 'logs', resource: 'log-group', arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME, resourceName: this.logGroup.logGroupName }),
                core_1.Stack.of(this).formatArn({ service: 'logs', resource: 'log-group', arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME, resourceName: `${this.logGroup.logGroupName}:log-stream:*` }),
            ],
            principals: [executionRole],
            sid: 'Allow-Logging',
        }));
        (_e = props.vpcEndpoints) === null || _e === void 0 ? void 0 : _e.stepFunctions.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'states:SendTaskFailure',
                'states:SendTaskHeartbeat',
                'states:SendTaskSuccess',
            ],
            resources: ['*'],
            principals: [this.taskDefinition.taskRole],
            sid: 'Allow-StepFunctions-Callbacks',
        }));
    }
    get taskDefinition() {
        return this.containerDefinition.taskDefinition;
    }
    createEcsRunTask(scope, id, opts) {
        return new aws_stepfunctions_tasks_1.EcsRunTask(scope, id, {
            // The container sends heartbeats every minute, but when the runloop will
            // actually get to submitting it is fairly dependent on the async
            // workload's nature... so we don't rely on it all too strongly, and
            // default to a 5 minutes timeout here as a minimal protection. Options
            // may override this liberally if they know better.
            heartbeat: core_1.Duration.minutes(5),
            ...opts,
            containerOverrides: [{
                    containerDefinition: this.containerDefinition,
                    command: aws_stepfunctions_1.JsonPath.listAt('$'),
                    environment: [
                        { name: 'SFN_TASK_TOKEN', value: aws_stepfunctions_1.JsonPath.taskToken },
                    ],
                }],
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            launchTarget: new aws_stepfunctions_tasks_1.EcsFargateLaunchTarget({ platformVersion: aws_ecs_1.FargatePlatformVersion.VERSION1_4 }),
            subnets: opts.vpcSubnets,
            taskDefinition: this.taskDefinition,
        });
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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