"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const catalog_builder_1 = require("../catalog-builder");
const constants_1 = require("../shared/constants");
const transliterator_1 = require("../transliterator");
const redrive_state_machine_1 = require("./redrive-state-machine");
/**
 * This retry policy is used for all items in the state machine and allows ample
 * retry attempts in order to avoid having to implement a custom backpressure
 * handling mehanism.
 *
 * This is meant as a stop-gap until we can implement a more resilient system,
 * which likely will involve more SQS queues, but will probably need to be
 * throughoutly vetted before it is rolled out everywhere.
 *
 * After 30 attempts, given the parameters, the last attempt will wait just
 * under 16 minutes, which should be enough for currently running Lambda
 * functions to complete (or time out after 15 minutes). The total time spent
 * waiting between retries by this time is just over 3 hours. This is a lot of
 * time, but in extreme burst situations (i.e: reprocessing everything), this
 * is actually a good thing.
 */
const THROTTLE_RETRY_POLICY = { backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 };
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            label: 'Dead-Letter Queue not empty',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).next(new aws_stepfunctions_1.Succeed(this, 'Sent to DLQ'));
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props);
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder.function,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" throttled', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['Lambda.TooManyRequestsException'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" failure', {
            parameters: { 'error.$': 'States.StringToJson($.Cause)' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.TaskFailed'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" fault', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.ALL'] });
        this.ecsCluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            containerInsights: true,
            enableFargateCapacityProviders: true,
            vpc: props.vpc,
        });
        this.transliterator = new transliterator_1.Transliterator(this, 'Transliterator', props);
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        })
            .next(new aws_stepfunctions_1.Pass(this, 'Prepare doc-gen ECS Command', {
            parameters: { 'command.$': 'States.Array(States.JsonToString($))' },
            resultPath: '$.docGen',
        }))
            .next(this.transliterator.createEcsRunTask(this, 'Generate docs', {
            cluster: this.ecsCluster,
            inputPath: '$.docGen.command',
            resultPath: '$.docGenOutput',
            // Expect this to complete within one hour
            timeout: core_1.Duration.hours(1),
            vpcSubnets: props.vpcSubnets,
        })
            // Do not retry NoSpaceLeftOnDevice errors, these are typically not transient.
            .addRetry({ errors: ['jsii-docgen.NoSpaceLeftOnDevice'], maxAttempts: 0 })
            .addRetry({
            errors: [
                'ECS.AmazonECSException',
                'ECS.InvalidParameterException',
                'jsii-docgen.NpmError.E429',
                'jsii-codgen.NpmError.EPROTO',
            ],
            ...THROTTLE_RETRY_POLICY,
        })
            .addRetry({
            errors: ['jsii-docgen.NpmError.ETARGET'],
            // We'll wait longer between retries. This is to account for CodeArtifact's lag behind npm
            backoffRate: 2,
            interval: core_1.Duration.minutes(5),
            maxAttempts: 3,
        })
            .addRetry({ maxAttempts: 3 })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Generate docs" timed out', { parameters: { error: 'Timed out!' } })
            .next(sendToDeadLetterQueue), { errors: ['States.Timeout'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Generate docs" service error', { parameters: { 'error.$': '$.Cause' } })
            .next(sendToDeadLetterQueue), { errors: ['ECS.AmazonECSException', 'ECS.InvalidParameterException'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Generate docs" failure', { parameters: { 'error.$': 'States.StringToJson($.Cause)' } })
            .next(sendToDeadLetterQueue), { errors: ['States.TaskFailed'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Generate docs" fault', { parameters: { 'error.$': '$.Cause' } })
            .next(sendToDeadLetterQueue), { errors: ['States.ALL'] }).next(addToCatalog));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.days(1),
            tracingEnabled: true,
        });
        if (props.vpc) {
            // Ensure the State Machine does not get to run before the VPC can be used.
            this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        }
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // The workflow is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the orchestrator.
        this.regenerateAllDocumentation = new RegenerateAllDocumentation(this, 'RegenerateAllDocumentation', {
            bucket: props.bucket,
            stateMachine: this.stateMachine,
        }).stateMachine;
    }
    metricEcsTaskCount(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'TaskCount',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % CPU utilization from the CPU units utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mCpuUtilized, 0) / FILL(mCpuReserved, REPEAT)',
            usingMetrics: {
                mCpuReserved: this.metricEcsCpuReserved(),
                mCpuUtilized: this.metricEcsCpuUtilized(),
            },
        });
    }
    metricEcsMemoryReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % memory utilization from the RAM utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mMemoryUtilized, 0) / FILL(mMemoryReserved, REPEAT)',
            usingMetrics: {
                mMemoryReserved: this.metricEcsMemoryReserved(),
                mMemoryUtilized: this.metricEcsMemoryUtilized(),
            },
        });
    }
    metricEcsNetworkRxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkRxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsNetworkTxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkTxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
}
exports.Orchestration = Orchestration;
class RegenerateAllDocumentation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const processVersions = new aws_stepfunctions_1.Choice(this, 'Get package versions page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each key prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Start Orchestration Workflow', {
            stateMachine: props.stateMachine,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                bucket: props.bucket.bucketName,
                assembly: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.ASSEMBLY_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                metadata: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.METADATA_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                package: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.PACKAGE_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] })));
        processVersions.next(new aws_stepfunctions_1.Choice(this, 'Has more versions?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processVersions)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success')));
        const processPackageVersions = new aws_stepfunctions_1.StateMachine(this, 'PerPackage', {
            definition: processVersions,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // This workflow is broken into two sub-workflows because otherwise it hits the 25K events limit
        // of StepFunction executions relatively quickly.
        const processNamespace = new aws_stepfunctions_1.Choice(this, 'Get @scope page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each @scope/pkg', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Process scoped package', {
            stateMachine: processPackageVersions,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] })));
        processNamespace.next(new aws_stepfunctions_1.Choice(this, 'Has more packages?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processNamespace)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'All Done')));
        const start = new aws_stepfunctions_1.Choice(this, 'Get prefix page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next prefixes page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First prefix page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        })).afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new aws_stepfunctions_1.Choice(this, 'Is this a @scope/ prefix?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Prefix', `${constants_1.STORAGE_KEY_PREFIX}@*`), processNamespace)
            .otherwise(new tasks.StepFunctionsStartExecution(this, 'Process unscoped package', {
            stateMachine: processPackageVersions,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] }))
            .afterwards()));
        start.next(new aws_stepfunctions_1.Choice(this, 'Has more prefixes?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), start)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition: start,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.hours(4),
            tracingEnabled: true,
        });
        props.bucket.grantRead(processPackageVersions);
        props.bucket.grantRead(this.stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvYmFja2VuZC9vcmNoZXN0cmF0aW9uL2luZGV4LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLDREQUFzSTtBQUV0SSw4Q0FBcUQ7QUFDckQsb0RBQXlEO0FBR3pELDhDQUFrRTtBQUNsRSxrRUFBeUo7QUFDekosMERBQTBEO0FBQzFELHdDQUFvRDtBQUVwRCwrQ0FBK0Q7QUFFL0QsbURBQWdEO0FBQ2hELHdEQUFvRDtBQUVwRCxtREFBdUg7QUFDdkgsc0RBQStFO0FBQy9FLG1FQUE4RDtBQUU5RDs7Ozs7Ozs7Ozs7Ozs7O0dBZUc7QUFDSCxNQUFNLHFCQUFxQixHQUFHLEVBQUUsV0FBVyxFQUFFLEdBQUcsRUFBRSxRQUFRLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxXQUFXLEVBQUUsRUFBRSxFQUFFLENBQUM7QUE4Q25HOztHQUVHO0FBQ0gsTUFBYSxhQUFjLFNBQVEsZ0JBQVM7SUF3QzFDLFlBQW1CLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXlCO1FBQ3hFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFFakIsSUFBSSxDQUFDLGVBQWUsR0FBRyxJQUFJLGVBQUssQ0FBQyxJQUFJLEVBQUUsS0FBSyxFQUFFO1lBQzVDLFVBQVUsRUFBRSx5QkFBZSxDQUFDLFdBQVc7WUFDdkMsZUFBZSxFQUFFLGVBQVEsQ0FBQyxJQUFJLENBQUMsRUFBRSxDQUFDO1lBQ2xDLGlCQUFpQixFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1NBQ3hDLENBQUMsQ0FBQztRQUVILEtBQUssQ0FBQyxVQUFVLENBQUMsb0JBQW9CLENBQ25DLHNEQUFzRCxFQUN0RCxJQUFJLCtCQUFjLENBQUM7WUFDakIsVUFBVSxFQUFFLFNBQVM7WUFDckIsS0FBSyxFQUFFLDZCQUE2QjtZQUNwQyxZQUFZLEVBQUU7Z0JBQ1osRUFBRSxFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsd0NBQXdDLENBQUMsRUFBRSxNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2dCQUNsRyxFQUFFLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQywyQ0FBMkMsQ0FBQyxFQUFFLE1BQU0sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7YUFDdEc7U0FDRixDQUFDLENBQUMsV0FBVyxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDL0IsU0FBUyxFQUFFLEdBQUcsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsSUFBSSxXQUFXO1lBQ3ZELGdCQUFnQixFQUFFO2dCQUNoQix1REFBdUQ7Z0JBQ3ZELEVBQUU7Z0JBQ0YsWUFBWSx5QkFBVyxFQUFFO2dCQUN6QixFQUFFO2dCQUNGLHlCQUF5Qix1QkFBVyxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsRUFBRTtnQkFDNUQsNkZBQTZGO2FBQzlGLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztZQUNaLGtCQUFrQixFQUFFLG1DQUFrQixDQUFDLGtDQUFrQztZQUN6RSxpQkFBaUIsRUFBRSxDQUFDO1lBQ3BCLFNBQVMsRUFBRSxDQUFDO1NBQ2IsQ0FBQyxDQUNILENBQUM7UUFFRixNQUFNLHFCQUFxQixHQUFHLElBQUksS0FBSyxDQUFDLGNBQWMsQ0FBQyxJQUFJLEVBQUUsMkJBQTJCLEVBQUU7WUFDeEYsV0FBVyxFQUFFLDZCQUFTLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQztZQUMxQyxLQUFLLEVBQUUsSUFBSSxDQUFDLGVBQWU7WUFDM0IsVUFBVSxFQUFFLDRCQUFRLENBQUMsT0FBTztTQUM3QixDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksMkJBQU8sQ0FBQyxJQUFJLEVBQUUsYUFBYSxDQUFDLENBQUMsQ0FBQztRQUUxQyxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksZ0NBQWMsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFeEUsTUFBTSxZQUFZLEdBQUcsSUFBSSxLQUFLLENBQUMsWUFBWSxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRTtZQUN2RSxjQUFjLEVBQUUsSUFBSSxDQUFDLGNBQWMsQ0FBQyxRQUFRO1lBQzVDLFVBQVUsRUFBRSx3QkFBd0I7WUFDcEMsY0FBYyxFQUFFO2dCQUNkLFFBQVEsRUFBRSxnQkFBZ0I7Z0JBQzFCLGFBQWEsRUFBRSxxQkFBcUI7YUFDckM7U0FDRixDQUFDO1lBQ0Esc0ZBQXNGO2FBQ3JGLFFBQVEsQ0FBQyxFQUFFLE1BQU0sRUFBRSxDQUFDLGlDQUFpQyxDQUFDLEVBQUUsR0FBRyxxQkFBcUIsRUFBRSxDQUFDO2FBQ25GLFFBQVEsQ0FDUCxJQUFJLHdCQUFJLENBQUMsSUFBSSxFQUFFLGlDQUFpQyxFQUFFO1lBQ2hELFVBQVUsRUFBRSxFQUFFLFNBQVMsRUFBRSxTQUFTLEVBQUU7WUFDcEMsVUFBVSxFQUFFLFNBQVM7U0FDdEIsQ0FBQyxDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxFQUM5QixFQUFFLE1BQU0sRUFBRSxDQUFDLGlDQUFpQyxDQUFDLEVBQUUsQ0FDaEQ7YUFDQSxRQUFRLENBQ1AsSUFBSSx3QkFBSSxDQUFDLElBQUksRUFBRSwrQkFBK0IsRUFBRTtZQUM5QyxVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUUsOEJBQThCLEVBQUU7WUFDekQsVUFBVSxFQUFFLFNBQVM7U0FDdEIsQ0FBQyxDQUFDLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxFQUM5QixFQUFFLE1BQU0sRUFBRSxDQUFDLG1CQUFtQixDQUFDLEVBQUUsQ0FDbEM7YUFDQSxRQUFRLENBQUMsSUFBSSx3QkFBSSxDQUFDLElBQUksRUFBRSw2QkFBNkIsRUFBRTtZQUN0RCxVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUUsU0FBUyxFQUFFO1lBQ3BDLFVBQVUsRUFBRSxTQUFTO1NBQ3RCLENBQUMsQ0FBQyxJQUFJLENBQUMscUJBQXFCLENBQUMsRUFBRSxFQUFFLE1BQU0sRUFBRSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQUMsQ0FBQztRQUU5RCxJQUFJLENBQUMsVUFBVSxHQUFHLElBQUksaUJBQU8sQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQzdDLGlCQUFpQixFQUFFLElBQUk7WUFDdkIsOEJBQThCLEVBQUUsSUFBSTtZQUNwQyxHQUFHLEVBQUUsS0FBSyxDQUFDLEdBQUc7U0FDZixDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsY0FBYyxHQUFHLElBQUksK0JBQWMsQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFFeEUsTUFBTSxVQUFVLEdBQUcsSUFBSSx3QkFBSSxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRTtZQUN6RCxTQUFTLEVBQUUsY0FBYztZQUN6QixVQUFVLEVBQUU7Z0JBQ1YsTUFBTSxFQUFFLE1BQU07Z0JBQ2QsUUFBUSxFQUFFLFFBQVE7Z0JBQ2xCLFdBQVcsRUFBRSxXQUFXO2dCQUN4QixhQUFhLEVBQUUsYUFBYTthQUM3QjtZQUNELFVBQVUsRUFBRSxrQkFBa0I7U0FDL0IsQ0FBQzthQUNDLElBQUksQ0FDSCxJQUFJLHdCQUFJLENBQUMsSUFBSSxFQUFFLDZCQUE2QixFQUFFO1lBQzVDLFVBQVUsRUFBRSxFQUFFLFdBQVcsRUFBRSxzQ0FBc0MsRUFBRTtZQUNuRSxVQUFVLEVBQUUsVUFBVTtTQUN2QixDQUFDLENBQ0g7YUFDQSxJQUFJLENBQ0gsSUFBSSxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQyxJQUFJLEVBQUUsZUFBZSxFQUFFO1lBQzFELE9BQU8sRUFBRSxJQUFJLENBQUMsVUFBVTtZQUN4QixTQUFTLEVBQUUsa0JBQWtCO1lBQzdCLFVBQVUsRUFBRSxnQkFBZ0I7WUFDNUIsMENBQTBDO1lBQzFDLE9BQU8sRUFBRSxlQUFRLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztZQUMxQixVQUFVLEVBQUUsS0FBSyxDQUFDLFVBQVU7U0FDN0IsQ0FBQztZQUNGLDhFQUE4RTthQUMzRSxRQUFRLENBQUMsRUFBRSxNQUFNLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxFQUFFLFdBQVcsRUFBRSxDQUFDLEVBQUUsQ0FBQzthQUN6RSxRQUFRLENBQUM7WUFDUixNQUFNLEVBQUU7Z0JBQ04sd0JBQXdCO2dCQUN4QiwrQkFBK0I7Z0JBQy9CLDJCQUEyQjtnQkFDM0IsNkJBQTZCO2FBQzlCO1lBQ0QsR0FBRyxxQkFBcUI7U0FDekIsQ0FBQzthQUNELFFBQVEsQ0FBQztZQUNSLE1BQU0sRUFBRSxDQUFDLDhCQUE4QixDQUFDO1lBQ3hDLDBGQUEwRjtZQUMxRixXQUFXLEVBQUUsQ0FBQztZQUNkLFFBQVEsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztZQUM3QixXQUFXLEVBQUUsQ0FBQztTQUNmLENBQUM7YUFDRCxRQUFRLENBQUMsRUFBRSxXQUFXLEVBQUUsQ0FBQyxFQUFFLENBQUM7YUFDNUIsUUFBUSxDQUNQLElBQUksd0JBQUksQ0FBQyxJQUFJLEVBQUUsMkJBQTJCLEVBQUUsRUFBRSxVQUFVLEVBQUUsRUFBRSxLQUFLLEVBQUUsWUFBWSxFQUFFLEVBQUUsQ0FBQzthQUNqRixJQUFJLENBQUMscUJBQXFCLENBQUMsRUFDOUIsRUFBRSxNQUFNLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUFFLENBQy9CO2FBQ0EsUUFBUSxDQUNQLElBQUksd0JBQUksQ0FBQyxJQUFJLEVBQUUsK0JBQStCLEVBQUUsRUFBRSxVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUUsU0FBUyxFQUFFLEVBQUUsQ0FBQzthQUN0RixJQUFJLENBQUMscUJBQXFCLENBQUMsRUFDOUIsRUFBRSxNQUFNLEVBQUUsQ0FBQyx3QkFBd0IsRUFBRSwrQkFBK0IsQ0FBQyxFQUFFLENBQ3hFO2FBQ0EsUUFBUSxDQUNQLElBQUksd0JBQUksQ0FBQyxJQUFJLEVBQUUseUJBQXlCLEVBQUUsRUFBRSxVQUFVLEVBQUUsRUFBRSxTQUFTLEVBQUUsOEJBQThCLEVBQUUsRUFBRSxDQUFDO2FBQ3JHLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxFQUM5QixFQUFFLE1BQU0sRUFBRSxDQUFDLG1CQUFtQixDQUFDLEVBQUUsQ0FDbEM7YUFDQSxRQUFRLENBQ1AsSUFBSSx3QkFBSSxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRSxFQUFFLFVBQVUsRUFBRSxFQUFFLFNBQVMsRUFBRSxTQUFTLEVBQUUsRUFBRSxDQUFDO2FBQzlFLElBQUksQ0FBQyxxQkFBcUIsQ0FBQyxFQUM5QixFQUFFLE1BQU0sRUFBRSxDQUFDLFlBQVksQ0FBQyxFQUFFLENBQzNCLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUN2QixDQUFDO1FBRUosSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLGdDQUFZLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNyRCxVQUFVO1lBQ1YsZ0JBQWdCLEVBQUUsb0JBQW9CLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7WUFDdEQsT0FBTyxFQUFFLGVBQVEsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1lBQ3pCLGNBQWMsRUFBRSxJQUFJO1NBQ3JCLENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxDQUFDLEdBQUcsRUFBRTtZQUNiLDJFQUEyRTtZQUMzRSxJQUFJLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO1NBQ2pGO1FBRUQsS0FBSyxDQUFDLFVBQVUsQ0FBQyxvQkFBb0IsQ0FDbkMsOEJBQThCLEVBQzlCLElBQUksQ0FBQyxZQUFZLENBQUMsWUFBWSxFQUFFO2FBQzdCLFdBQVcsQ0FBQyxJQUFJLEVBQUUscUJBQXFCLEVBQUU7WUFDeEMsU0FBUyxFQUFFLEdBQUcsSUFBSSxDQUFDLFlBQVksQ0FBQyxJQUFJLENBQUMsSUFBSSxJQUFJLElBQUksQ0FBQyxZQUFZLENBQUMsWUFBWSxFQUFFLENBQUMsVUFBVSxFQUFFO1lBQzFGLGdCQUFnQixFQUFFO2dCQUNoQiwrQkFBK0I7Z0JBQy9CLEVBQUU7Z0JBQ0YsWUFBWSx5QkFBVyxFQUFFO2dCQUN6QixFQUFFO2dCQUNGLGlDQUFpQywyQkFBZSxDQUFDLElBQUksQ0FBQyxZQUFZLENBQUMsRUFBRTtnQkFDckUsK0VBQStFO2FBQ2hGLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQztZQUNaLGtCQUFrQixFQUFFLG1DQUFrQixDQUFDLGtDQUFrQztZQUN6RSxpQkFBaUIsRUFBRSxDQUFDO1lBQ3BCLFNBQVMsRUFBRSxDQUFDO1NBQ2IsQ0FBQyxDQUFDLENBQUM7UUFFUix3RUFBd0U7UUFDeEUsMkNBQTJDO1FBQzNDLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSwyQ0FBbUIsQ0FBQyxJQUFJLEVBQUUsU0FBUyxFQUFFO1lBQzlELFdBQVcsRUFBRSwwRkFBMEY7WUFDdkcsV0FBVyxFQUFFO2dCQUNYLGlCQUFpQixFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsZUFBZTtnQkFDcEQsU0FBUyxFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsUUFBUTthQUN6QztZQUNELFVBQVUsRUFBRSxJQUFLO1lBQ2pCLE9BQU8sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixPQUFPLEVBQUUsb0JBQU8sQ0FBQyxNQUFNO1NBQ3hCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBQzVELElBQUksQ0FBQyxlQUFlLENBQUMsb0JBQW9CLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxDQUFDO1FBRWhFLHNFQUFzRTtRQUN0RSw4RUFBOEU7UUFDOUUsSUFBSSxDQUFDLDBCQUEwQixHQUFHLElBQUksMEJBQTBCLENBQUMsSUFBSSxFQUFFLDRCQUE0QixFQUFFO1lBQ25HLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTTtZQUNwQixZQUFZLEVBQUUsSUFBSSxDQUFDLFlBQVk7U0FDaEMsQ0FBQyxDQUFDLFlBQVksQ0FBQztJQUNsQixDQUFDO0lBRU0sa0JBQWtCLENBQUMsSUFBbUI7UUFDM0MsT0FBTyxJQUFJLHVCQUFNLENBQUM7WUFDaEIsU0FBUyxFQUFFLDBCQUFTLENBQUMsR0FBRztZQUN4QixHQUFHLElBQUk7WUFDUCxhQUFhLEVBQUUsRUFBRSxXQUFXLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLEVBQUU7WUFDM0QsVUFBVSxFQUFFLFdBQVc7WUFDdkIsU0FBUyxFQUFFLHVCQUF1QjtTQUNuQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU0sb0JBQW9CLENBQUMsSUFBb0I7UUFDOUMsT0FBTyxJQUFJLHVCQUFNLENBQUM7WUFDaEIsU0FBUyxFQUFFLDBCQUFTLENBQUMsT0FBTztZQUM1QixHQUFHLElBQUk7WUFDUCxhQUFhLEVBQUUsRUFBRSxXQUFXLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLEVBQUU7WUFDM0QsVUFBVSxFQUFFLGFBQWE7WUFDekIsU0FBUyxFQUFFLHVCQUF1QjtTQUNuQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU0sb0JBQW9CLENBQUMsSUFBb0I7UUFDOUMsT0FBTyxJQUFJLHVCQUFNLENBQUM7WUFDaEIsU0FBUyxFQUFFLDBCQUFTLENBQUMsT0FBTztZQUM1QixHQUFHLElBQUk7WUFDUCxhQUFhLEVBQUUsRUFBRSxXQUFXLEVBQUUsSUFBSSxDQUFDLFVBQVUsQ0FBQyxXQUFXLEVBQUU7WUFDM0QsVUFBVSxFQUFFLGFBQWE7WUFDekIsU0FBUyxFQUFFLHVCQUF1QjtTQUNuQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU0sdUJBQXVCLENBQUMsSUFBNEI7UUFDekQsT0FBTyxJQUFJLCtCQUFjLENBQUM7WUFDeEIsR0FBRyxJQUFJO1lBQ1Asb0VBQW9FO1lBQ3BFLDBFQUEwRTtZQUMxRSxtQ0FBbUM7WUFDbkMsVUFBVSxFQUFFLDBEQUEwRDtZQUN0RSxZQUFZLEVBQUU7Z0JBQ1osWUFBWSxFQUFFLElBQUksQ0FBQyxvQkFBb0IsRUFBRTtnQkFDekMsWUFBWSxFQUFFLElBQUksQ0FBQyxvQkFBb0IsRUFBRTthQUMxQztTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSx1QkFBdUIsQ0FBQyxJQUFvQjtRQUNqRCxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixTQUFTLEVBQUUsMEJBQVMsQ0FBQyxPQUFPO1lBQzVCLEdBQUcsSUFBSTtZQUNQLGFBQWEsRUFBRSxFQUFFLFdBQVcsRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVcsRUFBRTtZQUMzRCxVQUFVLEVBQUUsZ0JBQWdCO1lBQzVCLFNBQVMsRUFBRSx1QkFBdUI7U0FDbkMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVNLHVCQUF1QixDQUFDLElBQW9CO1FBQ2pELE9BQU8sSUFBSSx1QkFBTSxDQUFDO1lBQ2hCLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxJQUFJO1lBQ1AsYUFBYSxFQUFFLEVBQUUsV0FBVyxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsV0FBVyxFQUFFO1lBQzNELFVBQVUsRUFBRSxnQkFBZ0I7WUFDNUIsU0FBUyxFQUFFLHVCQUF1QjtTQUNuQyxDQUFDLENBQUM7SUFDTCxDQUFDO0lBRU0sMEJBQTBCLENBQUMsSUFBNEI7UUFDNUQsT0FBTyxJQUFJLCtCQUFjLENBQUM7WUFDeEIsR0FBRyxJQUFJO1lBQ1AsaUVBQWlFO1lBQ2pFLDBFQUEwRTtZQUMxRSxtQ0FBbUM7WUFDbkMsVUFBVSxFQUFFLGdFQUFnRTtZQUM1RSxZQUFZLEVBQUU7Z0JBQ1osZUFBZSxFQUFFLElBQUksQ0FBQyx1QkFBdUIsRUFBRTtnQkFDL0MsZUFBZSxFQUFFLElBQUksQ0FBQyx1QkFBdUIsRUFBRTthQUNoRDtTQUNGLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSx1QkFBdUIsQ0FBQyxJQUFvQjtRQUNqRCxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixTQUFTLEVBQUUsMEJBQVMsQ0FBQyxPQUFPO1lBQzVCLEdBQUcsSUFBSTtZQUNQLGFBQWEsRUFBRSxFQUFFLFdBQVcsRUFBRSxJQUFJLENBQUMsVUFBVSxDQUFDLFdBQVcsRUFBRTtZQUMzRCxVQUFVLEVBQUUsZ0JBQWdCO1lBQzVCLFNBQVMsRUFBRSx1QkFBdUI7U0FDbkMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVNLHVCQUF1QixDQUFDLElBQW9CO1FBQ2pELE9BQU8sSUFBSSx1QkFBTSxDQUFDO1lBQ2hCLFNBQVMsRUFBRSwwQkFBUyxDQUFDLE9BQU87WUFDNUIsR0FBRyxJQUFJO1lBQ1AsYUFBYSxFQUFFLEVBQUUsV0FBVyxFQUFFLElBQUksQ0FBQyxVQUFVLENBQUMsV0FBVyxFQUFFO1lBQzNELFVBQVUsRUFBRSxnQkFBZ0I7WUFDNUIsU0FBUyxFQUFFLHVCQUF1QjtTQUNuQyxDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUEvVUQsc0NBK1VDO0FBT0QsTUFBTSwwQkFBMkIsU0FBUSxnQkFBUztJQUdoRCxZQUFtQixLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFzQztRQUNyRixLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCLE1BQU0sZUFBZSxHQUFHLElBQUksMEJBQU0sQ0FBQyxJQUFJLEVBQUUsMkJBQTJCLENBQUM7YUFDbEUsSUFBSSxDQUFDLDZCQUFTLENBQUMsU0FBUyxDQUFDLGtDQUFrQyxDQUFDLEVBQUUsSUFBSSxLQUFLLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxvQkFBb0IsRUFBRTtZQUNsSCxPQUFPLEVBQUUsSUFBSTtZQUNiLE1BQU0sRUFBRSxlQUFlO1lBQ3ZCLFNBQVMsRUFBRSxlQUFlO1lBQzFCLFlBQVksRUFBRSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDO1lBQ3RDLFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVO2dCQUMvQixpQkFBaUIsRUFBRSw0QkFBUSxDQUFDLFFBQVEsQ0FBQyxrQ0FBa0MsQ0FBQztnQkFDeEUsU0FBUyxFQUFFLEdBQUc7Z0JBQ2QsTUFBTSxFQUFFLDRCQUFRLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQzthQUN0QztZQUNELFVBQVUsRUFBRSxZQUFZO1NBQ3pCLENBQUMsQ0FBQzthQUNGLFNBQVMsQ0FBQyxJQUFJLEtBQUssQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLHFCQUFxQixFQUFFO1lBQy9ELE9BQU8sRUFBRSxJQUFJO1lBQ2IsTUFBTSxFQUFFLGVBQWU7WUFDdkIsU0FBUyxFQUFFLGVBQWU7WUFDMUIsWUFBWSxFQUFFLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUM7WUFDdEMsVUFBVSxFQUFFO2dCQUNWLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVU7Z0JBQy9CLFNBQVMsRUFBRSxHQUFHO2dCQUNkLE1BQU0sRUFBRSw0QkFBUSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUM7YUFDdEM7WUFDRCxVQUFVLEVBQUUsWUFBWTtTQUN6QixDQUFDLENBQUM7YUFDRixVQUFVLEVBQUU7YUFDWixJQUFJLENBQUMsSUFBSSx1QkFBRyxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRSxFQUFFLFNBQVMsRUFBRSwyQkFBMkIsRUFBRSxVQUFVLEVBQUUsNEJBQVEsQ0FBQyxPQUFPLEVBQUUsQ0FBQzthQUNqSCxRQUFRLENBQUMsSUFBSSxLQUFLLENBQUMsMkJBQTJCLENBQUMsSUFBSSxFQUFFLDhCQUE4QixFQUFFO1lBQ3BGLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtZQUNoQyxtQkFBbUIsRUFBRSxJQUFJO1lBQ3pCLEtBQUssRUFBRSw2QkFBUyxDQUFDLFVBQVUsQ0FBQztnQkFDMUIsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVTtnQkFDL0IsUUFBUSxFQUFFLEVBQUUsR0FBRyxFQUFFLDRCQUFRLENBQUMsUUFBUSxDQUFDLG9CQUFvQiwrQkFBbUIsQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxFQUFFO2dCQUNyRyxRQUFRLEVBQUUsRUFBRSxHQUFHLEVBQUUsNEJBQVEsQ0FBQyxRQUFRLENBQUMsb0JBQW9CLCtCQUFtQixDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsY0FBYyxDQUFDLEVBQUU7Z0JBQ3JHLE9BQU8sRUFBRSxFQUFFLEdBQUcsRUFBRSw0QkFBUSxDQUFDLFFBQVEsQ0FBQyxvQkFBb0IsOEJBQWtCLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsRUFBRTthQUNwRyxDQUFDO1lBQ0Ysa0JBQWtCLEVBQUUsc0NBQWtCLENBQUMsZ0JBQWdCO1NBQ3hELENBQUMsQ0FBQyxRQUFRLENBQUMsRUFBRSxNQUFNLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDeEUsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLDBCQUFNLENBQUMsSUFBSSxFQUFFLG9CQUFvQixDQUFDO2FBQ3hELElBQUksQ0FBQyw2QkFBUyxDQUFDLFNBQVMsQ0FBQyxrQ0FBa0MsQ0FBQyxFQUFFLGVBQWUsQ0FBQzthQUM5RSxTQUFTLENBQUMsSUFBSSwyQkFBTyxDQUFDLElBQUksRUFBRSxTQUFTLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDNUMsTUFBTSxzQkFBc0IsR0FBRyxJQUFJLGdDQUFZLENBQUMsSUFBSSxFQUFFLFlBQVksRUFBRTtZQUNsRSxVQUFVLEVBQUUsZUFBZTtZQUMzQixPQUFPLEVBQUUsZUFBUSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7WUFDMUIsY0FBYyxFQUFFLElBQUk7U0FDckIsQ0FBQyxDQUFDO1FBRUgsZ0dBQWdHO1FBQ2hHLGlEQUFpRDtRQUNqRCxNQUFNLGdCQUFnQixHQUFHLElBQUksMEJBQU0sQ0FBQyxJQUFJLEVBQUUsaUJBQWlCLENBQUM7YUFDekQsSUFBSSxDQUFDLDZCQUFTLENBQUMsU0FBUyxDQUFDLGtDQUFrQyxDQUFDLEVBQUUsSUFBSSxLQUFLLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxrQkFBa0IsRUFBRTtZQUNoSCxPQUFPLEVBQUUsSUFBSTtZQUNiLE1BQU0sRUFBRSxlQUFlO1lBQ3ZCLFNBQVMsRUFBRSxlQUFlO1lBQzFCLFlBQVksRUFBRSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDO1lBQ3RDLFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVO2dCQUMvQixpQkFBaUIsRUFBRSw0QkFBUSxDQUFDLFFBQVEsQ0FBQyxrQ0FBa0MsQ0FBQztnQkFDeEUsU0FBUyxFQUFFLEdBQUc7Z0JBQ2QsTUFBTSxFQUFFLDRCQUFRLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQzthQUN0QztZQUNELFVBQVUsRUFBRSxZQUFZO1NBQ3pCLENBQUMsQ0FBQzthQUNGLFNBQVMsQ0FBQyxJQUFJLEtBQUssQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQzdELE9BQU8sRUFBRSxJQUFJO1lBQ2IsTUFBTSxFQUFFLGVBQWU7WUFDdkIsU0FBUyxFQUFFLGVBQWU7WUFDMUIsWUFBWSxFQUFFLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUM7WUFDdEMsVUFBVSxFQUFFO2dCQUNWLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVU7Z0JBQy9CLFNBQVMsRUFBRSxHQUFHO2dCQUNkLE1BQU0sRUFBRSw0QkFBUSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUM7YUFDdEM7WUFDRCxVQUFVLEVBQUUsWUFBWTtTQUN6QixDQUFDLENBQUM7YUFDRixVQUFVLEVBQUU7YUFDWixJQUFJLENBQUMsSUFBSSx1QkFBRyxDQUFDLElBQUksRUFBRSxxQkFBcUIsRUFBRSxFQUFFLFNBQVMsRUFBRSwyQkFBMkIsRUFBRSxVQUFVLEVBQUUsNEJBQVEsQ0FBQyxPQUFPLEVBQUUsQ0FBQzthQUNqSCxRQUFRLENBQUMsSUFBSSxLQUFLLENBQUMsMkJBQTJCLENBQUMsSUFBSSxFQUFFLHdCQUF3QixFQUFFO1lBQzlFLFlBQVksRUFBRSxzQkFBc0I7WUFDcEMsbUJBQW1CLEVBQUUsSUFBSTtZQUN6QixLQUFLLEVBQUUsNkJBQVMsQ0FBQyxVQUFVLENBQUM7Z0JBQzFCLE1BQU0sRUFBRSw0QkFBUSxDQUFDLFFBQVEsQ0FBQyxVQUFVLENBQUM7YUFDdEMsQ0FBQztZQUNGLGtCQUFrQixFQUFFLHNDQUFrQixDQUFDLE9BQU87U0FDL0MsQ0FBQyxDQUFDLFFBQVEsQ0FBQyxFQUFFLE1BQU0sRUFBRSxDQUFDLHNDQUFzQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUN4RSxnQkFBZ0IsQ0FBQyxJQUFJLENBQUMsSUFBSSwwQkFBTSxDQUFDLElBQUksRUFBRSxvQkFBb0IsQ0FBQzthQUN6RCxJQUFJLENBQUMsNkJBQVMsQ0FBQyxTQUFTLENBQUMsa0NBQWtDLENBQUMsRUFBRSxnQkFBZ0IsQ0FBQzthQUMvRSxTQUFTLENBQUMsSUFBSSwyQkFBTyxDQUFDLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFFN0MsTUFBTSxLQUFLLEdBQUcsSUFBSSwwQkFBTSxDQUFDLElBQUksRUFBRSxpQkFBaUIsQ0FBQzthQUM5QyxJQUFJLENBQ0gsNkJBQVMsQ0FBQyxTQUFTLENBQUMsa0NBQWtDLENBQUMsRUFDdkQsSUFBSSxLQUFLLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxvQkFBb0IsRUFBRTtZQUNuRCxPQUFPLEVBQUUsSUFBSTtZQUNiLE1BQU0sRUFBRSxlQUFlO1lBQ3ZCLFNBQVMsRUFBRSxlQUFlO1lBQzFCLFlBQVksRUFBRSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDO1lBQ3RDLFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVO2dCQUMvQixpQkFBaUIsRUFBRSw0QkFBUSxDQUFDLFFBQVEsQ0FBQyxrQ0FBa0MsQ0FBQztnQkFDeEUsU0FBUyxFQUFFLEdBQUc7Z0JBQ2QsTUFBTSxFQUFFLDhCQUFrQjthQUMzQjtZQUNELFVBQVUsRUFBRSxZQUFZO1NBQ3pCLENBQUMsQ0FDSDthQUNBLFNBQVMsQ0FDUixJQUFJLEtBQUssQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFO1lBQ2xELE9BQU8sRUFBRSxJQUFJO1lBQ2IsTUFBTSxFQUFFLGVBQWU7WUFDdkIsU0FBUyxFQUFFLGVBQWU7WUFDMUIsWUFBWSxFQUFFLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUM7WUFDdEMsVUFBVSxFQUFFO2dCQUNWLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVU7Z0JBQy9CLFNBQVMsRUFBRSxHQUFHO2dCQUNkLE1BQU0sRUFBRSw4QkFBa0I7YUFDM0I7WUFDRCxVQUFVLEVBQUUsWUFBWTtTQUN6QixDQUFDLENBQ0gsQ0FBQyxVQUFVLEVBQUU7YUFDYixJQUFJLENBQUMsSUFBSSx1QkFBRyxDQUFDLElBQUksRUFBRSxpQkFBaUIsRUFBRSxFQUFFLFNBQVMsRUFBRSwyQkFBMkIsRUFBRSxVQUFVLEVBQUUsNEJBQVEsQ0FBQyxPQUFPLEVBQUUsQ0FBQzthQUM3RyxRQUFRLENBQ1AsSUFBSSwwQkFBTSxDQUFDLElBQUksRUFBRSwyQkFBMkIsQ0FBQzthQUMxQyxJQUFJLENBQUMsNkJBQVMsQ0FBQyxhQUFhLENBQUMsVUFBVSxFQUFFLEdBQUcsOEJBQWtCLElBQUksQ0FBQyxFQUFFLGdCQUFnQixDQUFDO2FBQ3RGLFNBQVMsQ0FBQyxJQUFJLEtBQUssQ0FBQywyQkFBMkIsQ0FBQyxJQUFJLEVBQUUsMEJBQTBCLEVBQUU7WUFDakYsWUFBWSxFQUFFLHNCQUFzQjtZQUNwQyxtQkFBbUIsRUFBRSxJQUFJO1lBQ3pCLEtBQUssRUFBRSw2QkFBUyxDQUFDLFVBQVUsQ0FBQztnQkFDMUIsTUFBTSxFQUFFLDRCQUFRLENBQUMsUUFBUSxDQUFDLFVBQVUsQ0FBQzthQUN0QyxDQUFDO1lBQ0Ysa0JBQWtCLEVBQUUsc0NBQWtCLENBQUMsT0FBTztTQUMvQyxDQUFDLENBQUMsUUFBUSxDQUFDLEVBQUUsTUFBTSxFQUFFLENBQUMsc0NBQXNDLENBQUMsRUFBRSxDQUFDLENBQUM7YUFDakUsVUFBVSxFQUFFLENBQ2hCLENBQUMsQ0FBQztRQUVQLEtBQUssQ0FBQyxJQUFJLENBQUMsSUFBSSwwQkFBTSxDQUFDLElBQUksRUFBRSxvQkFBb0IsQ0FBQzthQUM5QyxJQUFJLENBQUMsNkJBQVMsQ0FBQyxTQUFTLENBQUMsa0NBQWtDLENBQUMsRUFBRSxLQUFLLENBQUM7YUFDcEUsU0FBUyxDQUFDLElBQUksMkJBQU8sQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBRXpDLElBQUksQ0FBQyxZQUFZLEdBQUcsSUFBSSxnQ0FBWSxDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDckQsVUFBVSxFQUFFLEtBQUs7WUFDakIsZ0JBQWdCLEVBQUUsb0JBQW9CLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUM7WUFDdEQsT0FBTyxFQUFFLGVBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1lBQzFCLGNBQWMsRUFBRSxJQUFJO1NBQ3JCLENBQUMsQ0FBQztRQUVILEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDL0MsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDO0lBQzVDLENBQUM7Q0FDRjtBQUVEOzs7O0dBSUc7QUFDSCxTQUFTLG9CQUFvQixDQUFDLFFBQWdCO0lBQzVDLDRCQUE0QjtJQUM1QixPQUFPLFFBQVEsQ0FBQyxLQUFLLENBQUMsdUJBQXVCLENBQUMsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLENBQUM7QUFDM0QsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENvbXBhcmlzb25PcGVyYXRvciwgTWF0aEV4cHJlc3Npb24sIE1hdGhFeHByZXNzaW9uT3B0aW9ucywgTWV0cmljLCBNZXRyaWNPcHRpb25zLCBTdGF0aXN0aWMgfSBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWR3YXRjaCc7XG5pbXBvcnQgeyBTdWJuZXRTZWxlY3Rpb24sIFZwYyB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0IHsgQ2x1c3RlciwgSUNsdXN0ZXIgfSBmcm9tICdAYXdzLWNkay9hd3MtZWNzJztcbmltcG9ydCB7IElGdW5jdGlvbiwgVHJhY2luZyB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1sYW1iZGEnO1xuaW1wb3J0IHsgUmV0ZW50aW9uRGF5cyB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1sb2dzJztcbmltcG9ydCB7IElCdWNrZXQgfSBmcm9tICdAYXdzLWNkay9hd3MtczMnO1xuaW1wb3J0IHsgSVF1ZXVlLCBRdWV1ZSwgUXVldWVFbmNyeXB0aW9uIH0gZnJvbSAnQGF3cy1jZGsvYXdzLXNxcyc7XG5pbXBvcnQgeyBDaG9pY2UsIENvbmRpdGlvbiwgSW50ZWdyYXRpb25QYXR0ZXJuLCBJU3RhdGVNYWNoaW5lLCBKc29uUGF0aCwgTWFwLCBQYXNzLCBTdGF0ZU1hY2hpbmUsIFN1Y2NlZWQsIFRhc2tJbnB1dCB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCAqIGFzIHRhc2tzIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zLXRhc2tzJztcbmltcG9ydCB7IENvbnN0cnVjdCwgRHVyYXRpb24gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFJlcG9zaXRvcnkgfSBmcm9tICcuLi8uLi9jb2RlYXJ0aWZhY3QvcmVwb3NpdG9yeSc7XG5pbXBvcnQgeyBzcXNRdWV1ZVVybCwgc3RhdGVNYWNoaW5lVXJsIH0gZnJvbSAnLi4vLi4vZGVlcC1saW5rJztcbmltcG9ydCB7IE1vbml0b3JpbmcgfSBmcm9tICcuLi8uLi9tb25pdG9yaW5nJztcbmltcG9ydCB7IFJVTkJPT0tfVVJMIH0gZnJvbSAnLi4vLi4vcnVuYm9vay11cmwnO1xuaW1wb3J0IHsgQ2F0YWxvZ0J1aWxkZXIgfSBmcm9tICcuLi9jYXRhbG9nLWJ1aWxkZXInO1xuaW1wb3J0IHsgRGVueUxpc3QgfSBmcm9tICcuLi9kZW55LWxpc3QnO1xuaW1wb3J0IHsgQVNTRU1CTFlfS0VZX1NVRkZJWCwgTUVUQURBVEFfS0VZX1NVRkZJWCwgUEFDS0FHRV9LRVlfU1VGRklYLCBTVE9SQUdFX0tFWV9QUkVGSVggfSBmcm9tICcuLi9zaGFyZWQvY29uc3RhbnRzJztcbmltcG9ydCB7IFRyYW5zbGl0ZXJhdG9yLCBUcmFuc2xpdGVyYXRvclZwY0VuZHBvaW50cyB9IGZyb20gJy4uL3RyYW5zbGl0ZXJhdG9yJztcbmltcG9ydCB7IFJlZHJpdmVTdGF0ZU1hY2hpbmUgfSBmcm9tICcuL3JlZHJpdmUtc3RhdGUtbWFjaGluZSc7XG5cbi8qKlxuICogVGhpcyByZXRyeSBwb2xpY3kgaXMgdXNlZCBmb3IgYWxsIGl0ZW1zIGluIHRoZSBzdGF0ZSBtYWNoaW5lIGFuZCBhbGxvd3MgYW1wbGVcbiAqIHJldHJ5IGF0dGVtcHRzIGluIG9yZGVyIHRvIGF2b2lkIGhhdmluZyB0byBpbXBsZW1lbnQgYSBjdXN0b20gYmFja3ByZXNzdXJlXG4gKiBoYW5kbGluZyBtZWhhbmlzbS5cbiAqXG4gKiBUaGlzIGlzIG1lYW50IGFzIGEgc3RvcC1nYXAgdW50aWwgd2UgY2FuIGltcGxlbWVudCBhIG1vcmUgcmVzaWxpZW50IHN5c3RlbSxcbiAqIHdoaWNoIGxpa2VseSB3aWxsIGludm9sdmUgbW9yZSBTUVMgcXVldWVzLCBidXQgd2lsbCBwcm9iYWJseSBuZWVkIHRvIGJlXG4gKiB0aHJvdWdob3V0bHkgdmV0dGVkIGJlZm9yZSBpdCBpcyByb2xsZWQgb3V0IGV2ZXJ5d2hlcmUuXG4gKlxuICogQWZ0ZXIgMzAgYXR0ZW1wdHMsIGdpdmVuIHRoZSBwYXJhbWV0ZXJzLCB0aGUgbGFzdCBhdHRlbXB0IHdpbGwgd2FpdCBqdXN0XG4gKiB1bmRlciAxNiBtaW51dGVzLCB3aGljaCBzaG91bGQgYmUgZW5vdWdoIGZvciBjdXJyZW50bHkgcnVubmluZyBMYW1iZGFcbiAqIGZ1bmN0aW9ucyB0byBjb21wbGV0ZSAob3IgdGltZSBvdXQgYWZ0ZXIgMTUgbWludXRlcykuIFRoZSB0b3RhbCB0aW1lIHNwZW50XG4gKiB3YWl0aW5nIGJldHdlZW4gcmV0cmllcyBieSB0aGlzIHRpbWUgaXMganVzdCBvdmVyIDMgaG91cnMuIFRoaXMgaXMgYSBsb3Qgb2ZcbiAqIHRpbWUsIGJ1dCBpbiBleHRyZW1lIGJ1cnN0IHNpdHVhdGlvbnMgKGkuZTogcmVwcm9jZXNzaW5nIGV2ZXJ5dGhpbmcpLCB0aGlzXG4gKiBpcyBhY3R1YWxseSBhIGdvb2QgdGhpbmcuXG4gKi9cbmNvbnN0IFRIUk9UVExFX1JFVFJZX1BPTElDWSA9IHsgYmFja29mZlJhdGU6IDEuMSwgaW50ZXJ2YWw6IER1cmF0aW9uLm1pbnV0ZXMoMSksIG1heEF0dGVtcHRzOiAzMCB9O1xuXG5leHBvcnQgaW50ZXJmYWNlIE9yY2hlc3RyYXRpb25Qcm9wcyB7XG4gIC8qKlxuICAgKiBUaGUgYnVja2V0IGluIHdoaWNoIHRvIHNvdXJjZSBhc3NlbWJsaWVzIHRvIHRyYW5zbGl0ZXJhdGUuXG4gICAqL1xuICByZWFkb25seSBidWNrZXQ6IElCdWNrZXQ7XG5cbiAgLyoqXG4gICAqIFRoZSBDb2RlQXJ0aWZhY3QgcmVnaXN0cnkgdG8gdXNlIGZvciByZWd1bGFyIG9wZXJhdGlvbnMuXG4gICAqL1xuICByZWFkb25seSBjb2RlQXJ0aWZhY3Q/OiBSZXBvc2l0b3J5O1xuXG4gIC8qKlxuICAgKiBUaGUgbW9uaXRvcmluZyBoYW5kbGVyIHRvIHJlZ2lzdGVyIGFsYXJtcyB3aXRoLlxuICAgKi9cbiAgcmVhZG9ubHkgbW9uaXRvcmluZzogTW9uaXRvcmluZztcblxuICAvKipcbiAgICogVGhlIFZQQyBpbiB3aGljaCB0byBwbGFjZSBuZXR3b3JrZWQgcmVzb3VyY2VzLlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjPzogVnBjO1xuXG4gIC8qKlxuICAgKiBUaGUgVlBDIHN1Ym5ldCBzZWxlY3Rpb24gdG8gdXNlLlxuICAgKi9cbiAgcmVhZG9ubHkgdnBjU3VibmV0cz86IFN1Ym5ldFNlbGVjdGlvbjtcblxuICAvKipcbiAgICogVlBDIGVuZHBvaW50cyB0byB1c2UgZm9yIGludGVyYWN0aW5nIHdpdGggQ29kZUFydGlmYWN0IGFuZCBTMy5cbiAgICovXG4gIHJlYWRvbmx5IHZwY0VuZHBvaW50cz86IFRyYW5zbGl0ZXJhdG9yVnBjRW5kcG9pbnRzO1xuXG4gIC8qKlxuICAgKiBIb3cgbG9uZyBzaG91bGQgZXhlY3V0aW9uIGxvZ3MgYmUgcmV0YWluZWQ/XG4gICAqXG4gICAqIEBkZWZhdWx0IFJldGVudGlvbkRheXMuVEVOX1lFQVJTXG4gICAqL1xuICByZWFkb25seSBsb2dSZXRlbnRpb24/OiBSZXRlbnRpb25EYXlzO1xuXG4gIC8qKlxuICAgKiBUaGUgZGVueSBsaXN0LlxuICAgKi9cbiAgcmVhZG9ubHkgZGVueUxpc3Q6IERlbnlMaXN0O1xufVxuXG4vKipcbiAqIE9yY2hlc3RyYXRlcyB0aGUgYmFja2VuZCBwcm9jZXNzaW5nIHRhc2tzIHVzaW5nIGEgU3RlcEZ1bmN0aW9ucyBTdGF0ZSBNYWNoaW5lLlxuICovXG5leHBvcnQgY2xhc3MgT3JjaGVzdHJhdGlvbiBleHRlbmRzIENvbnN0cnVjdCB7XG4gIC8qKlxuICAgKiBUaGUgc3RhdGUgbWFjaGluZSB0aGF0IHNob3VsZCBiZSB0cmlnZ2VyZWQgZm9yIHN0YXJ0aW5nIGJhY2stZW5kIHByb2Nlc3NpbmdcbiAgICogZm9yIGEgbmV3bHkgZGlzY292ZXJlZCBwYWNrYWdlLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHN0YXRlTWFjaGluZTogSVN0YXRlTWFjaGluZTtcblxuICAvKipcbiAgICogVGhlIGRlYWQgbGV0dGVyIHF1ZXVlIGZyb20gdGhlIHN0YXRlIG1hY2hpbmUuIElucHV0cyBhbmQgZXJyb3JzIGFyZSB3cml0dGVuXG4gICAqIHRoZXJlIGlmIHRoZSBzdGF0ZSBtYWNoaW5lIGZhaWxzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRlYWRMZXR0ZXJRdWV1ZTogSVF1ZXVlO1xuXG4gIC8qKlxuICAgKiBUaGUgZnVuY3Rpb24gb3BlcmF0b3JzIGNhbiB1c2UgdG8gcmVkcml2ZSBtZXNzYWdlcyBmcm9tIHRoZSBkZWFkIGxldHRlclxuICAgKiBxdWV1ZS5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSByZWRyaXZlRnVuY3Rpb246IElGdW5jdGlvbjtcblxuICAvKipcbiAgICogVGhlIGZ1bmN0aW9uIG9wZXJhdG9ycyBjYW4gdXNlIHRvIHJlcHJvY2VzcyBhbGwgaW5kZXhlZCBwYWNrYWdlcyB0aHJvdWdoXG4gICAqIHRoZSBiYWNrZW5kIGRhdGEgcGlwZWxpbmUuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcmVnZW5lcmF0ZUFsbERvY3VtZW50YXRpb246IElTdGF0ZU1hY2hpbmU7XG5cbiAgLyoqXG4gICAqIFRoZSBmdW5jdGlvbiB0aGF0IGJ1aWxkcyB0aGUgY2F0YWxvZy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBjYXRhbG9nQnVpbGRlcjogQ2F0YWxvZ0J1aWxkZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBFQ1MgY2x1c3RlciB1c2VkIHRvIHJ1biB0YXNrcy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBlY3NDbHVzdGVyOiBJQ2x1c3RlcjtcblxuICAvKipcbiAgICogVGhlIHRyYW5zbGl0ZXJhdG9yIHVzZWQgYnkgdGhpcyBvcmNoZXN0cmF0aW9uIHdvcmtmbG93LlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IHRyYW5zbGl0ZXJhdG9yOiBUcmFuc2xpdGVyYXRvcjtcblxuICBwdWJsaWMgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IE9yY2hlc3RyYXRpb25Qcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICB0aGlzLmRlYWRMZXR0ZXJRdWV1ZSA9IG5ldyBRdWV1ZSh0aGlzLCAnRExRJywge1xuICAgICAgZW5jcnlwdGlvbjogUXVldWVFbmNyeXB0aW9uLktNU19NQU5BR0VELFxuICAgICAgcmV0ZW50aW9uUGVyaW9kOiBEdXJhdGlvbi5kYXlzKDE0KSxcbiAgICAgIHZpc2liaWxpdHlUaW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDE1KSxcbiAgICB9KTtcblxuICAgIHByb3BzLm1vbml0b3JpbmcuYWRkSGlnaFNldmVyaXR5QWxhcm0oXG4gICAgICAnQmFja2VuZCBPcmNoZXN0cmF0aW9uIERlYWQtTGV0dGVyIFF1ZXVlIGlzIG5vdCBlbXB0eScsXG4gICAgICBuZXcgTWF0aEV4cHJlc3Npb24oe1xuICAgICAgICBleHByZXNzaW9uOiAnbTEgKyBtMicsXG4gICAgICAgIGxhYmVsOiAnRGVhZC1MZXR0ZXIgUXVldWUgbm90IGVtcHR5JyxcbiAgICAgICAgdXNpbmdNZXRyaWNzOiB7XG4gICAgICAgICAgbTE6IHRoaXMuZGVhZExldHRlclF1ZXVlLm1ldHJpY0FwcHJveGltYXRlTnVtYmVyT2ZNZXNzYWdlc1Zpc2libGUoeyBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoMSkgfSksXG4gICAgICAgICAgbTI6IHRoaXMuZGVhZExldHRlclF1ZXVlLm1ldHJpY0FwcHJveGltYXRlTnVtYmVyT2ZNZXNzYWdlc05vdFZpc2libGUoeyBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoMSkgfSksXG4gICAgICAgIH0sXG4gICAgICB9KS5jcmVhdGVBbGFybSh0aGlzLCAnRExRQWxhcm0nLCB7XG4gICAgICAgIGFsYXJtTmFtZTogYCR7dGhpcy5kZWFkTGV0dGVyUXVldWUubm9kZS5wYXRofS9Ob3RFbXB0eWAsXG4gICAgICAgIGFsYXJtRGVzY3JpcHRpb246IFtcbiAgICAgICAgICAnQmFja2VuZCBvcmNoZXN0cmF0aW9uIGRlYWQtbGV0dGVyIHF1ZXVlIGlzIG5vdCBlbXB0eS4nLFxuICAgICAgICAgICcnLFxuICAgICAgICAgIGBSdW5Cb29rOiAke1JVTkJPT0tfVVJMfWAsXG4gICAgICAgICAgJycsXG4gICAgICAgICAgYERpcmVjdCBsaW5rIHRvIHF1ZXVlOiAke3Nxc1F1ZXVlVXJsKHRoaXMuZGVhZExldHRlclF1ZXVlKX1gLFxuICAgICAgICAgICdXYXJuaW5nOiBTdGF0ZSBNYWNoaW5lcyBleGVjdXRpb25zIHRoYXQgc2VudCBtZXNzYWdlcyB0byB0aGUgRExRIHdpbGwgbm90IHNob3cgYXMgXCJmYWlsZWRcIi4nLFxuICAgICAgICBdLmpvaW4oJ1xcbicpLFxuICAgICAgICBjb21wYXJpc29uT3BlcmF0b3I6IENvbXBhcmlzb25PcGVyYXRvci5HUkVBVEVSX1RIQU5fT1JfRVFVQUxfVE9fVEhSRVNIT0xELFxuICAgICAgICBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICAgICAgdGhyZXNob2xkOiAxLFxuICAgICAgfSksXG4gICAgKTtcblxuICAgIGNvbnN0IHNlbmRUb0RlYWRMZXR0ZXJRdWV1ZSA9IG5ldyB0YXNrcy5TcXNTZW5kTWVzc2FnZSh0aGlzLCAnU2VuZCB0byBEZWFkIExldHRlciBRdWV1ZScsIHtcbiAgICAgIG1lc3NhZ2VCb2R5OiBUYXNrSW5wdXQuZnJvbUpzb25QYXRoQXQoJyQnKSxcbiAgICAgIHF1ZXVlOiB0aGlzLmRlYWRMZXR0ZXJRdWV1ZSxcbiAgICAgIHJlc3VsdFBhdGg6IEpzb25QYXRoLkRJU0NBUkQsXG4gICAgfSkubmV4dChuZXcgU3VjY2VlZCh0aGlzLCAnU2VudCB0byBETFEnKSk7XG5cbiAgICB0aGlzLmNhdGFsb2dCdWlsZGVyID0gbmV3IENhdGFsb2dCdWlsZGVyKHRoaXMsICdDYXRhbG9nQnVpbGRlcicsIHByb3BzKTtcblxuICAgIGNvbnN0IGFkZFRvQ2F0YWxvZyA9IG5ldyB0YXNrcy5MYW1iZGFJbnZva2UodGhpcywgJ0FkZCB0byBjYXRhbG9nLmpzb24nLCB7XG4gICAgICBsYW1iZGFGdW5jdGlvbjogdGhpcy5jYXRhbG9nQnVpbGRlci5mdW5jdGlvbixcbiAgICAgIHJlc3VsdFBhdGg6ICckLmNhdGFsb2dCdWlsZGVyT3V0cHV0JyxcbiAgICAgIHJlc3VsdFNlbGVjdG9yOiB7XG4gICAgICAgICdFVGFnLiQnOiAnJC5QYXlsb2FkLkVUYWcnLFxuICAgICAgICAnVmVyc2lvbklkLiQnOiAnJC5QYXlsb2FkLlZlcnNpb25JZCcsXG4gICAgICB9LFxuICAgIH0pXG4gICAgICAvLyBUaGlzIGhhcyBhIGNvbmN1cnJlbmN5IG9mIDEsIHNvIHdlIHdhbnQgdG8gYWdncmVzc2l2ZWx5IHJldHJ5IGJlaW5nIHRocm90dGxlZCBoZXJlLlxuICAgICAgLmFkZFJldHJ5KHsgZXJyb3JzOiBbJ0xhbWJkYS5Ub29NYW55UmVxdWVzdHNFeGNlcHRpb24nXSwgLi4uVEhST1RUTEVfUkVUUllfUE9MSUNZIH0pXG4gICAgICAuYWRkQ2F0Y2goXG4gICAgICAgIG5ldyBQYXNzKHRoaXMsICdcIkFkZCB0byBjYXRhbG9nLmpzb25cIiB0aHJvdHRsZWQnLCB7XG4gICAgICAgICAgcGFyYW1ldGVyczogeyAnZXJyb3IuJCc6ICckLkNhdXNlJyB9LFxuICAgICAgICAgIHJlc3VsdFBhdGg6ICckLmVycm9yJyxcbiAgICAgICAgfSkubmV4dChzZW5kVG9EZWFkTGV0dGVyUXVldWUpLFxuICAgICAgICB7IGVycm9yczogWydMYW1iZGEuVG9vTWFueVJlcXVlc3RzRXhjZXB0aW9uJ10gfSxcbiAgICAgIClcbiAgICAgIC5hZGRDYXRjaChcbiAgICAgICAgbmV3IFBhc3ModGhpcywgJ1wiQWRkIHRvIGNhdGFsb2cuanNvblwiIGZhaWx1cmUnLCB7XG4gICAgICAgICAgcGFyYW1ldGVyczogeyAnZXJyb3IuJCc6ICdTdGF0ZXMuU3RyaW5nVG9Kc29uKCQuQ2F1c2UpJyB9LFxuICAgICAgICAgIHJlc3VsdFBhdGg6ICckLmVycm9yJyxcbiAgICAgICAgfSkubmV4dChzZW5kVG9EZWFkTGV0dGVyUXVldWUpLFxuICAgICAgICB7IGVycm9yczogWydTdGF0ZXMuVGFza0ZhaWxlZCddIH0sXG4gICAgICApXG4gICAgICAuYWRkQ2F0Y2gobmV3IFBhc3ModGhpcywgJ1wiQWRkIHRvIGNhdGFsb2cuanNvblwiIGZhdWx0Jywge1xuICAgICAgICBwYXJhbWV0ZXJzOiB7ICdlcnJvci4kJzogJyQuQ2F1c2UnIH0sXG4gICAgICAgIHJlc3VsdFBhdGg6ICckLmVycm9yJyxcbiAgICAgIH0pLm5leHQoc2VuZFRvRGVhZExldHRlclF1ZXVlKSwgeyBlcnJvcnM6IFsnU3RhdGVzLkFMTCddIH0pO1xuXG4gICAgdGhpcy5lY3NDbHVzdGVyID0gbmV3IENsdXN0ZXIodGhpcywgJ0NsdXN0ZXInLCB7XG4gICAgICBjb250YWluZXJJbnNpZ2h0czogdHJ1ZSxcbiAgICAgIGVuYWJsZUZhcmdhdGVDYXBhY2l0eVByb3ZpZGVyczogdHJ1ZSxcbiAgICAgIHZwYzogcHJvcHMudnBjLFxuICAgIH0pO1xuXG4gICAgdGhpcy50cmFuc2xpdGVyYXRvciA9IG5ldyBUcmFuc2xpdGVyYXRvcih0aGlzLCAnVHJhbnNsaXRlcmF0b3InLCBwcm9wcyk7XG5cbiAgICBjb25zdCBkZWZpbml0aW9uID0gbmV3IFBhc3ModGhpcywgJ1RyYWNrIEV4ZWN1dGlvbiBJbmZvcycsIHtcbiAgICAgIGlucHV0UGF0aDogJyQkLkV4ZWN1dGlvbicsXG4gICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICdJZC4kJzogJyQuSWQnLFxuICAgICAgICAnTmFtZS4kJzogJyQuTmFtZScsXG4gICAgICAgICdSb2xlQXJuLiQnOiAnJC5Sb2xlQXJuJyxcbiAgICAgICAgJ1N0YXJ0VGltZS4kJzogJyQuU3RhcnRUaW1lJyxcbiAgICAgIH0sXG4gICAgICByZXN1bHRQYXRoOiAnJC4kVGFza0V4ZWN1dGlvbicsXG4gICAgfSlcbiAgICAgIC5uZXh0KFxuICAgICAgICBuZXcgUGFzcyh0aGlzLCAnUHJlcGFyZSBkb2MtZ2VuIEVDUyBDb21tYW5kJywge1xuICAgICAgICAgIHBhcmFtZXRlcnM6IHsgJ2NvbW1hbmQuJCc6ICdTdGF0ZXMuQXJyYXkoU3RhdGVzLkpzb25Ub1N0cmluZygkKSknIH0sXG4gICAgICAgICAgcmVzdWx0UGF0aDogJyQuZG9jR2VuJyxcbiAgICAgICAgfSksXG4gICAgICApXG4gICAgICAubmV4dChcbiAgICAgICAgdGhpcy50cmFuc2xpdGVyYXRvci5jcmVhdGVFY3NSdW5UYXNrKHRoaXMsICdHZW5lcmF0ZSBkb2NzJywge1xuICAgICAgICAgIGNsdXN0ZXI6IHRoaXMuZWNzQ2x1c3RlcixcbiAgICAgICAgICBpbnB1dFBhdGg6ICckLmRvY0dlbi5jb21tYW5kJyxcbiAgICAgICAgICByZXN1bHRQYXRoOiAnJC5kb2NHZW5PdXRwdXQnLFxuICAgICAgICAgIC8vIEV4cGVjdCB0aGlzIHRvIGNvbXBsZXRlIHdpdGhpbiBvbmUgaG91clxuICAgICAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLmhvdXJzKDEpLFxuICAgICAgICAgIHZwY1N1Ym5ldHM6IHByb3BzLnZwY1N1Ym5ldHMsXG4gICAgICAgIH0pXG4gICAgICAgIC8vIERvIG5vdCByZXRyeSBOb1NwYWNlTGVmdE9uRGV2aWNlIGVycm9ycywgdGhlc2UgYXJlIHR5cGljYWxseSBub3QgdHJhbnNpZW50LlxuICAgICAgICAgIC5hZGRSZXRyeSh7IGVycm9yczogWydqc2lpLWRvY2dlbi5Ob1NwYWNlTGVmdE9uRGV2aWNlJ10sIG1heEF0dGVtcHRzOiAwIH0pXG4gICAgICAgICAgLmFkZFJldHJ5KHtcbiAgICAgICAgICAgIGVycm9yczogW1xuICAgICAgICAgICAgICAnRUNTLkFtYXpvbkVDU0V4Y2VwdGlvbicsIC8vIFRhc2sgZmFpbGVkIHN0YXJ0aW5nLCB1c3VhbGx5IGR1ZSB0byB0aHJvdHRsZSAvIG91dCBvZiBjYXBhY2l0eVxuICAgICAgICAgICAgICAnRUNTLkludmFsaWRQYXJhbWV0ZXJFeGNlcHRpb24nLCAvLyBUaGlzIGlzIHJldHVybmVkIHdoZW4gRUNTIGdldHMgdGhyb3R0bGVkIHdoZW4gdHJ5aW5nIHRvIGFjY2VzcyBWUEMvU0dzLlxuICAgICAgICAgICAgICAnanNpaS1kb2NnZW4uTnBtRXJyb3IuRTQyOScsIC8vIEhUVFAgNDI5IChcIlRvbyBNYW55IFJlcXVlc3RzXCIpIGZyb20gQ29kZUFydGlmYWN0J3MgUzMgYnVja2V0XG4gICAgICAgICAgICAgICdqc2lpLWNvZGdlbi5OcG1FcnJvci5FUFJPVE8nLCAvLyBTcG9yYWRpYyBUTFMgbmVnb3RpYXRpb24gZmFpbHVyZXMgd2Ugc2VlIGluIGxvZ3MsIHRyYW5zaWVudFxuICAgICAgICAgICAgXSxcbiAgICAgICAgICAgIC4uLlRIUk9UVExFX1JFVFJZX1BPTElDWSxcbiAgICAgICAgICB9KVxuICAgICAgICAgIC5hZGRSZXRyeSh7XG4gICAgICAgICAgICBlcnJvcnM6IFsnanNpaS1kb2NnZW4uTnBtRXJyb3IuRVRBUkdFVCddLCAvLyBTZWVuIHdoZW4gZGVwZW5kZW5jaWVzIGFyZW4ndCBhdmFpbGFibGUgeWV0XG4gICAgICAgICAgICAvLyBXZSdsbCB3YWl0IGxvbmdlciBiZXR3ZWVuIHJldHJpZXMuIFRoaXMgaXMgdG8gYWNjb3VudCBmb3IgQ29kZUFydGlmYWN0J3MgbGFnIGJlaGluZCBucG1cbiAgICAgICAgICAgIGJhY2tvZmZSYXRlOiAyLFxuICAgICAgICAgICAgaW50ZXJ2YWw6IER1cmF0aW9uLm1pbnV0ZXMoNSksXG4gICAgICAgICAgICBtYXhBdHRlbXB0czogMyxcbiAgICAgICAgICB9KVxuICAgICAgICAgIC5hZGRSZXRyeSh7IG1heEF0dGVtcHRzOiAzIH0pXG4gICAgICAgICAgLmFkZENhdGNoKFxuICAgICAgICAgICAgbmV3IFBhc3ModGhpcywgJ1wiR2VuZXJhdGUgZG9jc1wiIHRpbWVkIG91dCcsIHsgcGFyYW1ldGVyczogeyBlcnJvcjogJ1RpbWVkIG91dCEnIH0gfSlcbiAgICAgICAgICAgICAgLm5leHQoc2VuZFRvRGVhZExldHRlclF1ZXVlKSxcbiAgICAgICAgICAgIHsgZXJyb3JzOiBbJ1N0YXRlcy5UaW1lb3V0J10gfSxcbiAgICAgICAgICApXG4gICAgICAgICAgLmFkZENhdGNoKFxuICAgICAgICAgICAgbmV3IFBhc3ModGhpcywgJ1wiR2VuZXJhdGUgZG9jc1wiIHNlcnZpY2UgZXJyb3InLCB7IHBhcmFtZXRlcnM6IHsgJ2Vycm9yLiQnOiAnJC5DYXVzZScgfSB9KVxuICAgICAgICAgICAgICAubmV4dChzZW5kVG9EZWFkTGV0dGVyUXVldWUpLFxuICAgICAgICAgICAgeyBlcnJvcnM6IFsnRUNTLkFtYXpvbkVDU0V4Y2VwdGlvbicsICdFQ1MuSW52YWxpZFBhcmFtZXRlckV4Y2VwdGlvbiddIH0sXG4gICAgICAgICAgKVxuICAgICAgICAgIC5hZGRDYXRjaChcbiAgICAgICAgICAgIG5ldyBQYXNzKHRoaXMsICdcIkdlbmVyYXRlIGRvY3NcIiBmYWlsdXJlJywgeyBwYXJhbWV0ZXJzOiB7ICdlcnJvci4kJzogJ1N0YXRlcy5TdHJpbmdUb0pzb24oJC5DYXVzZSknIH0gfSlcbiAgICAgICAgICAgICAgLm5leHQoc2VuZFRvRGVhZExldHRlclF1ZXVlKSxcbiAgICAgICAgICAgIHsgZXJyb3JzOiBbJ1N0YXRlcy5UYXNrRmFpbGVkJ10gfSxcbiAgICAgICAgICApXG4gICAgICAgICAgLmFkZENhdGNoKFxuICAgICAgICAgICAgbmV3IFBhc3ModGhpcywgJ1wiR2VuZXJhdGUgZG9jc1wiIGZhdWx0JywgeyBwYXJhbWV0ZXJzOiB7ICdlcnJvci4kJzogJyQuQ2F1c2UnIH0gfSlcbiAgICAgICAgICAgICAgLm5leHQoc2VuZFRvRGVhZExldHRlclF1ZXVlKSxcbiAgICAgICAgICAgIHsgZXJyb3JzOiBbJ1N0YXRlcy5BTEwnXSB9LFxuICAgICAgICAgICkubmV4dChhZGRUb0NhdGFsb2cpLFxuICAgICAgKTtcblxuICAgIHRoaXMuc3RhdGVNYWNoaW5lID0gbmV3IFN0YXRlTWFjaGluZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBkZWZpbml0aW9uLFxuICAgICAgc3RhdGVNYWNoaW5lTmFtZTogc3RhdGVNYWNoaW5lTmFtZUZyb20odGhpcy5ub2RlLnBhdGgpLFxuICAgICAgdGltZW91dDogRHVyYXRpb24uZGF5cygxKSwgLy8gQW1wbGUgdGltZSBmb3IgcmV0cmllcywgZXRjLi4uXG4gICAgICB0cmFjaW5nRW5hYmxlZDogdHJ1ZSxcbiAgICB9KTtcblxuICAgIGlmIChwcm9wcy52cGMpIHtcbiAgICAgIC8vIEVuc3VyZSB0aGUgU3RhdGUgTWFjaGluZSBkb2VzIG5vdCBnZXQgdG8gcnVuIGJlZm9yZSB0aGUgVlBDIGNhbiBiZSB1c2VkLlxuICAgICAgdGhpcy5zdGF0ZU1hY2hpbmUubm9kZS5hZGREZXBlbmRlbmN5KHByb3BzLnZwYy5pbnRlcm5ldENvbm5lY3Rpdml0eUVzdGFibGlzaGVkKTtcbiAgICB9XG5cbiAgICBwcm9wcy5tb25pdG9yaW5nLmFkZEhpZ2hTZXZlcml0eUFsYXJtKFxuICAgICAgJ0JhY2tlbmQgT3JjaGVzdHJhdGlvbiBGYWlsZWQnLFxuICAgICAgdGhpcy5zdGF0ZU1hY2hpbmUubWV0cmljRmFpbGVkKClcbiAgICAgICAgLmNyZWF0ZUFsYXJtKHRoaXMsICdPcmNoZXN0cmF0aW9uRmFpbGVkJywge1xuICAgICAgICAgIGFsYXJtTmFtZTogYCR7dGhpcy5zdGF0ZU1hY2hpbmUubm9kZS5wYXRofS8ke3RoaXMuc3RhdGVNYWNoaW5lLm1ldHJpY0ZhaWxlZCgpLm1ldHJpY05hbWV9YCxcbiAgICAgICAgICBhbGFybURlc2NyaXB0aW9uOiBbXG4gICAgICAgICAgICAnQmFja2VuZCBvcmNoZXN0cmF0aW9uIGZhaWxlZCEnLFxuICAgICAgICAgICAgJycsXG4gICAgICAgICAgICBgUnVuQm9vazogJHtSVU5CT09LX1VSTH1gLFxuICAgICAgICAgICAgJycsXG4gICAgICAgICAgICBgRGlyZWN0IGxpbmsgdG8gc3RhdGUgbWFjaGluZTogJHtzdGF0ZU1hY2hpbmVVcmwodGhpcy5zdGF0ZU1hY2hpbmUpfWAsXG4gICAgICAgICAgICAnV2FybmluZzogbWVzc2FnZXMgdGhhdCByZXN1bHRlZCBpbiBhIGZhaWxlZCBleGVjdHVpb24gd2lsbCBOT1QgYmUgaW4gdGhlIERMUSEnLFxuICAgICAgICAgIF0uam9pbignXFxuJyksXG4gICAgICAgICAgY29tcGFyaXNvbk9wZXJhdG9yOiBDb21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICAgICAgICBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICAgICAgICB0aHJlc2hvbGQ6IDEsXG4gICAgICAgIH0pKTtcblxuICAgIC8vIFRoaXMgZnVuY3Rpb24gaXMgaW50ZW5kZWQgdG8gYmUgbWFudWFsbHkgdHJpZ2dlcmVkIGJ5IGFuIG9wZXJyYXRvciB0b1xuICAgIC8vIGF0dGVtcHQgcmVkcml2aW5nIG1lc3NhZ2VzIGZyb20gdGhlIERMUS5cbiAgICB0aGlzLnJlZHJpdmVGdW5jdGlvbiA9IG5ldyBSZWRyaXZlU3RhdGVNYWNoaW5lKHRoaXMsICdSZWRyaXZlJywge1xuICAgICAgZGVzY3JpcHRpb246ICdbQ29uc3RydWN0SHViL1JlZHJpdmVdIE1hbnVhbGx5IHJlZHJpdmVzIGFsbCBtZXNzYWdlcyBmcm9tIHRoZSBiYWNrZW5kIGRlYWQgbGV0dGVyIHF1ZXVlJyxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIFNUQVRFX01BQ0hJTkVfQVJOOiB0aGlzLnN0YXRlTWFjaGluZS5zdGF0ZU1hY2hpbmVBcm4sXG4gICAgICAgIFFVRVVFX1VSTDogdGhpcy5kZWFkTGV0dGVyUXVldWUucXVldWVVcmwsXG4gICAgICB9LFxuICAgICAgbWVtb3J5U2l6ZTogMV8wMjQsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDE1KSxcbiAgICAgIHRyYWNpbmc6IFRyYWNpbmcuQUNUSVZFLFxuICAgIH0pO1xuICAgIHRoaXMuc3RhdGVNYWNoaW5lLmdyYW50U3RhcnRFeGVjdXRpb24odGhpcy5yZWRyaXZlRnVuY3Rpb24pO1xuICAgIHRoaXMuZGVhZExldHRlclF1ZXVlLmdyYW50Q29uc3VtZU1lc3NhZ2VzKHRoaXMucmVkcml2ZUZ1bmN0aW9uKTtcblxuICAgIC8vIFRoZSB3b3JrZmxvdyBpcyBpbnRlbmRlZCB0byBiZSBtYW51YWxseSB0cmlnZ2VyZWQgYnkgYW4gb3BlcmF0b3IgdG9cbiAgICAvLyByZXByb2Nlc3MgYWxsIHBhY2thZ2UgdmVyc2lvbnMgY3VycmVudGx5IGluIHN0b3JlIHRocm91Z2ggdGhlIG9yY2hlc3RyYXRvci5cbiAgICB0aGlzLnJlZ2VuZXJhdGVBbGxEb2N1bWVudGF0aW9uID0gbmV3IFJlZ2VuZXJhdGVBbGxEb2N1bWVudGF0aW9uKHRoaXMsICdSZWdlbmVyYXRlQWxsRG9jdW1lbnRhdGlvbicsIHtcbiAgICAgIGJ1Y2tldDogcHJvcHMuYnVja2V0LFxuICAgICAgc3RhdGVNYWNoaW5lOiB0aGlzLnN0YXRlTWFjaGluZSxcbiAgICB9KS5zdGF0ZU1hY2hpbmU7XG4gIH1cblxuICBwdWJsaWMgbWV0cmljRWNzVGFza0NvdW50KG9wdHM6IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiBuZXcgTWV0cmljKHtcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLlNVTSxcbiAgICAgIC4uLm9wdHMsXG4gICAgICBkaW1lbnNpb25zTWFwOiB7IENsdXN0ZXJOYW1lOiB0aGlzLmVjc0NsdXN0ZXIuY2x1c3Rlck5hbWUgfSxcbiAgICAgIG1ldHJpY05hbWU6ICdUYXNrQ291bnQnLFxuICAgICAgbmFtZXNwYWNlOiAnRUNTL0NvbnRhaW5lckluc2lnaHRzJyxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBtZXRyaWNFY3NDcHVSZXNlcnZlZChvcHRzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIG5ldyBNZXRyaWMoe1xuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuTUFYSU1VTSxcbiAgICAgIC4uLm9wdHMsXG4gICAgICBkaW1lbnNpb25zTWFwOiB7IENsdXN0ZXJOYW1lOiB0aGlzLmVjc0NsdXN0ZXIuY2x1c3Rlck5hbWUgfSxcbiAgICAgIG1ldHJpY05hbWU6ICdDcHVSZXNlcnZlZCcsXG4gICAgICBuYW1lc3BhY2U6ICdFQ1MvQ29udGFpbmVySW5zaWdodHMnLFxuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIG1ldHJpY0Vjc0NwdVV0aWxpemVkKG9wdHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gbmV3IE1ldHJpYyh7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgICAgLi4ub3B0cyxcbiAgICAgIGRpbWVuc2lvbnNNYXA6IHsgQ2x1c3Rlck5hbWU6IHRoaXMuZWNzQ2x1c3Rlci5jbHVzdGVyTmFtZSB9LFxuICAgICAgbWV0cmljTmFtZTogJ0NwdVV0aWxpemVkJyxcbiAgICAgIG5hbWVzcGFjZTogJ0VDUy9Db250YWluZXJJbnNpZ2h0cycsXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgbWV0cmljRWNzQ3B1VXRpbGl6YXRpb24ob3B0cz86IE1hdGhFeHByZXNzaW9uT3B0aW9ucyk6IE1hdGhFeHByZXNzaW9uIHtcbiAgICByZXR1cm4gbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgIC4uLm9wdHMsXG4gICAgICAvLyBDYWxjdWxhdGVzIHRoZSAlIENQVSB1dGlsaXphdGlvbiBmcm9tIHRoZSBDUFUgdW5pdHMgdXRpbGl6YXRpb24gJlxuICAgICAgLy8gcmVzZXJ2YXRpb24uIEZJTEwgaXMgdXNlZCB0byBtYWtlIGEgbm9uLXNwYXJzZSB0aW1lLXNlcmllcyAodGhlIG1ldHJpY3NcbiAgICAgIC8vIGFyZSBub3QgZW1pdHRlZCBpZiBubyB0YXNrIHJ1bnMpXG4gICAgICBleHByZXNzaW9uOiAnMTAwICogRklMTChtQ3B1VXRpbGl6ZWQsIDApIC8gRklMTChtQ3B1UmVzZXJ2ZWQsIFJFUEVBVCknLFxuICAgICAgdXNpbmdNZXRyaWNzOiB7XG4gICAgICAgIG1DcHVSZXNlcnZlZDogdGhpcy5tZXRyaWNFY3NDcHVSZXNlcnZlZCgpLFxuICAgICAgICBtQ3B1VXRpbGl6ZWQ6IHRoaXMubWV0cmljRWNzQ3B1VXRpbGl6ZWQoKSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgbWV0cmljRWNzTWVtb3J5UmVzZXJ2ZWQob3B0cz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiBuZXcgTWV0cmljKHtcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLk1BWElNVU0sXG4gICAgICAuLi5vcHRzLFxuICAgICAgZGltZW5zaW9uc01hcDogeyBDbHVzdGVyTmFtZTogdGhpcy5lY3NDbHVzdGVyLmNsdXN0ZXJOYW1lIH0sXG4gICAgICBtZXRyaWNOYW1lOiAnTWVtb3J5UmVzZXJ2ZWQnLFxuICAgICAgbmFtZXNwYWNlOiAnRUNTL0NvbnRhaW5lckluc2lnaHRzJyxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBtZXRyaWNFY3NNZW1vcnlVdGlsaXplZChvcHRzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIG5ldyBNZXRyaWMoe1xuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuTUFYSU1VTSxcbiAgICAgIC4uLm9wdHMsXG4gICAgICBkaW1lbnNpb25zTWFwOiB7IENsdXN0ZXJOYW1lOiB0aGlzLmVjc0NsdXN0ZXIuY2x1c3Rlck5hbWUgfSxcbiAgICAgIG1ldHJpY05hbWU6ICdNZW1vcnlVdGlsaXplZCcsXG4gICAgICBuYW1lc3BhY2U6ICdFQ1MvQ29udGFpbmVySW5zaWdodHMnLFxuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIG1ldHJpY0Vjc01lbW9yeVV0aWxpemF0aW9uKG9wdHM/OiBNYXRoRXhwcmVzc2lvbk9wdGlvbnMpOiBNYXRoRXhwcmVzc2lvbiB7XG4gICAgcmV0dXJuIG5ldyBNYXRoRXhwcmVzc2lvbih7XG4gICAgICAuLi5vcHRzLFxuICAgICAgLy8gQ2FsY3VsYXRlcyB0aGUgJSBtZW1vcnkgdXRpbGl6YXRpb24gZnJvbSB0aGUgUkFNIHV0aWxpemF0aW9uICZcbiAgICAgIC8vIHJlc2VydmF0aW9uLiBGSUxMIGlzIHVzZWQgdG8gbWFrZSBhIG5vbi1zcGFyc2UgdGltZS1zZXJpZXMgKHRoZSBtZXRyaWNzXG4gICAgICAvLyBhcmUgbm90IGVtaXR0ZWQgaWYgbm8gdGFzayBydW5zKVxuICAgICAgZXhwcmVzc2lvbjogJzEwMCAqIEZJTEwobU1lbW9yeVV0aWxpemVkLCAwKSAvIEZJTEwobU1lbW9yeVJlc2VydmVkLCBSRVBFQVQpJyxcbiAgICAgIHVzaW5nTWV0cmljczoge1xuICAgICAgICBtTWVtb3J5UmVzZXJ2ZWQ6IHRoaXMubWV0cmljRWNzTWVtb3J5UmVzZXJ2ZWQoKSxcbiAgICAgICAgbU1lbW9yeVV0aWxpemVkOiB0aGlzLm1ldHJpY0Vjc01lbW9yeVV0aWxpemVkKCksXG4gICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIG1ldHJpY0Vjc05ldHdvcmtSeEJ5dGVzKG9wdHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gbmV3IE1ldHJpYyh7XG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5NQVhJTVVNLFxuICAgICAgLi4ub3B0cyxcbiAgICAgIGRpbWVuc2lvbnNNYXA6IHsgQ2x1c3Rlck5hbWU6IHRoaXMuZWNzQ2x1c3Rlci5jbHVzdGVyTmFtZSB9LFxuICAgICAgbWV0cmljTmFtZTogJ05ldHdvcmtSeEJ5dGVzJyxcbiAgICAgIG5hbWVzcGFjZTogJ0VDUy9Db250YWluZXJJbnNpZ2h0cycsXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgbWV0cmljRWNzTmV0d29ya1R4Qnl0ZXMob3B0cz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiBuZXcgTWV0cmljKHtcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLk1BWElNVU0sXG4gICAgICAuLi5vcHRzLFxuICAgICAgZGltZW5zaW9uc01hcDogeyBDbHVzdGVyTmFtZTogdGhpcy5lY3NDbHVzdGVyLmNsdXN0ZXJOYW1lIH0sXG4gICAgICBtZXRyaWNOYW1lOiAnTmV0d29ya1R4Qnl0ZXMnLFxuICAgICAgbmFtZXNwYWNlOiAnRUNTL0NvbnRhaW5lckluc2lnaHRzJyxcbiAgICB9KTtcbiAgfVxufVxuXG5pbnRlcmZhY2UgUmVnZW5lcmF0ZUFsbERvY3VtZW50YXRpb25Qcm9wcyB7XG4gIHJlYWRvbmx5IGJ1Y2tldDogSUJ1Y2tldDtcbiAgcmVhZG9ubHkgc3RhdGVNYWNoaW5lOiBJU3RhdGVNYWNoaW5lO1xufVxuXG5jbGFzcyBSZWdlbmVyYXRlQWxsRG9jdW1lbnRhdGlvbiBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHB1YmxpYyByZWFkb25seSBzdGF0ZU1hY2hpbmU6IFN0YXRlTWFjaGluZTtcblxuICBwdWJsaWMgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IFJlZ2VuZXJhdGVBbGxEb2N1bWVudGF0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgY29uc3QgcHJvY2Vzc1ZlcnNpb25zID0gbmV3IENob2ljZSh0aGlzLCAnR2V0IHBhY2thZ2UgdmVyc2lvbnMgcGFnZScpXG4gICAgICAud2hlbihDb25kaXRpb24uaXNQcmVzZW50KCckLnJlc3BvbnNlLk5leHRDb250aW51YXRpb25Ub2tlbicpLCBuZXcgdGFza3MuQ2FsbEF3c1NlcnZpY2UodGhpcywgJ05leHQgdmVyc2lvbnMgcGFnZScsIHtcbiAgICAgICAgc2VydmljZTogJ3MzJyxcbiAgICAgICAgYWN0aW9uOiAnbGlzdE9iamVjdHNWMicsXG4gICAgICAgIGlhbUFjdGlvbjogJ3MzOkxpc3RCdWNrZXQnLFxuICAgICAgICBpYW1SZXNvdXJjZXM6IFtwcm9wcy5idWNrZXQuYnVja2V0QXJuXSxcbiAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgIEJ1Y2tldDogcHJvcHMuYnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgICAgQ29udGludWF0aW9uVG9rZW46IEpzb25QYXRoLnN0cmluZ0F0KCckLnJlc3BvbnNlLk5leHRDb250aW51YXRpb25Ub2tlbicpLFxuICAgICAgICAgIERlbGltaXRlcjogJy8nLFxuICAgICAgICAgIFByZWZpeDogSnNvblBhdGguc3RyaW5nQXQoJyQuUHJlZml4JyksXG4gICAgICAgIH0sXG4gICAgICAgIHJlc3VsdFBhdGg6ICckLnJlc3BvbnNlJyxcbiAgICAgIH0pKVxuICAgICAgLm90aGVyd2lzZShuZXcgdGFza3MuQ2FsbEF3c1NlcnZpY2UodGhpcywgJ0ZpcnN0IHZlcnNpb25zIHBhZ2UnLCB7XG4gICAgICAgIHNlcnZpY2U6ICdzMycsXG4gICAgICAgIGFjdGlvbjogJ2xpc3RPYmplY3RzVjInLFxuICAgICAgICBpYW1BY3Rpb246ICdzMzpMaXN0QnVja2V0JyxcbiAgICAgICAgaWFtUmVzb3VyY2VzOiBbcHJvcHMuYnVja2V0LmJ1Y2tldEFybl0sXG4gICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICBCdWNrZXQ6IHByb3BzLmJ1Y2tldC5idWNrZXROYW1lLFxuICAgICAgICAgIERlbGltaXRlcjogJy8nLFxuICAgICAgICAgIFByZWZpeDogSnNvblBhdGguc3RyaW5nQXQoJyQuUHJlZml4JyksXG4gICAgICAgIH0sXG4gICAgICAgIHJlc3VsdFBhdGg6ICckLnJlc3BvbnNlJyxcbiAgICAgIH0pKVxuICAgICAgLmFmdGVyd2FyZHMoKVxuICAgICAgLm5leHQobmV3IE1hcCh0aGlzLCAnRm9yIGVhY2gga2V5IHByZWZpeCcsIHsgaXRlbXNQYXRoOiAnJC5yZXNwb25zZS5Db21tb25QcmVmaXhlcycsIHJlc3VsdFBhdGg6IEpzb25QYXRoLkRJU0NBUkQgfSlcbiAgICAgICAgLml0ZXJhdG9yKG5ldyB0YXNrcy5TdGVwRnVuY3Rpb25zU3RhcnRFeGVjdXRpb24odGhpcywgJ1N0YXJ0IE9yY2hlc3RyYXRpb24gV29ya2Zsb3cnLCB7XG4gICAgICAgICAgc3RhdGVNYWNoaW5lOiBwcm9wcy5zdGF0ZU1hY2hpbmUsXG4gICAgICAgICAgYXNzb2NpYXRlV2l0aFBhcmVudDogdHJ1ZSxcbiAgICAgICAgICBpbnB1dDogVGFza0lucHV0LmZyb21PYmplY3Qoe1xuICAgICAgICAgICAgYnVja2V0OiBwcm9wcy5idWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAgICAgIGFzc2VtYmx5OiB7IGtleTogSnNvblBhdGguc3RyaW5nQXQoYFN0YXRlcy5Gb3JtYXQoJ3t9JHtBU1NFTUJMWV9LRVlfU1VGRklYLnN1YnN0cigxKX0nLCAkLlByZWZpeClgKSB9LFxuICAgICAgICAgICAgbWV0YWRhdGE6IHsga2V5OiBKc29uUGF0aC5zdHJpbmdBdChgU3RhdGVzLkZvcm1hdCgne30ke01FVEFEQVRBX0tFWV9TVUZGSVguc3Vic3RyKDEpfScsICQuUHJlZml4KWApIH0sXG4gICAgICAgICAgICBwYWNrYWdlOiB7IGtleTogSnNvblBhdGguc3RyaW5nQXQoYFN0YXRlcy5Gb3JtYXQoJ3t9JHtQQUNLQUdFX0tFWV9TVUZGSVguc3Vic3RyKDEpfScsICQuUHJlZml4KWApIH0sXG4gICAgICAgICAgfSksXG4gICAgICAgICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBJbnRlZ3JhdGlvblBhdHRlcm4uUkVRVUVTVF9SRVNQT05TRSxcbiAgICAgICAgfSkuYWRkUmV0cnkoeyBlcnJvcnM6IFsnU3RlcEZ1bmN0aW9ucy5FeGVjdXRpb25MaW1pdEV4Y2VlZGVkJ10gfSkpKTtcbiAgICBwcm9jZXNzVmVyc2lvbnMubmV4dChuZXcgQ2hvaWNlKHRoaXMsICdIYXMgbW9yZSB2ZXJzaW9ucz8nKVxuICAgICAgLndoZW4oQ29uZGl0aW9uLmlzUHJlc2VudCgnJC5yZXNwb25zZS5OZXh0Q29udGludWF0aW9uVG9rZW4nKSwgcHJvY2Vzc1ZlcnNpb25zKVxuICAgICAgLm90aGVyd2lzZShuZXcgU3VjY2VlZCh0aGlzLCAnU3VjY2VzcycpKSk7XG4gICAgY29uc3QgcHJvY2Vzc1BhY2thZ2VWZXJzaW9ucyA9IG5ldyBTdGF0ZU1hY2hpbmUodGhpcywgJ1BlclBhY2thZ2UnLCB7XG4gICAgICBkZWZpbml0aW9uOiBwcm9jZXNzVmVyc2lvbnMsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5ob3VycygxKSxcbiAgICAgIHRyYWNpbmdFbmFibGVkOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgLy8gVGhpcyB3b3JrZmxvdyBpcyBicm9rZW4gaW50byB0d28gc3ViLXdvcmtmbG93cyBiZWNhdXNlIG90aGVyd2lzZSBpdCBoaXRzIHRoZSAyNUsgZXZlbnRzIGxpbWl0XG4gICAgLy8gb2YgU3RlcEZ1bmN0aW9uIGV4ZWN1dGlvbnMgcmVsYXRpdmVseSBxdWlja2x5LlxuICAgIGNvbnN0IHByb2Nlc3NOYW1lc3BhY2UgPSBuZXcgQ2hvaWNlKHRoaXMsICdHZXQgQHNjb3BlIHBhZ2UnKVxuICAgICAgLndoZW4oQ29uZGl0aW9uLmlzUHJlc2VudCgnJC5yZXNwb25zZS5OZXh0Q29udGludWF0aW9uVG9rZW4nKSwgbmV3IHRhc2tzLkNhbGxBd3NTZXJ2aWNlKHRoaXMsICdOZXh0IEBzY29wZSBwYWdlJywge1xuICAgICAgICBzZXJ2aWNlOiAnczMnLFxuICAgICAgICBhY3Rpb246ICdsaXN0T2JqZWN0c1YyJyxcbiAgICAgICAgaWFtQWN0aW9uOiAnczM6TGlzdEJ1Y2tldCcsXG4gICAgICAgIGlhbVJlc291cmNlczogW3Byb3BzLmJ1Y2tldC5idWNrZXRBcm5dLFxuICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgQnVja2V0OiBwcm9wcy5idWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAgICBDb250aW51YXRpb25Ub2tlbjogSnNvblBhdGguc3RyaW5nQXQoJyQucmVzcG9uc2UuTmV4dENvbnRpbnVhdGlvblRva2VuJyksXG4gICAgICAgICAgRGVsaW1pdGVyOiAnLycsXG4gICAgICAgICAgUHJlZml4OiBKc29uUGF0aC5zdHJpbmdBdCgnJC5QcmVmaXgnKSxcbiAgICAgICAgfSxcbiAgICAgICAgcmVzdWx0UGF0aDogJyQucmVzcG9uc2UnLFxuICAgICAgfSkpXG4gICAgICAub3RoZXJ3aXNlKG5ldyB0YXNrcy5DYWxsQXdzU2VydmljZSh0aGlzLCAnRmlyc3QgQHNjb3BlIHBhZ2UnLCB7XG4gICAgICAgIHNlcnZpY2U6ICdzMycsXG4gICAgICAgIGFjdGlvbjogJ2xpc3RPYmplY3RzVjInLFxuICAgICAgICBpYW1BY3Rpb246ICdzMzpMaXN0QnVja2V0JyxcbiAgICAgICAgaWFtUmVzb3VyY2VzOiBbcHJvcHMuYnVja2V0LmJ1Y2tldEFybl0sXG4gICAgICAgIHBhcmFtZXRlcnM6IHtcbiAgICAgICAgICBCdWNrZXQ6IHByb3BzLmJ1Y2tldC5idWNrZXROYW1lLFxuICAgICAgICAgIERlbGltaXRlcjogJy8nLFxuICAgICAgICAgIFByZWZpeDogSnNvblBhdGguc3RyaW5nQXQoJyQuUHJlZml4JyksXG4gICAgICAgIH0sXG4gICAgICAgIHJlc3VsdFBhdGg6ICckLnJlc3BvbnNlJyxcbiAgICAgIH0pKVxuICAgICAgLmFmdGVyd2FyZHMoKVxuICAgICAgLm5leHQobmV3IE1hcCh0aGlzLCAnRm9yIGVhY2ggQHNjb3BlL3BrZycsIHsgaXRlbXNQYXRoOiAnJC5yZXNwb25zZS5Db21tb25QcmVmaXhlcycsIHJlc3VsdFBhdGg6IEpzb25QYXRoLkRJU0NBUkQgfSlcbiAgICAgICAgLml0ZXJhdG9yKG5ldyB0YXNrcy5TdGVwRnVuY3Rpb25zU3RhcnRFeGVjdXRpb24odGhpcywgJ1Byb2Nlc3Mgc2NvcGVkIHBhY2thZ2UnLCB7XG4gICAgICAgICAgc3RhdGVNYWNoaW5lOiBwcm9jZXNzUGFja2FnZVZlcnNpb25zLFxuICAgICAgICAgIGFzc29jaWF0ZVdpdGhQYXJlbnQ6IHRydWUsXG4gICAgICAgICAgaW5wdXQ6IFRhc2tJbnB1dC5mcm9tT2JqZWN0KHtcbiAgICAgICAgICAgIFByZWZpeDogSnNvblBhdGguc3RyaW5nQXQoJyQuUHJlZml4JyksXG4gICAgICAgICAgfSksXG4gICAgICAgICAgaW50ZWdyYXRpb25QYXR0ZXJuOiBJbnRlZ3JhdGlvblBhdHRlcm4uUlVOX0pPQixcbiAgICAgICAgfSkuYWRkUmV0cnkoeyBlcnJvcnM6IFsnU3RlcEZ1bmN0aW9ucy5FeGVjdXRpb25MaW1pdEV4Y2VlZGVkJ10gfSkpKTtcbiAgICBwcm9jZXNzTmFtZXNwYWNlLm5leHQobmV3IENob2ljZSh0aGlzLCAnSGFzIG1vcmUgcGFja2FnZXM/JylcbiAgICAgIC53aGVuKENvbmRpdGlvbi5pc1ByZXNlbnQoJyQucmVzcG9uc2UuTmV4dENvbnRpbnVhdGlvblRva2VuJyksIHByb2Nlc3NOYW1lc3BhY2UpXG4gICAgICAub3RoZXJ3aXNlKG5ldyBTdWNjZWVkKHRoaXMsICdBbGwgRG9uZScpKSk7XG5cbiAgICBjb25zdCBzdGFydCA9IG5ldyBDaG9pY2UodGhpcywgJ0dldCBwcmVmaXggcGFnZScpXG4gICAgICAud2hlbihcbiAgICAgICAgQ29uZGl0aW9uLmlzUHJlc2VudCgnJC5yZXNwb25zZS5OZXh0Q29udGludWF0aW9uVG9rZW4nKSxcbiAgICAgICAgbmV3IHRhc2tzLkNhbGxBd3NTZXJ2aWNlKHRoaXMsICdOZXh0IHByZWZpeGVzIHBhZ2UnLCB7XG4gICAgICAgICAgc2VydmljZTogJ3MzJyxcbiAgICAgICAgICBhY3Rpb246ICdsaXN0T2JqZWN0c1YyJyxcbiAgICAgICAgICBpYW1BY3Rpb246ICdzMzpMaXN0QnVja2V0JyxcbiAgICAgICAgICBpYW1SZXNvdXJjZXM6IFtwcm9wcy5idWNrZXQuYnVja2V0QXJuXSxcbiAgICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICBCdWNrZXQ6IHByb3BzLmJ1Y2tldC5idWNrZXROYW1lLFxuICAgICAgICAgICAgQ29udGludWF0aW9uVG9rZW46IEpzb25QYXRoLnN0cmluZ0F0KCckLnJlc3BvbnNlLk5leHRDb250aW51YXRpb25Ub2tlbicpLFxuICAgICAgICAgICAgRGVsaW1pdGVyOiAnLycsXG4gICAgICAgICAgICBQcmVmaXg6IFNUT1JBR0VfS0VZX1BSRUZJWCxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHJlc3VsdFBhdGg6ICckLnJlc3BvbnNlJyxcbiAgICAgICAgfSksXG4gICAgICApXG4gICAgICAub3RoZXJ3aXNlKFxuICAgICAgICBuZXcgdGFza3MuQ2FsbEF3c1NlcnZpY2UodGhpcywgJ0ZpcnN0IHByZWZpeCBwYWdlJywge1xuICAgICAgICAgIHNlcnZpY2U6ICdzMycsXG4gICAgICAgICAgYWN0aW9uOiAnbGlzdE9iamVjdHNWMicsXG4gICAgICAgICAgaWFtQWN0aW9uOiAnczM6TGlzdEJ1Y2tldCcsXG4gICAgICAgICAgaWFtUmVzb3VyY2VzOiBbcHJvcHMuYnVja2V0LmJ1Y2tldEFybl0sXG4gICAgICAgICAgcGFyYW1ldGVyczoge1xuICAgICAgICAgICAgQnVja2V0OiBwcm9wcy5idWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAgICAgIERlbGltaXRlcjogJy8nLFxuICAgICAgICAgICAgUHJlZml4OiBTVE9SQUdFX0tFWV9QUkVGSVgsXG4gICAgICAgICAgfSxcbiAgICAgICAgICByZXN1bHRQYXRoOiAnJC5yZXNwb25zZScsXG4gICAgICAgIH0pLFxuICAgICAgKS5hZnRlcndhcmRzKClcbiAgICAgIC5uZXh0KG5ldyBNYXAodGhpcywgJ0ZvciBlYWNoIHByZWZpeCcsIHsgaXRlbXNQYXRoOiAnJC5yZXNwb25zZS5Db21tb25QcmVmaXhlcycsIHJlc3VsdFBhdGg6IEpzb25QYXRoLkRJU0NBUkQgfSlcbiAgICAgICAgLml0ZXJhdG9yKFxuICAgICAgICAgIG5ldyBDaG9pY2UodGhpcywgJ0lzIHRoaXMgYSBAc2NvcGUvIHByZWZpeD8nKVxuICAgICAgICAgICAgLndoZW4oQ29uZGl0aW9uLnN0cmluZ01hdGNoZXMoJyQuUHJlZml4JywgYCR7U1RPUkFHRV9LRVlfUFJFRklYfUAqYCksIHByb2Nlc3NOYW1lc3BhY2UpXG4gICAgICAgICAgICAub3RoZXJ3aXNlKG5ldyB0YXNrcy5TdGVwRnVuY3Rpb25zU3RhcnRFeGVjdXRpb24odGhpcywgJ1Byb2Nlc3MgdW5zY29wZWQgcGFja2FnZScsIHtcbiAgICAgICAgICAgICAgc3RhdGVNYWNoaW5lOiBwcm9jZXNzUGFja2FnZVZlcnNpb25zLFxuICAgICAgICAgICAgICBhc3NvY2lhdGVXaXRoUGFyZW50OiB0cnVlLFxuICAgICAgICAgICAgICBpbnB1dDogVGFza0lucHV0LmZyb21PYmplY3Qoe1xuICAgICAgICAgICAgICAgIFByZWZpeDogSnNvblBhdGguc3RyaW5nQXQoJyQuUHJlZml4JyksXG4gICAgICAgICAgICAgIH0pLFxuICAgICAgICAgICAgICBpbnRlZ3JhdGlvblBhdHRlcm46IEludGVncmF0aW9uUGF0dGVybi5SVU5fSk9CLFxuICAgICAgICAgICAgfSkuYWRkUmV0cnkoeyBlcnJvcnM6IFsnU3RlcEZ1bmN0aW9ucy5FeGVjdXRpb25MaW1pdEV4Y2VlZGVkJ10gfSkpXG4gICAgICAgICAgICAuYWZ0ZXJ3YXJkcygpLFxuICAgICAgICApKTtcblxuICAgIHN0YXJ0Lm5leHQobmV3IENob2ljZSh0aGlzLCAnSGFzIG1vcmUgcHJlZml4ZXM/JylcbiAgICAgIC53aGVuKENvbmRpdGlvbi5pc1ByZXNlbnQoJyQucmVzcG9uc2UuTmV4dENvbnRpbnVhdGlvblRva2VuJyksIHN0YXJ0KVxuICAgICAgLm90aGVyd2lzZShuZXcgU3VjY2VlZCh0aGlzLCAnRG9uZScpKSk7XG5cbiAgICB0aGlzLnN0YXRlTWFjaGluZSA9IG5ldyBTdGF0ZU1hY2hpbmUodGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgZGVmaW5pdGlvbjogc3RhcnQsXG4gICAgICBzdGF0ZU1hY2hpbmVOYW1lOiBzdGF0ZU1hY2hpbmVOYW1lRnJvbSh0aGlzLm5vZGUucGF0aCksXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5ob3Vycyg0KSxcbiAgICAgIHRyYWNpbmdFbmFibGVkOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgcHJvcHMuYnVja2V0LmdyYW50UmVhZChwcm9jZXNzUGFja2FnZVZlcnNpb25zKTtcbiAgICBwcm9wcy5idWNrZXQuZ3JhbnRSZWFkKHRoaXMuc3RhdGVNYWNoaW5lKTtcbiAgfVxufVxuXG4vKipcbiAqIFRoaXMgdHVybnMgYSBub2RlIHBhdGggaW50byBhIHZhbGlkIHN0YXRlIG1hY2hpbmUgbmFtZSwgdG8gdHJ5IGFuZCBpbXByb3ZlXG4gKiB0aGUgU3RlcEZ1bmN0aW9uJ3MgQVdTIGNvbnNvbGUgZXhwZXJpZW5jZSB3aGlsZSBtaW5pbWl6aW5nIHRoZSByaXNrIGZvclxuICogY29sbGlzb25zLlxuICovXG5mdW5jdGlvbiBzdGF0ZU1hY2hpbmVOYW1lRnJvbShub2RlUGF0aDogc3RyaW5nKTogc3RyaW5nIHtcbiAgLy8gUG9vciBtYW4ncyByZXBsYWNlIGFsbC4uLlxuICByZXR1cm4gbm9kZVBhdGguc3BsaXQoL1teYS16MC05KyFALigpPV8nLV0rL2kpLmpvaW4oJy4nKTtcbn1cbiJdfQ==