"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ingestion = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_s3_deployment_1 = require("@aws-cdk/aws-s3-deployment");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const config_file_1 = require("../../config-file");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const constants_1 = require("../shared/constants");
const constants_2 = require("./constants");
const ingestion_1 = require("./ingestion");
const re_ingest_1 = require("./re-ingest");
/**
 * The ingestion function receives messages from discovery integrations and
 * processes their payloads into the provided S3 Bucket.
 *
 * This function is also an `IGrantable`, so that it can be granted permissions
 * to read from the source S3 buckets.
 */
class Ingestion extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d;
        super(scope, id);
        this.queueRetentionPeriod = core_1.Duration.days(14);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        this.queue = new aws_sqs_1.Queue(this, 'Queue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const configFilename = 'config.json';
        const config = new config_file_1.ConfigFile(configFilename, JSON.stringify({
            packageLinks: (_a = props.packageLinks) !== null && _a !== void 0 ? _a : [],
            packageTags: (_b = props.packageTags) !== null && _b !== void 0 ? _b : [],
        }));
        const configBucket = new aws_s3_1.Bucket(this, 'ConfigBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
        });
        new aws_s3_deployment_1.BucketDeployment(this, 'DeployIngestionConfiguration', {
            sources: [aws_s3_deployment_1.Source.asset(config.dir)],
            destinationBucket: configBucket,
        });
        const environment = {
            AWS_EMF_ENVIRONMENT: 'Local',
            BUCKET_NAME: props.bucket.bucketName,
            CONFIG_BUCKET_NAME: configBucket.bucketName,
            CONFIG_FILE_KEY: configFilename,
            STATE_MACHINE_ARN: props.orchestration.stateMachine.stateMachineArn,
        };
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.publishingRepositoryNpmEndpoint;
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
        }
        const handler = new ingestion_1.Ingestion(this, 'Default', {
            description: '[ConstructHub/Ingestion] Ingests new package versions into the Construct Hub',
            environment,
            logRetention: (_c = props.logRetention) !== null && _c !== void 0 ? _c : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.function = handler;
        configBucket.grantRead(handler);
        props.bucket.grantWrite(this.function);
        (_d = props.codeArtifact) === null || _d === void 0 ? void 0 : _d.grantPublishToRepository(handler);
        props.orchestration.stateMachine.grantStartExecution(this.function);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.queue, { batchSize: 1 }));
        // This event source is disabled, and can be used to re-process dead-letter-queue messages
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.deadLetterQueue, { batchSize: 1, enabled: false }));
        // Reprocess workflow
        const reprocessQueue = new aws_sqs_1.Queue(this, 'ReprocessQueue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            // Visibility timeout of 15 minutes matches the Lambda maximum execution time.
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.bucket.grantRead(this.function, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.PACKAGE_KEY_SUFFIX}`);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(reprocessQueue, { batchSize: 1 }));
        new ReprocessIngestionWorkflow(this, 'ReprocessWorkflow', { bucket: props.bucket, queue: reprocessQueue });
        this.grantPrincipal = this.function.grantPrincipal;
        props.monitoring.addHighSeverityAlarm('Ingestion Dead-Letter Queue not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.node.path}/DLQNotEmpty`,
            alarmDescription: [
                'The dead-letter queue for the Ingestion function is not empty!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            // SQS does not emit metrics if the queue has been empty for a while, which is GOOD.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
        props.monitoring.addHighSeverityAlarm('Ingestion failures', this.function.metricErrors().createAlarm(this, 'FailureAlarm', {
            alarmName: `${this.node.path}/Failure`,
            alarmDescription: [
                'The Ingestion function is failing!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            // Lambda only emits metrics when the function is invoked. No invokation => no errors.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
    }
    metricFoundLicenseFile(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "FoundLicenseFile" /* FOUND_LICENSE_FILE */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricIneligibleLicense(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "IneligibleLicense" /* INELIGIBLE_LICENSE */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricInvalidAssembly(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidAssembly" /* INVALID_ASSEMBLY */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricInvalidTarball(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidTarball" /* INVALID_TARBALL */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * This metrics is the total count of packages that were rejected due to
     * mismatched identity (name, version, license) between the `package.json`
     * file and te `.jsii` attribute.
     */
    metricMismatchedIdentityRejections(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
}
exports.Ingestion = Ingestion;
/**
 * A StepFunctions State Machine to reprocess every currently indexed package
 * through the ingestion function. This should not be frequently required, but
 * can come in handy at times.
 *
 * For more information, refer to the runbook at
 * https://github.com/cdklabs/construct-hub/blob/main/docs/operator-runbook.md
 */
class ReprocessIngestionWorkflow extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const lambdaFunction = new re_ingest_1.ReIngest(this, 'Function', {
            description: '[ConstructHub/Ingestion/ReIngest] The function used to reprocess packages through ingestion',
            environment: { BUCKET_NAME: props.bucket.bucketName, QUEUE_URL: props.queue.queueUrl },
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        props.queue.grantSendMessages(lambdaFunction);
        props.bucket.grantRead(lambdaFunction, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.METADATA_KEY_SUFFIX}`);
        props.bucket.grantRead(lambdaFunction, `${constants_1.STORAGE_KEY_PREFIX}*${constants_1.PACKAGE_KEY_SUFFIX}`);
        const listBucket = new aws_stepfunctions_1.Choice(this, 'Has a ContinuationToken?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.ContinuationToken'), new aws_stepfunctions_tasks_1.CallAwsService(this, 'S3.ListObjectsV2(NextPage)', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.ContinuationToken'),
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }))
            .otherwise(new aws_stepfunctions_tasks_1.CallAwsService(this, 'S3.ListObjectsV2(FirstPage)', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        })).afterwards();
        const process = new aws_stepfunctions_1.Map(this, 'Process Result', {
            itemsPath: '$.response.Contents',
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).iterator(new aws_stepfunctions_1.Choice(this, 'Is metadata object?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Key', `*${constants_1.METADATA_KEY_SUFFIX}`), new aws_stepfunctions_tasks_1.LambdaInvoke(this, 'Send for reprocessing', { lambdaFunction })
            // Ample retries here... We should never fail because of throttling....
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 }))
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Nothing to do')));
        // Need to physical-name the state machine so it can self-invoke.
        const stateMachineName = stateMachineNameFrom(this.node.path);
        listBucket.next(process.next(new aws_stepfunctions_1.Choice(this, 'Is there more?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new aws_stepfunctions_tasks_1.StepFunctionsStartExecution(this, 'Continue as new', {
            associateWithParent: true,
            stateMachine: aws_stepfunctions_1.StateMachine.fromStateMachineArn(this, 'ThisStateMachine', core_1.Stack.of(this).formatArn({
                arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME,
                service: 'states',
                resource: 'stateMachine',
                resourceName: stateMachineName,
            })),
            input: aws_stepfunctions_1.TaskInput.fromObject({ ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken') }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] }))
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'All Done'))));
        const stateMachine = new aws_stepfunctions_1.StateMachine(this, 'StateMachine', {
            definition: listBucket,
            stateMachineName,
            timeout: core_1.Duration.hours(1),
        });
        props.bucket.grantRead(stateMachine);
        props.queue.grantSendMessages(stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvYmFja2VuZC9pbmdlc3Rpb24vaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNERBQWlJO0FBRWpJLG9EQUF3RTtBQUN4RSxnRkFBbUU7QUFDbkUsZ0RBQWtEO0FBQ2xELDRDQUFxRTtBQUNyRSxrRUFBc0U7QUFDdEUsOENBQWtFO0FBQ2xFLGtFQUFvSTtBQUNwSSw4RUFBNkc7QUFDN0csd0NBQXNFO0FBRXRFLG1EQUErQztBQUMvQywrQ0FBaUU7QUFHakUsbURBQWdEO0FBR2hELG1EQUFrRztBQUNsRywyQ0FBNEQ7QUFDNUQsMkNBQW1EO0FBQ25ELDJDQUF1QztBQTJDdkM7Ozs7OztHQU1HO0FBQ0gsTUFBYSxTQUFVLFNBQVEsZ0JBQVM7SUFtQnRDLFlBQW1CLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXFCOztRQUNwRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBTEgseUJBQW9CLEdBQUcsZUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztRQU92RCxJQUFJLENBQUMsZUFBZSxHQUFHLElBQUksZUFBSyxDQUFDLElBQUksRUFBRSxLQUFLLEVBQUU7WUFDNUMsVUFBVSxFQUFFLHlCQUFlLENBQUMsV0FBVztZQUN2QyxlQUFlLEVBQUUsSUFBSSxDQUFDLG9CQUFvQjtZQUMxQyxpQkFBaUIsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztTQUN4QyxDQUFDLENBQUM7UUFFSCxJQUFJLENBQUMsS0FBSyxHQUFHLElBQUksZUFBSyxDQUFDLElBQUksRUFBRSxPQUFPLEVBQUU7WUFDcEMsZUFBZSxFQUFFO2dCQUNmLGVBQWUsRUFBRSxDQUFDO2dCQUNsQixLQUFLLEVBQUUsSUFBSSxDQUFDLGVBQWU7YUFDNUI7WUFDRCxVQUFVLEVBQUUseUJBQWUsQ0FBQyxXQUFXO1lBQ3ZDLGVBQWUsRUFBRSxJQUFJLENBQUMsb0JBQW9CO1lBQzFDLGlCQUFpQixFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1NBQ3hDLENBQUMsQ0FBQztRQUVILE1BQU0sY0FBYyxHQUFHLGFBQWEsQ0FBQztRQUNyQyxNQUFNLE1BQU0sR0FBRyxJQUFJLHdCQUFVLENBQUMsY0FBYyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUM7WUFDM0QsWUFBWSxRQUFFLEtBQUssQ0FBQyxZQUFZLG1DQUFJLEVBQUU7WUFDdEMsV0FBVyxRQUFFLEtBQUssQ0FBQyxXQUFXLG1DQUFJLEVBQUU7U0FDckMsQ0FBQyxDQUFDLENBQUM7UUFFSixNQUFNLFlBQVksR0FBRyxJQUFJLGVBQU0sQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ3BELGlCQUFpQixFQUFFLDBCQUFpQixDQUFDLFNBQVM7WUFDOUMsVUFBVSxFQUFFLElBQUk7U0FDakIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxvQ0FBZ0IsQ0FBQyxJQUFJLEVBQUUsOEJBQThCLEVBQUU7WUFDekQsT0FBTyxFQUFFLENBQUMsMEJBQU0sQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1lBQ25DLGlCQUFpQixFQUFFLFlBQVk7U0FDaEMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxXQUFXLEdBQWlDO1lBQ2hELG1CQUFtQixFQUFFLE9BQU87WUFDNUIsV0FBVyxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVTtZQUNwQyxrQkFBa0IsRUFBRSxZQUFZLENBQUMsVUFBVTtZQUMzQyxlQUFlLEVBQUUsY0FBYztZQUMvQixpQkFBaUIsRUFBRSxLQUFLLENBQUMsYUFBYSxDQUFDLFlBQVksQ0FBQyxlQUFlO1NBQ3BFLENBQUM7UUFFRixJQUFJLEtBQUssQ0FBQyxZQUFZLEVBQUU7WUFDdEIsV0FBVyxDQUFDLGlDQUFpQyxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsK0JBQStCLENBQUM7WUFDbkcsV0FBVyxDQUFDLHlCQUF5QixHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsb0JBQW9CLENBQUM7WUFDaEYsV0FBVyxDQUFDLDBCQUEwQixHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMscUJBQXFCLENBQUM7U0FDbkY7UUFFRCxNQUFNLE9BQU8sR0FBRyxJQUFJLHFCQUFPLENBQUMsSUFBSSxFQUFFLFNBQVMsRUFBRTtZQUMzQyxXQUFXLEVBQUUsOEVBQThFO1lBQzNGLFdBQVc7WUFDWCxZQUFZLFFBQUUsS0FBSyxDQUFDLFlBQVksbUNBQUksd0JBQWEsQ0FBQyxTQUFTO1lBQzNELFVBQVUsRUFBRSxLQUFNO1lBQ2xCLE9BQU8sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixPQUFPLEVBQUUsb0JBQU8sQ0FBQyxNQUFNO1NBQ3hCLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxRQUFRLEdBQUcsT0FBTyxDQUFDO1FBRXhCLFlBQVksQ0FBQyxTQUFTLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDaEMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3ZDLE1BQUEsS0FBSyxDQUFDLFlBQVksMENBQUUsd0JBQXdCLENBQUMsT0FBTyxFQUFFO1FBQ3RELEtBQUssQ0FBQyxhQUFhLENBQUMsWUFBWSxDQUFDLG1CQUFtQixDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQztRQUVwRSxJQUFJLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQyxJQUFJLHlDQUFjLENBQUMsSUFBSSxDQUFDLEtBQUssRUFBRSxFQUFFLFNBQVMsRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDL0UsMEZBQTBGO1FBQzFGLElBQUksQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLElBQUkseUNBQWMsQ0FBQyxJQUFJLENBQUMsZUFBZSxFQUFFLEVBQUUsU0FBUyxFQUFFLENBQUMsRUFBRSxPQUFPLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBR3pHLHFCQUFxQjtRQUNyQixNQUFNLGNBQWMsR0FBRyxJQUFJLGVBQUssQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLEVBQUU7WUFDdkQsZUFBZSxFQUFFO2dCQUNmLGVBQWUsRUFBRSxDQUFDO2dCQUNsQixLQUFLLEVBQUUsSUFBSSxDQUFDLGVBQWU7YUFDNUI7WUFDRCxVQUFVLEVBQUUseUJBQWUsQ0FBQyxXQUFXO1lBQ3ZDLGVBQWUsRUFBRSxJQUFJLENBQUMsb0JBQW9CO1lBQzFDLDhFQUE4RTtZQUM5RSxpQkFBaUIsRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztTQUN4QyxDQUFDLENBQUM7UUFDSCxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxJQUFJLENBQUMsUUFBUSxFQUFFLEdBQUcsOEJBQWtCLElBQUksOEJBQWtCLEVBQUUsQ0FBQyxDQUFDO1FBQ3JGLElBQUksQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLElBQUkseUNBQWMsQ0FBQyxjQUFjLEVBQUUsRUFBRSxTQUFTLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ25GLElBQUksMEJBQTBCLENBQUMsSUFBSSxFQUFFLG1CQUFtQixFQUFFLEVBQUUsTUFBTSxFQUFFLEtBQUssQ0FBQyxNQUFNLEVBQUUsS0FBSyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUM7UUFFM0csSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQztRQUVuRCxLQUFLLENBQUMsVUFBVSxDQUFDLG9CQUFvQixDQUNuQyx1Q0FBdUMsRUFDdkMsSUFBSSwrQkFBYyxDQUFDO1lBQ2pCLFVBQVUsRUFBRSxTQUFTO1lBQ3JCLFlBQVksRUFBRTtnQkFDWixFQUFFLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyx3Q0FBd0MsQ0FBQyxFQUFFLE1BQU0sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7Z0JBQ2xHLEVBQUUsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLDJDQUEyQyxDQUFDLEVBQUUsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzthQUN0RztTQUNGLENBQUMsQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUMvQixTQUFTLEVBQUUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksY0FBYztZQUMxQyxnQkFBZ0IsRUFBRTtnQkFDaEIsZ0VBQWdFO2dCQUNoRSxFQUFFO2dCQUNGLFlBQVkseUJBQVcsRUFBRTtnQkFDekIsRUFBRTtnQkFDRiw2QkFBNkIsdUJBQVcsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLEVBQUU7Z0JBQ2hFLGdDQUFnQyw2QkFBaUIsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUU7YUFDbkUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO1lBQ1osa0JBQWtCLEVBQUUsbUNBQWtCLENBQUMsa0NBQWtDO1lBQ3pFLGlCQUFpQixFQUFFLENBQUM7WUFDcEIsU0FBUyxFQUFFLENBQUM7WUFDWixvRkFBb0Y7WUFDcEYsZ0JBQWdCLEVBQUUsaUNBQWdCLENBQUMsYUFBYTtTQUNqRCxDQUFDLENBQ0gsQ0FBQztRQUNGLEtBQUssQ0FBQyxVQUFVLENBQUMsb0JBQW9CLENBQ25DLG9CQUFvQixFQUNwQixJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVksRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQzdELFNBQVMsRUFBRSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxVQUFVO1lBQ3RDLGdCQUFnQixFQUFFO2dCQUNoQixvQ0FBb0M7Z0JBQ3BDLEVBQUU7Z0JBQ0YsWUFBWSx5QkFBVyxFQUFFO2dCQUN6QixFQUFFO2dCQUNGLGdDQUFnQyw2QkFBaUIsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUU7YUFDbkUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO1lBQ1osa0JBQWtCLEVBQUUsbUNBQWtCLENBQUMsa0NBQWtDO1lBQ3pFLGlCQUFpQixFQUFFLENBQUM7WUFDcEIsU0FBUyxFQUFFLENBQUM7WUFDWixzRkFBc0Y7WUFDdEYsZ0JBQWdCLEVBQUUsaUNBQWdCLENBQUMsYUFBYTtTQUNqRCxDQUFDLENBQ0gsQ0FBQztJQUNKLENBQUM7SUFFTSxzQkFBc0IsQ0FBQyxJQUFvQjtRQUNoRCxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDM0IsU0FBUyxFQUFFLDBCQUFTLENBQUMsR0FBRztZQUN4QixHQUFHLElBQUk7WUFDUCxVQUFVLDZDQUErQjtZQUN6QyxTQUFTLEVBQUUsNkJBQWlCO1NBQzdCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSx1QkFBdUIsQ0FBQyxJQUFvQjtRQUNqRCxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDM0IsU0FBUyxFQUFFLDBCQUFTLENBQUMsR0FBRztZQUN4QixHQUFHLElBQUk7WUFDUCxVQUFVLDhDQUErQjtZQUN6QyxTQUFTLEVBQUUsNkJBQWlCO1NBQzdCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSxxQkFBcUIsQ0FBQyxJQUFvQjtRQUMvQyxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDM0IsU0FBUyxFQUFFLDBCQUFTLENBQUMsR0FBRztZQUN4QixHQUFHLElBQUk7WUFDUCxVQUFVLDBDQUE2QjtZQUN2QyxTQUFTLEVBQUUsNkJBQWlCO1NBQzdCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSxvQkFBb0IsQ0FBQyxJQUFvQjtRQUM5QyxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDM0IsU0FBUyxFQUFFLDBCQUFTLENBQUMsR0FBRztZQUN4QixHQUFHLElBQUk7WUFDUCxVQUFVLHdDQUE0QjtZQUN0QyxTQUFTLEVBQUUsNkJBQWlCO1NBQzdCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksa0NBQWtDLENBQUMsSUFBb0I7UUFDNUQsT0FBTyxJQUFJLHVCQUFNLENBQUM7WUFDaEIsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQzNCLFNBQVMsRUFBRSwwQkFBUyxDQUFDLEdBQUc7WUFDeEIsR0FBRyxJQUFJO1lBQ1AsVUFBVSxxRUFBMkM7WUFDckQsU0FBUyxFQUFFLDZCQUFpQjtTQUM3QixDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUE1TUQsOEJBNE1DO0FBT0Q7Ozs7Ozs7R0FPRztBQUNILE1BQU0sMEJBQTJCLFNBQVEsZ0JBQVM7SUFDaEQsWUFBbUIsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBc0M7UUFDckYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUVqQixNQUFNLGNBQWMsR0FBRyxJQUFJLG9CQUFRLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUNwRCxXQUFXLEVBQUUsNkZBQTZGO1lBQzFHLFdBQVcsRUFBRSxFQUFFLFdBQVcsRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVUsRUFBRSxTQUFTLEVBQUUsS0FBSyxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUU7WUFDdEYsT0FBTyxFQUFFLG9CQUFPLENBQUMsTUFBTTtTQUN4QixDQUFDLENBQUM7UUFFSCxLQUFLLENBQUMsS0FBSyxDQUFDLGlCQUFpQixDQUFDLGNBQWMsQ0FBQyxDQUFDO1FBQzlDLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLGNBQWMsRUFBRSxHQUFHLDhCQUFrQixJQUFJLCtCQUFtQixFQUFFLENBQUMsQ0FBQztRQUN2RixLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxjQUFjLEVBQUUsR0FBRyw4QkFBa0IsSUFBSSw4QkFBa0IsRUFBRSxDQUFDLENBQUM7UUFFdEYsTUFBTSxVQUFVLEdBQUcsSUFBSSwwQkFBTSxDQUFDLElBQUksRUFBRSwwQkFBMEIsQ0FBQzthQUM1RCxJQUFJLENBQUMsNkJBQVMsQ0FBQyxTQUFTLENBQUMscUJBQXFCLENBQUMsRUFDOUMsSUFBSSx3Q0FBYyxDQUFDLElBQUksRUFBRSw0QkFBNEIsRUFBRTtZQUNyRCxPQUFPLEVBQUUsSUFBSTtZQUNiLE1BQU0sRUFBRSxlQUFlO1lBQ3ZCLFNBQVMsRUFBRSxlQUFlO1lBQzFCLFlBQVksRUFBRSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDO1lBQ3RDLFVBQVUsRUFBRTtnQkFDVixNQUFNLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVO2dCQUMvQixpQkFBaUIsRUFBRSw0QkFBUSxDQUFDLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQztnQkFDM0QsTUFBTSxFQUFFLDhCQUFrQjthQUMzQjtZQUNELFVBQVUsRUFBRSxZQUFZO1NBQ3pCLENBQUMsQ0FBQzthQUNKLFNBQVMsQ0FBQyxJQUFJLHdDQUFjLENBQUMsSUFBSSxFQUFFLDZCQUE2QixFQUFFO1lBQ2pFLE9BQU8sRUFBRSxJQUFJO1lBQ2IsTUFBTSxFQUFFLGVBQWU7WUFDdkIsU0FBUyxFQUFFLGVBQWU7WUFDMUIsWUFBWSxFQUFFLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTLENBQUM7WUFDdEMsVUFBVSxFQUFFO2dCQUNWLE1BQU0sRUFBRSxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVU7Z0JBQy9CLE1BQU0sRUFBRSw4QkFBa0I7YUFDM0I7WUFDRCxVQUFVLEVBQUUsWUFBWTtTQUN6QixDQUFDLENBQUMsQ0FBQyxVQUFVLEVBQUUsQ0FBQztRQUVuQixNQUFNLE9BQU8sR0FBRyxJQUFJLHVCQUFHLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFO1lBQzlDLFNBQVMsRUFBRSxxQkFBcUI7WUFDaEMsVUFBVSxFQUFFLDRCQUFRLENBQUMsT0FBTztTQUM3QixDQUFDLENBQUMsUUFBUSxDQUNULElBQUksMEJBQU0sQ0FBQyxJQUFJLEVBQUUscUJBQXFCLENBQUM7YUFDcEMsSUFBSSxDQUNILDZCQUFTLENBQUMsYUFBYSxDQUFDLE9BQU8sRUFBRSxJQUFJLCtCQUFtQixFQUFFLENBQUMsRUFDM0QsSUFBSSxzQ0FBWSxDQUFDLElBQUksRUFBRSx1QkFBdUIsRUFBRSxFQUFFLGNBQWMsRUFBRSxDQUFDO1lBQ2pFLHVFQUF1RTthQUN0RSxRQUFRLENBQUMsRUFBRSxNQUFNLEVBQUUsQ0FBQyxpQ0FBaUMsQ0FBQyxFQUFFLFdBQVcsRUFBRSxHQUFHLEVBQUUsUUFBUSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsV0FBVyxFQUFFLEVBQUUsRUFBRSxDQUFDLENBQy9IO2FBQ0EsU0FBUyxDQUFDLElBQUksMkJBQU8sQ0FBQyxJQUFJLEVBQUUsZUFBZSxDQUFDLENBQUMsQ0FDakQsQ0FBQztRQUVGLGlFQUFpRTtRQUNqRSxNQUFNLGdCQUFnQixHQUFHLG9CQUFvQixDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDOUQsVUFBVSxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLElBQUksMEJBQU0sQ0FBQyxJQUFJLEVBQUUsZ0JBQWdCLENBQUM7YUFDNUQsSUFBSSxDQUNILDZCQUFTLENBQUMsU0FBUyxDQUFDLGtDQUFrQyxDQUFDLEVBQ3ZELElBQUkscURBQTJCLENBQUMsSUFBSSxFQUFFLGlCQUFpQixFQUFFO1lBQ3ZELG1CQUFtQixFQUFFLElBQUk7WUFDekIsWUFBWSxFQUFFLGdDQUFZLENBQUMsbUJBQW1CLENBQUMsSUFBSSxFQUFFLGtCQUFrQixFQUFFLFlBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsU0FBUyxDQUFDO2dCQUNoRyxTQUFTLEVBQUUsZ0JBQVMsQ0FBQyxtQkFBbUI7Z0JBQ3hDLE9BQU8sRUFBRSxRQUFRO2dCQUNqQixRQUFRLEVBQUUsY0FBYztnQkFDeEIsWUFBWSxFQUFFLGdCQUFnQjthQUMvQixDQUFDLENBQUM7WUFDSCxLQUFLLEVBQUUsNkJBQVMsQ0FBQyxVQUFVLENBQUMsRUFBRSxpQkFBaUIsRUFBRSw0QkFBUSxDQUFDLFFBQVEsQ0FBQyxrQ0FBa0MsQ0FBQyxFQUFFLENBQUM7WUFDekcsa0JBQWtCLEVBQUUsc0NBQWtCLENBQUMsZ0JBQWdCO1NBQ3hELENBQUMsQ0FBQyxRQUFRLENBQUMsRUFBRSxNQUFNLEVBQUUsQ0FBQyxzQ0FBc0MsQ0FBQyxFQUFFLENBQUMsQ0FDbEU7YUFDQSxTQUFTLENBQUMsSUFBSSwyQkFBTyxDQUFDLElBQUksRUFBRSxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQztRQUU5QyxNQUFNLFlBQVksR0FBRyxJQUFJLGdDQUFZLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUMxRCxVQUFVLEVBQUUsVUFBVTtZQUN0QixnQkFBZ0I7WUFDaEIsT0FBTyxFQUFFLGVBQVEsQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO1NBQzNCLENBQUMsQ0FBQztRQUVILEtBQUssQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLFlBQVksQ0FBQyxDQUFDO1FBQ3JDLEtBQUssQ0FBQyxLQUFLLENBQUMsaUJBQWlCLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDOUMsQ0FBQztDQUNGO0FBRUQ7Ozs7R0FJRztBQUNILFNBQVMsb0JBQW9CLENBQUMsUUFBZ0I7SUFDNUMsNEJBQTRCO0lBQzVCLE9BQU8sUUFBUSxDQUFDLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztBQUMzRCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29tcGFyaXNvbk9wZXJhdG9yLCBNYXRoRXhwcmVzc2lvbiwgTWV0cmljLCBNZXRyaWNPcHRpb25zLCBTdGF0aXN0aWMsIFRyZWF0TWlzc2luZ0RhdGEgfSBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWR3YXRjaCc7XG5pbXBvcnQgeyBJR3JhbnRhYmxlLCBJUHJpbmNpcGFsIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgeyBGdW5jdGlvblByb3BzLCBJRnVuY3Rpb24sIFRyYWNpbmcgfSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCB7IFNxc0V2ZW50U291cmNlIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYS1ldmVudC1zb3VyY2VzJztcbmltcG9ydCB7IFJldGVudGlvbkRheXMgfSBmcm9tICdAYXdzLWNkay9hd3MtbG9ncyc7XG5pbXBvcnQgeyBCbG9ja1B1YmxpY0FjY2VzcywgQnVja2V0LCBJQnVja2V0IH0gZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IEJ1Y2tldERlcGxveW1lbnQsIFNvdXJjZSB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1zMy1kZXBsb3ltZW50JztcbmltcG9ydCB7IElRdWV1ZSwgUXVldWUsIFF1ZXVlRW5jcnlwdGlvbiB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1zcXMnO1xuaW1wb3J0IHsgU3RhdGVNYWNoaW5lLCBKc29uUGF0aCwgQ2hvaWNlLCBTdWNjZWVkLCBDb25kaXRpb24sIE1hcCwgVGFza0lucHV0LCBJbnRlZ3JhdGlvblBhdHRlcm4gfSBmcm9tICdAYXdzLWNkay9hd3Mtc3RlcGZ1bmN0aW9ucyc7XG5pbXBvcnQgeyBDYWxsQXdzU2VydmljZSwgTGFtYmRhSW52b2tlLCBTdGVwRnVuY3Rpb25zU3RhcnRFeGVjdXRpb24gfSBmcm9tICdAYXdzLWNkay9hd3Mtc3RlcGZ1bmN0aW9ucy10YXNrcyc7XG5pbXBvcnQgeyBDb25zdHJ1Y3QsIER1cmF0aW9uLCBTdGFjaywgQXJuRm9ybWF0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBSZXBvc2l0b3J5IH0gZnJvbSAnLi4vLi4vY29kZWFydGlmYWN0L3JlcG9zaXRvcnknO1xuaW1wb3J0IHsgQ29uZmlnRmlsZSB9IGZyb20gJy4uLy4uL2NvbmZpZy1maWxlJztcbmltcG9ydCB7IGxhbWJkYUZ1bmN0aW9uVXJsLCBzcXNRdWV1ZVVybCB9IGZyb20gJy4uLy4uL2RlZXAtbGluayc7XG5pbXBvcnQgeyBNb25pdG9yaW5nIH0gZnJvbSAnLi4vLi4vbW9uaXRvcmluZyc7XG5pbXBvcnQgeyBQYWNrYWdlVGFnQ29uZmlnIH0gZnJvbSAnLi4vLi4vcGFja2FnZS10YWcnO1xuaW1wb3J0IHsgUlVOQk9PS19VUkwgfSBmcm9tICcuLi8uLi9ydW5ib29rLXVybCc7XG5pbXBvcnQgdHlwZSB7IFBhY2thZ2VMaW5rQ29uZmlnIH0gZnJvbSAnLi4vLi4vd2ViYXBwJztcbmltcG9ydCB7IE9yY2hlc3RyYXRpb24gfSBmcm9tICcuLi9vcmNoZXN0cmF0aW9uJztcbmltcG9ydCB7IFNUT1JBR0VfS0VZX1BSRUZJWCwgTUVUQURBVEFfS0VZX1NVRkZJWCwgUEFDS0FHRV9LRVlfU1VGRklYIH0gZnJvbSAnLi4vc2hhcmVkL2NvbnN0YW50cyc7XG5pbXBvcnQgeyBNZXRyaWNOYW1lLCBNRVRSSUNTX05BTUVTUEFDRSB9IGZyb20gJy4vY29uc3RhbnRzJztcbmltcG9ydCB7IEluZ2VzdGlvbiBhcyBIYW5kbGVyIH0gZnJvbSAnLi9pbmdlc3Rpb24nO1xuaW1wb3J0IHsgUmVJbmdlc3QgfSBmcm9tICcuL3JlLWluZ2VzdCc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgSW5nZXN0aW9uUHJvcHMge1xuICAvKipcbiAgICogVGhlIGJ1Y2tldCBpbiB3aGljaCBpbmdlc3RlZCBvYmplY3RzIGFyZSBkdWUgdG8gYmUgaW5zZXJ0ZWQuXG4gICAqL1xuICByZWFkb25seSBidWNrZXQ6IElCdWNrZXQ7XG5cbiAgLyoqXG4gICAqIFRoZSBDb2RlQXJ0aWZhY3QgcmVwb3NpdG9yeSB0byB3aGljaCBwYWNrYWdlcyBzaG91bGQgYmUgcHVibGlzaGVkLiBUaGlzIGlzXG4gICAqIHRoZSBDb25zdHJ1Y3RIdWIgaW50ZXJuYWwgQ29kZUFydGlmYWN0IHJlcG9zaXRvcnksIGlmIG9uZSBleGlzdHMuXG4gICAqL1xuICByZWFkb25seSBjb2RlQXJ0aWZhY3Q/OiBSZXBvc2l0b3J5O1xuXG4gIC8qKlxuICAgKiBUaGUgbW9uaXRvcmluZyBoYW5kbGVyIHRvIHJlZ2lzdGVyIGFsYXJtcyB3aXRoLlxuICAgKi9cbiAgcmVhZG9ubHkgbW9uaXRvcmluZzogTW9uaXRvcmluZztcblxuICAvKipcbiAgICogVGhlIGJhY2tlbmQgb3JjaGVzdHJhdGlvbiB0byBpbnZva2Ugb25jZSB0aGUgcGFja2FnZSBtZXRhZGF0YSBoYXMgYmVlblxuICAgKiBzdWNjZXNzZnVsbHkgcmVnaXN0ZXJlZC5cbiAgICovXG4gIHJlYWRvbmx5IG9yY2hlc3RyYXRpb246IE9yY2hlc3RyYXRpb247XG5cbiAgLyoqXG4gICAqIEhvdyBsb25nIHRvIHJldGFpbiB0aGUgQ2xvdWRXYXRjaCBsb2dzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBSZXRlbnRpb25EYXlzLlRFTl9ZRUFSU1xuICAgKi9cbiAgcmVhZG9ubHkgbG9nUmV0ZW50aW9uPzogUmV0ZW50aW9uRGF5cztcblxuICAvKipcbiAgICogQ29uZmlndXJhdGlvbiBmb3IgY3VzdG9tIHBhY2thZ2UgcGFnZSBsaW5rcy5cbiAgICovXG4gIHJlYWRvbmx5IHBhY2thZ2VMaW5rcz86IFBhY2thZ2VMaW5rQ29uZmlnW107XG5cbiAgLyoqXG4gICAqIFNlcmlhbGl6ZWQgY29uZmlndXJhdGlvbiBmb3IgY3VzdG9tIHBhY2thZ2UgdGFncy5cbiAgICovXG4gIHJlYWRvbmx5IHBhY2thZ2VUYWdzPzogUGFja2FnZVRhZ0NvbmZpZ1tdO1xufVxuXG4vKipcbiAqIFRoZSBpbmdlc3Rpb24gZnVuY3Rpb24gcmVjZWl2ZXMgbWVzc2FnZXMgZnJvbSBkaXNjb3ZlcnkgaW50ZWdyYXRpb25zIGFuZFxuICogcHJvY2Vzc2VzIHRoZWlyIHBheWxvYWRzIGludG8gdGhlIHByb3ZpZGVkIFMzIEJ1Y2tldC5cbiAqXG4gKiBUaGlzIGZ1bmN0aW9uIGlzIGFsc28gYW4gYElHcmFudGFibGVgLCBzbyB0aGF0IGl0IGNhbiBiZSBncmFudGVkIHBlcm1pc3Npb25zXG4gKiB0byByZWFkIGZyb20gdGhlIHNvdXJjZSBTMyBidWNrZXRzLlxuICovXG5leHBvcnQgY2xhc3MgSW5nZXN0aW9uIGV4dGVuZHMgQ29uc3RydWN0IGltcGxlbWVudHMgSUdyYW50YWJsZSB7XG4gIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbDogSVByaW5jaXBhbDtcblxuICAvKipcbiAgICogVGhlIFNRUyBxdWV1ZSB0aGF0IHRyaWdnZXJzIHRoZSBpbmdlc3Rpb24gZnVuY3Rpb24uXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcXVldWU6IElRdWV1ZTtcblxuICAvKipcbiAgICogVGhlIGluZ2VzdGlvbiBkZWFkIGxldHRlciBxdWV1ZSwgd2hpY2ggd2lsbCBob2xkIG1lc3NhZ2VzIHRoYXQgZmFpbGVkXG4gICAqIGluZ2VzdGlvbiBvbmUgdG9vIG1hbnkgdGltZXMsIHNvIHRoYXQgcG9pc29uIHBpbGxzIGRvbid0IGVuZGxlc3NseSBjb25zdW1lXG4gICAqIHJlc291cmNlcy5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBkZWFkTGV0dGVyUXVldWU6IElRdWV1ZTtcblxuICBwdWJsaWMgcmVhZG9ubHkgcXVldWVSZXRlbnRpb25QZXJpb2QgPSBEdXJhdGlvbi5kYXlzKDE0KTtcblxuICBwdWJsaWMgcmVhZG9ubHkgZnVuY3Rpb246IElGdW5jdGlvbjtcblxuICBwdWJsaWMgY29uc3RydWN0b3Ioc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEluZ2VzdGlvblByb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkKTtcblxuICAgIHRoaXMuZGVhZExldHRlclF1ZXVlID0gbmV3IFF1ZXVlKHRoaXMsICdETFEnLCB7XG4gICAgICBlbmNyeXB0aW9uOiBRdWV1ZUVuY3J5cHRpb24uS01TX01BTkFHRUQsXG4gICAgICByZXRlbnRpb25QZXJpb2Q6IHRoaXMucXVldWVSZXRlbnRpb25QZXJpb2QsXG4gICAgICB2aXNpYmlsaXR5VGltZW91dDogRHVyYXRpb24ubWludXRlcygxNSksXG4gICAgfSk7XG5cbiAgICB0aGlzLnF1ZXVlID0gbmV3IFF1ZXVlKHRoaXMsICdRdWV1ZScsIHtcbiAgICAgIGRlYWRMZXR0ZXJRdWV1ZToge1xuICAgICAgICBtYXhSZWNlaXZlQ291bnQ6IDUsXG4gICAgICAgIHF1ZXVlOiB0aGlzLmRlYWRMZXR0ZXJRdWV1ZSxcbiAgICAgIH0sXG4gICAgICBlbmNyeXB0aW9uOiBRdWV1ZUVuY3J5cHRpb24uS01TX01BTkFHRUQsXG4gICAgICByZXRlbnRpb25QZXJpb2Q6IHRoaXMucXVldWVSZXRlbnRpb25QZXJpb2QsXG4gICAgICB2aXNpYmlsaXR5VGltZW91dDogRHVyYXRpb24ubWludXRlcygxNSksXG4gICAgfSk7XG5cbiAgICBjb25zdCBjb25maWdGaWxlbmFtZSA9ICdjb25maWcuanNvbic7XG4gICAgY29uc3QgY29uZmlnID0gbmV3IENvbmZpZ0ZpbGUoY29uZmlnRmlsZW5hbWUsIEpTT04uc3RyaW5naWZ5KHtcbiAgICAgIHBhY2thZ2VMaW5rczogcHJvcHMucGFja2FnZUxpbmtzID8/IFtdLFxuICAgICAgcGFja2FnZVRhZ3M6IHByb3BzLnBhY2thZ2VUYWdzID8/IFtdLFxuICAgIH0pKTtcblxuICAgIGNvbnN0IGNvbmZpZ0J1Y2tldCA9IG5ldyBCdWNrZXQodGhpcywgJ0NvbmZpZ0J1Y2tldCcsIHtcbiAgICAgIGJsb2NrUHVibGljQWNjZXNzOiBCbG9ja1B1YmxpY0FjY2Vzcy5CTE9DS19BTEwsXG4gICAgICBlbmZvcmNlU1NMOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgbmV3IEJ1Y2tldERlcGxveW1lbnQodGhpcywgJ0RlcGxveUluZ2VzdGlvbkNvbmZpZ3VyYXRpb24nLCB7XG4gICAgICBzb3VyY2VzOiBbU291cmNlLmFzc2V0KGNvbmZpZy5kaXIpXSxcbiAgICAgIGRlc3RpbmF0aW9uQnVja2V0OiBjb25maWdCdWNrZXQsXG4gICAgfSk7XG5cbiAgICBjb25zdCBlbnZpcm9ubWVudDogRnVuY3Rpb25Qcm9wc1snZW52aXJvbm1lbnQnXSA9IHtcbiAgICAgIEFXU19FTUZfRU5WSVJPTk1FTlQ6ICdMb2NhbCcsXG4gICAgICBCVUNLRVRfTkFNRTogcHJvcHMuYnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICBDT05GSUdfQlVDS0VUX05BTUU6IGNvbmZpZ0J1Y2tldC5idWNrZXROYW1lLFxuICAgICAgQ09ORklHX0ZJTEVfS0VZOiBjb25maWdGaWxlbmFtZSxcbiAgICAgIFNUQVRFX01BQ0hJTkVfQVJOOiBwcm9wcy5vcmNoZXN0cmF0aW9uLnN0YXRlTWFjaGluZS5zdGF0ZU1hY2hpbmVBcm4sXG4gICAgfTtcblxuICAgIGlmIChwcm9wcy5jb2RlQXJ0aWZhY3QpIHtcbiAgICAgIGVudmlyb25tZW50LkNPREVfQVJUSUZBQ1RfUkVQT1NJVE9SWV9FTkRQT0lOVCA9IHByb3BzLmNvZGVBcnRpZmFjdC5wdWJsaXNoaW5nUmVwb3NpdG9yeU5wbUVuZHBvaW50O1xuICAgICAgZW52aXJvbm1lbnQuQ09ERV9BUlRJRkFDVF9ET01BSU5fTkFNRSA9IHByb3BzLmNvZGVBcnRpZmFjdC5yZXBvc2l0b3J5RG9tYWluTmFtZTtcbiAgICAgIGVudmlyb25tZW50LkNPREVfQVJUSUZBQ1RfRE9NQUlOX09XTkVSID0gcHJvcHMuY29kZUFydGlmYWN0LnJlcG9zaXRvcnlEb21haW5Pd25lcjtcbiAgICB9XG5cbiAgICBjb25zdCBoYW5kbGVyID0gbmV3IEhhbmRsZXIodGhpcywgJ0RlZmF1bHQnLCB7XG4gICAgICBkZXNjcmlwdGlvbjogJ1tDb25zdHJ1Y3RIdWIvSW5nZXN0aW9uXSBJbmdlc3RzIG5ldyBwYWNrYWdlIHZlcnNpb25zIGludG8gdGhlIENvbnN0cnVjdCBIdWInLFxuICAgICAgZW52aXJvbm1lbnQsXG4gICAgICBsb2dSZXRlbnRpb246IHByb3BzLmxvZ1JldGVudGlvbiA/PyBSZXRlbnRpb25EYXlzLlRFTl9ZRUFSUyxcbiAgICAgIG1lbW9yeVNpemU6IDEwXzI0MCwgLy8gQ3VycmVudGx5IHRoZSBtYXhpbXVtIHBvc3NpYmxlIHNldHRpbmdcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLm1pbnV0ZXMoMTUpLFxuICAgICAgdHJhY2luZzogVHJhY2luZy5BQ1RJVkUsXG4gICAgfSk7XG4gICAgdGhpcy5mdW5jdGlvbiA9IGhhbmRsZXI7XG5cbiAgICBjb25maWdCdWNrZXQuZ3JhbnRSZWFkKGhhbmRsZXIpO1xuICAgIHByb3BzLmJ1Y2tldC5ncmFudFdyaXRlKHRoaXMuZnVuY3Rpb24pO1xuICAgIHByb3BzLmNvZGVBcnRpZmFjdD8uZ3JhbnRQdWJsaXNoVG9SZXBvc2l0b3J5KGhhbmRsZXIpO1xuICAgIHByb3BzLm9yY2hlc3RyYXRpb24uc3RhdGVNYWNoaW5lLmdyYW50U3RhcnRFeGVjdXRpb24odGhpcy5mdW5jdGlvbik7XG5cbiAgICB0aGlzLmZ1bmN0aW9uLmFkZEV2ZW50U291cmNlKG5ldyBTcXNFdmVudFNvdXJjZSh0aGlzLnF1ZXVlLCB7IGJhdGNoU2l6ZTogMSB9KSk7XG4gICAgLy8gVGhpcyBldmVudCBzb3VyY2UgaXMgZGlzYWJsZWQsIGFuZCBjYW4gYmUgdXNlZCB0byByZS1wcm9jZXNzIGRlYWQtbGV0dGVyLXF1ZXVlIG1lc3NhZ2VzXG4gICAgdGhpcy5mdW5jdGlvbi5hZGRFdmVudFNvdXJjZShuZXcgU3FzRXZlbnRTb3VyY2UodGhpcy5kZWFkTGV0dGVyUXVldWUsIHsgYmF0Y2hTaXplOiAxLCBlbmFibGVkOiBmYWxzZSB9KSk7XG5cblxuICAgIC8vIFJlcHJvY2VzcyB3b3JrZmxvd1xuICAgIGNvbnN0IHJlcHJvY2Vzc1F1ZXVlID0gbmV3IFF1ZXVlKHRoaXMsICdSZXByb2Nlc3NRdWV1ZScsIHtcbiAgICAgIGRlYWRMZXR0ZXJRdWV1ZToge1xuICAgICAgICBtYXhSZWNlaXZlQ291bnQ6IDUsXG4gICAgICAgIHF1ZXVlOiB0aGlzLmRlYWRMZXR0ZXJRdWV1ZSxcbiAgICAgIH0sXG4gICAgICBlbmNyeXB0aW9uOiBRdWV1ZUVuY3J5cHRpb24uS01TX01BTkFHRUQsXG4gICAgICByZXRlbnRpb25QZXJpb2Q6IHRoaXMucXVldWVSZXRlbnRpb25QZXJpb2QsXG4gICAgICAvLyBWaXNpYmlsaXR5IHRpbWVvdXQgb2YgMTUgbWludXRlcyBtYXRjaGVzIHRoZSBMYW1iZGEgbWF4aW11bSBleGVjdXRpb24gdGltZS5cbiAgICAgIHZpc2liaWxpdHlUaW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDE1KSxcbiAgICB9KTtcbiAgICBwcm9wcy5idWNrZXQuZ3JhbnRSZWFkKHRoaXMuZnVuY3Rpb24sIGAke1NUT1JBR0VfS0VZX1BSRUZJWH0qJHtQQUNLQUdFX0tFWV9TVUZGSVh9YCk7XG4gICAgdGhpcy5mdW5jdGlvbi5hZGRFdmVudFNvdXJjZShuZXcgU3FzRXZlbnRTb3VyY2UocmVwcm9jZXNzUXVldWUsIHsgYmF0Y2hTaXplOiAxIH0pKTtcbiAgICBuZXcgUmVwcm9jZXNzSW5nZXN0aW9uV29ya2Zsb3codGhpcywgJ1JlcHJvY2Vzc1dvcmtmbG93JywgeyBidWNrZXQ6IHByb3BzLmJ1Y2tldCwgcXVldWU6IHJlcHJvY2Vzc1F1ZXVlIH0pO1xuXG4gICAgdGhpcy5ncmFudFByaW5jaXBhbCA9IHRoaXMuZnVuY3Rpb24uZ3JhbnRQcmluY2lwYWw7XG5cbiAgICBwcm9wcy5tb25pdG9yaW5nLmFkZEhpZ2hTZXZlcml0eUFsYXJtKFxuICAgICAgJ0luZ2VzdGlvbiBEZWFkLUxldHRlciBRdWV1ZSBub3QgZW1wdHknLFxuICAgICAgbmV3IE1hdGhFeHByZXNzaW9uKHtcbiAgICAgICAgZXhwcmVzc2lvbjogJ20xICsgbTInLFxuICAgICAgICB1c2luZ01ldHJpY3M6IHtcbiAgICAgICAgICBtMTogdGhpcy5kZWFkTGV0dGVyUXVldWUubWV0cmljQXBwcm94aW1hdGVOdW1iZXJPZk1lc3NhZ2VzVmlzaWJsZSh7IHBlcmlvZDogRHVyYXRpb24ubWludXRlcygxKSB9KSxcbiAgICAgICAgICBtMjogdGhpcy5kZWFkTGV0dGVyUXVldWUubWV0cmljQXBwcm94aW1hdGVOdW1iZXJPZk1lc3NhZ2VzTm90VmlzaWJsZSh7IHBlcmlvZDogRHVyYXRpb24ubWludXRlcygxKSB9KSxcbiAgICAgICAgfSxcbiAgICAgIH0pLmNyZWF0ZUFsYXJtKHRoaXMsICdETFFBbGFybScsIHtcbiAgICAgICAgYWxhcm1OYW1lOiBgJHt0aGlzLm5vZGUucGF0aH0vRExRTm90RW1wdHlgLFxuICAgICAgICBhbGFybURlc2NyaXB0aW9uOiBbXG4gICAgICAgICAgJ1RoZSBkZWFkLWxldHRlciBxdWV1ZSBmb3IgdGhlIEluZ2VzdGlvbiBmdW5jdGlvbiBpcyBub3QgZW1wdHkhJyxcbiAgICAgICAgICAnJyxcbiAgICAgICAgICBgUnVuQm9vazogJHtSVU5CT09LX1VSTH1gLFxuICAgICAgICAgICcnLFxuICAgICAgICAgIGBEaXJlY3QgbGluayB0byB0aGUgcXVldWU6ICR7c3FzUXVldWVVcmwodGhpcy5kZWFkTGV0dGVyUXVldWUpfWAsXG4gICAgICAgICAgYERpcmVjdCBsaW5rIHRvIHRoZSBmdW5jdGlvbjogJHtsYW1iZGFGdW5jdGlvblVybCh0aGlzLmZ1bmN0aW9uKX1gLFxuICAgICAgICBdLmpvaW4oJ1xcbicpLFxuICAgICAgICBjb21wYXJpc29uT3BlcmF0b3I6IENvbXBhcmlzb25PcGVyYXRvci5HUkVBVEVSX1RIQU5fT1JfRVFVQUxfVE9fVEhSRVNIT0xELFxuICAgICAgICBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICAgICAgdGhyZXNob2xkOiAxLFxuICAgICAgICAvLyBTUVMgZG9lcyBub3QgZW1pdCBtZXRyaWNzIGlmIHRoZSBxdWV1ZSBoYXMgYmVlbiBlbXB0eSBmb3IgYSB3aGlsZSwgd2hpY2ggaXMgR09PRC5cbiAgICAgICAgdHJlYXRNaXNzaW5nRGF0YTogVHJlYXRNaXNzaW5nRGF0YS5OT1RfQlJFQUNISU5HLFxuICAgICAgfSksXG4gICAgKTtcbiAgICBwcm9wcy5tb25pdG9yaW5nLmFkZEhpZ2hTZXZlcml0eUFsYXJtKFxuICAgICAgJ0luZ2VzdGlvbiBmYWlsdXJlcycsXG4gICAgICB0aGlzLmZ1bmN0aW9uLm1ldHJpY0Vycm9ycygpLmNyZWF0ZUFsYXJtKHRoaXMsICdGYWlsdXJlQWxhcm0nLCB7XG4gICAgICAgIGFsYXJtTmFtZTogYCR7dGhpcy5ub2RlLnBhdGh9L0ZhaWx1cmVgLFxuICAgICAgICBhbGFybURlc2NyaXB0aW9uOiBbXG4gICAgICAgICAgJ1RoZSBJbmdlc3Rpb24gZnVuY3Rpb24gaXMgZmFpbGluZyEnLFxuICAgICAgICAgICcnLFxuICAgICAgICAgIGBSdW5Cb29rOiAke1JVTkJPT0tfVVJMfWAsXG4gICAgICAgICAgJycsXG4gICAgICAgICAgYERpcmVjdCBsaW5rIHRvIHRoZSBmdW5jdGlvbjogJHtsYW1iZGFGdW5jdGlvblVybCh0aGlzLmZ1bmN0aW9uKX1gLFxuICAgICAgICBdLmpvaW4oJ1xcbicpLFxuICAgICAgICBjb21wYXJpc29uT3BlcmF0b3I6IENvbXBhcmlzb25PcGVyYXRvci5HUkVBVEVSX1RIQU5fT1JfRVFVQUxfVE9fVEhSRVNIT0xELFxuICAgICAgICBldmFsdWF0aW9uUGVyaW9kczogMixcbiAgICAgICAgdGhyZXNob2xkOiAxLFxuICAgICAgICAvLyBMYW1iZGEgb25seSBlbWl0cyBtZXRyaWNzIHdoZW4gdGhlIGZ1bmN0aW9uIGlzIGludm9rZWQuIE5vIGludm9rYXRpb24gPT4gbm8gZXJyb3JzLlxuICAgICAgICB0cmVhdE1pc3NpbmdEYXRhOiBUcmVhdE1pc3NpbmdEYXRhLk5PVF9CUkVBQ0hJTkcsXG4gICAgICB9KSxcbiAgICApO1xuICB9XG5cbiAgcHVibGljIG1ldHJpY0ZvdW5kTGljZW5zZUZpbGUob3B0cz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiBuZXcgTWV0cmljKHtcbiAgICAgIHBlcmlvZDogRHVyYXRpb24ubWludXRlcyg1KSxcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLlNVTSxcbiAgICAgIC4uLm9wdHMsXG4gICAgICBtZXRyaWNOYW1lOiBNZXRyaWNOYW1lLkZPVU5EX0xJQ0VOU0VfRklMRSxcbiAgICAgIG5hbWVzcGFjZTogTUVUUklDU19OQU1FU1BBQ0UsXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgbWV0cmljSW5lbGlnaWJsZUxpY2Vuc2Uob3B0cz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiBuZXcgTWV0cmljKHtcbiAgICAgIHBlcmlvZDogRHVyYXRpb24ubWludXRlcyg1KSxcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLlNVTSxcbiAgICAgIC4uLm9wdHMsXG4gICAgICBtZXRyaWNOYW1lOiBNZXRyaWNOYW1lLklORUxJR0lCTEVfTElDRU5TRSxcbiAgICAgIG5hbWVzcGFjZTogTUVUUklDU19OQU1FU1BBQ0UsXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgbWV0cmljSW52YWxpZEFzc2VtYmx5KG9wdHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gbmV3IE1ldHJpYyh7XG4gICAgICBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoNSksXG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5TVU0sXG4gICAgICAuLi5vcHRzLFxuICAgICAgbWV0cmljTmFtZTogTWV0cmljTmFtZS5JTlZBTElEX0FTU0VNQkxZLFxuICAgICAgbmFtZXNwYWNlOiBNRVRSSUNTX05BTUVTUEFDRSxcbiAgICB9KTtcbiAgfVxuXG4gIHB1YmxpYyBtZXRyaWNJbnZhbGlkVGFyYmFsbChvcHRzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIG5ldyBNZXRyaWMoe1xuICAgICAgcGVyaW9kOiBEdXJhdGlvbi5taW51dGVzKDUpLFxuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuU1VNLFxuICAgICAgLi4ub3B0cyxcbiAgICAgIG1ldHJpY05hbWU6IE1ldHJpY05hbWUuSU5WQUxJRF9UQVJCQUxMLFxuICAgICAgbmFtZXNwYWNlOiBNRVRSSUNTX05BTUVTUEFDRSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGlzIG1ldHJpY3MgaXMgdGhlIHRvdGFsIGNvdW50IG9mIHBhY2thZ2VzIHRoYXQgd2VyZSByZWplY3RlZCBkdWUgdG9cbiAgICogbWlzbWF0Y2hlZCBpZGVudGl0eSAobmFtZSwgdmVyc2lvbiwgbGljZW5zZSkgYmV0d2VlbiB0aGUgYHBhY2thZ2UuanNvbmBcbiAgICogZmlsZSBhbmQgdGUgYC5qc2lpYCBhdHRyaWJ1dGUuXG4gICAqL1xuICBwdWJsaWMgbWV0cmljTWlzbWF0Y2hlZElkZW50aXR5UmVqZWN0aW9ucyhvcHRzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIG5ldyBNZXRyaWMoe1xuICAgICAgcGVyaW9kOiBEdXJhdGlvbi5taW51dGVzKDUpLFxuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuU1VNLFxuICAgICAgLi4ub3B0cyxcbiAgICAgIG1ldHJpY05hbWU6IE1ldHJpY05hbWUuTUlTTUFUQ0hFRF9JREVOVElUWV9SRUpFQ1RJT05TLFxuICAgICAgbmFtZXNwYWNlOiBNRVRSSUNTX05BTUVTUEFDRSxcbiAgICB9KTtcbiAgfVxufVxuXG5pbnRlcmZhY2UgUmVwcm9jZXNzSW5nZXN0aW9uV29ya2Zsb3dQcm9wcyB7XG4gIHJlYWRvbmx5IGJ1Y2tldDogSUJ1Y2tldDtcbiAgcmVhZG9ubHkgcXVldWU6IElRdWV1ZTtcbn1cblxuLyoqXG4gKiBBIFN0ZXBGdW5jdGlvbnMgU3RhdGUgTWFjaGluZSB0byByZXByb2Nlc3MgZXZlcnkgY3VycmVudGx5IGluZGV4ZWQgcGFja2FnZVxuICogdGhyb3VnaCB0aGUgaW5nZXN0aW9uIGZ1bmN0aW9uLiBUaGlzIHNob3VsZCBub3QgYmUgZnJlcXVlbnRseSByZXF1aXJlZCwgYnV0XG4gKiBjYW4gY29tZSBpbiBoYW5keSBhdCB0aW1lcy5cbiAqXG4gKiBGb3IgbW9yZSBpbmZvcm1hdGlvbiwgcmVmZXIgdG8gdGhlIHJ1bmJvb2sgYXRcbiAqIGh0dHBzOi8vZ2l0aHViLmNvbS9jZGtsYWJzL2NvbnN0cnVjdC1odWIvYmxvYi9tYWluL2RvY3Mvb3BlcmF0b3ItcnVuYm9vay5tZFxuICovXG5jbGFzcyBSZXByb2Nlc3NJbmdlc3Rpb25Xb3JrZmxvdyBleHRlbmRzIENvbnN0cnVjdCB7XG4gIHB1YmxpYyBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogUmVwcm9jZXNzSW5nZXN0aW9uV29ya2Zsb3dQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBsYW1iZGFGdW5jdGlvbiA9IG5ldyBSZUluZ2VzdCh0aGlzLCAnRnVuY3Rpb24nLCB7XG4gICAgICBkZXNjcmlwdGlvbjogJ1tDb25zdHJ1Y3RIdWIvSW5nZXN0aW9uL1JlSW5nZXN0XSBUaGUgZnVuY3Rpb24gdXNlZCB0byByZXByb2Nlc3MgcGFja2FnZXMgdGhyb3VnaCBpbmdlc3Rpb24nLFxuICAgICAgZW52aXJvbm1lbnQ6IHsgQlVDS0VUX05BTUU6IHByb3BzLmJ1Y2tldC5idWNrZXROYW1lLCBRVUVVRV9VUkw6IHByb3BzLnF1ZXVlLnF1ZXVlVXJsIH0sXG4gICAgICB0cmFjaW5nOiBUcmFjaW5nLkFDVElWRSxcbiAgICB9KTtcblxuICAgIHByb3BzLnF1ZXVlLmdyYW50U2VuZE1lc3NhZ2VzKGxhbWJkYUZ1bmN0aW9uKTtcbiAgICBwcm9wcy5idWNrZXQuZ3JhbnRSZWFkKGxhbWJkYUZ1bmN0aW9uLCBgJHtTVE9SQUdFX0tFWV9QUkVGSVh9KiR7TUVUQURBVEFfS0VZX1NVRkZJWH1gKTtcbiAgICBwcm9wcy5idWNrZXQuZ3JhbnRSZWFkKGxhbWJkYUZ1bmN0aW9uLCBgJHtTVE9SQUdFX0tFWV9QUkVGSVh9KiR7UEFDS0FHRV9LRVlfU1VGRklYfWApO1xuXG4gICAgY29uc3QgbGlzdEJ1Y2tldCA9IG5ldyBDaG9pY2UodGhpcywgJ0hhcyBhIENvbnRpbnVhdGlvblRva2VuPycpXG4gICAgICAud2hlbihDb25kaXRpb24uaXNQcmVzZW50KCckLkNvbnRpbnVhdGlvblRva2VuJyksXG4gICAgICAgIG5ldyBDYWxsQXdzU2VydmljZSh0aGlzLCAnUzMuTGlzdE9iamVjdHNWMihOZXh0UGFnZSknLCB7XG4gICAgICAgICAgc2VydmljZTogJ3MzJyxcbiAgICAgICAgICBhY3Rpb246ICdsaXN0T2JqZWN0c1YyJyxcbiAgICAgICAgICBpYW1BY3Rpb246ICdzMzpMaXN0QnVja2V0JyxcbiAgICAgICAgICBpYW1SZXNvdXJjZXM6IFtwcm9wcy5idWNrZXQuYnVja2V0QXJuXSxcbiAgICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgICBCdWNrZXQ6IHByb3BzLmJ1Y2tldC5idWNrZXROYW1lLFxuICAgICAgICAgICAgQ29udGludWF0aW9uVG9rZW46IEpzb25QYXRoLnN0cmluZ0F0KCckLkNvbnRpbnVhdGlvblRva2VuJyksXG4gICAgICAgICAgICBQcmVmaXg6IFNUT1JBR0VfS0VZX1BSRUZJWCxcbiAgICAgICAgICB9LFxuICAgICAgICAgIHJlc3VsdFBhdGg6ICckLnJlc3BvbnNlJyxcbiAgICAgICAgfSkpXG4gICAgICAub3RoZXJ3aXNlKG5ldyBDYWxsQXdzU2VydmljZSh0aGlzLCAnUzMuTGlzdE9iamVjdHNWMihGaXJzdFBhZ2UpJywge1xuICAgICAgICBzZXJ2aWNlOiAnczMnLFxuICAgICAgICBhY3Rpb246ICdsaXN0T2JqZWN0c1YyJyxcbiAgICAgICAgaWFtQWN0aW9uOiAnczM6TGlzdEJ1Y2tldCcsXG4gICAgICAgIGlhbVJlc291cmNlczogW3Byb3BzLmJ1Y2tldC5idWNrZXRBcm5dLFxuICAgICAgICBwYXJhbWV0ZXJzOiB7XG4gICAgICAgICAgQnVja2V0OiBwcm9wcy5idWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAgICBQcmVmaXg6IFNUT1JBR0VfS0VZX1BSRUZJWCxcbiAgICAgICAgfSxcbiAgICAgICAgcmVzdWx0UGF0aDogJyQucmVzcG9uc2UnLFxuICAgICAgfSkpLmFmdGVyd2FyZHMoKTtcblxuICAgIGNvbnN0IHByb2Nlc3MgPSBuZXcgTWFwKHRoaXMsICdQcm9jZXNzIFJlc3VsdCcsIHtcbiAgICAgIGl0ZW1zUGF0aDogJyQucmVzcG9uc2UuQ29udGVudHMnLFxuICAgICAgcmVzdWx0UGF0aDogSnNvblBhdGguRElTQ0FSRCxcbiAgICB9KS5pdGVyYXRvcihcbiAgICAgIG5ldyBDaG9pY2UodGhpcywgJ0lzIG1ldGFkYXRhIG9iamVjdD8nKVxuICAgICAgICAud2hlbihcbiAgICAgICAgICBDb25kaXRpb24uc3RyaW5nTWF0Y2hlcygnJC5LZXknLCBgKiR7TUVUQURBVEFfS0VZX1NVRkZJWH1gKSxcbiAgICAgICAgICBuZXcgTGFtYmRhSW52b2tlKHRoaXMsICdTZW5kIGZvciByZXByb2Nlc3NpbmcnLCB7IGxhbWJkYUZ1bmN0aW9uIH0pXG4gICAgICAgICAgICAvLyBBbXBsZSByZXRyaWVzIGhlcmUuLi4gV2Ugc2hvdWxkIG5ldmVyIGZhaWwgYmVjYXVzZSBvZiB0aHJvdHRsaW5nLi4uLlxuICAgICAgICAgICAgLmFkZFJldHJ5KHsgZXJyb3JzOiBbJ0xhbWJkYS5Ub29NYW55UmVxdWVzdHNFeGNlcHRpb24nXSwgYmFja29mZlJhdGU6IDEuMSwgaW50ZXJ2YWw6IER1cmF0aW9uLm1pbnV0ZXMoMSksIG1heEF0dGVtcHRzOiAzMCB9KSxcbiAgICAgICAgKVxuICAgICAgICAub3RoZXJ3aXNlKG5ldyBTdWNjZWVkKHRoaXMsICdOb3RoaW5nIHRvIGRvJykpLFxuICAgICk7XG5cbiAgICAvLyBOZWVkIHRvIHBoeXNpY2FsLW5hbWUgdGhlIHN0YXRlIG1hY2hpbmUgc28gaXQgY2FuIHNlbGYtaW52b2tlLlxuICAgIGNvbnN0IHN0YXRlTWFjaGluZU5hbWUgPSBzdGF0ZU1hY2hpbmVOYW1lRnJvbSh0aGlzLm5vZGUucGF0aCk7XG4gICAgbGlzdEJ1Y2tldC5uZXh0KHByb2Nlc3MubmV4dChuZXcgQ2hvaWNlKHRoaXMsICdJcyB0aGVyZSBtb3JlPycpXG4gICAgICAud2hlbihcbiAgICAgICAgQ29uZGl0aW9uLmlzUHJlc2VudCgnJC5yZXNwb25zZS5OZXh0Q29udGludWF0aW9uVG9rZW4nKSxcbiAgICAgICAgbmV3IFN0ZXBGdW5jdGlvbnNTdGFydEV4ZWN1dGlvbih0aGlzLCAnQ29udGludWUgYXMgbmV3Jywge1xuICAgICAgICAgIGFzc29jaWF0ZVdpdGhQYXJlbnQ6IHRydWUsXG4gICAgICAgICAgc3RhdGVNYWNoaW5lOiBTdGF0ZU1hY2hpbmUuZnJvbVN0YXRlTWFjaGluZUFybih0aGlzLCAnVGhpc1N0YXRlTWFjaGluZScsIFN0YWNrLm9mKHRoaXMpLmZvcm1hdEFybih7XG4gICAgICAgICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FLFxuICAgICAgICAgICAgc2VydmljZTogJ3N0YXRlcycsXG4gICAgICAgICAgICByZXNvdXJjZTogJ3N0YXRlTWFjaGluZScsXG4gICAgICAgICAgICByZXNvdXJjZU5hbWU6IHN0YXRlTWFjaGluZU5hbWUsXG4gICAgICAgICAgfSkpLFxuICAgICAgICAgIGlucHV0OiBUYXNrSW5wdXQuZnJvbU9iamVjdCh7IENvbnRpbnVhdGlvblRva2VuOiBKc29uUGF0aC5zdHJpbmdBdCgnJC5yZXNwb25zZS5OZXh0Q29udGludWF0aW9uVG9rZW4nKSB9KSxcbiAgICAgICAgICBpbnRlZ3JhdGlvblBhdHRlcm46IEludGVncmF0aW9uUGF0dGVybi5SRVFVRVNUX1JFU1BPTlNFLFxuICAgICAgICB9KS5hZGRSZXRyeSh7IGVycm9yczogWydTdGVwRnVuY3Rpb25zLkV4ZWN1dGlvbkxpbWl0RXhjZWVkZWQnXSB9KSxcbiAgICAgIClcbiAgICAgIC5vdGhlcndpc2UobmV3IFN1Y2NlZWQodGhpcywgJ0FsbCBEb25lJykpKSk7XG5cbiAgICBjb25zdCBzdGF0ZU1hY2hpbmUgPSBuZXcgU3RhdGVNYWNoaW5lKHRoaXMsICdTdGF0ZU1hY2hpbmUnLCB7XG4gICAgICBkZWZpbml0aW9uOiBsaXN0QnVja2V0LFxuICAgICAgc3RhdGVNYWNoaW5lTmFtZSxcbiAgICAgIHRpbWVvdXQ6IER1cmF0aW9uLmhvdXJzKDEpLFxuICAgIH0pO1xuXG4gICAgcHJvcHMuYnVja2V0LmdyYW50UmVhZChzdGF0ZU1hY2hpbmUpO1xuICAgIHByb3BzLnF1ZXVlLmdyYW50U2VuZE1lc3NhZ2VzKHN0YXRlTWFjaGluZSk7XG4gIH1cbn1cblxuLyoqXG4gKiBUaGlzIHR1cm5zIGEgbm9kZSBwYXRoIGludG8gYSB2YWxpZCBzdGF0ZSBtYWNoaW5lIG5hbWUsIHRvIHRyeSBhbmQgaW1wcm92ZVxuICogdGhlIFN0ZXBGdW5jdGlvbidzIEFXUyBjb25zb2xlIGV4cGVyaWVuY2Ugd2hpbGUgbWluaW1pemluZyB0aGUgcmlzayBmb3JcbiAqIGNvbGxpc29ucy5cbiAqL1xuZnVuY3Rpb24gc3RhdGVNYWNoaW5lTmFtZUZyb20obm9kZVBhdGg6IHN0cmluZyk6IHN0cmluZyB7XG4gIC8vIFBvb3IgbWFuJ3MgcmVwbGFjZSBhbGwuLi5cbiAgcmV0dXJuIG5vZGVQYXRoLnNwbGl0KC9bXmEtejAtOSshQC4oKT1fJy1dKy9pKS5qb2luKCcuJyk7XG59XG4iXX0=