"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackendDashboard = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const language_1 = require("./backend/shared/language");
const deep_link_1 = require("./deep-link");
const metric_utils_1 = require("./metric-utils");
class BackendDashboard extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        new aws_cloudwatch_1.Dashboard(this, 'Resource', {
            dashboardName: props.dashboardName,
            periodOverride: aws_cloudwatch_1.PeriodOverride.AUTO,
            start: '-P1W',
            widgets: [
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Catalog Overview',
                            '',
                            `[button:primary:Package Data](${deep_link_1.s3ObjectUrl(props.packageData)})`,
                            `[button:Catalog Builder](${deep_link_1.lambdaFunctionUrl(props.orchestration.catalogBuilder.function)})`,
                            `[button:Inventory Canary](${deep_link_1.lambdaFunctionUrl(props.inventory.function)})`,
                            `[button:Search Canary Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.inventory.function)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Size',
                        left: [
                            props.inventory.metricSubmoduleCount({ label: 'Submodules' }),
                            props.inventory.metricPackageVersionCount({ label: 'Package Versions' }),
                            props.inventory.metricPackageMajorCount({ label: 'Package Majors' }),
                            props.inventory.metricPackageCount({ label: 'Packages' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Issues',
                        left: [
                            props.inventory.metricUnknownObjectCount({ label: 'Unknown' }),
                            props.inventory.metricMissingAssemblyCount({ label: 'Missing Assembly' }),
                            props.inventory.metricMissingPackageMetadataCount({ label: 'Missing Metadata' }),
                            props.inventory.metricMissingPackageTarballCount({ label: 'Missing Tarball' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.catalogBuilder.metricMissingConstructFrameworkCount({ label: 'No Construct Framework' }),
                            props.orchestration.catalogBuilder.metricMissingConstructFrameworkVersionCount({ label: 'No Construct Framework Version' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                ],
                ...this.catalogOverviewLanguageSections(props),
                ...renderPackageSourcesWidgets(props.packageSources),
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Ingestion Function',
                            '',
                            `[button:Ingestion Function](${deep_link_1.lambdaFunctionUrl(props.ingestion.function)})`,
                            `[button:primary:Search Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.ingestion.function)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.ingestion.deadLetterQueue)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            metric_utils_1.fillMetric(props.ingestion.function.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(props.ingestion.function.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Queue',
                        left: [
                            props.ingestion.queue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            props.ingestion.queue.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.queue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightAnnotations: [{
                                color: '#ffa500',
                                label: '10 Minutes',
                                value: core_1.Duration.minutes(10).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Quality',
                        left: [
                            metric_utils_1.fillMetric(props.ingestion.metricInvalidAssembly({ label: 'Invalid Assemblies' })),
                            metric_utils_1.fillMetric(props.ingestion.metricInvalidTarball({ label: 'Invalid Tarball' })),
                            metric_utils_1.fillMetric(props.ingestion.metricIneligibleLicense({ label: 'Ineligible License' })),
                            metric_utils_1.fillMetric(props.ingestion.metricMismatchedIdentityRejections({ label: 'Mismatched Identity' })),
                            metric_utils_1.fillMetric(props.ingestion.metricFoundLicenseFile({ label: 'Found License file' })),
                        ],
                        leftYAxis: { label: 'Count', min: 0, showUnits: false },
                        stacked: true,
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letters',
                        left: [
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightAnnotations: [{
                                color: '#ff7f0e',
                                label: '10 days',
                                value: core_1.Duration.days(10).toSeconds(),
                            }, {
                                color: '#ff0000',
                                label: '14 days (DLQ Retention)',
                                value: core_1.Duration.days(14).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Orchestration',
                            '',
                            `[button:primary:State Machine](${deep_link_1.stateMachineUrl(props.orchestration.stateMachine)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.orchestration.deadLetterQueue)})`,
                            `[button:Redrive DLQ](${deep_link_1.lambdaFunctionUrl(props.orchestration.redriveFunction)})`,
                            `[button:Regenerate All Documentation](${deep_link_1.stateMachineUrl(props.orchestration.regenerateAllDocumentation)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'State Machine Executions',
                        left: [
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricStarted({ label: 'Started' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricSucceeded({ label: 'Succeeded' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricAborted({ label: 'Aborted' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricFailed({ label: 'Failed' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricThrottled({ label: 'Throttled' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricTimedOut({ label: 'Timed Out' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.stateMachine.metricTime({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letter Queue',
                        left: [
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightAnnotations: [{
                                color: '#ff7f0e',
                                label: '10 days',
                                value: core_1.Duration.days(10).toSeconds(),
                            }, {
                                color: '#ff0000',
                                label: '14 days (DLQ Retention)',
                                value: core_1.Duration.days(14).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                // deny list
                // ----------------------------------------------
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Deny List',
                            '',
                            `[button:primary:Deny List Object](${deep_link_1.s3ObjectUrl(props.denyList.bucket, props.denyList.objectKey)})`,
                            `[button:Prune Function](${deep_link_1.lambdaFunctionUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Prune Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Delete Queue](${deep_link_1.sqsQueueUrl(props.denyList.prune.queue)})`,
                            `[button:Delete Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.deleteHandler)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Deny List',
                        left: [
                            metric_utils_1.fillMetric(props.denyList.metricDenyListRules({ label: 'Rules' }), 'REPEAT'),
                            props.denyList.prune.queue.metricNumberOfMessagesDeleted({ label: 'Deleted Files' }),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Prune Function Health',
                        left: [
                            metric_utils_1.fillMetric(props.denyList.prune.pruneHandler.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(props.denyList.prune.pruneHandler.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                ],
                ...(props.packageStats ? renderPackageStatsWidgets(props.packageStats) : []),
            ],
        });
    }
    *catalogOverviewLanguageSections({ inventory, orchestration }) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    '# Documentation Generation',
                    '',
                    `[button:primary:Transliterator Logs](${deep_link_1.logGroupUrl(orchestration.transliterator.logGroup)})`,
                    `[button:Transliterator ECS Cluster](${deep_link_1.ecsClusterUrl(orchestration.ecsCluster)})`,
                ].join('\n'),
            }),
        ];
        const mFargateUsage = new aws_cloudwatch_1.Metric({
            dimensionsMap: {
                Class: 'None',
                Resource: 'OnDemand',
                Service: 'Fargate',
                Type: 'Resource',
            },
            metricName: 'ResourceCount',
            namespace: 'AWS/Usage',
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
        });
        yield [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Fargate Resources',
                left: [
                    mFargateUsage.with({ label: 'Fargate Usage (On-Demand)' }),
                    new aws_cloudwatch_1.MathExpression({
                        expression: 'SERVICE_QUOTA(mFargateUsage)',
                        label: 'Fargate Quota (On-Demand)',
                        usingMetrics: { mFargateUsage },
                    }),
                ],
                leftYAxis: { min: 0 },
                right: [
                    orchestration.metricEcsCpuUtilization({ label: 'CPU Utilization' }),
                    orchestration.metricEcsMemoryUtilization({ label: 'Memory Utilization' }),
                ],
                rightYAxis: { label: 'Percent', min: 0, max: 100, showUnits: false },
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'ECS Resources',
                left: [
                    metric_utils_1.fillMetric(orchestration.metricEcsNetworkRxBytes({ label: 'Received Bytes' })),
                    metric_utils_1.fillMetric(orchestration.metricEcsNetworkTxBytes({ label: 'Transmitted Bytes' })),
                ],
                leftYAxis: { min: 0 },
                right: [
                    metric_utils_1.fillMetric(orchestration.metricEcsTaskCount({ label: 'Task Count' })),
                ],
                rightYAxis: { min: 0 },
            }),
        ];
        for (const language of language_1.DocumentationLanguage.ALL) {
            yield [
                new aws_cloudwatch_1.TextWidget({
                    height: 1,
                    width: 24,
                    markdown: `## Language: ${language.toString()}`,
                }),
            ];
            yield [
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    stacked: true,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 6,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    stacked: true,
                }),
            ];
        }
    }
}
exports.BackendDashboard = BackendDashboard;
function* renderPackageSourcesWidgets(packageSources) {
    var _a;
    for (const packageSource of packageSources) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    `# ${packageSource.name}`,
                    '',
                    ...((_a = packageSource.links) !== null && _a !== void 0 ? _a : [])
                        .map(({ name, primary, url }) => `[${primary ? 'button:primary' : 'button'}:${name}](${url})`),
                ].join('\n'),
            }),
        ];
        yield* packageSource.dashboardWidgets;
    }
    return;
}
function renderPackageStatsWidgets(packageStats) {
    return [
        [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    '# Package Stats',
                    '',
                    `[button:primary:Package Stats Object](${deep_link_1.s3ObjectUrl(packageStats.bucket, packageStats.statsKey)})`,
                    `[button:Package Stats Function](${deep_link_1.lambdaFunctionUrl(packageStats.handler)})`,
                    `[button:Package Stats Logs](${deep_link_1.lambdaSearchLogGroupUrl(packageStats.handler)})`,
                ].join('\n'),
            }),
        ],
        [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Number of Package Stats Recorded',
                left: [
                    metric_utils_1.fillMetric(packageStats.metricPackagesCount({ label: 'Packages with stats' }), 'REPEAT'),
                ],
                leftYAxis: { min: 0 },
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Invocation Duration',
                left: [
                    packageStats.handler.metricDuration({ label: 'Duration' }),
                ],
                leftYAxis: { min: 0 },
                rightAnnotations: [{
                        color: '#ffa500',
                        label: '15 minutes (Lambda timeout)',
                        value: core_1.Duration.minutes(15).toSeconds(),
                    }],
            }),
        ],
    ];
}
//# sourceMappingURL=data:application/json;base64,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