"""
Exception classes and tests for prompts, LLMs, and workflows.
"""

from .llms import ChatPrompt


def isAcceptableLLMResponse(response_given, acceptable_options):
    """
    Tests to confirm the response_given is in the list of acceptable_options. acceptable_options can also be a single string.

    Returns True if the response is 'acceptable', otherwise throws an LLMResponseException.
    """

    compare_to = None
    if isinstance(acceptable_options, str):
        compare_to = [acceptable_options]
    elif isinstance(acceptable_options, list):
        compare_to = acceptable_options

    if compare_to is None:
        raise Exception("testLLMResponse() only accepts a list or string object for acceptable_options.")

    if response_given not in acceptable_options:
        raise LLMResponseException(response_given, compare_to)

    return True


def isLLMCodeExecutable(llm_code):
    """
    Runs code and checks if any errors occur. Returns True if there are no errors.
    """
    try:
        exec(llm_code)
    except Exception as e:
        raise LLMCodeException(llm_code, e)

    return True


def isProperlyStructuredChat(messages, force_roles=False):
    """
    Checks if messages are an array of dicts with (role, content) keys.

    force_roles=True also confirms we only have roles of "system", "user", and "assistant" to abide by OpenAI's API.
    """
    for m in messages:
        keys = m.keys()
        if not (len(keys) == 2 and "role" in keys and "content" in keys):
            return False
        if force_roles:
            role = m["role"]
            if role not in ["system", "user", "assistant"]:
                return False
    return True


def reviewOutputWithLLM(text, requirements, llm):
    """
    Has an LLM review an output and determines whether the output is OK or not.
    """
    prompt = ChatPrompt(
        [{"role": "system",
          "content": "Follow the user's instructions exactly, and only respond with YES or NO (with additional info)."},
         {"role": "user",
          "content": "I'm working with a large language model and hope you can confirm if the following text abides by "
                     "a set of requirements I've provided. Here is the text:\n-----\n{output}\n-----\n\nBelow are the "
                     "requirements the text above is supposed to meet.\n\n-----\n{requirements}\n-----\n\nDoes the "
                     "text meet the requirements? Please only answer YES or NO. If NO, you can provide additional "
                     "information on what the text is missing."}
         ])

    result = llm.complete_chat(prompt.fill(text=text, requirements=requirements))
    if result == "YES":
        return True
    else:
        raise LLMReviewException(result)


class LLMReviewException(Exception):
    """
    Exception that gets thrown 
    """

    def __init__(self, message):
        super().__init__("LLM Review Exception: text does not meet requirements.\nInfo: " + message)
        self.message = message

    def __repr__(self):
        return "LLM Review Exception: text does not meet requirements.\nInfo: " + self.message


class ChatStructureException(Exception):
    """
    Exception that gets thrown when a chat structure isn't correct (i.e., role, content pairs are not pairs)
    """

    def __init__(self):
        super().__init__("Chat Structure Exception: chat messages are not following the proper chat structure.")

    def __repr__(self):
        return f"Chat Structure Exception: chat messages are not following the proper chat structure."


class LLMCodeException(Exception):
    """
    Exception to track exceptions from code generated by LLMs.
    """

    def __init__(self, code, exc):
        super().__init__("LLM Code Exception: code is raising an error.")
        self.code = code
        self.exception = exc
        self.exception_string = str(exc)

    def __repr__(self):
        return f"LLM Code Exception: code is raising an error."


class LLMResponseException(Exception):
    """
    Exception to track acceptable responses from an LLM.
    """

    def __init__(self, response_given, acceptable_options):
        super().__init__("LLM Response Exception: response given is not in the list of acceptable options.")
        self.response_given = response_given
        self.acceptable_options = acceptable_options

    def __repr__(self):
        return f"LLM Response Exception: response given is not in the list of acceptable options."
