"""
Stream method tests using actual receiver binary outputs for pyubx2.UBXReader

Created on 3 Oct 2020

*** NB: must be saved in UTF-8 format ***

@author: semuadmin
"""
# pylint: disable=line-too-long, invalid-name, missing-docstring, no-member

import sys
import os
import unittest
from io import StringIO

from pyubx2 import (
    UBXReader,
    VALCKSUM,
    UBX_PROTOCOL,
    NMEA_PROTOCOL,
    ERR_RAISE,
    ERR_LOG,
    ERR_IGNORE,
)
from pyubx2.exceptions import UBXParseError
import pyubx2.ubxtypes_core as ubt


class StreamTest(unittest.TestCase):
    def setUp(self):
        self.maxDiff = None
        dirname = os.path.dirname(__file__)
        # self.testdump = open(os.path.join(dirname, "testdump.log"), "wb")
        self.streamNAV = open(os.path.join(dirname, "pygpsdata-NAV.log"), "rb")
        self.streamNAVHPPOS = open(
            os.path.join(dirname, "pygpsdata-NAVHPPOS.log"), "rb"
        )
        self.streamHNR = open(os.path.join(dirname, "pygpsdata-HNR.log"), "rb")
        self.streamRXM = open(os.path.join(dirname, "pygpsdata-RXM.log"), "rb")
        self.streamESF = open(os.path.join(dirname, "pygpsdata-ESF.log"), "rb")
        self.streamINF = open(os.path.join(dirname, "pygpsdata-INF.log"), "rb")
        self.streamCFG = open(os.path.join(dirname, "pygpsdata-CFG.log"), "rb")
        self.streamMON = open(os.path.join(dirname, "pygpsdata-MON.log"), "rb")
        self.streamITER = open(os.path.join(dirname, "pygpsdata-ITER.log"), "rb")
        self.streamMIX = open(os.path.join(dirname, "pygpsdata-MIXED3.log"), "rb")
        self.streamDEBUG = open(
            os.path.join(dirname, "ucenter-ZEDF9P-configdebug.log"), "rb"
        )

        # self.streamMIXRTCM3 = open(
        #     os.path.join(dirname, "pygpsdata-MIXED-RTCM3.log"), "rb"
        # )
        self.streamMIXBADCK = open(
            os.path.join(dirname, "pygpsdata-MIXED3BADCK.log"), "rb"
        )
        self.streamBADHDR = open(os.path.join(dirname, "pygpsdata-BADHDR.log"), "rb")
        self.streamBADEOF1 = open(os.path.join(dirname, "pygpsdata-BADEOF1.log"), "rb")
        self.streamBADEOF2 = open(os.path.join(dirname, "pygpsdata-BADEOF2.log"), "rb")
        self.streamBADEOF3 = open(os.path.join(dirname, "pygpsdata-BADEOF3.log"), "rb")
        self.streamBADCK2 = open(os.path.join(dirname, "pygpsdata-BADCK2.log"), "rb")
        self.streamBADNMEAEOF = open(
            os.path.join(dirname, "pygpsdata-BADNMEAEOF.log"), "rb"
        )

    def tearDown(self):
        # self.testdump.close()
        self.streamNAV.close()
        self.streamNAVHPPOS.close()
        self.streamHNR.close()
        self.streamRXM.close()
        self.streamESF.close()
        self.streamINF.close()
        self.streamCFG.close()
        self.streamMON.close()
        self.streamITER.close()
        self.streamMIX.close()
        # self.streamMIXRTCM3.close()
        self.streamMIXBADCK.close()
        self.streamBADHDR.close()
        self.streamBADEOF1.close()
        self.streamBADEOF2.close()
        self.streamBADEOF3.close()
        self.streamBADCK2.close()
        self.streamBADNMEAEOF.close()
        self.streamDEBUG.close()

    def catchio(self):
        """
        Capture stdout as string.
        """

        self._saved_stdout = sys.stdout
        self._strout = StringIO()
        sys.stdout = self._strout

    def restoreio(self) -> str:
        """
        Return captured output and restore stdout.
        """

        sys.stdout = self._saved_stdout
        return self._strout.getvalue().strip()

    def testNAVLOG(
        self,
    ):  # test stream of UBX NAV messages
        EXPECTED_RESULTS = (
            "<UBX(NAV-PVT, iTOW=11:34:59, year=2021, month=12, day=4, hour=11, min=34, second=59, validDate=1, validTime=1, fullyResolved=1, validMag=0, tAcc=26, nano=-361668, fixType=3, gnssFixOk=1, difSoln=1, psmState=0, headVehValid=0, carrSoln=0, confirmedAvai=1, confirmedDate=1, confirmedTime=1, numSV=26, lon=-2.2402855, lat=53.4507228, height=91184, hMSL=42701, hAcc=1491, vAcc=2065, velN=-7, velE=4, velD=9, gSpeed=8, headMot=0.0, sAcc=259, headAcc=180.0, pDOP=1.01, invalidLlh=0, lastCorrectionAge=0, reserved0=793711598, headVeh=0.0, magDec=0.0, magAcc=0.0)>",
            "<UBX(NAV-ORB, iTOW=11:34:59, version=1, numSv=107, reserved0=0, gnssId_01=GPS, svId_01=1, health_01=1, visibility_01=1, ephUsability_01=0, ephSource_01=0, almUsability_01=17, almSource_01=1, anoAopUsability_01=0, type_01=0, gnssId_02=GPS, svId_02=2, health_02=1, visibility_02=3, ephUsability_02=10, ephSource_02=1, almUsability_02=17, almSource_02=1, anoAopUsability_02=7, type_02=2, gnssId_03=GPS, svId_03=3, health_03=1, visibility_03=1, ephUsability_03=0, ephSource_03=0, almUsability_03=17, almSource_03=1, anoAopUsability_03=0, type_03=0, gnssId_04=GPS, svId_04=4, health_04=1, visibility_04=1, ephUsability_04=10, ephSource_04=1, almUsability_04=17, almSource_04=1, anoAopUsability_04=7, type_04=2, gnssId_05=GPS, svId_05=5, health_05=1, visibility_05=3, ephUsability_05=10, ephSource_05=1, almUsability_05=17, almSource_05=1, anoAopUsability_05=7, type_05=2, gnssId_06=GPS, svId_06=6, health_06=1, visibility_06=1, ephUsability_06=10, ephSource_06=1, almUsability_06=17, almSource_06=1, anoAopUsability_06=7, type_06=2, gnssId_07=GPS, svId_07=7, health_07=1, visibility_07=3, ephUsability_07=10, ephSource_07=1, almUsability_07=17, almSource_07=1, anoAopUsability_07=7, type_07=2, gnssId_08=GPS, svId_08=8, health_08=1, visibility_08=1, ephUsability_08=0, ephSource_08=0, almUsability_08=17, almSource_08=1, anoAopUsability_08=0, type_08=0, gnssId_09=GPS, svId_09=9, health_09=1, visibility_09=3, ephUsability_09=10, ephSource_09=1, almUsability_09=17, almSource_09=1, anoAopUsability_09=7, type_09=2, gnssId_10=GPS, svId_10=10, health_10=1, visibility_10=1, ephUsability_10=0, ephSource_10=0, almUsability_10=17, almSource_10=1, anoAopUsability_10=0, type_10=0, gnssId_11=GPS, svId_11=11, health_11=2, visibility_11=3, ephUsability_11=10, ephSource_11=1, almUsability_11=17, almSource_11=1, anoAopUsability_11=0, type_11=0, gnssId_12=GPS, svId_12=12, health_12=1, visibility_12=1, ephUsability_12=0, ephSource_12=0, almUsability_12=17, almSource_12=1, anoAopUsability_12=0, type_12=0, gnssId_13=GPS, svId_13=13, health_13=1, visibility_13=3, ephUsability_13=10, ephSource_13=1, almUsability_13=17, almSource_13=1, anoAopUsability_13=7, type_13=2, gnssId_14=GPS, svId_14=14, health_14=1, visibility_14=3, ephUsability_14=0, ephSource_14=0, almUsability_14=17, almSource_14=1, anoAopUsability_14=0, type_14=0, gnssId_15=GPS, svId_15=15, health_15=1, visibility_15=3, ephUsability_15=10, ephSource_15=1, almUsability_15=17, almSource_15=1, anoAopUsability_15=7, type_15=2, gnssId_16=GPS, svId_16=16, health_16=1, visibility_16=2, ephUsability_16=0, ephSource_16=0, almUsability_16=17, almSource_16=1, anoAopUsability_16=0, type_16=0, gnssId_17=GPS, svId_17=17, health_17=1, visibility_17=1, ephUsability_17=0, ephSource_17=0, almUsability_17=17, almSource_17=1, anoAopUsability_17=0, type_17=0, gnssId_18=GPS, svId_18=18, health_18=1, visibility_18=3, ephUsability_18=10, ephSource_18=1, almUsability_18=17, almSource_18=1, anoAopUsability_18=7, type_18=2, gnssId_19=GPS, svId_19=19, health_19=1, visibility_19=1, ephUsability_19=0, ephSource_19=0, almUsability_19=17, almSource_19=1, anoAopUsability_19=0, type_19=0, gnssId_20=GPS, svId_20=20, health_20=1, visibility_20=3, ephUsability_20=10, ephSource_20=1, almUsability_20=17, almSource_20=1, anoAopUsability_20=7, type_20=2, gnssId_21=GPS, svId_21=21, health_21=1, visibility_21=1, ephUsability_21=0, ephSource_21=0, almUsability_21=17, almSource_21=1, anoAopUsability_21=0, type_21=0, gnssId_22=GPS, svId_22=22, health_22=2, visibility_22=1, ephUsability_22=0, ephSource_22=0, almUsability_22=17, almSource_22=1, anoAopUsability_22=0, type_22=0, gnssId_23=GPS, svId_23=23, health_23=1, visibility_23=1, ephUsability_23=0, ephSource_23=0, almUsability_23=17, almSource_23=1, anoAopUsability_23=0, type_23=0, gnssId_24=GPS, svId_24=24, health_24=1, visibility_24=1, ephUsability_24=0, ephSource_24=0, almUsability_24=17, almSource_24=1, anoAopUsability_24=0, type_24=0, gnssId_25=GPS, svId_25=25, health_25=1, visibility_25=1, ephUsability_25=0, ephSource_25=0, almUsability_25=17, almSource_25=1, anoAopUsability_25=0, type_25=0, gnssId_26=GPS, svId_26=26, health_26=1, visibility_26=1, ephUsability_26=0, ephSource_26=0, almUsability_26=17, almSource_26=1, anoAopUsability_26=0, type_26=0, gnssId_27=GPS, svId_27=27, health_27=1, visibility_27=2, ephUsability_27=0, ephSource_27=0, almUsability_27=17, almSource_27=1, anoAopUsability_27=0, type_27=0, gnssId_28=GPS, svId_28=29, health_28=1, visibility_28=1, ephUsability_28=10, ephSource_28=1, almUsability_28=17, almSource_28=1, anoAopUsability_28=7, type_28=2, gnssId_29=GPS, svId_29=30, health_29=1, visibility_29=3, ephUsability_29=10, ephSource_29=1, almUsability_29=17, almSource_29=1, anoAopUsability_29=7, type_29=2, gnssId_30=GPS, svId_30=31, health_30=1, visibility_30=1, ephUsability_30=0, ephSource_30=0, almUsability_30=17, almSource_30=1, anoAopUsability_30=0, type_30=0, gnssId_31=GPS, svId_31=32, health_31=1, visibility_31=1, ephUsability_31=0, ephSource_31=0, almUsability_31=17, almSource_31=1, anoAopUsability_31=0, type_31=0, gnssId_32=SBAS, svId_32=123, health_32=1, visibility_32=3, ephUsability_32=0, ephSource_32=0, almUsability_32=31, almSource_32=1, anoAopUsability_32=0, type_32=0, gnssId_33=SBAS, svId_33=136, health_33=1, visibility_33=3, ephUsability_33=1, ephSource_33=1, almUsability_33=31, almSource_33=1, anoAopUsability_33=0, type_33=0, gnssId_34=Galileo, svId_34=1, health_34=1, visibility_34=1, ephUsability_34=0, ephSource_34=0, almUsability_34=14, almSource_34=1, anoAopUsability_34=0, type_34=0, gnssId_35=Galileo, svId_35=2, health_35=1, visibility_35=3, ephUsability_35=0, ephSource_35=0, almUsability_35=14, almSource_35=1, anoAopUsability_35=0, type_35=0, gnssId_36=Galileo, svId_36=3, health_36=1, visibility_36=3, ephUsability_36=13, ephSource_36=1, almUsability_36=14, almSource_36=1, anoAopUsability_36=7, type_36=2, gnssId_37=Galileo, svId_37=4, health_37=1, visibility_37=1, ephUsability_37=0, ephSource_37=0, almUsability_37=14, almSource_37=1, anoAopUsability_37=0, type_37=0, gnssId_38=Galileo, svId_38=5, health_38=1, visibility_38=1, ephUsability_38=9, ephSource_38=1, almUsability_38=14, almSource_38=1, anoAopUsability_38=7, type_38=2, gnssId_39=Galileo, svId_39=7, health_39=1, visibility_39=3, ephUsability_39=15, ephSource_39=1, almUsability_39=14, almSource_39=1, anoAopUsability_39=7, type_39=2, gnssId_40=Galileo, svId_40=8, health_40=1, visibility_40=3, ephUsability_40=13, ephSource_40=1, almUsability_40=14, almSource_40=1, anoAopUsability_40=7, type_40=2, gnssId_41=Galileo, svId_41=9, health_41=1, visibility_41=1, ephUsability_41=0, ephSource_41=0, almUsability_41=14, almSource_41=1, anoAopUsability_41=0, type_41=0, gnssId_42=Galileo, svId_42=11, health_42=1, visibility_42=1, ephUsability_42=0, ephSource_42=0, almUsability_42=14, almSource_42=1, anoAopUsability_42=0, type_42=0, gnssId_43=Galileo, svId_43=12, health_43=1, visibility_43=1, ephUsability_43=0, ephSource_43=0, almUsability_43=14, almSource_43=1, anoAopUsability_43=0, type_43=0, gnssId_44=Galileo, svId_44=13, health_44=1, visibility_44=3, ephUsability_44=13, ephSource_44=1, almUsability_44=14, almSource_44=1, anoAopUsability_44=7, type_44=2, gnssId_45=Galileo, svId_45=15, health_45=1, visibility_45=1, ephUsability_45=9, ephSource_45=1, almUsability_45=14, almSource_45=1, anoAopUsability_45=4, type_45=2, gnssId_46=Galileo, svId_46=19, health_46=1, visibility_46=1, ephUsability_46=0, ephSource_46=0, almUsability_46=14, almSource_46=1, anoAopUsability_46=0, type_46=0, gnssId_47=Galileo, svId_47=21, health_47=1, visibility_47=1, ephUsability_47=0, ephSource_47=0, almUsability_47=14, almSource_47=1, anoAopUsability_47=0, type_47=0, gnssId_48=Galileo, svId_48=24, health_48=1, visibility_48=3, ephUsability_48=11, ephSource_48=1, almUsability_48=14, almSource_48=1, anoAopUsability_48=7, type_48=2, gnssId_49=Galileo, svId_49=25, health_49=1, visibility_49=3, ephUsability_49=15, ephSource_49=1, almUsability_49=14, almSource_49=1, anoAopUsability_49=7, type_49=2, gnssId_50=Galileo, svId_50=26, health_50=1, visibility_50=3, ephUsability_50=15, ephSource_50=1, almUsability_50=14, almSource_50=1, anoAopUsability_50=7, type_50=2, gnssId_51=Galileo, svId_51=27, health_51=1, visibility_51=1, ephUsability_51=0, ephSource_51=0, almUsability_51=14, almSource_51=1, anoAopUsability_51=0, type_51=0, gnssId_52=Galileo, svId_52=30, health_52=1, visibility_52=2, ephUsability_52=0, ephSource_52=0, almUsability_52=14, almSource_52=1, anoAopUsability_52=0, type_52=0, gnssId_53=Galileo, svId_53=31, health_53=1, visibility_53=1, ephUsability_53=0, ephSource_53=0, almUsability_53=14, almSource_53=1, anoAopUsability_53=0, type_53=0, gnssId_54=Galileo, svId_54=33, health_54=1, visibility_54=3, ephUsability_54=12, ephSource_54=1, almUsability_54=14, almSource_54=1, anoAopUsability_54=7, type_54=2, gnssId_55=Galileo, svId_55=36, health_55=1, visibility_55=1, ephUsability_55=0, ephSource_55=0, almUsability_55=14, almSource_55=1, anoAopUsability_55=0, type_55=0, gnssId_56=BeiDou, svId_56=1, health_56=1, visibility_56=1, ephUsability_56=0, ephSource_56=0, almUsability_56=8, almSource_56=1, anoAopUsability_56=0, type_56=0, gnssId_57=BeiDou, svId_57=2, health_57=1, visibility_57=1, ephUsability_57=0, ephSource_57=0, almUsability_57=8, almSource_57=1, anoAopUsability_57=0, type_57=0, gnssId_58=BeiDou, svId_58=3, health_58=1, visibility_58=1, ephUsability_58=0, ephSource_58=0, almUsability_58=8, almSource_58=1, anoAopUsability_58=0, type_58=0, gnssId_59=BeiDou, svId_59=4, health_59=1, visibility_59=1, ephUsability_59=0, ephSource_59=0, almUsability_59=8, almSource_59=1, anoAopUsability_59=0, type_59=0, gnssId_60=BeiDou, svId_60=5, health_60=1, visibility_60=3, ephUsability_60=0, ephSource_60=0, almUsability_60=8, almSource_60=1, anoAopUsability_60=0, type_60=0, gnssId_61=BeiDou, svId_61=6, health_61=1, visibility_61=1, ephUsability_61=0, ephSource_61=0, almUsability_61=8, almSource_61=1, anoAopUsability_61=0, type_61=0, gnssId_62=BeiDou, svId_62=7, health_62=1, visibility_62=3, ephUsability_62=10, ephSource_62=1, almUsability_62=8, almSource_62=1, anoAopUsability_62=7, type_62=2, gnssId_63=BeiDou, svId_63=8, health_63=1, visibility_63=1, ephUsability_63=0, ephSource_63=0, almUsability_63=8, almSource_63=1, anoAopUsability_63=0, type_63=0, gnssId_64=BeiDou, svId_64=9, health_64=1, visibility_64=1, ephUsability_64=0, ephSource_64=0, almUsability_64=8, almSource_64=1, anoAopUsability_64=0, type_64=0, gnssId_65=BeiDou, svId_65=10, health_65=1, visibility_65=3, ephUsability_65=10, ephSource_65=1, almUsability_65=8, almSource_65=1, anoAopUsability_65=7, type_65=2, gnssId_66=BeiDou, svId_66=11, health_66=1, visibility_66=3, ephUsability_66=10, ephSource_66=1, almUsability_66=8, almSource_66=1, anoAopUsability_66=7, type_66=2, gnssId_67=BeiDou, svId_67=12, health_67=1, visibility_67=1, ephUsability_67=6, ephSource_67=1, almUsability_67=8, almSource_67=1, anoAopUsability_67=7, type_67=2, gnssId_68=BeiDou, svId_68=13, health_68=1, visibility_68=1, ephUsability_68=0, ephSource_68=0, almUsability_68=8, almSource_68=1, anoAopUsability_68=0, type_68=0, gnssId_69=BeiDou, svId_69=14, health_69=1, visibility_69=3, ephUsability_69=10, ephSource_69=1, almUsability_69=8, almSource_69=1, anoAopUsability_69=7, type_69=2, gnssId_70=BeiDou, svId_70=16, health_70=1, visibility_70=1, ephUsability_70=0, ephSource_70=0, almUsability_70=8, almSource_70=1, anoAopUsability_70=0, type_70=0, gnssId_71=BeiDou, svId_71=19, health_71=1, visibility_71=1, ephUsability_71=2, ephSource_71=1, almUsability_71=8, almSource_71=1, anoAopUsability_71=7, type_71=2, gnssId_72=BeiDou, svId_72=20, health_72=1, visibility_72=1, ephUsability_72=0, ephSource_72=0, almUsability_72=8, almSource_72=1, anoAopUsability_72=0, type_72=0, gnssId_73=BeiDou, svId_73=21, health_73=1, visibility_73=3, ephUsability_73=10, ephSource_73=1, almUsability_73=8, almSource_73=1, anoAopUsability_73=7, type_73=2, gnssId_74=BeiDou, svId_74=22, health_74=1, visibility_74=3, ephUsability_74=10, ephSource_74=1, almUsability_74=8, almSource_74=1, anoAopUsability_74=7, type_74=2, gnssId_75=BeiDou, svId_75=23, health_75=1, visibility_75=1, ephUsability_75=0, ephSource_75=0, almUsability_75=8, almSource_75=1, anoAopUsability_75=0, type_75=0, gnssId_76=BeiDou, svId_76=24, health_76=1, visibility_76=1, ephUsability_76=0, ephSource_76=0, almUsability_76=8, almSource_76=1, anoAopUsability_76=0, type_76=0, gnssId_77=BeiDou, svId_77=25, health_77=1, visibility_77=1, ephUsability_77=0, ephSource_77=0, almUsability_77=8, almSource_77=1, anoAopUsability_77=0, type_77=0, gnssId_78=BeiDou, svId_78=26, health_78=1, visibility_78=3, ephUsability_78=10, ephSource_78=1, almUsability_78=8, almSource_78=1, anoAopUsability_78=7, type_78=2, gnssId_79=BeiDou, svId_79=27, health_79=1, visibility_79=1, ephUsability_79=0, ephSource_79=0, almUsability_79=8, almSource_79=1, anoAopUsability_79=0, type_79=0, gnssId_80=BeiDou, svId_80=28, health_80=1, visibility_80=3, ephUsability_80=10, ephSource_80=1, almUsability_80=8, almSource_80=1, anoAopUsability_80=7, type_80=2, gnssId_81=BeiDou, svId_81=29, health_81=1, visibility_81=1, ephUsability_81=0, ephSource_81=0, almUsability_81=8, almSource_81=1, anoAopUsability_81=0, type_81=0, gnssId_82=BeiDou, svId_82=30, health_82=1, visibility_82=1, ephUsability_82=0, ephSource_82=0, almUsability_82=8, almSource_82=1, anoAopUsability_82=0, type_82=0, gnssId_83=BeiDou, svId_83=34, health_83=1, visibility_83=1, ephUsability_83=6, ephSource_83=1, almUsability_83=0, almSource_83=0, anoAopUsability_83=4, type_83=2, gnssId_84=GLONASS, svId_84=1, health_84=1, visibility_84=3, ephUsability_84=6, ephSource_84=1, almUsability_84=21, almSource_84=1, anoAopUsability_84=7, type_84=2, gnssId_85=GLONASS, svId_85=2, health_85=1, visibility_85=3, ephUsability_85=6, ephSource_85=1, almUsability_85=21, almSource_85=1, anoAopUsability_85=7, type_85=2, gnssId_86=GLONASS, svId_86=3, health_86=1, visibility_86=3, ephUsability_86=6, ephSource_86=1, almUsability_86=21, almSource_86=1, anoAopUsability_86=7, type_86=2, gnssId_87=GLONASS, svId_87=4, health_87=1, visibility_87=1, ephUsability_87=0, ephSource_87=0, almUsability_87=21, almSource_87=1, anoAopUsability_87=0, type_87=0, gnssId_88=GLONASS, svId_88=5, health_88=1, visibility_88=1, ephUsability_88=0, ephSource_88=0, almUsability_88=21, almSource_88=1, anoAopUsability_88=0, type_88=0, gnssId_89=GLONASS, svId_89=6, health_89=1, visibility_89=1, ephUsability_89=0, ephSource_89=0, almUsability_89=21, almSource_89=1, anoAopUsability_89=0, type_89=0, gnssId_90=GLONASS, svId_90=7, health_90=1, visibility_90=1, ephUsability_90=0, ephSource_90=1, almUsability_90=21, almSource_90=1, anoAopUsability_90=7, type_90=2, gnssId_91=GLONASS, svId_91=8, health_91=1, visibility_91=1, ephUsability_91=4, ephSource_91=1, almUsability_91=21, almSource_91=1, anoAopUsability_91=7, type_91=2, gnssId_92=GLONASS, svId_92=9, health_92=1, visibility_92=1, ephUsability_92=0, ephSource_92=1, almUsability_92=21, almSource_92=1, anoAopUsability_92=7, type_92=2, gnssId_93=GLONASS, svId_93=10, health_93=1, visibility_93=3, ephUsability_93=2, ephSource_93=1, almUsability_93=21, almSource_93=1, anoAopUsability_93=4, type_93=2, gnssId_94=GLONASS, svId_94=11, health_94=2, visibility_94=3, ephUsability_94=6, ephSource_94=1, almUsability_94=21, almSource_94=1, anoAopUsability_94=0, type_94=0, gnssId_95=GLONASS, svId_95=12, health_95=1, visibility_95=3, ephUsability_95=6, ephSource_95=1, almUsability_95=21, almSource_95=1, anoAopUsability_95=7, type_95=2, gnssId_96=GLONASS, svId_96=13, health_96=1, visibility_96=1, ephUsability_96=0, ephSource_96=0, almUsability_96=21, almSource_96=1, anoAopUsability_96=0, type_96=0, gnssId_97=GLONASS, svId_97=14, health_97=1, visibility_97=1, ephUsability_97=0, ephSource_97=0, almUsability_97=21, almSource_97=1, anoAopUsability_97=0, type_97=0, gnssId_98=GLONASS, svId_98=15, health_98=1, visibility_98=1, ephUsability_98=0, ephSource_98=0, almUsability_98=21, almSource_98=1, anoAopUsability_98=0, type_98=0, gnssId_99=GLONASS, svId_99=16, health_99=1, visibility_99=1, ephUsability_99=0, ephSource_99=0, almUsability_99=21, almSource_99=1, anoAopUsability_99=0, type_99=0, gnssId_100=GLONASS, svId_100=17, health_100=1, visibility_100=3, ephUsability_100=6, ephSource_100=1, almUsability_100=21, almSource_100=1, anoAopUsability_100=7, type_100=2, gnssId_101=GLONASS, svId_101=18, health_101=1, visibility_101=3, ephUsability_101=6, ephSource_101=1, almUsability_101=21, almSource_101=1, anoAopUsability_101=7, type_101=2, gnssId_102=GLONASS, svId_102=19, health_102=1, visibility_102=3, ephUsability_102=0, ephSource_102=0, almUsability_102=21, almSource_102=1, anoAopUsability_102=0, type_102=0, gnssId_103=GLONASS, svId_103=20, health_103=1, visibility_103=1, ephUsability_103=0, ephSource_103=0, almUsability_103=21, almSource_103=1, anoAopUsability_103=0, type_103=0, gnssId_104=GLONASS, svId_104=21, health_104=1, visibility_104=1, ephUsability_104=0, ephSource_104=0, almUsability_104=21, almSource_104=1, anoAopUsability_104=0, type_104=0, gnssId_105=GLONASS, svId_105=22, health_105=1, visibility_105=1, ephUsability_105=0, ephSource_105=0, almUsability_105=21, almSource_105=1, anoAopUsability_105=0, type_105=0, gnssId_106=GLONASS, svId_106=23, health_106=1, visibility_106=1, ephUsability_106=0, ephSource_106=1, almUsability_106=21, almSource_106=1, anoAopUsability_106=7, type_106=2, gnssId_107=GLONASS, svId_107=24, health_107=1, visibility_107=1, ephUsability_107=2, ephSource_107=1, almUsability_107=21, almSource_107=1, anoAopUsability_107=7, type_107=2)>",
            "<UBX(NAV-SAT, iTOW=11:34:59, version=1, numSvs=43, reserved0=0, gnssId_01=GPS, svId_01=2, cno_01=31, elev_01=18, azim_01=221, prRes_01=1.7, qualityInd_01=7, svUsed_01=1, health_01=1, diffCorr_01=1, smoothed_01=0, orbitSource_01=1, ephAvail_01=1, almAvail_01=1, anoAvail_01=0, aopAvail_01=1, sbasCorrUsed_01=1, rtcmCorrUsed_01=0, slasCorrUsed_01=0, spartnCorrUsed_01=0, prCorrUsed_01=1, crCorrUsed_01=0, doCorrUsed_01=1, gnssId_02=GPS, svId_02=5, cno_02=17, elev_02=61, azim_02=283, prRes_02=0.9, qualityInd_02=4, svUsed_02=1, health_02=1, diffCorr_02=1, smoothed_02=0, orbitSource_02=1, ephAvail_02=1, almAvail_02=1, anoAvail_02=0, aopAvail_02=1, sbasCorrUsed_02=1, rtcmCorrUsed_02=0, slasCorrUsed_02=0, spartnCorrUsed_02=0, prCorrUsed_02=1, crCorrUsed_02=0, doCorrUsed_02=1, gnssId_03=GPS, svId_03=7, cno_03=40, elev_03=51, azim_03=67, prRes_03=0.3, qualityInd_03=7, svUsed_03=1, health_03=1, diffCorr_03=1, smoothed_03=0, orbitSource_03=1, ephAvail_03=1, almAvail_03=1, anoAvail_03=0, aopAvail_03=1, sbasCorrUsed_03=1, rtcmCorrUsed_03=0, slasCorrUsed_03=0, spartnCorrUsed_03=0, prCorrUsed_03=1, crCorrUsed_03=0, doCorrUsed_03=1, gnssId_04=GPS, svId_04=9, cno_04=0, elev_04=15, azim_04=89, prRes_04=0.0, qualityInd_04=0, svUsed_04=0, health_04=1, diffCorr_04=0, smoothed_04=0, orbitSource_04=1, ephAvail_04=1, almAvail_04=1, anoAvail_04=0, aopAvail_04=1, sbasCorrUsed_04=0, rtcmCorrUsed_04=0, slasCorrUsed_04=0, spartnCorrUsed_04=0, prCorrUsed_04=0, crCorrUsed_04=0, doCorrUsed_04=0, gnssId_05=GPS, svId_05=11, cno_05=0, elev_05=17, azim_05=208, prRes_05=0.0, qualityInd_05=0, svUsed_05=0, health_05=2, diffCorr_05=0, smoothed_05=0, orbitSource_05=2, ephAvail_05=1, almAvail_05=1, anoAvail_05=0, aopAvail_05=0, sbasCorrUsed_05=0, rtcmCorrUsed_05=0, slasCorrUsed_05=0, spartnCorrUsed_05=0, prCorrUsed_05=0, crCorrUsed_05=0, doCorrUsed_05=0, gnssId_06=GPS, svId_06=13, cno_06=32, elev_06=41, azim_06=265, prRes_06=-8.1, qualityInd_06=7, svUsed_06=1, health_06=1, diffCorr_06=1, smoothed_06=0, orbitSource_06=1, ephAvail_06=1, almAvail_06=1, anoAvail_06=0, aopAvail_06=1, sbasCorrUsed_06=1, rtcmCorrUsed_06=0, slasCorrUsed_06=0, spartnCorrUsed_06=0, prCorrUsed_06=1, crCorrUsed_06=0, doCorrUsed_06=1, gnssId_07=GPS, svId_07=14, cno_07=0, elev_07=14, azim_07=146, prRes_07=0.0, qualityInd_07=0, svUsed_07=0, health_07=1, diffCorr_07=0, smoothed_07=0, orbitSource_07=2, ephAvail_07=0, almAvail_07=1, anoAvail_07=0, aopAvail_07=0, sbasCorrUsed_07=0, rtcmCorrUsed_07=0, slasCorrUsed_07=0, spartnCorrUsed_07=0, prCorrUsed_07=0, crCorrUsed_07=0, doCorrUsed_07=0, gnssId_08=GPS, svId_08=15, cno_08=41, elev_08=12, azim_08=275, prRes_08=0.1, qualityInd_08=7, svUsed_08=1, health_08=1, diffCorr_08=1, smoothed_08=0, orbitSource_08=1, ephAvail_08=1, almAvail_08=1, anoAvail_08=0, aopAvail_08=1, sbasCorrUsed_08=1, rtcmCorrUsed_08=0, slasCorrUsed_08=0, spartnCorrUsed_08=0, prCorrUsed_08=1, crCorrUsed_08=0, doCorrUsed_08=1, gnssId_09=GPS, svId_09=16, cno_09=0, elev_09=1, azim_09=12, prRes_09=0.0, qualityInd_09=0, svUsed_09=0, health_09=1, diffCorr_09=0, smoothed_09=0, orbitSource_09=2, ephAvail_09=0, almAvail_09=1, anoAvail_09=0, aopAvail_09=0, sbasCorrUsed_09=0, rtcmCorrUsed_09=0, slasCorrUsed_09=0, spartnCorrUsed_09=0, prCorrUsed_09=0, crCorrUsed_09=0, doCorrUsed_09=0, gnssId_10=GPS, svId_10=18, cno_10=30, elev_10=12, azim_10=331, prRes_10=-0.5, qualityInd_10=6, svUsed_10=1, health_10=1, diffCorr_10=1, smoothed_10=0, orbitSource_10=1, ephAvail_10=1, almAvail_10=1, anoAvail_10=0, aopAvail_10=1, sbasCorrUsed_10=1, rtcmCorrUsed_10=0, slasCorrUsed_10=0, spartnCorrUsed_10=0, prCorrUsed_10=1, crCorrUsed_10=0, doCorrUsed_10=1, gnssId_11=GPS, svId_11=20, cno_11=19, elev_11=65, azim_11=205, prRes_11=-2.8, qualityInd_11=4, svUsed_11=1, health_11=1, diffCorr_11=1, smoothed_11=0, orbitSource_11=1, ephAvail_11=1, almAvail_11=1, anoAvail_11=0, aopAvail_11=1, sbasCorrUsed_11=1, rtcmCorrUsed_11=0, slasCorrUsed_11=0, spartnCorrUsed_11=0, prCorrUsed_11=1, crCorrUsed_11=0, doCorrUsed_11=1, gnssId_12=GPS, svId_12=27, cno_12=0, elev_12=2, azim_12=32, prRes_12=0.0, qualityInd_12=0, svUsed_12=0, health_12=1, diffCorr_12=0, smoothed_12=0, orbitSource_12=2, ephAvail_12=0, almAvail_12=1, anoAvail_12=0, aopAvail_12=0, sbasCorrUsed_12=0, rtcmCorrUsed_12=0, slasCorrUsed_12=0, spartnCorrUsed_12=0, prCorrUsed_12=0, crCorrUsed_12=0, doCorrUsed_12=0, gnssId_13=GPS, svId_13=30, cno_13=41, elev_13=71, azim_13=128, prRes_13=-0.2, qualityInd_13=7, svUsed_13=1, health_13=1, diffCorr_13=1, smoothed_13=0, orbitSource_13=1, ephAvail_13=1, almAvail_13=1, anoAvail_13=0, aopAvail_13=1, sbasCorrUsed_13=1, rtcmCorrUsed_13=0, slasCorrUsed_13=0, spartnCorrUsed_13=0, prCorrUsed_13=1, crCorrUsed_13=0, doCorrUsed_13=1, gnssId_14=SBAS, svId_14=123, cno_14=0, elev_14=22, azim_14=140, prRes_14=0.0, qualityInd_14=1, svUsed_14=0, health_14=1, diffCorr_14=0, smoothed_14=0, orbitSource_14=2, ephAvail_14=0, almAvail_14=1, anoAvail_14=0, aopAvail_14=0, sbasCorrUsed_14=0, rtcmCorrUsed_14=0, slasCorrUsed_14=0, spartnCorrUsed_14=0, prCorrUsed_14=0, crCorrUsed_14=0, doCorrUsed_14=0, gnssId_15=SBAS, svId_15=136, cno_15=31, elev_15=29, azim_15=171, prRes_15=0.0, qualityInd_15=7, svUsed_15=0, health_15=1, diffCorr_15=0, smoothed_15=0, orbitSource_15=2, ephAvail_15=1, almAvail_15=1, anoAvail_15=0, aopAvail_15=0, sbasCorrUsed_15=0, rtcmCorrUsed_15=0, slasCorrUsed_15=0, spartnCorrUsed_15=0, prCorrUsed_15=0, crCorrUsed_15=0, doCorrUsed_15=0, gnssId_16=Galileo, svId_16=2, cno_16=0, elev_16=33, azim_16=135, prRes_16=0.0, qualityInd_16=0, svUsed_16=0, health_16=1, diffCorr_16=0, smoothed_16=0, orbitSource_16=2, ephAvail_16=0, almAvail_16=1, anoAvail_16=0, aopAvail_16=0, sbasCorrUsed_16=0, rtcmCorrUsed_16=0, slasCorrUsed_16=0, spartnCorrUsed_16=0, prCorrUsed_16=0, crCorrUsed_16=0, doCorrUsed_16=0, gnssId_17=Galileo, svId_17=3, cno_17=31, elev_17=39, azim_17=95, prRes_17=4.9, qualityInd_17=7, svUsed_17=1, health_17=1, diffCorr_17=0, smoothed_17=0, orbitSource_17=1, ephAvail_17=1, almAvail_17=1, anoAvail_17=0, aopAvail_17=1, sbasCorrUsed_17=0, rtcmCorrUsed_17=0, slasCorrUsed_17=0, spartnCorrUsed_17=0, prCorrUsed_17=0, crCorrUsed_17=0, doCorrUsed_17=0, gnssId_18=Galileo, svId_18=7, cno_18=33, elev_18=30, azim_18=277, prRes_18=0.4, qualityInd_18=7, svUsed_18=1, health_18=1, diffCorr_18=0, smoothed_18=0, orbitSource_18=1, ephAvail_18=1, almAvail_18=1, anoAvail_18=0, aopAvail_18=1, sbasCorrUsed_18=0, rtcmCorrUsed_18=0, slasCorrUsed_18=0, spartnCorrUsed_18=0, prCorrUsed_18=0, crCorrUsed_18=0, doCorrUsed_18=0, gnssId_19=Galileo, svId_19=8, cno_19=37, elev_19=84, azim_19=294, prRes_19=-0.7, qualityInd_19=7, svUsed_19=1, health_19=1, diffCorr_19=0, smoothed_19=0, orbitSource_19=1, ephAvail_19=1, almAvail_19=1, anoAvail_19=0, aopAvail_19=1, sbasCorrUsed_19=0, rtcmCorrUsed_19=0, slasCorrUsed_19=0, spartnCorrUsed_19=0, prCorrUsed_19=0, crCorrUsed_19=0, doCorrUsed_19=0, gnssId_20=Galileo, svId_20=13, cno_20=34, elev_20=23, azim_20=237, prRes_20=1.0, qualityInd_20=7, svUsed_20=1, health_20=1, diffCorr_20=0, smoothed_20=0, orbitSource_20=1, ephAvail_20=1, almAvail_20=1, anoAvail_20=0, aopAvail_20=1, sbasCorrUsed_20=0, rtcmCorrUsed_20=0, slasCorrUsed_20=0, spartnCorrUsed_20=0, prCorrUsed_20=0, crCorrUsed_20=0, doCorrUsed_20=0, gnssId_21=Galileo, svId_21=24, cno_21=0, elev_21=8, azim_21=23, prRes_21=0.0, qualityInd_21=0, svUsed_21=0, health_21=1, diffCorr_21=0, smoothed_21=0, orbitSource_21=1, ephAvail_21=1, almAvail_21=1, anoAvail_21=0, aopAvail_21=1, sbasCorrUsed_21=0, rtcmCorrUsed_21=0, slasCorrUsed_21=0, spartnCorrUsed_21=0, prCorrUsed_21=0, crCorrUsed_21=0, doCorrUsed_21=0, gnssId_22=Galileo, svId_22=25, cno_22=29, elev_22=38, azim_22=72, prRes_22=5.1, qualityInd_22=7, svUsed_22=1, health_22=1, diffCorr_22=0, smoothed_22=0, orbitSource_22=1, ephAvail_22=1, almAvail_22=1, anoAvail_22=0, aopAvail_22=1, sbasCorrUsed_22=0, rtcmCorrUsed_22=0, slasCorrUsed_22=0, spartnCorrUsed_22=0, prCorrUsed_22=0, crCorrUsed_22=0, doCorrUsed_22=0, gnssId_23=Galileo, svId_23=26, cno_23=29, elev_23=29, azim_23=297, prRes_23=3.4, qualityInd_23=7, svUsed_23=1, health_23=1, diffCorr_23=0, smoothed_23=0, orbitSource_23=1, ephAvail_23=1, almAvail_23=1, anoAvail_23=0, aopAvail_23=1, sbasCorrUsed_23=0, rtcmCorrUsed_23=0, slasCorrUsed_23=0, spartnCorrUsed_23=0, prCorrUsed_23=0, crCorrUsed_23=0, doCorrUsed_23=0, gnssId_24=Galileo, svId_24=30, cno_24=0, elev_24=2, azim_24=174, prRes_24=0.0, qualityInd_24=0, svUsed_24=0, health_24=1, diffCorr_24=0, smoothed_24=0, orbitSource_24=2, ephAvail_24=0, almAvail_24=1, anoAvail_24=0, aopAvail_24=0, sbasCorrUsed_24=0, rtcmCorrUsed_24=0, slasCorrUsed_24=0, spartnCorrUsed_24=0, prCorrUsed_24=0, crCorrUsed_24=0, doCorrUsed_24=0, gnssId_25=Galileo, svId_25=33, cno_25=26, elev_25=9, azim_25=342, prRes_25=-4.0, qualityInd_25=7, svUsed_25=1, health_25=1, diffCorr_25=0, smoothed_25=0, orbitSource_25=1, ephAvail_25=1, almAvail_25=1, anoAvail_25=0, aopAvail_25=1, sbasCorrUsed_25=0, rtcmCorrUsed_25=0, slasCorrUsed_25=0, spartnCorrUsed_25=0, prCorrUsed_25=0, crCorrUsed_25=0, doCorrUsed_25=0, gnssId_26=BeiDou, svId_26=5, cno_26=0, elev_26=7, azim_26=115, prRes_26=0.0, qualityInd_26=1, svUsed_26=0, health_26=1, diffCorr_26=0, smoothed_26=0, orbitSource_26=2, ephAvail_26=0, almAvail_26=1, anoAvail_26=0, aopAvail_26=0, sbasCorrUsed_26=0, rtcmCorrUsed_26=0, slasCorrUsed_26=0, spartnCorrUsed_26=0, prCorrUsed_26=0, crCorrUsed_26=0, doCorrUsed_26=0, gnssId_27=BeiDou, svId_27=7, cno_27=27, elev_27=22, azim_27=57, prRes_27=-13.4, qualityInd_27=7, svUsed_27=1, health_27=1, diffCorr_27=0, smoothed_27=0, orbitSource_27=1, ephAvail_27=1, almAvail_27=1, anoAvail_27=0, aopAvail_27=1, sbasCorrUsed_27=0, rtcmCorrUsed_27=0, slasCorrUsed_27=0, spartnCorrUsed_27=0, prCorrUsed_27=0, crCorrUsed_27=0, doCorrUsed_27=0, gnssId_28=BeiDou, svId_28=10, cno_28=0, elev_28=24, azim_28=72, prRes_28=0.0, qualityInd_28=0, svUsed_28=0, health_28=1, diffCorr_28=0, smoothed_28=0, orbitSource_28=1, ephAvail_28=1, almAvail_28=1, anoAvail_28=0, aopAvail_28=1, sbasCorrUsed_28=0, rtcmCorrUsed_28=0, slasCorrUsed_28=0, spartnCorrUsed_28=0, prCorrUsed_28=0, crCorrUsed_28=0, doCorrUsed_28=0, gnssId_29=BeiDou, svId_29=11, cno_29=17, elev_29=5, azim_29=258, prRes_29=-8.9, qualityInd_29=4, svUsed_29=0, health_29=1, diffCorr_29=0, smoothed_29=0, orbitSource_29=1, ephAvail_29=1, almAvail_29=1, anoAvail_29=0, aopAvail_29=1, sbasCorrUsed_29=0, rtcmCorrUsed_29=0, slasCorrUsed_29=0, spartnCorrUsed_29=0, prCorrUsed_29=0, crCorrUsed_29=0, doCorrUsed_29=0, gnssId_30=BeiDou, svId_30=14, cno_30=39, elev_30=19, azim_30=297, prRes_30=0.7, qualityInd_30=7, svUsed_30=1, health_30=1, diffCorr_30=0, smoothed_30=0, orbitSource_30=1, ephAvail_30=1, almAvail_30=1, anoAvail_30=0, aopAvail_30=1, sbasCorrUsed_30=0, rtcmCorrUsed_30=0, slasCorrUsed_30=0, spartnCorrUsed_30=0, prCorrUsed_30=0, crCorrUsed_30=0, doCorrUsed_30=0, gnssId_31=BeiDou, svId_31=21, cno_31=29, elev_31=79, azim_31=145, prRes_31=3.4, qualityInd_31=7, svUsed_31=1, health_31=1, diffCorr_31=0, smoothed_31=0, orbitSource_31=1, ephAvail_31=1, almAvail_31=1, anoAvail_31=0, aopAvail_31=1, sbasCorrUsed_31=0, rtcmCorrUsed_31=0, slasCorrUsed_31=0, spartnCorrUsed_31=0, prCorrUsed_31=0, crCorrUsed_31=0, doCorrUsed_31=0, gnssId_32=BeiDou, svId_32=22, cno_32=26, elev_32=26, azim_32=122, prRes_32=-6.7, qualityInd_32=4, svUsed_32=1, health_32=1, diffCorr_32=0, smoothed_32=0, orbitSource_32=1, ephAvail_32=1, almAvail_32=1, anoAvail_32=0, aopAvail_32=1, sbasCorrUsed_32=0, rtcmCorrUsed_32=0, slasCorrUsed_32=0, spartnCorrUsed_32=0, prCorrUsed_32=0, crCorrUsed_32=0, doCorrUsed_32=0, gnssId_33=BeiDou, svId_33=26, cno_33=0, elev_33=22, azim_33=191, prRes_33=0.0, qualityInd_33=0, svUsed_33=0, health_33=1, diffCorr_33=0, smoothed_33=0, orbitSource_33=1, ephAvail_33=1, almAvail_33=1, anoAvail_33=0, aopAvail_33=1, sbasCorrUsed_33=0, rtcmCorrUsed_33=0, slasCorrUsed_33=0, spartnCorrUsed_33=0, prCorrUsed_33=0, crCorrUsed_33=0, doCorrUsed_33=0, gnssId_34=BeiDou, svId_34=28, cno_34=28, elev_34=10, azim_34=332, prRes_34=-2.7, qualityInd_34=7, svUsed_34=1, health_34=1, diffCorr_34=0, smoothed_34=0, orbitSource_34=1, ephAvail_34=1, almAvail_34=1, anoAvail_34=0, aopAvail_34=1, sbasCorrUsed_34=0, rtcmCorrUsed_34=0, slasCorrUsed_34=0, spartnCorrUsed_34=0, prCorrUsed_34=0, crCorrUsed_34=0, doCorrUsed_34=0, gnssId_35=GLONASS, svId_35=1, cno_35=29, elev_35=42, azim_35=105, prRes_35=3.3, qualityInd_35=7, svUsed_35=1, health_35=1, diffCorr_35=0, smoothed_35=0, orbitSource_35=1, ephAvail_35=1, almAvail_35=1, anoAvail_35=0, aopAvail_35=1, sbasCorrUsed_35=0, rtcmCorrUsed_35=0, slasCorrUsed_35=0, spartnCorrUsed_35=0, prCorrUsed_35=0, crCorrUsed_35=0, doCorrUsed_35=0, gnssId_36=GLONASS, svId_36=2, cno_36=34, elev_36=68, azim_36=323, prRes_36=2.4, qualityInd_36=7, svUsed_36=1, health_36=1, diffCorr_36=0, smoothed_36=0, orbitSource_36=1, ephAvail_36=1, almAvail_36=1, anoAvail_36=0, aopAvail_36=1, sbasCorrUsed_36=0, rtcmCorrUsed_36=0, slasCorrUsed_36=0, spartnCorrUsed_36=0, prCorrUsed_36=0, crCorrUsed_36=0, doCorrUsed_36=0, gnssId_37=GLONASS, svId_37=3, cno_37=31, elev_37=17, azim_37=298, prRes_37=-5.9, qualityInd_37=7, svUsed_37=1, health_37=1, diffCorr_37=0, smoothed_37=0, orbitSource_37=1, ephAvail_37=1, almAvail_37=1, anoAvail_37=0, aopAvail_37=1, sbasCorrUsed_37=0, rtcmCorrUsed_37=0, slasCorrUsed_37=0, spartnCorrUsed_37=0, prCorrUsed_37=0, crCorrUsed_37=0, doCorrUsed_37=0, gnssId_38=GLONASS, svId_38=10, cno_38=13, elev_38=14, azim_38=26, prRes_38=-28.5, qualityInd_38=4, svUsed_38=0, health_38=1, diffCorr_38=0, smoothed_38=0, orbitSource_38=1, ephAvail_38=1, almAvail_38=1, anoAvail_38=0, aopAvail_38=1, sbasCorrUsed_38=0, rtcmCorrUsed_38=0, slasCorrUsed_38=0, spartnCorrUsed_38=0, prCorrUsed_38=0, crCorrUsed_38=0, doCorrUsed_38=0, gnssId_39=GLONASS, svId_39=11, cno_39=30, elev_39=56, azim_39=71, prRes_39=0.0, qualityInd_39=7, svUsed_39=0, health_39=2, diffCorr_39=0, smoothed_39=0, orbitSource_39=2, ephAvail_39=1, almAvail_39=1, anoAvail_39=0, aopAvail_39=0, sbasCorrUsed_39=0, rtcmCorrUsed_39=0, slasCorrUsed_39=0, spartnCorrUsed_39=0, prCorrUsed_39=0, crCorrUsed_39=0, doCorrUsed_39=0, gnssId_40=GLONASS, svId_40=12, cno_40=26, elev_40=39, azim_40=163, prRes_40=-0.5, qualityInd_40=7, svUsed_40=1, health_40=1, diffCorr_40=0, smoothed_40=0, orbitSource_40=1, ephAvail_40=1, almAvail_40=1, anoAvail_40=0, aopAvail_40=1, sbasCorrUsed_40=0, rtcmCorrUsed_40=0, slasCorrUsed_40=0, spartnCorrUsed_40=0, prCorrUsed_40=0, crCorrUsed_40=0, doCorrUsed_40=0, gnssId_41=GLONASS, svId_41=17, cno_41=29, elev_41=13, azim_41=253, prRes_41=1.6, qualityInd_41=7, svUsed_41=1, health_41=1, diffCorr_41=0, smoothed_41=0, orbitSource_41=1, ephAvail_41=1, almAvail_41=1, anoAvail_41=0, aopAvail_41=1, sbasCorrUsed_41=0, rtcmCorrUsed_41=0, slasCorrUsed_41=0, spartnCorrUsed_41=0, prCorrUsed_41=0, crCorrUsed_41=0, doCorrUsed_41=0, gnssId_42=GLONASS, svId_42=18, cno_42=39, elev_42=26, azim_42=302, prRes_42=-1.9, qualityInd_42=7, svUsed_42=1, health_42=1, diffCorr_42=0, smoothed_42=0, orbitSource_42=1, ephAvail_42=1, almAvail_42=1, anoAvail_42=0, aopAvail_42=1, sbasCorrUsed_42=0, rtcmCorrUsed_42=0, slasCorrUsed_42=0, spartnCorrUsed_42=0, prCorrUsed_42=0, crCorrUsed_42=0, doCorrUsed_42=0, gnssId_43=GLONASS, svId_43=19, cno_43=0, elev_43=16, azim_43=349, prRes_43=0.0, qualityInd_43=0, svUsed_43=0, health_43=1, diffCorr_43=0, smoothed_43=0, orbitSource_43=2, ephAvail_43=0, almAvail_43=1, anoAvail_43=0, aopAvail_43=0, sbasCorrUsed_43=0, rtcmCorrUsed_43=0, slasCorrUsed_43=0, spartnCorrUsed_43=0, prCorrUsed_43=0, crCorrUsed_43=0, doCorrUsed_43=0)>",
            "<UBX(NAV-SIG, iTOW=11:34:59, version=0, numSigs=32, reserved0=0, gnssId_01=GPS, svId_01=2, sigId_01=0, freqId_01=0, prRes_01=1.7, cno_01=31, qualityInd_01=7, corrSource_01=1, ionoModel_01=2, health_01=1, prSmoothed_01=0, prUsed_01=1, crUsed_01=0, doUsed_01=1, prCorrUsed_01=1, crCorrUsed_01=0, doCorrUsed_01=1, reserved1_01=0, gnssId_02=GPS, svId_02=5, sigId_02=0, freqId_02=0, prRes_02=0.9, cno_02=17, qualityInd_02=4, corrSource_02=1, ionoModel_02=2, health_02=1, prSmoothed_02=0, prUsed_02=1, crUsed_02=0, doUsed_02=1, prCorrUsed_02=1, crCorrUsed_02=0, doCorrUsed_02=1, reserved1_02=0, gnssId_03=GPS, svId_03=7, sigId_03=0, freqId_03=0, prRes_03=0.3, cno_03=40, qualityInd_03=7, corrSource_03=1, ionoModel_03=2, health_03=1, prSmoothed_03=0, prUsed_03=1, crUsed_03=0, doUsed_03=1, prCorrUsed_03=1, crCorrUsed_03=0, doCorrUsed_03=1, reserved1_03=0, gnssId_04=GPS, svId_04=13, sigId_04=0, freqId_04=0, prRes_04=-8.1, cno_04=32, qualityInd_04=7, corrSource_04=1, ionoModel_04=2, health_04=1, prSmoothed_04=0, prUsed_04=1, crUsed_04=0, doUsed_04=1, prCorrUsed_04=1, crCorrUsed_04=0, doCorrUsed_04=1, reserved1_04=0, gnssId_05=GPS, svId_05=15, sigId_05=0, freqId_05=0, prRes_05=0.1, cno_05=41, qualityInd_05=7, corrSource_05=1, ionoModel_05=2, health_05=1, prSmoothed_05=0, prUsed_05=1, crUsed_05=0, doUsed_05=1, prCorrUsed_05=1, crCorrUsed_05=0, doCorrUsed_05=1, reserved1_05=0, gnssId_06=GPS, svId_06=18, sigId_06=0, freqId_06=0, prRes_06=-0.5, cno_06=30, qualityInd_06=6, corrSource_06=1, ionoModel_06=2, health_06=1, prSmoothed_06=0, prUsed_06=1, crUsed_06=0, doUsed_06=1, prCorrUsed_06=1, crCorrUsed_06=0, doCorrUsed_06=1, reserved1_06=0, gnssId_07=GPS, svId_07=20, sigId_07=0, freqId_07=0, prRes_07=-2.8, cno_07=19, qualityInd_07=4, corrSource_07=1, ionoModel_07=2, health_07=1, prSmoothed_07=0, prUsed_07=1, crUsed_07=0, doUsed_07=1, prCorrUsed_07=1, crCorrUsed_07=0, doCorrUsed_07=1, reserved1_07=0, gnssId_08=GPS, svId_08=30, sigId_08=0, freqId_08=0, prRes_08=-0.2, cno_08=41, qualityInd_08=7, corrSource_08=1, ionoModel_08=2, health_08=1, prSmoothed_08=0, prUsed_08=1, crUsed_08=0, doUsed_08=1, prCorrUsed_08=1, crCorrUsed_08=0, doCorrUsed_08=1, reserved1_08=0, gnssId_09=SBAS, svId_09=123, sigId_09=0, freqId_09=0, prRes_09=0.0, cno_09=0, qualityInd_09=1, corrSource_09=0, ionoModel_09=0, health_09=0, prSmoothed_09=0, prUsed_09=0, crUsed_09=0, doUsed_09=0, prCorrUsed_09=0, crCorrUsed_09=0, doCorrUsed_09=0, reserved1_09=0, gnssId_10=SBAS, svId_10=136, sigId_10=0, freqId_10=0, prRes_10=0.0, cno_10=31, qualityInd_10=7, corrSource_10=0, ionoModel_10=0, health_10=0, prSmoothed_10=0, prUsed_10=0, crUsed_10=0, doUsed_10=0, prCorrUsed_10=0, crCorrUsed_10=0, doCorrUsed_10=0, reserved1_10=0, gnssId_11=Galileo, svId_11=3, sigId_11=0, freqId_11=0, prRes_11=4.9, cno_11=31, qualityInd_11=7, corrSource_11=0, ionoModel_11=2, health_11=1, prSmoothed_11=0, prUsed_11=1, crUsed_11=0, doUsed_11=1, prCorrUsed_11=0, crCorrUsed_11=0, doCorrUsed_11=0, reserved1_11=0, gnssId_12=Galileo, svId_12=7, sigId_12=0, freqId_12=0, prRes_12=0.4, cno_12=33, qualityInd_12=7, corrSource_12=0, ionoModel_12=2, health_12=1, prSmoothed_12=0, prUsed_12=1, crUsed_12=0, doUsed_12=1, prCorrUsed_12=0, crCorrUsed_12=0, doCorrUsed_12=0, reserved1_12=0, gnssId_13=Galileo, svId_13=8, sigId_13=0, freqId_13=0, prRes_13=-0.7, cno_13=37, qualityInd_13=7, corrSource_13=0, ionoModel_13=2, health_13=1, prSmoothed_13=0, prUsed_13=1, crUsed_13=0, doUsed_13=1, prCorrUsed_13=0, crCorrUsed_13=0, doCorrUsed_13=0, reserved1_13=0, gnssId_14=Galileo, svId_14=13, sigId_14=0, freqId_14=0, prRes_14=1.0, cno_14=34, qualityInd_14=7, corrSource_14=0, ionoModel_14=2, health_14=1, prSmoothed_14=0, prUsed_14=1, crUsed_14=0, doUsed_14=1, prCorrUsed_14=0, crCorrUsed_14=0, doCorrUsed_14=0, reserved1_14=0, gnssId_15=Galileo, svId_15=25, sigId_15=0, freqId_15=0, prRes_15=5.1, cno_15=29, qualityInd_15=7, corrSource_15=0, ionoModel_15=2, health_15=1, prSmoothed_15=0, prUsed_15=1, crUsed_15=0, doUsed_15=1, prCorrUsed_15=0, crCorrUsed_15=0, doCorrUsed_15=0, reserved1_15=0, gnssId_16=Galileo, svId_16=26, sigId_16=0, freqId_16=0, prRes_16=3.4, cno_16=29, qualityInd_16=7, corrSource_16=0, ionoModel_16=2, health_16=1, prSmoothed_16=0, prUsed_16=1, crUsed_16=0, doUsed_16=1, prCorrUsed_16=0, crCorrUsed_16=0, doCorrUsed_16=0, reserved1_16=0, gnssId_17=Galileo, svId_17=33, sigId_17=0, freqId_17=0, prRes_17=-4.0, cno_17=26, qualityInd_17=7, corrSource_17=0, ionoModel_17=2, health_17=1, prSmoothed_17=0, prUsed_17=1, crUsed_17=0, doUsed_17=1, prCorrUsed_17=0, crCorrUsed_17=0, doCorrUsed_17=0, reserved1_17=0, gnssId_18=BeiDou, svId_18=5, sigId_18=1, freqId_18=0, prRes_18=0.0, cno_18=0, qualityInd_18=1, corrSource_18=0, ionoModel_18=0, health_18=1, prSmoothed_18=0, prUsed_18=0, crUsed_18=0, doUsed_18=0, prCorrUsed_18=0, crCorrUsed_18=0, doCorrUsed_18=0, reserved1_18=0, gnssId_19=BeiDou, svId_19=7, sigId_19=0, freqId_19=0, prRes_19=-13.4, cno_19=27, qualityInd_19=7, corrSource_19=0, ionoModel_19=2, health_19=1, prSmoothed_19=0, prUsed_19=1, crUsed_19=0, doUsed_19=1, prCorrUsed_19=0, crCorrUsed_19=0, doCorrUsed_19=0, reserved1_19=0, gnssId_20=BeiDou, svId_20=11, sigId_20=0, freqId_20=0, prRes_20=-8.9, cno_20=17, qualityInd_20=4, corrSource_20=0, ionoModel_20=0, health_20=1, prSmoothed_20=0, prUsed_20=0, crUsed_20=0, doUsed_20=0, prCorrUsed_20=0, crCorrUsed_20=0, doCorrUsed_20=0, reserved1_20=0, gnssId_21=BeiDou, svId_21=14, sigId_21=0, freqId_21=0, prRes_21=0.7, cno_21=39, qualityInd_21=7, corrSource_21=0, ionoModel_21=2, health_21=1, prSmoothed_21=0, prUsed_21=1, crUsed_21=0, doUsed_21=1, prCorrUsed_21=0, crCorrUsed_21=0, doCorrUsed_21=0, reserved1_21=0, gnssId_22=BeiDou, svId_22=21, sigId_22=0, freqId_22=0, prRes_22=3.4, cno_22=29, qualityInd_22=7, corrSource_22=0, ionoModel_22=2, health_22=1, prSmoothed_22=0, prUsed_22=1, crUsed_22=0, doUsed_22=1, prCorrUsed_22=0, crCorrUsed_22=0, doCorrUsed_22=0, reserved1_22=0, gnssId_23=BeiDou, svId_23=22, sigId_23=0, freqId_23=0, prRes_23=-6.7, cno_23=26, qualityInd_23=4, corrSource_23=0, ionoModel_23=2, health_23=1, prSmoothed_23=0, prUsed_23=1, crUsed_23=0, doUsed_23=1, prCorrUsed_23=0, crCorrUsed_23=0, doCorrUsed_23=0, reserved1_23=0, gnssId_24=BeiDou, svId_24=28, sigId_24=0, freqId_24=0, prRes_24=-2.7, cno_24=28, qualityInd_24=7, corrSource_24=0, ionoModel_24=2, health_24=1, prSmoothed_24=0, prUsed_24=1, crUsed_24=0, doUsed_24=1, prCorrUsed_24=0, crCorrUsed_24=0, doCorrUsed_24=0, reserved1_24=0, gnssId_25=GLONASS, svId_25=1, sigId_25=0, freqId_25=8, prRes_25=3.3, cno_25=29, qualityInd_25=7, corrSource_25=0, ionoModel_25=2, health_25=1, prSmoothed_25=0, prUsed_25=1, crUsed_25=0, doUsed_25=1, prCorrUsed_25=0, crCorrUsed_25=0, doCorrUsed_25=0, reserved1_25=0, gnssId_26=GLONASS, svId_26=2, sigId_26=0, freqId_26=3, prRes_26=2.4, cno_26=34, qualityInd_26=7, corrSource_26=0, ionoModel_26=2, health_26=1, prSmoothed_26=0, prUsed_26=1, crUsed_26=0, doUsed_26=1, prCorrUsed_26=0, crCorrUsed_26=0, doCorrUsed_26=0, reserved1_26=0, gnssId_27=GLONASS, svId_27=3, sigId_27=0, freqId_27=12, prRes_27=-5.9, cno_27=31, qualityInd_27=7, corrSource_27=0, ionoModel_27=2, health_27=1, prSmoothed_27=0, prUsed_27=1, crUsed_27=0, doUsed_27=1, prCorrUsed_27=0, crCorrUsed_27=0, doCorrUsed_27=0, reserved1_27=0, gnssId_28=GLONASS, svId_28=10, sigId_28=0, freqId_28=0, prRes_28=-28.5, cno_28=13, qualityInd_28=4, corrSource_28=0, ionoModel_28=0, health_28=1, prSmoothed_28=0, prUsed_28=0, crUsed_28=0, doUsed_28=0, prCorrUsed_28=0, crCorrUsed_28=0, doCorrUsed_28=0, reserved1_28=0, gnssId_29=GLONASS, svId_29=11, sigId_29=0, freqId_29=7, prRes_29=0.0, cno_29=30, qualityInd_29=7, corrSource_29=0, ionoModel_29=0, health_29=0, prSmoothed_29=0, prUsed_29=0, crUsed_29=0, doUsed_29=0, prCorrUsed_29=0, crCorrUsed_29=0, doCorrUsed_29=0, reserved1_29=0, gnssId_30=GLONASS, svId_30=12, sigId_30=0, freqId_30=6, prRes_30=-0.5, cno_30=26, qualityInd_30=7, corrSource_30=0, ionoModel_30=2, health_30=1, prSmoothed_30=0, prUsed_30=1, crUsed_30=0, doUsed_30=1, prCorrUsed_30=0, crCorrUsed_30=0, doCorrUsed_30=0, reserved1_30=0, gnssId_31=GLONASS, svId_31=17, sigId_31=0, freqId_31=11, prRes_31=1.6, cno_31=29, qualityInd_31=7, corrSource_31=0, ionoModel_31=2, health_31=1, prSmoothed_31=0, prUsed_31=1, crUsed_31=0, doUsed_31=1, prCorrUsed_31=0, crCorrUsed_31=0, doCorrUsed_31=0, reserved1_31=0, gnssId_32=GLONASS, svId_32=18, sigId_32=0, freqId_32=4, prRes_32=-1.9, cno_32=39, qualityInd_32=7, corrSource_32=0, ionoModel_32=2, health_32=1, prSmoothed_32=0, prUsed_32=1, crUsed_32=0, doUsed_32=1, prCorrUsed_32=0, crCorrUsed_32=0, doCorrUsed_32=0, reserved1_32=0)>",
            "<UBX(NAV-STATUS, iTOW=11:34:59, gpsFix=3, gpsFixOk=1, diffSoln=1, wknSet=1, towSet=1, diffCorr=0, carrSolnValid=0, mapMatching=0, psmState=0, spoofDetState=1, carrSoln=0, ttff=23352, msss=10402347)>",
            "<UBX(NAV-POSECEF, iTOW=11:34:59, ecefX=380364578, ecefY=-14879975, ecefZ=510064662, pAcc=255)>",
            "<UBX(NAV-POSLLH, iTOW=11:34:59, lon=-2.2402855, lat=53.4507228, height=91184, hMSL=42701, hAcc=1491, vAcc=2065)>",
            "<UBX(NAV-DOP, iTOW=11:34:59, gDOP=1.14, pDOP=1.01, tDOP=0.53, vDOP=0.83, hDOP=0.58, nDOP=0.46, eDOP=0.35)>",
            "<UBX(NAV-VELECEF, iTOW=11:34:59, ecefVX=0, ecefVY=0, ecefVZ=-1, sAcc=26)>",
            "<UBX(NAV-VELNED, iTOW=11:34:59, velN=-1, velE=0, velD=1, speed=1, gSpeed=1, heading=0.0, sAcc=26, cAcc=180.0)>",
            "<UBX(NAV-TIMEGPS, iTOW=11:34:59, fTOW=-361668, week=2186, leapS=18, towValid=1, weekValid=1, leapSValid=1, tAcc=6)>",
            "<UBX(NAV-TIMEGLO, iTOW=11:34:59, TOD=52499, fTOD=-361682, Nt=704, N4=7, todValid=1, dateValid=1, tAcc=5)>",
            "<UBX(NAV-TIMEBDS, iTOW=11:34:59, SOW=560103, fSOW=-361693, week=830, leapS=4, sowValid=1, weekValid=1, leapSValid=1, tAcc=5)>",
            "<UBX(NAV-TIMEGAL, iTOW=11:34:59, galTow=560117, fGalTow=-361655, galWno=1162, leapS=18, galTowValid=1, galWnoValid=1, leapSValid=1, tAcc=4)>",
            "<UBX(NAV-TIMEUTC, iTOW=11:34:59, tAcc=26, nano=-361668, year=2021, month=12, day=4, hour=11, min=34, sec=59, validTOW=1, validWKN=1, validUTC=1, utcStandard=3)>",
            "<UBX(NAV-TIMELS, iTOW=11:34:59, version=0, reserved0=0, srcOfCurrLs=2, currLs=18, srcOfLsChange=2, lsChange=0, timeToLsEvent=-560098, dateOfLsGpsWn=2185, dateOfLsGpsDn=7, reserved1=0, validCurrLs=1, validTimeToLsEvent=1)>",
            "<UBX(NAV-TIMEQZSS, iTOW=11:34:59, qzssTow=560117, fQzssTow=-361655, qzssWno=2186, leapS=18, qzssTowValid=1, qzssWnoValid=1, leapSValid=1, tAcc=3340)>",
            "<UBX(NAV-CLOCK, iTOW=11:34:59, clkB=-638345, clkD=-48, tAcc=4, fAcc=599)>",
            "<UBX(NAV-SBAS, iTOW=11:34:59, geo=136, mode=1, sys=1, Ranging=0, Corrections=1, Integrity=1, Testmode=0, Bad=0, cnt=31, integrityUsed=0, reserved1=0, svid_01=243, flags_01=8, udre_01=14, svSys_01=255, svService_01=0, reserved2_01=0, prc_01=0, reserved3_01=0, ic_01=274, svid_02=5, flags_02=139, udre_02=5, svSys_02=16, svService_02=1, reserved2_02=0, prc_02=-113, reserved3_02=0, ic_02=140, svid_03=30, flags_03=139, udre_03=5, svSys_03=16, svService_03=1, reserved2_03=0, prc_03=-38, reserved3_03=0, ic_03=125, svid_04=15, flags_04=139, udre_04=5, svSys_04=16, svService_04=1, reserved2_04=0, prc_04=38, reserved3_04=0, ic_04=350, svid_05=2, flags_05=139, udre_05=5, svSys_05=16, svService_05=1, reserved2_05=0, prc_05=-163, reserved3_05=0, ic_05=348, svid_06=7, flags_06=139, udre_06=5, svSys_06=16, svService_06=1, reserved2_06=0, prc_06=-75, reserved3_06=0, ic_06=143, svid_07=20, flags_07=139, udre_07=5, svSys_07=16, svService_07=1, reserved2_07=0, prc_07=-50, reserved3_07=0, ic_07=134, svid_08=76, flags_08=8, udre_08=14, svSys_08=255, svService_08=0, reserved2_08=0, prc_08=0, reserved3_08=0, ic_08=175, svid_09=49, flags_09=8, udre_09=14, svSys_09=255, svService_09=0, reserved2_09=0, prc_09=0, reserved3_09=0, ic_09=246, svid_10=48, flags_10=8, udre_10=14, svSys_10=255, svService_10=0, reserved2_10=0, prc_10=0, reserved3_10=0, ic_10=124, svid_11=34, flags_11=8, udre_11=14, svSys_11=255, svService_11=0, reserved2_11=0, prc_11=0, reserved3_11=0, ic_11=251, svid_12=41, flags_12=8, udre_12=14, svSys_12=255, svService_12=0, reserved2_12=0, prc_12=0, reserved3_12=0, ic_12=309, svid_13=38, flags_13=8, udre_13=14, svSys_13=255, svService_13=0, reserved2_13=0, prc_13=0, reserved3_13=0, ic_13=432, svid_14=236, flags_14=8, udre_14=14, svSys_14=255, svService_14=0, reserved2_14=0, prc_14=0, reserved3_14=0, ic_14=226, svid_15=18, flags_15=139, udre_15=6, svSys_15=16, svService_15=1, reserved2_15=0, prc_15=13, reserved3_15=0, ic_15=321, svid_16=66, flags_16=8, udre_16=14, svSys_16=255, svService_16=0, reserved2_16=0, prc_16=0, reserved3_16=0, ic_16=126, svid_17=235, flags_17=8, udre_17=14, svSys_17=255, svService_17=0, reserved2_17=0, prc_17=0, reserved3_17=0, ic_17=171, svid_18=55, flags_18=8, udre_18=14, svSys_18=255, svService_18=0, reserved2_18=0, prc_18=0, reserved3_18=0, ic_18=283, svid_19=67, flags_19=8, udre_19=14, svSys_19=255, svService_19=0, reserved2_19=0, prc_19=0, reserved3_19=0, ic_19=312, svid_20=82, flags_20=8, udre_20=14, svSys_20=255, svService_20=0, reserved2_20=0, prc_20=0, reserved3_20=0, ic_20=238, svid_21=213, flags_21=8, udre_21=14, svSys_21=255, svService_21=0, reserved2_21=0, prc_21=0, reserved3_21=0, ic_21=173, svid_22=218, flags_22=8, udre_22=14, svSys_22=255, svService_22=0, reserved2_22=0, prc_22=0, reserved3_22=0, ic_22=121, svid_23=74, flags_23=8, udre_23=14, svSys_23=255, svService_23=0, reserved2_23=0, prc_23=0, reserved3_23=0, ic_23=289, svid_24=163, flags_24=0, udre_24=14, svSys_24=255, svService_24=0, reserved2_24=0, prc_24=0, reserved3_24=0, ic_24=0, svid_25=13, flags_25=139, udre_25=5, svSys_25=16, svService_25=1, reserved2_25=0, prc_25=-75, reserved3_25=0, ic_25=176, svid_26=65, flags_26=8, udre_26=14, svSys_26=255, svService_26=0, reserved2_26=0, prc_26=0, reserved3_26=0, ic_26=162, svid_27=217, flags_27=8, udre_27=14, svSys_27=255, svService_27=0, reserved2_27=0, prc_27=0, reserved3_27=0, ic_27=217, svid_28=223, flags_28=8, udre_28=14, svSys_28=255, svService_28=0, reserved2_28=0, prc_28=0, reserved3_28=0, ic_28=265, svid_29=81, flags_29=8, udre_29=14, svSys_29=255, svService_29=0, reserved2_29=0, prc_29=0, reserved3_29=0, ic_29=334, svid_30=123, flags_30=0, udre_30=14, svSys_30=1, svService_30=14, reserved2_30=0, prc_30=0, reserved3_30=0, ic_30=0, svid_31=136, flags_31=0, udre_31=14, svSys_31=1, svService_31=6, reserved2_31=243, prc_31=0, reserved3_31=0, ic_31=0)>",
            "<UBX(NAV-SLAS, iTOW=11:34:59, version=0, reserved0=0, gmsLon=0.0, gmsLat=0.0, gmsCode=0, qzssSvId=0, gmsAvailable=0, qzssSvAvailable=0, testMode=0, cnt=0)>",
            "<UBX(NAV-AOPSTATUS, iTOW=11:34:59, aopCfg=1, status=0, reserved1=2560)>",
            "<UBX(NAV-ODO, version=0, reserved0=2187957, iTOW=11:34:59, distance=0, totalDistance=0, distanceStd=0)>",
            "<UBX(NAV-COV, iTOW=11:34:59, version=0, posCovValid=1, velCovValid=1, reserved0=1171368248954360692736, posCovNN=1.5019601583480835, posCovNE=-0.3225536346435547, posCovND=-1.239114761352539, posCovEE=0.7207303047180176, posCovED=1.396087408065796, posCovDD=4.265473365783691, velCovNN=0.0426090806722641, velCovNE=0.0012168486136943102, velCovND=-0.009526630863547325, velCovEE=0.020430445671081543, velCovED=0.017686128616333008, velCovDD=0.06891409307718277)>",
            "<UBX(NAV-EELL, iTOW=11:34:59, version=0, reserved0=0, errEllipseOrient=160.23, errEllipseMajor=3162, errEllipseMinor=1933)>",
            "<UBX(NAV-GEOFENCE, iTOW=11:34:59, version=0, status=0, numFences=0, combState=0)>",
            "<UBX(NAV-EOE, iTOW=11:34:59)>",
            "<UBX(NAV-TIMENAVIC, iTOW=19:09:01, NavICTow=241759, fNavICTow=97081, NavICWno=1188, leapS=18, NavICTowValid=1, NavICWnoValid=1, leapSValid=1, tAcc=3337)>",
        )

        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamNAV)
        while raw is not None:
            (raw, parsed) = ubxreader.read()
            # print(f"{i} = {parsed}")
            if raw is not None:
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1

    def testHNRLOG(
        self,
    ):  # test stream of UBX HNR messages
        EXPECTED_RESULTS = (
            "<UBX(HNR-PVT, iTOW=09:14:59.400000, year=2021, month=4, day=16, hour=9, min=14, second=59, validDate=1, validTime=1, fullyResolved=1, nano=400000109, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=3999, lon=4.5161898, lat=51.8927505, height=68308, hMSL=22326, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
            "<UBX(HNR-PVT, iTOW=09:14:59.450000, year=2021, month=4, day=16, hour=9, min=14, second=59, validDate=1, validTime=1, fullyResolved=1, nano=450000110, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=4499, lon=4.5161895, lat=51.8927505, height=68316, hMSL=22334, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
            "<UBX(HNR-PVT, iTOW=09:14:59.500000, year=2021, month=4, day=16, hour=9, min=14, second=59, validDate=1, validTime=1, fullyResolved=1, nano=500000111, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=4999, lon=4.5161891, lat=51.8927505, height=68324, hMSL=22342, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
            "<UBX(HNR-PVT, iTOW=09:14:59.550000, year=2021, month=4, day=16, hour=9, min=14, second=59, validDate=1, validTime=1, fullyResolved=1, nano=550000112, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=5499, lon=4.5161888, lat=51.8927505, height=68332, hMSL=22350, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
            "<UBX(HNR-PVT, iTOW=09:14:59.600000, year=2021, month=4, day=16, hour=9, min=14, second=59, validDate=1, validTime=1, fullyResolved=1, nano=600000112, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=5999, lon=4.5161884, lat=51.8927505, height=68340, hMSL=22358, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
            "<UBX(HNR-PVT, iTOW=09:14:59.650000, year=2021, month=4, day=16, hour=9, min=14, second=59, validDate=1, validTime=1, fullyResolved=1, nano=650000113, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=6499, lon=4.5161881, lat=51.8927505, height=68348, hMSL=22366, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
            "<UBX(HNR-PVT, iTOW=09:14:59.700000, year=2021, month=4, day=16, hour=9, min=14, second=59, validDate=1, validTime=1, fullyResolved=1, nano=700000114, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=6999, lon=4.5161877, lat=51.8927505, height=68356, hMSL=22374, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
            "<UBX(HNR-PVT, iTOW=09:14:59.750000, year=2021, month=4, day=16, hour=9, min=14, second=59, validDate=1, validTime=1, fullyResolved=1, nano=750000115, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=7499, lon=4.5161874, lat=51.8927505, height=68364, hMSL=22382, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
            "<UBX(HNR-PVT, iTOW=09:14:59.800000, year=2021, month=4, day=16, hour=9, min=14, second=59, validDate=1, validTime=1, fullyResolved=1, nano=800000115, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=7999, lon=4.516187, lat=51.8927506, height=68372, hMSL=22390, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
            "<UBX(HNR-PVT, iTOW=09:14:59.850000, year=2021, month=4, day=16, hour=9, min=14, second=59, validDate=1, validTime=1, fullyResolved=1, nano=850000116, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=8499, lon=4.5161867, lat=51.8927506, height=68380, hMSL=22398, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
            "<UBX(HNR-PVT, iTOW=09:14:59.900000, year=2021, month=4, day=16, hour=9, min=14, second=59, validDate=1, validTime=1, fullyResolved=1, nano=900000117, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=8999, lon=4.5161863, lat=51.8927506, height=68388, hMSL=22406, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
            "<UBX(HNR-PVT, iTOW=09:14:59.950000, year=2021, month=4, day=16, hour=9, min=14, second=59, validDate=1, validTime=1, fullyResolved=1, nano=950000118, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=9499, lon=4.516186, lat=51.8927506, height=68396, hMSL=22414, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
            "<UBX(HNR-PVT, iTOW=09:15:00, year=2021, month=4, day=16, hour=9, min=15, second=0, validDate=1, validTime=1, fullyResolved=1, nano=118, gpsFix=3, GPSfixOK=1, DiffSoln=0, WKNSET=1, TOWSET=1, headVehValid=1, reserved1=9999, lon=4.5161857, lat=51.8927506, height=68404, hMSL=22422, gSpeed=477, speed=502, headMot=265.8509, headVeh=265.8509, hAcc=36541, vAcc=49102, sAcc=1557, headAcc=38.87061, reserved2=775501108)>",
        )

        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamHNR)
        while raw is not None:
            (raw, parsed) = ubxreader.read()
            # print(f"{i} = {parsed}")
            if raw is not None:
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1

    def testNAVHPPOS(
        self,
    ):  # test stream of UBX NAV-HPPOS* messages
        EXPECTED_RESULTS = (
            "<UBX(NAV-HPPOSECEF, version=0, reserved0=0, iTOW=12:11:19, ecefX=381429769.95, ecefY=-13697582.76, ecefZ=509330640.95, invalidEcef=0, pAcc=587.3)>",
            "<UBX(NAV-HPPOSLLH, version=0, reserved0=0, invalidLlh=0, iTOW=12:11:19, lon=-2.056673696, lat=53.337816927, height=281785.8, hMSL=233522.7, hAcc=335.0, vAcc=482.4)>",
            "<UBX(NAV-HPPOSECEF, version=0, reserved0=0, iTOW=12:11:20, ecefX=381429768.07, ecefY=-13697583.38, ecefZ=509330643.41, invalidEcef=0, pAcc=588.6)>",
            "<UBX(NAV-HPPOSLLH, version=0, reserved0=0, invalidLlh=0, iTOW=12:11:20, lon=-2.056673798, lat=53.337817193, height=281794.4, hMSL=233531.3, hAcc=336.0, vAcc=483.3)>",
        )

        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamNAVHPPOS)
        while raw is not None:
            (raw, parsed) = ubxreader.read()
            # print(f"{i} = {parsed}")
            if raw is not None:
                # print(parsed)
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1

    def testRXMLOG(
        self,
    ):  # test stream of UBX RXM messages
        EXPECTED_RESULTS = (
            "<UBX(RXM-MEASX, version=1, reserved0=0, gpsTOW=231234000, gloTOW=242016000, bdsTOW=231220000, reserved1=231234000, qzssTOW=1000, gpsTOWacc=0.0, gloTOWacc=0.0, bdsTOWacc=0.0, reserved2=0, qzssTOWacc=0.0, numSv=9, towSet=2, reserved3=0, gnssId_01=QZSS, svId_01=1, cNo_01=12, mpathIndic_01=1, dopplerMS_01=461.52, dopplerHz_01=2425.2, wholeChips_01=809, fracChips_01=24, codePhase_01=0.790835380554, intCodePhase_01=0, pseuRangeRMSErr_01=52, reserved4_01=0, gnssId_02=GPS, svId_02=18, cNo_02=17, mpathIndic_02=1, dopplerMS_02=105.84, dopplerHz_02=556.2, wholeChips_02=858, fracChips_02=265, codePhase_02=0.838963508606, intCodePhase_02=0, pseuRangeRMSErr_02=46, reserved4_02=0, gnssId_03=GPS, svId_03=28, cNo_03=18, mpathIndic_03=1, dopplerMS_03=423.04, dopplerHz_03=2223.0, wholeChips_03=536, fracChips_03=533, codePhase_03=0.524457931519, intCodePhase_03=0, pseuRangeRMSErr_03=46, reserved4_03=0, gnssId_04=GLONASS, svId_04=8, cNo_04=17, mpathIndic_04=1, dopplerMS_04=477.96, dopplerHz_04=2559.4, wholeChips_04=55, fracChips_04=693, codePhase_04=0.108956813812, intCodePhase_04=0, pseuRangeRMSErr_04=46, reserved4_04=0, gnssId_05=GLONASS, svId_05=9, cNo_05=25, mpathIndic_05=1, dopplerMS_05=172.8, dopplerHz_05=922.8, wholeChips_05=279, fracChips_05=102, codePhase_05=0.546183109283, intCodePhase_05=0, pseuRangeRMSErr_05=27, reserved4_05=0, gnssId_06=GLONASS, svId_06=7, cNo_06=24, mpathIndic_06=1, dopplerMS_06=-146.88, dopplerHz_06=-786.2, wholeChips_06=100, fracChips_06=156, codePhase_06=0.195994377136, intCodePhase_06=0, pseuRangeRMSErr_06=46, reserved4_06=0, gnssId_07=GPS, svId_07=7, cNo_07=13, mpathIndic_07=1, dopplerMS_07=-591.32, dopplerHz_07=-3107.4, wholeChips_07=947, fracChips_07=989, codePhase_07=0.926653862, intCodePhase_07=0, pseuRangeRMSErr_07=52, reserved4_07=0, gnssId_08=GPS, svId_08=13, cNo_08=28, mpathIndic_08=1, dopplerMS_08=225.96, dopplerHz_08=1187.4, wholeChips_08=239, fracChips_08=545, codePhase_08=0.234147548676, intCodePhase_08=0, pseuRangeRMSErr_08=15, reserved4_08=0, gnssId_09=GPS, svId_09=5, cNo_09=32, mpathIndic_09=1, dopplerMS_09=-384.24, dopplerHz_09=-2019.2, wholeChips_09=220, fracChips_09=411, codePhase_09=0.215446949005, intCodePhase_09=0, pseuRangeRMSErr_09=18, reserved4_09=0)>",
            "<UBX(RXM-SVSI, iTOW=16:13:36, week=2128, numVis=24, numSV=190, svid_01=1, ura_01=15, healthy_01=1, ephVal_01=0, almVal_01=1, notAvail_01=0, azim_01=82, elev_01=-49, almAge_01=2, ephAge_01=15, svid_02=2, ura_02=0, healthy_02=1, ephVal_02=1, almVal_02=1, notAvail_02=0, azim_02=212, elev_02=2, almAge_02=2, ephAge_02=4, svid_03=3, ura_03=15, healthy_03=1, ephVal_03=0, almVal_03=1, notAvail_03=0, azim_03=126, elev_03=-54, almAge_03=2, ephAge_03=15, svid_04=4, ura_04=15, healthy_04=1, ephVal_04=0, almVal_04=1, notAvail_04=0, azim_04=96, elev_04=-29, almAge_04=2, ephAge_04=15, svid_05=5, ura_05=0, healthy_05=1, ephVal_05=1, almVal_05=1, notAvail_05=0, azim_05=202, elev_05=63, almAge_05=2, ephAge_05=2, svid_06=6, ura_06=15, healthy_06=1, ephVal_06=0, almVal_06=1, notAvail_06=0, azim_06=178, elev_06=-21, almAge_06=2, ephAge_06=15, svid_07=7, ura_07=0, healthy_07=1, ephVal_07=1, almVal_07=1, notAvail_07=0, azim_07=59, elev_07=35, almAge_07=2, ephAge_07=2, svid_08=8, ura_08=15, healthy_08=1, ephVal_08=0, almVal_08=1, notAvail_08=0, azim_08=49, elev_08=5, almAge_08=2, ephAge_08=15, svid_09=9, ura_09=0, healthy_09=1, ephVal_09=1, almVal_09=1, notAvail_09=0, azim_09=98, elev_09=1, almAge_09=2, ephAge_09=4, svid_10=10, ura_10=15, healthy_10=1, ephVal_10=0, almVal_10=1, notAvail_10=0, azim_10=329, elev_10=-32, almAge_10=2, ephAge_10=15, svid_11=11, ura_11=15, healthy_11=1, ephVal_11=0, almVal_11=1, notAvail_11=0, azim_11=65, elev_11=-25, almAge_11=3, ephAge_11=15, svid_12=12, ura_12=15, healthy_12=1, ephVal_12=0, almVal_12=1, notAvail_12=0, azim_12=218, elev_12=-42, almAge_12=3, ephAge_12=15, svid_13=13, ura_13=0, healthy_13=1, ephVal_13=1, almVal_13=1, notAvail_13=0, azim_13=276, elev_13=60, almAge_13=3, ephAge_13=4, svid_14=14, ura_14=15, healthy_14=1, ephVal_14=0, almVal_14=0, notAvail_14=0, azim_14=0, elev_14=-91, almAge_14=15, ephAge_14=15, svid_15=15, ura_15=15, healthy_15=1, ephVal_15=0, almVal_15=1, notAvail_15=0, azim_15=281, elev_15=25, almAge_15=3, ephAge_15=15, svid_16=16, ura_16=15, healthy_16=1, ephVal_16=0, almVal_16=1, notAvail_16=0, azim_16=3, elev_16=-13, almAge_16=3, ephAge_16=15, svid_17=17, ura_17=15, healthy_17=1, ephVal_17=0, almVal_17=1, notAvail_17=0, azim_17=147, elev_17=-27, almAge_17=3, ephAge_17=15, svid_18=18, ura_18=15, healthy_18=1, ephVal_18=0, almVal_18=1, notAvail_18=0, azim_18=317, elev_18=20, almAge_18=3, ephAge_18=15, svid_19=19, ura_19=15, healthy_19=1, ephVal_19=0, almVal_19=1, notAvail_19=0, azim_19=164, elev_19=-30, almAge_19=3, ephAge_19=15, svid_20=20, ura_20=15, healthy_20=1, ephVal_20=0, almVal_20=1, notAvail_20=0, azim_20=324, elev_20=-6, almAge_20=3, ephAge_20=15, svid_21=21, ura_21=15, healthy_21=1, ephVal_21=0, almVal_21=1, notAvail_21=0, azim_21=40, elev_21=-33, almAge_21=3, ephAge_21=15, svid_22=22, ura_22=15, healthy_22=1, ephVal_22=0, almVal_22=1, notAvail_22=0, azim_22=104, elev_22=-66, almAge_22=3, ephAge_22=15, svid_23=23, ura_23=15, healthy_23=1, ephVal_23=0, almVal_23=1, notAvail_23=0, azim_23=323, elev_23=-9, almAge_23=3, ephAge_23=15, svid_24=24, ura_24=15, healthy_24=1, ephVal_24=0, almVal_24=1, notAvail_24=0, azim_24=233, elev_24=-10, almAge_24=3, ephAge_24=15, svid_25=25, ura_25=15, healthy_25=1, ephVal_25=0, almVal_25=1, notAvail_25=0, azim_25=253, elev_25=-56, almAge_25=2, ephAge_25=15, svid_26=26, ura_26=15, healthy_26=1, ephVal_26=0, almVal_26=1, notAvail_26=0, azim_26=348, elev_26=-37, almAge_26=2, ephAge_26=15, svid_27=27, ura_27=15, healthy_27=1, ephVal_27=0, almVal_27=1, notAvail_27=0, azim_27=17, elev_27=7, almAge_27=2, ephAge_27=15, svid_28=28, ura_28=15, healthy_28=1, ephVal_28=0, almVal_28=1, notAvail_28=0, azim_28=137, elev_28=26, almAge_28=2, ephAge_28=15, svid_29=29, ura_29=15, healthy_29=1, ephVal_29=0, almVal_29=1, notAvail_29=0, azim_29=277, elev_29=-15, almAge_29=2, ephAge_29=15, svid_30=30, ura_30=0, healthy_30=0, ephVal_30=1, almVal_30=1, notAvail_30=0, azim_30=78, elev_30=63, almAge_30=2, ephAge_30=2, svid_31=31, ura_31=15, healthy_31=1, ephVal_31=0, almVal_31=1, notAvail_31=0, azim_31=6, elev_31=-79, almAge_31=2, ephAge_31=15, svid_32=32, ura_32=15, healthy_32=1, ephVal_32=0, almVal_32=1, notAvail_32=0, azim_32=276, elev_32=-66, almAge_32=2, ephAge_32=15, svid_33=120, ura_33=15, healthy_33=1, ephVal_33=0, almVal_33=0, notAvail_33=0, azim_33=196, elev_33=28, almAge_33=15, ephAge_33=15, svid_34=121, ura_34=15, healthy_34=1, ephVal_34=0, almVal_34=0, notAvail_34=0, azim_34=0, elev_34=-91, almAge_34=15, ephAge_34=15, svid_35=122, ura_35=15, healthy_35=1, ephVal_35=0, almVal_35=0, notAvail_35=0, azim_35=0, elev_35=-91, almAge_35=15, ephAge_35=15, svid_36=123, ura_36=15, healthy_36=1, ephVal_36=0, almVal_36=0, notAvail_36=0, azim_36=140, elev_36=22, almAge_36=15, ephAge_36=15, svid_37=124, ura_37=15, healthy_37=1, ephVal_37=0, almVal_37=0, notAvail_37=0, azim_37=0, elev_37=-91, almAge_37=15, ephAge_37=15, svid_38=125, ura_38=15, healthy_38=1, ephVal_38=0, almVal_38=0, notAvail_38=0, azim_38=0, elev_38=-91, almAge_38=15, ephAge_38=15, svid_39=126, ura_39=15, healthy_39=1, ephVal_39=0, almVal_39=0, notAvail_39=0, azim_39=0, elev_39=-91, almAge_39=15, ephAge_39=15, svid_40=127, ura_40=15, healthy_40=1, ephVal_40=0, almVal_40=0, notAvail_40=0, azim_40=117, elev_40=10, almAge_40=15, ephAge_40=15, svid_41=128, ura_41=15, healthy_41=1, ephVal_41=0, almVal_41=0, notAvail_41=0, azim_41=94, elev_41=-6, almAge_41=15, ephAge_41=15, svid_42=129, ura_42=15, healthy_42=1, ephVal_42=0, almVal_42=0, notAvail_42=0, azim_42=44, elev_42=-35, almAge_42=15, ephAge_42=15, svid_43=130, ura_43=15, healthy_43=1, ephVal_43=0, almVal_43=0, notAvail_43=0, azim_43=0, elev_43=-91, almAge_43=15, ephAge_43=15, svid_44=131, ura_44=15, healthy_44=1, ephVal_44=0, almVal_44=0, notAvail_44=0, azim_44=0, elev_44=-91, almAge_44=15, ephAge_44=15, svid_45=132, ura_45=15, healthy_45=1, ephVal_45=0, almVal_45=0, notAvail_45=0, azim_45=0, elev_45=-91, almAge_45=15, ephAge_45=15, svid_46=133, ura_46=15, healthy_46=1, ephVal_46=0, almVal_46=0, notAvail_46=0, azim_46=275, elev_46=-12, almAge_46=15, ephAge_46=15, svid_47=134, ura_47=15, healthy_47=1, ephVal_47=0, almVal_47=0, notAvail_47=0, azim_47=0, elev_47=-91, almAge_47=15, ephAge_47=15, svid_48=135, ura_48=15, healthy_48=1, ephVal_48=0, almVal_48=0, notAvail_48=0, azim_48=305, elev_48=-30, almAge_48=15, ephAge_48=15, svid_49=136, ura_49=15, healthy_49=1, ephVal_49=0, almVal_49=0, notAvail_49=0, azim_49=171, elev_49=29, almAge_49=15, ephAge_49=15, svid_50=137, ura_50=15, healthy_50=1, ephVal_50=0, almVal_50=0, notAvail_50=0, azim_50=39, elev_50=-37, almAge_50=15, ephAge_50=15, svid_51=138, ura_51=15, healthy_51=1, ephVal_51=0, almVal_51=0, notAvail_51=0, azim_51=282, elev_51=-17, almAge_51=15, ephAge_51=15, svid_52=139, ura_52=15, healthy_52=1, ephVal_52=0, almVal_52=0, notAvail_52=0, azim_52=0, elev_52=-91, almAge_52=15, ephAge_52=15, svid_53=140, ura_53=15, healthy_53=1, ephVal_53=0, almVal_53=0, notAvail_53=0, azim_53=0, elev_53=-91, almAge_53=15, ephAge_53=15, svid_54=141, ura_54=15, healthy_54=1, ephVal_54=0, almVal_54=0, notAvail_54=0, azim_54=0, elev_54=-91, almAge_54=15, ephAge_54=15, svid_55=142, ura_55=15, healthy_55=1, ephVal_55=0, almVal_55=0, notAvail_55=0, azim_55=0, elev_55=-91, almAge_55=15, ephAge_55=15, svid_56=143, ura_56=15, healthy_56=1, ephVal_56=0, almVal_56=0, notAvail_56=0, azim_56=0, elev_56=-91, almAge_56=15, ephAge_56=15, svid_57=144, ura_57=15, healthy_57=1, ephVal_57=0, almVal_57=0, notAvail_57=0, azim_57=0, elev_57=-91, almAge_57=15, ephAge_57=15, svid_58=145, ura_58=15, healthy_58=1, ephVal_58=0, almVal_58=0, notAvail_58=0, azim_58=0, elev_58=-91, almAge_58=15, ephAge_58=15, svid_59=146, ura_59=15, healthy_59=1, ephVal_59=0, almVal_59=0, notAvail_59=0, azim_59=0, elev_59=-91, almAge_59=15, ephAge_59=15, svid_60=147, ura_60=15, healthy_60=1, ephVal_60=0, almVal_60=0, notAvail_60=0, azim_60=0, elev_60=-91, almAge_60=15, ephAge_60=15, svid_61=148, ura_61=15, healthy_61=1, ephVal_61=0, almVal_61=0, notAvail_61=0, azim_61=0, elev_61=-91, almAge_61=15, ephAge_61=15, svid_62=149, ura_62=15, healthy_62=1, ephVal_62=0, almVal_62=0, notAvail_62=0, azim_62=0, elev_62=-91, almAge_62=15, ephAge_62=15, svid_63=150, ura_63=15, healthy_63=1, ephVal_63=0, almVal_63=0, notAvail_63=0, azim_63=0, elev_63=-91, almAge_63=15, ephAge_63=15, svid_64=151, ura_64=15, healthy_64=1, ephVal_64=0, almVal_64=0, notAvail_64=0, azim_64=0, elev_64=-91, almAge_64=15, ephAge_64=15, svid_65=152, ura_65=15, healthy_65=1, ephVal_65=0, almVal_65=0, notAvail_65=0, azim_65=0, elev_65=-91, almAge_65=15, ephAge_65=15, svid_66=153, ura_66=15, healthy_66=1, ephVal_66=0, almVal_66=0, notAvail_66=0, azim_66=0, elev_66=-91, almAge_66=15, ephAge_66=15, svid_67=154, ura_67=15, healthy_67=1, ephVal_67=0, almVal_67=0, notAvail_67=0, azim_67=0, elev_67=-91, almAge_67=15, ephAge_67=15, svid_68=155, ura_68=15, healthy_68=1, ephVal_68=0, almVal_68=0, notAvail_68=0, azim_68=0, elev_68=-91, almAge_68=15, ephAge_68=15, svid_69=156, ura_69=15, healthy_69=1, ephVal_69=0, almVal_69=0, notAvail_69=0, azim_69=0, elev_69=-91, almAge_69=15, ephAge_69=15, svid_70=157, ura_70=15, healthy_70=1, ephVal_70=0, almVal_70=0, notAvail_70=0, azim_70=0, elev_70=-91, almAge_70=15, ephAge_70=15, svid_71=158, ura_71=15, healthy_71=1, ephVal_71=0, almVal_71=0, notAvail_71=0, azim_71=0, elev_71=-91, almAge_71=15, ephAge_71=15, svid_72=211, ura_72=15, healthy_72=1, ephVal_72=0, almVal_72=0, notAvail_72=0, azim_72=0, elev_72=-91, almAge_72=15, ephAge_72=15, svid_73=212, ura_73=15, healthy_73=1, ephVal_73=0, almVal_73=0, notAvail_73=0, azim_73=0, elev_73=-91, almAge_73=15, ephAge_73=15, svid_74=213, ura_74=15, healthy_74=1, ephVal_74=0, almVal_74=0, notAvail_74=0, azim_74=0, elev_74=-91, almAge_74=15, ephAge_74=15, svid_75=214, ura_75=15, healthy_75=1, ephVal_75=0, almVal_75=0, notAvail_75=0, azim_75=0, elev_75=-91, almAge_75=15, ephAge_75=15, svid_76=215, ura_76=15, healthy_76=1, ephVal_76=0, almVal_76=0, notAvail_76=0, azim_76=0, elev_76=-91, almAge_76=15, ephAge_76=15, svid_77=216, ura_77=15, healthy_77=1, ephVal_77=0, almVal_77=0, notAvail_77=0, azim_77=0, elev_77=-91, almAge_77=15, ephAge_77=15, svid_78=217, ura_78=15, healthy_78=1, ephVal_78=0, almVal_78=0, notAvail_78=0, azim_78=0, elev_78=-91, almAge_78=15, ephAge_78=15, svid_79=218, ura_79=15, healthy_79=1, ephVal_79=0, almVal_79=0, notAvail_79=0, azim_79=0, elev_79=-91, almAge_79=15, ephAge_79=15, svid_80=219, ura_80=15, healthy_80=1, ephVal_80=0, almVal_80=0, notAvail_80=0, azim_80=0, elev_80=-91, almAge_80=15, ephAge_80=15, svid_81=220, ura_81=15, healthy_81=1, ephVal_81=0, almVal_81=0, notAvail_81=0, azim_81=0, elev_81=-91, almAge_81=15, ephAge_81=15, svid_82=221, ura_82=15, healthy_82=1, ephVal_82=0, almVal_82=0, notAvail_82=0, azim_82=0, elev_82=-91, almAge_82=15, ephAge_82=15, svid_83=222, ura_83=15, healthy_83=1, ephVal_83=0, almVal_83=0, notAvail_83=0, azim_83=0, elev_83=-91, almAge_83=15, ephAge_83=15, svid_84=223, ura_84=15, healthy_84=1, ephVal_84=0, almVal_84=0, notAvail_84=0, azim_84=0, elev_84=-91, almAge_84=15, ephAge_84=15, svid_85=224, ura_85=15, healthy_85=1, ephVal_85=0, almVal_85=0, notAvail_85=0, azim_85=0, elev_85=-91, almAge_85=15, ephAge_85=15, svid_86=225, ura_86=15, healthy_86=1, ephVal_86=0, almVal_86=0, notAvail_86=0, azim_86=0, elev_86=-91, almAge_86=15, ephAge_86=15, svid_87=226, ura_87=15, healthy_87=1, ephVal_87=0, almVal_87=0, notAvail_87=0, azim_87=0, elev_87=-91, almAge_87=15, ephAge_87=15, svid_88=227, ura_88=15, healthy_88=1, ephVal_88=0, almVal_88=0, notAvail_88=0, azim_88=0, elev_88=-91, almAge_88=15, ephAge_88=15, svid_89=228, ura_89=15, healthy_89=1, ephVal_89=0, almVal_89=0, notAvail_89=0, azim_89=0, elev_89=-91, almAge_89=15, ephAge_89=15, svid_90=229, ura_90=15, healthy_90=1, ephVal_90=0, almVal_90=0, notAvail_90=0, azim_90=0, elev_90=-91, almAge_90=15, ephAge_90=15, svid_91=230, ura_91=15, healthy_91=1, ephVal_91=0, almVal_91=0, notAvail_91=0, azim_91=0, elev_91=-91, almAge_91=15, ephAge_91=15, svid_92=231, ura_92=15, healthy_92=1, ephVal_92=0, almVal_92=0, notAvail_92=0, azim_92=0, elev_92=-91, almAge_92=15, ephAge_92=15, svid_93=232, ura_93=15, healthy_93=1, ephVal_93=0, almVal_93=0, notAvail_93=0, azim_93=0, elev_93=-91, almAge_93=15, ephAge_93=15, svid_94=233, ura_94=15, healthy_94=1, ephVal_94=0, almVal_94=0, notAvail_94=0, azim_94=0, elev_94=-91, almAge_94=15, ephAge_94=15, svid_95=234, ura_95=15, healthy_95=1, ephVal_95=0, almVal_95=0, notAvail_95=0, azim_95=0, elev_95=-91, almAge_95=15, ephAge_95=15, svid_96=235, ura_96=15, healthy_96=1, ephVal_96=0, almVal_96=0, notAvail_96=0, azim_96=0, elev_96=-91, almAge_96=15, ephAge_96=15, svid_97=236, ura_97=15, healthy_97=1, ephVal_97=0, almVal_97=0, notAvail_97=0, azim_97=0, elev_97=-91, almAge_97=15, ephAge_97=15, svid_98=237, ura_98=15, healthy_98=1, ephVal_98=0, almVal_98=0, notAvail_98=0, azim_98=0, elev_98=-91, almAge_98=15, ephAge_98=15, svid_99=238, ura_99=15, healthy_99=1, ephVal_99=0, almVal_99=0, notAvail_99=0, azim_99=0, elev_99=-91, almAge_99=15, ephAge_99=15, svid_100=239, ura_100=15, healthy_100=1, ephVal_100=0, almVal_100=0, notAvail_100=0, azim_100=0, elev_100=-91, almAge_100=15, ephAge_100=15, svid_101=240, ura_101=15, healthy_101=1, ephVal_101=0, almVal_101=0, notAvail_101=0, azim_101=0, elev_101=-91, almAge_101=15, ephAge_101=15, svid_102=241, ura_102=15, healthy_102=1, ephVal_102=0, almVal_102=0, notAvail_102=0, azim_102=0, elev_102=-91, almAge_102=15, ephAge_102=15, svid_103=242, ura_103=15, healthy_103=1, ephVal_103=0, almVal_103=0, notAvail_103=0, azim_103=0, elev_103=-91, almAge_103=15, ephAge_103=15, svid_104=243, ura_104=15, healthy_104=1, ephVal_104=0, almVal_104=0, notAvail_104=0, azim_104=0, elev_104=-91, almAge_104=15, ephAge_104=15, svid_105=244, ura_105=15, healthy_105=1, ephVal_105=0, almVal_105=0, notAvail_105=0, azim_105=0, elev_105=-91, almAge_105=15, ephAge_105=15, svid_106=245, ura_106=15, healthy_106=1, ephVal_106=0, almVal_106=0, notAvail_106=0, azim_106=0, elev_106=-91, almAge_106=15, ephAge_106=15, svid_107=246, ura_107=15, healthy_107=1, ephVal_107=0, almVal_107=0, notAvail_107=0, azim_107=0, elev_107=-91, almAge_107=15, ephAge_107=15, svid_108=159, ura_108=15, healthy_108=1, ephVal_108=0, almVal_108=0, notAvail_108=0, azim_108=0, elev_108=-91, almAge_108=15, ephAge_108=15, svid_109=160, ura_109=15, healthy_109=1, ephVal_109=0, almVal_109=0, notAvail_109=0, azim_109=0, elev_109=-91, almAge_109=15, ephAge_109=15, svid_110=161, ura_110=15, healthy_110=1, ephVal_110=0, almVal_110=0, notAvail_110=0, azim_110=0, elev_110=-91, almAge_110=15, ephAge_110=15, svid_111=162, ura_111=15, healthy_111=1, ephVal_111=0, almVal_111=0, notAvail_111=0, azim_111=0, elev_111=-91, almAge_111=15, ephAge_111=15, svid_112=163, ura_112=15, healthy_112=1, ephVal_112=0, almVal_112=0, notAvail_112=0, azim_112=0, elev_112=-91, almAge_112=15, ephAge_112=15, svid_113=33, ura_113=15, healthy_113=1, ephVal_113=0, almVal_113=0, notAvail_113=0, azim_113=0, elev_113=-91, almAge_113=15, ephAge_113=15, svid_114=34, ura_114=15, healthy_114=1, ephVal_114=0, almVal_114=0, notAvail_114=0, azim_114=0, elev_114=-91, almAge_114=15, ephAge_114=15, svid_115=35, ura_115=15, healthy_115=1, ephVal_115=0, almVal_115=0, notAvail_115=0, azim_115=0, elev_115=-91, almAge_115=15, ephAge_115=15, svid_116=36, ura_116=15, healthy_116=1, ephVal_116=0, almVal_116=0, notAvail_116=0, azim_116=0, elev_116=-91, almAge_116=15, ephAge_116=15, svid_117=37, ura_117=15, healthy_117=1, ephVal_117=0, almVal_117=0, notAvail_117=0, azim_117=0, elev_117=-91, almAge_117=15, ephAge_117=15, svid_118=38, ura_118=15, healthy_118=1, ephVal_118=0, almVal_118=0, notAvail_118=0, azim_118=0, elev_118=-91, almAge_118=15, ephAge_118=15, svid_119=39, ura_119=15, healthy_119=1, ephVal_119=0, almVal_119=0, notAvail_119=0, azim_119=0, elev_119=-91, almAge_119=15, ephAge_119=15, svid_120=40, ura_120=15, healthy_120=1, ephVal_120=0, almVal_120=0, notAvail_120=0, azim_120=0, elev_120=-91, almAge_120=15, ephAge_120=15, svid_121=41, ura_121=15, healthy_121=1, ephVal_121=0, almVal_121=0, notAvail_121=0, azim_121=0, elev_121=-91, almAge_121=15, ephAge_121=15, svid_122=42, ura_122=15, healthy_122=1, ephVal_122=0, almVal_122=0, notAvail_122=0, azim_122=0, elev_122=-91, almAge_122=15, ephAge_122=15, svid_123=43, ura_123=15, healthy_123=1, ephVal_123=0, almVal_123=0, notAvail_123=0, azim_123=0, elev_123=-91, almAge_123=15, ephAge_123=15, svid_124=44, ura_124=15, healthy_124=1, ephVal_124=0, almVal_124=0, notAvail_124=0, azim_124=0, elev_124=-91, almAge_124=15, ephAge_124=15, svid_125=45, ura_125=15, healthy_125=1, ephVal_125=0, almVal_125=0, notAvail_125=0, azim_125=0, elev_125=-91, almAge_125=15, ephAge_125=15, svid_126=46, ura_126=15, healthy_126=1, ephVal_126=0, almVal_126=0, notAvail_126=0, azim_126=0, elev_126=-91, almAge_126=15, ephAge_126=15, svid_127=47, ura_127=15, healthy_127=1, ephVal_127=0, almVal_127=0, notAvail_127=0, azim_127=0, elev_127=-91, almAge_127=15, ephAge_127=15, svid_128=48, ura_128=15, healthy_128=1, ephVal_128=0, almVal_128=0, notAvail_128=0, azim_128=0, elev_128=-91, almAge_128=15, ephAge_128=15, svid_129=49, ura_129=15, healthy_129=1, ephVal_129=0, almVal_129=0, notAvail_129=0, azim_129=0, elev_129=-91, almAge_129=15, ephAge_129=15, svid_130=50, ura_130=15, healthy_130=1, ephVal_130=0, almVal_130=0, notAvail_130=0, azim_130=0, elev_130=-91, almAge_130=15, ephAge_130=15, svid_131=51, ura_131=15, healthy_131=1, ephVal_131=0, almVal_131=0, notAvail_131=0, azim_131=0, elev_131=-91, almAge_131=15, ephAge_131=15, svid_132=52, ura_132=15, healthy_132=1, ephVal_132=0, almVal_132=0, notAvail_132=0, azim_132=0, elev_132=-91, almAge_132=15, ephAge_132=15, svid_133=53, ura_133=15, healthy_133=1, ephVal_133=0, almVal_133=0, notAvail_133=0, azim_133=0, elev_133=-91, almAge_133=15, ephAge_133=15, svid_134=54, ura_134=15, healthy_134=1, ephVal_134=0, almVal_134=0, notAvail_134=0, azim_134=0, elev_134=-91, almAge_134=15, ephAge_134=15, svid_135=55, ura_135=15, healthy_135=1, ephVal_135=0, almVal_135=0, notAvail_135=0, azim_135=0, elev_135=-91, almAge_135=15, ephAge_135=15, svid_136=56, ura_136=15, healthy_136=1, ephVal_136=0, almVal_136=0, notAvail_136=0, azim_136=0, elev_136=-91, almAge_136=15, ephAge_136=15, svid_137=57, ura_137=15, healthy_137=1, ephVal_137=0, almVal_137=0, notAvail_137=0, azim_137=0, elev_137=-91, almAge_137=15, ephAge_137=15, svid_138=58, ura_138=15, healthy_138=1, ephVal_138=0, almVal_138=0, notAvail_138=0, azim_138=0, elev_138=-91, almAge_138=15, ephAge_138=15, svid_139=59, ura_139=15, healthy_139=1, ephVal_139=0, almVal_139=0, notAvail_139=0, azim_139=0, elev_139=-91, almAge_139=15, ephAge_139=15, svid_140=60, ura_140=15, healthy_140=1, ephVal_140=0, almVal_140=0, notAvail_140=0, azim_140=0, elev_140=-91, almAge_140=15, ephAge_140=15, svid_141=61, ura_141=15, healthy_141=1, ephVal_141=0, almVal_141=0, notAvail_141=0, azim_141=0, elev_141=-91, almAge_141=15, ephAge_141=15, svid_142=62, ura_142=15, healthy_142=1, ephVal_142=0, almVal_142=0, notAvail_142=0, azim_142=0, elev_142=-91, almAge_142=15, ephAge_142=15, svid_143=63, ura_143=15, healthy_143=1, ephVal_143=0, almVal_143=0, notAvail_143=0, azim_143=0, elev_143=-91, almAge_143=15, ephAge_143=15, svid_144=64, ura_144=15, healthy_144=1, ephVal_144=0, almVal_144=0, notAvail_144=0, azim_144=0, elev_144=-91, almAge_144=15, ephAge_144=15, svid_145=173, ura_145=15, healthy_145=1, ephVal_145=0, almVal_145=0, notAvail_145=0, azim_145=0, elev_145=-91, almAge_145=15, ephAge_145=15, svid_146=174, ura_146=15, healthy_146=1, ephVal_146=0, almVal_146=0, notAvail_146=0, azim_146=0, elev_146=-91, almAge_146=15, ephAge_146=15, svid_147=175, ura_147=15, healthy_147=1, ephVal_147=0, almVal_147=0, notAvail_147=0, azim_147=0, elev_147=-91, almAge_147=15, ephAge_147=15, svid_148=176, ura_148=15, healthy_148=1, ephVal_148=0, almVal_148=0, notAvail_148=0, azim_148=0, elev_148=-91, almAge_148=15, ephAge_148=15, svid_149=177, ura_149=15, healthy_149=1, ephVal_149=0, almVal_149=0, notAvail_149=0, azim_149=0, elev_149=-91, almAge_149=15, ephAge_149=15, svid_150=178, ura_150=15, healthy_150=1, ephVal_150=0, almVal_150=0, notAvail_150=0, azim_150=0, elev_150=-91, almAge_150=15, ephAge_150=15, svid_151=179, ura_151=15, healthy_151=1, ephVal_151=0, almVal_151=0, notAvail_151=0, azim_151=0, elev_151=-91, almAge_151=15, ephAge_151=15, svid_152=180, ura_152=15, healthy_152=1, ephVal_152=0, almVal_152=0, notAvail_152=0, azim_152=0, elev_152=-91, almAge_152=15, ephAge_152=15, svid_153=181, ura_153=15, healthy_153=1, ephVal_153=0, almVal_153=0, notAvail_153=0, azim_153=0, elev_153=-91, almAge_153=15, ephAge_153=15, svid_154=182, ura_154=15, healthy_154=1, ephVal_154=0, almVal_154=0, notAvail_154=0, azim_154=0, elev_154=-91, almAge_154=15, ephAge_154=15, svid_155=193, ura_155=15, healthy_155=1, ephVal_155=0, almVal_155=0, notAvail_155=0, azim_155=0, elev_155=-91, almAge_155=15, ephAge_155=15, svid_156=194, ura_156=15, healthy_156=1, ephVal_156=0, almVal_156=0, notAvail_156=0, azim_156=0, elev_156=-91, almAge_156=15, ephAge_156=15, svid_157=195, ura_157=15, healthy_157=1, ephVal_157=0, almVal_157=0, notAvail_157=0, azim_157=0, elev_157=-91, almAge_157=15, ephAge_157=15, svid_158=196, ura_158=15, healthy_158=1, ephVal_158=0, almVal_158=0, notAvail_158=0, azim_158=0, elev_158=-91, almAge_158=15, ephAge_158=15, svid_159=197, ura_159=15, healthy_159=1, ephVal_159=0, almVal_159=0, notAvail_159=0, azim_159=0, elev_159=-91, almAge_159=15, ephAge_159=15, svid_160=65, ura_160=15, healthy_160=1, ephVal_160=0, almVal_160=1, notAvail_160=0, azim_160=305, elev_160=0, almAge_160=6, ephAge_160=15, svid_161=66, ura_161=15, healthy_161=1, ephVal_161=0, almVal_161=1, notAvail_161=0, azim_161=299, elev_161=-49, almAge_161=6, ephAge_161=15, svid_162=67, ura_162=15, healthy_162=1, ephVal_162=0, almVal_162=1, notAvail_162=0, azim_162=263, elev_162=-84, almAge_162=6, ephAge_162=15, svid_163=68, ura_163=15, healthy_163=1, ephVal_163=0, almVal_163=1, notAvail_163=0, azim_163=130, elev_163=-58, almAge_163=6, ephAge_163=15, svid_164=69, ura_164=15, healthy_164=1, ephVal_164=0, almVal_164=1, notAvail_164=0, azim_164=124, elev_164=-21, almAge_164=6, ephAge_164=15, svid_165=70, ura_165=15, healthy_165=1, ephVal_165=0, almVal_165=1, notAvail_165=0, azim_165=121, elev_165=30, almAge_165=6, ephAge_165=15, svid_166=71, ura_166=1, healthy_166=1, ephVal_166=1, almVal_166=1, notAvail_166=0, azim_166=71, elev_166=82, almAge_166=6, ephAge_166=4, svid_167=72, ura_167=15, healthy_167=1, ephVal_167=0, almVal_167=1, notAvail_167=0, azim_167=309, elev_167=40, almAge_167=6, ephAge_167=15, svid_168=73, ura_168=1, healthy_168=1, ephVal_168=1, almVal_168=1, notAvail_168=0, azim_168=145, elev_168=65, almAge_168=6, ephAge_168=4, svid_169=74, ura_169=15, healthy_169=1, ephVal_169=0, almVal_169=1, notAvail_169=0, azim_169=191, elev_169=22, almAge_169=6, ephAge_169=15, svid_170=75, ura_170=15, healthy_170=1, ephVal_170=0, almVal_170=1, notAvail_170=0, azim_170=205, elev_170=-24, almAge_170=6, ephAge_170=15, svid_171=76, ura_171=15, healthy_171=1, ephVal_171=0, almVal_171=1, notAvail_171=0, azim_171=223, elev_171=-59, almAge_171=6, ephAge_171=15, svid_172=77, ura_172=15, healthy_172=1, ephVal_172=0, almVal_172=1, notAvail_172=0, azim_172=324, elev_172=-75, almAge_172=6, ephAge_172=15, svid_173=78, ura_173=15, healthy_173=1, ephVal_173=0, almVal_173=1, notAvail_173=0, azim_173=11, elev_173=-44, almAge_173=6, ephAge_173=15, svid_174=79, ura_174=15, healthy_174=1, ephVal_174=0, almVal_174=1, notAvail_174=0, azim_174=24, elev_174=-5, almAge_174=6, ephAge_174=15, svid_175=80, ura_175=2, healthy_175=1, ephVal_175=1, almVal_175=1, notAvail_175=0, azim_175=45, elev_175=43, almAge_175=6, ephAge_175=4, svid_176=81, ura_176=15, healthy_176=1, ephVal_176=0, almVal_176=1, notAvail_176=0, azim_176=21, elev_176=4, almAge_176=6, ephAge_176=15, svid_177=82, ura_177=15, healthy_177=1, ephVal_177=0, almVal_177=1, notAvail_177=0, azim_177=57, elev_177=-18, almAge_177=6, ephAge_177=15, svid_178=83, ura_178=15, healthy_178=1, ephVal_178=0, almVal_178=1, notAvail_178=0, azim_178=94, elev_178=-37, almAge_178=6, ephAge_178=15, svid_179=84, ura_179=15, healthy_179=1, ephVal_179=0, almVal_179=1, notAvail_179=0, azim_179=152, elev_179=-44, almAge_179=6, ephAge_179=15, svid_180=85, ura_180=15, healthy_180=1, ephVal_180=0, almVal_180=1, notAvail_180=0, azim_180=203, elev_180=-30, almAge_180=6, ephAge_180=15, svid_181=86, ura_181=15, healthy_181=1, ephVal_181=0, almVal_181=1, notAvail_181=0, azim_181=240, elev_181=-8, almAge_181=6, ephAge_181=15, svid_182=87, ura_182=15, healthy_182=1, ephVal_182=0, almVal_182=1, notAvail_182=0, azim_182=295, elev_182=18, almAge_182=6, ephAge_182=15, svid_183=88, ura_183=15, healthy_183=1, ephVal_183=0, almVal_183=1, notAvail_183=0, azim_183=334, elev_183=19, almAge_183=6, ephAge_183=15, svid_184=89, ura_184=15, healthy_184=1, ephVal_184=0, almVal_184=0, notAvail_184=0, azim_184=0, elev_184=-91, almAge_184=15, ephAge_184=15, svid_185=90, ura_185=15, healthy_185=1, ephVal_185=0, almVal_185=0, notAvail_185=0, azim_185=0, elev_185=-91, almAge_185=15, ephAge_185=15, svid_186=91, ura_186=15, healthy_186=1, ephVal_186=0, almVal_186=0, notAvail_186=0, azim_186=0, elev_186=-91, almAge_186=15, ephAge_186=15, svid_187=92, ura_187=15, healthy_187=1, ephVal_187=0, almVal_187=0, notAvail_187=0, azim_187=0, elev_187=-91, almAge_187=15, ephAge_187=15, svid_188=93, ura_188=15, healthy_188=1, ephVal_188=0, almVal_188=0, notAvail_188=0, azim_188=0, elev_188=-91, almAge_188=15, ephAge_188=15, svid_189=94, ura_189=15, healthy_189=1, ephVal_189=0, almVal_189=0, notAvail_189=0, azim_189=0, elev_189=-91, almAge_189=15, ephAge_189=15, svid_190=95, ura_190=15, healthy_190=1, ephVal_190=0, almVal_190=0, notAvail_190=0, azim_190=0, elev_190=-91, almAge_190=15, ephAge_190=15)>",
            "<UBX(RXM-IMES, numTx=0, version=1, reserved1=0)>",
            "<UBX(RXM-SFRBX, gnssId=GPS, svId=5, sigId=0, freqId=0, numWords=10, chn=0, version=2, reserved0=0, dwrd_01=583028782, dwrd_02=2463198336, dwrd_03=394902765, dwrd_04=2566867280, dwrd_05=1062207503, dwrd_06=675481840, dwrd_07=616371498, dwrd_08=2740700967, dwrd_09=768066377, dwrd_10=3045061856)>",
            "<UBX(RXM-SPARTN-KEY, version=1, numKeys=2, reserved0=0, encryptAlgorithm_01=0, keyLengthBytes_01=2, validFromWno_01=43, validFromTow_01=231234000, encryptAlgorithm_02=0, keyLengthBytes_02=3, validFromWno_02=43, validFromTow_02=242016000, key_01=12, key_02=0, key_03=41, key_04=9, key_05=0)>",
        )
        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamRXM)
        while raw is not None:
            (raw, parsed) = ubxreader.read()
            # print(f"{i} = {parsed}")
            if raw is not None:
                # print(f'{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1

    def testESFLOG(
        self,
    ):  # test stream of UBX ESF messages
        EXPECTED_RESULTS = (
            "<UBX(ESF-MEAS, timeTag=514162, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=3, id=0, dataField_01=16776523, dataType_01=16, dataField_02=576, dataType_02=17, dataField_03=10275, dataType_03=18, dataField_04=514162, dataType_04=0)>",
            "<UBX(ESF-MEAS, timeTag=514157, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=4, id=0, dataField_01=1741, dataType_01=14, dataField_02=16776932, dataType_02=13, dataField_03=16775683, dataType_03=5, dataField_04=2825, dataType_04=12, dataField_05=514157, dataType_05=0)>",
            "<UBX(ESF-MEAS, timeTag=514261, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=3, id=0, dataField_01=16776525, dataType_01=16, dataField_02=581, dataType_02=17, dataField_03=10271, dataType_03=18, dataField_04=514261, dataType_04=0)>",
            "<UBX(ESF-MEAS, timeTag=514256, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=4, id=0, dataField_01=1660, dataType_01=14, dataField_02=16776907, dataType_02=13, dataField_03=16775596, dataType_03=5, dataField_04=2825, dataType_04=12, dataField_05=514256, dataType_05=0)>",
            "<UBX(ESF-MEAS, timeTag=514360, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=3, id=0, dataField_01=16776522, dataType_01=16, dataField_02=577, dataType_02=17, dataField_03=10279, dataType_03=18, dataField_04=514360, dataType_04=0)>",
            "<UBX(ESF-MEAS, timeTag=514355, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=4, id=0, dataField_01=1551, dataType_01=14, dataField_02=16776726, dataType_02=13, dataField_03=16775771, dataType_03=5, dataField_04=2826, dataType_04=12, dataField_05=514355, dataType_05=0)>",
            "<UBX(ESF-MEAS, timeTag=514460, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=3, id=0, dataField_01=16776526, dataType_01=16, dataField_02=590, dataType_02=17, dataField_03=10272, dataType_03=18, dataField_04=514460, dataType_04=0)>",
            "<UBX(ESF-MEAS, timeTag=514455, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=4, id=0, dataField_01=1669, dataType_01=14, dataField_02=16776823, dataType_02=13, dataField_03=16775649, dataType_03=5, dataField_04=2826, dataType_04=12, dataField_05=514455, dataType_05=0)>",
            "<UBX(ESF-MEAS, timeTag=514559, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=3, id=0, dataField_01=16776524, dataType_01=16, dataField_02=574, dataType_02=17, dataField_03=10276, dataType_03=18, dataField_04=514559, dataType_04=0)>",
            "<UBX(ESF-MEAS, timeTag=514554, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=4, id=0, dataField_01=1682, dataType_01=14, dataField_02=16776801, dataType_02=13, dataField_03=16775583, dataType_03=5, dataField_04=2826, dataType_04=12, dataField_05=514554, dataType_05=0)>",
            "<UBX(ESF-MEAS, timeTag=514659, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=3, id=0, dataField_01=16776519, dataType_01=16, dataField_02=580, dataType_02=17, dataField_03=10268, dataType_03=18, dataField_04=514659, dataType_04=0)>",
            "<UBX(ESF-MEAS, timeTag=514654, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=4, id=0, dataField_01=1775, dataType_01=14, dataField_02=16776888, dataType_02=13, dataField_03=16775624, dataType_03=5, dataField_04=2826, dataType_04=12, dataField_05=514654, dataType_05=0)>",
            "<UBX(ESF-MEAS, timeTag=514758, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=3, id=0, dataField_01=16776522, dataType_01=16, dataField_02=590, dataType_02=17, dataField_03=10273, dataType_03=18, dataField_04=514758, dataType_04=0)>",
            "<UBX(ESF-MEAS, timeTag=514753, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=4, id=0, dataField_01=1666, dataType_01=14, dataField_02=16776795, dataType_02=13, dataField_03=16775624, dataType_03=5, dataField_04=2825, dataType_04=12, dataField_05=514753, dataType_05=0)>",
            "<UBX(ESF-MEAS, timeTag=514858, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=3, id=0, dataField_01=16776520, dataType_01=16, dataField_02=583, dataType_02=17, dataField_03=10279, dataType_03=18, dataField_04=514858, dataType_04=0)>",
            "<UBX(ESF-MEAS, timeTag=514853, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=4, id=0, dataField_01=1819, dataType_01=14, dataField_02=16776941, dataType_02=13, dataField_03=16775674, dataType_03=5, dataField_04=2825, dataType_04=12, dataField_05=514853, dataType_05=0)>",
            "<UBX(ESF-MEAS, timeTag=75227, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=4, id=0, dataField_01=630, dataType_01=14, dataField_02=16775174, dataType_02=13, dataField_03=16775134, dataType_03=5, dataField_04=2644, dataType_04=12, dataField_05=75227, dataType_05=0)>",
            "<UBX(ESF-MEAS, timeTag=74734, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=3, id=0, dataField_01=4584, dataType_01=16, dataField_02=2042, dataType_02=17, dataField_03=8865, dataType_03=18, dataField_04=74734, dataType_04=0)>",
            "<UBX(ESF-MEAS, timeTag=74132, timeMarkSent=0, timeMarkEdge=0, calibTtagValid=1, numMeas=4, id=0, dataField_01=1535, dataType_01=14, dataField_02=16776947, dataType_02=13, dataField_03=2893, dataType_03=5, dataField_04=2641, dataType_04=12, dataField_05=74132, dataType_05=0)>",
        )

        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamESF)
        while raw is not None:
            (raw, parsed) = ubxreader.read()
            # print(f"{i} = {parsed}")
            if raw is not None:
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1

    def testINFLOG(
        self,
    ):  # test stream of UBX INF messages
        EXPECTED_RESULTS = (
            "<UBX(INF-NOTICE, message=u-blox AG - www.u-blox.com)>",
            "<UBX(INF-NOTICE, message=HW UBX-M8030 00080000)>",
            "<UBX(INF-NOTICE, message=ROM CORE 3.01 (107888))>",
            "<UBX(INF-NOTICE, message=FWVER=SPG 3.01)>",
            "<UBX(INF-NOTICE, message=PROTVER=18.00)>",
            "<UBX(INF-NOTICE, message=GPS;GLO;GAL;BDS)>",
            "<UBX(INF-NOTICE, message=SBAS;IMES;QZSS)>",
            "<UBX(INF-NOTICE, message=GNSS OTP=GPS;GLO)>",
            "<UBX(INF-NOTICE, message=LLC=FFFFFFFF-FFFFFFFF-FFFFFFFF-FFFFFFFF-FFFFFFFD)>",
            "<UBX(INF-NOTICE, message=ANTSUPERV=AC SD PDoS SR)>",
            "<UBX(INF-NOTICE, message=ANTSTATUS=OK)>",
            "<UBX(INF-NOTICE, message=PF=3FF)>",
        )

        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamINF)
        while raw is not None:
            (raw, parsed) = ubxreader.read()
            # print(f"{i} = {parsed}")
            if raw is not None:
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1

    def testMONLOG(
        self,
    ):  # test stream of UBX MON messages
        EXPECTED_RESULTS = (
            "<UBX(MON-MSGPP, msg1_01=0, msg1_02=0, msg1_03=0, msg1_04=0, msg1_05=0, msg1_06=0, msg1_07=0, msg1_08=0, msg2_01=0, msg2_02=0, msg2_03=0, msg2_04=0, msg2_05=0, msg2_06=0, msg2_07=0, msg2_08=0, msg3_01=0, msg3_02=0, msg3_03=0, msg3_04=0, msg3_05=0, msg3_06=0, msg3_07=0, msg3_08=0, msg4_01=123, msg4_02=0, msg4_03=0, msg4_04=0, msg4_05=0, msg4_06=0, msg4_07=0, msg4_08=0, msg5_01=0, msg5_02=0, msg5_03=0, msg5_04=0, msg5_05=0, msg5_06=0, msg5_07=0, msg5_08=0, msg6_01=0, msg6_02=0, msg6_03=0, msg6_04=0, msg6_05=0, msg6_06=0, msg6_07=0, msg6_08=0, skipped_01=0, skipped_02=0, skipped_03=0, skipped_04=0, skipped_05=0, skipped_06=0)>",
            "<UBX(MON-TXBUF, pending_01=0, pending_02=0, pending_03=0, pending_04=0, pending_05=0, pending_06=0, usage_01=0, usage_02=0, usage_03=0, usage_04=0, usage_05=0, usage_06=0, peakUsage_01=0, peakUsage_02=12, peakUsage_03=0, peakUsage_04=24, peakUsage_05=0, peakUsage_06=0, tUsage=0, tPeakUsage=24, limit=0, lem=0, alloc=0, reserved0=0)>",
            "<UBX(MON-RXBUF, pending_01=0, pending_02=0, pending_03=0, pending_04=0, pending_05=0, pending_06=0, usage_01=0, usage_02=0, usage_03=0, usage_04=0, usage_05=0, usage_06=0, peakUsage_01=0, peakUsage_02=0, peakUsage_03=0, peakUsage_04=4, peakUsage_05=0, peakUsage_06=0)>",
            "<UBX(MON-IO, rxBytes=0, txBytes=0, parityErrs=0, framingErrs=0, overrunErrs=0, breakCond=0, rxBusy=0, txBusy=0, reserved1=0)>",
            "<UBX(MON-COMMS, version=0, nPorts=2, mem=0, alloc=0, reserved0=0, protIds_01=0, protIds_02=1, protIds_03=5, protIds_04=255, portId_01=256, txPending_01=0, txBytes_01=18620, txUsage_01=0, txPeakUsage_01=12, rxPending_01=0, rxBytes_01=0, rxUsage_01=0, rxPeakUsage_01=0, overrunErrs_01=0, msgs_01_01=0, msgs_01_02=0, msgs_01_03=0, msgs_01_04=0, reserved1_01=0, skipped_01=0, portId_02=768, txPending_02=0, txBytes_02=13105, txUsage_02=0, txPeakUsage_02=24, rxPending_02=0, rxBytes_02=1937, rxUsage_02=0, rxPeakUsage_02=4, overrunErrs_02=0, msgs_02_01=123, msgs_02_02=0, msgs_02_03=0, msgs_02_04=0, reserved1_02=0, skipped_02=0)>",
            "<UBX(MON-HW, pinSel=b'\\xc1\\x81\\x01\\x00', pinBank=b'\\x00\\x00\\x00\\x00', pinDir=b'\\x00\\x00\\x01\\x00', pinVal=b'\\xdb\\xe7\\x01\\x00', noisePerMS=92, agcCnt=1404, aStatus=2, aPower=1, rtcCalib=0, safeBoot=0, jammingState=0, xtalAbsent=0, reserved0=133, usedMask=b'\\xbe\\x7f\\x01\\x00', VP_01=b'\\xff', VP_02=b'\\x00', VP_03=b'\\x01', VP_04=b'\\x03', VP_05=b'\\x02', VP_06=b'\\x10', VP_07=b'\\xff', VP_08=b'\\x12', VP_09=b'\\x13', VP_10=b'\\x14', VP_11=b'\\x15', VP_12=b'\\x0e', VP_13=b'\\x0a', VP_14=b'\\x0b', VP_15=b'\\x0f', VP_16=b'\\xff', VP_17=b'\\x35', jamInd=7, reserved1=31390, pinIrq=b'\\x00\\x00\\x00\\x00', pullH=b'\\xdb\\xfb\\x00\\x00', pullL=b'\\x00\\x00\\x00\\x00')>",
            "<UBX(MON-HW2, ofsI=26, magI=163, ofsQ=23, magQ=163, cfgSource=0, reserved0=0, lowLevCfg=4294967295, reserved1=18446744073709551615, postStatus=0, reserved2=0)>",
            "<UBX(MON-HW3, version=0, nPins=17, rtcCalib=0, safeBoot=0, xtalAbsent=0, hwVersion=b'\\x30\\x30\\x31\\x39\\x30\\x30\\x30\\x30\\x00\\x00', reserved0=8056321, pinId_01=0, periphPIO_01=1, pinBank_01=0, direction_01=0, pinValue_01=1, vpManager_01=0, pioIrq_01=0, pioPullHigh_01=1, pioPullLow_01=0, VP_01=255, reserved1_01=0, pinId_02=256, periphPIO_02=0, pinBank_02=0, direction_02=0, pinValue_02=1, vpManager_02=1, pioIrq_02=0, pioPullHigh_02=1, pioPullLow_02=0, VP_02=0, reserved1_02=0, pinId_03=512, periphPIO_03=0, pinBank_03=0, direction_03=0, pinValue_03=0, vpManager_03=1, pioIrq_03=0, pioPullHigh_03=0, pioPullLow_03=0, VP_03=1, reserved1_03=0, pinId_04=768, periphPIO_04=0, pinBank_04=0, direction_04=0, pinValue_04=1, vpManager_04=1, pioIrq_04=0, pioPullHigh_04=1, pioPullLow_04=0, VP_04=3, reserved1_04=0, pinId_05=1024, periphPIO_05=0, pinBank_05=0, direction_05=0, pinValue_05=1, vpManager_05=1, pioIrq_05=0, pioPullHigh_05=1, pioPullLow_05=0, VP_05=2, reserved1_05=0, pinId_06=1280, periphPIO_06=0, pinBank_06=0, direction_06=0, pinValue_06=0, vpManager_06=1, pioIrq_06=0, pioPullHigh_06=0, pioPullLow_06=0, VP_06=16, reserved1_06=0, pinId_07=1536, periphPIO_07=1, pinBank_07=0, direction_07=0, pinValue_07=1, vpManager_07=0, pioIrq_07=0, pioPullHigh_07=1, pioPullLow_07=0, VP_07=255, reserved1_07=0, pinId_08=1792, periphPIO_08=1, pinBank_08=0, direction_08=0, pinValue_08=1, vpManager_08=1, pioIrq_08=0, pioPullHigh_08=1, pioPullLow_08=0, VP_08=18, reserved1_08=0, pinId_09=2048, periphPIO_09=1, pinBank_09=0, direction_09=0, pinValue_09=1, vpManager_09=1, pioIrq_09=0, pioPullHigh_09=1, pioPullLow_09=0, VP_09=19, reserved1_09=0, pinId_10=2304, periphPIO_10=0, pinBank_10=0, direction_10=0, pinValue_10=1, vpManager_10=1, pioIrq_10=0, pioPullHigh_10=1, pioPullLow_10=0, VP_10=20, reserved1_10=0, pinId_11=2560, periphPIO_11=0, pinBank_11=0, direction_11=0, pinValue_11=1, vpManager_11=1, pioIrq_11=0, pioPullHigh_11=0, pioPullLow_11=0, VP_11=21, reserved1_11=0, pinId_12=2816, periphPIO_12=0, pinBank_12=0, direction_12=0, pinValue_12=0, vpManager_12=1, pioIrq_12=0, pioPullHigh_12=1, pioPullLow_12=0, VP_12=14, reserved1_12=0, pinId_13=3072, periphPIO_13=0, pinBank_13=0, direction_13=0, pinValue_13=0, vpManager_13=1, pioIrq_13=0, pioPullHigh_13=1, pioPullLow_13=0, VP_13=10, reserved1_13=0, pinId_14=3328, periphPIO_14=0, pinBank_14=0, direction_14=0, pinValue_14=1, vpManager_14=1, pioIrq_14=0, pioPullHigh_14=1, pioPullLow_14=0, VP_14=11, reserved1_14=0, pinId_15=3584, periphPIO_15=0, pinBank_15=0, direction_15=0, pinValue_15=1, vpManager_15=1, pioIrq_15=0, pioPullHigh_15=1, pioPullLow_15=0, VP_15=15, reserved1_15=0, pinId_16=3840, periphPIO_16=1, pinBank_16=0, direction_16=0, pinValue_16=1, vpManager_16=0, pioIrq_16=0, pioPullHigh_16=1, pioPullLow_16=0, VP_16=255, reserved1_16=0, pinId_17=4096, periphPIO_17=1, pinBank_17=0, direction_17=1, pinValue_17=1, vpManager_17=1, pioIrq_17=0, pioPullHigh_17=0, pioPullLow_17=0, VP_17=53, reserved1_17=0)>",
            "<UBX(MON-RF, version=0, nBlocks=1, reserved0=0, blockId_01=0, jammingState_01=0, antStatus_01=2, antPower_01=1, postStatus_01=0, reserved1_01=0, noisePerMS_01=92, agcCnt_01=1404, jamInd_01=7, ofsI_01=26, magI_01=163, ofsQ_01=23, magQ_01=163, reserved2_01=0)>",
            "<UBX(MON-SPAN, version=0, numRfBlocks=1, reserved0=0, spectrum_01=[40, 40, 40, 39, 41, 41, 41, 42, 40, 41, 39, 40, 40, 39, 38, 41, 41, 42, 42, 42, 42, 42, 42, 43, 43, 44, 44, 42, 44, 43, 46, 42, 65, 45, 52, 58, 51, 51, 50, 57, 51, 59, 59, 49, 63, 62, 54, 53, 55, 63, 59, 62, 62, 70, 68, 72, 75, 79, 88, 84, 92, 98, 99, 102, 107, 109, 113, 122, 126, 128, 129, 133, 136, 137, 140, 143, 139, 140, 138, 139, 141, 140, 138, 139, 139, 141, 140, 141, 143, 144, 145, 146, 147, 149, 154, 154, 153, 155, 157, 159, 159, 161, 161, 162, 163, 162, 161, 159, 160, 157, 156, 154, 153, 153, 148, 148, 145, 145, 141, 142, 141, 139, 138, 137, 136, 136, 136, 138, 138, 137, 137, 136, 137, 136, 138, 139, 137, 138, 138, 138, 138, 137, 135, 137, 138, 136, 137, 139, 136, 137, 137, 138, 137, 137, 136, 135, 136, 136, 135, 136, 136, 136, 135, 135, 137, 137, 138, 138, 139, 140, 140, 141, 143, 140, 143, 144, 145, 145, 145, 147, 145, 145, 147, 145, 143, 144, 142, 142, 142, 138, 135, 134, 128, 125, 120, 117, 111, 105, 100, 97, 90, 85, 77, 73, 68, 63, 60, 56, 54, 54, 52, 52, 50, 51, 49, 48, 48, 46, 46, 46, 44, 47, 45, 45, 45, 44, 43, 45, 43, 42, 42, 41, 40, 42, 41, 41, 39, 40, 42, 41, 39, 41, 40, 41, 41, 40, 39, 40, 40, 39, 39, 40, 40, 40, 40, 39], span_01=128000000, res_01=500000, center_01=1583400000, pga_01=12, reserved1_01=0)>",
            "<UBX(MON-RF, version=0, nBlocks=1, reserved0=0, blockId_01=0, jammingState_01=3, antStatus_01=2, antPower_01=1, postStatus_01=0, reserved1_01=0, noisePerMS_01=92, agcCnt_01=1404, jamInd_01=255, ofsI_01=26, magI_01=163, ofsQ_01=23, magQ_01=163, reserved2_01=0)>",
        )

        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamMON)
        while raw is not None:
            (raw, parsed) = ubxreader.read()
            if raw is not None:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1

    def testCFGLOG(
        self,
    ):  # test stream of UBX CFG messages
        EXPECTED_RESULTS = (
            "<UBX(CFG-ANT, svcs=1, scd=0, ocd=0, pdwnOnSCD=0, recovery=0, pinSwitch=16, pinSCD=15, pinOCD=8, reconfig=0)>",
            "<UBX(CFG-BATCH, version=0, enable=0, extraPvt=0, extraOdo=0, pioEnable=0, pioActiveLow=0, bufSize=0, notifThrs=0, pioId=0, reserved0=0)>",
            "<UBX(CFG-DAT, datumNum=0, datumName=b'WGS84\\x00', majA=6378137.0, flat=298.257223563, dX=0.0, dY=0.0, dZ=0.0, rotX=0.0, rotY=0.0, rotZ=0.0, scale=0.0)>",
            "<UBX(CFG-GEOFENCE, version=0, numFences=0, confLvl=0, reserved0=0, pioEnabled=0, pinPolarity=0, pin=0, reserved1=0)>",
            "<UBX(CFG-GNSS, msgVer=0, numTrkChHw=42, numTrkChUse=42, numConfigBlocks=6, gnssId_01=GPS, resTrkCh_01=8, maxTrkCh_01=16, reserved0_01=0, enable_01=1, sigCfMask_01=1, gnssId_02=SBAS, resTrkCh_02=3, maxTrkCh_02=3, reserved0_02=0, enable_02=1, sigCfMask_02=1, gnssId_03=Galileo, resTrkCh_03=8, maxTrkCh_03=12, reserved0_03=0, enable_03=1, sigCfMask_03=1, gnssId_04=BeiDou, resTrkCh_04=2, maxTrkCh_04=5, reserved0_04=0, enable_04=1, sigCfMask_04=1, gnssId_05=QZSS, resTrkCh_05=3, maxTrkCh_05=4, reserved0_05=0, enable_05=1, sigCfMask_05=5, gnssId_06=GLONASS, resTrkCh_06=8, maxTrkCh_06=12, reserved0_06=0, enable_06=1, sigCfMask_06=1)>",
            "<UBX(CFG-ITFM, bbThreshold=3, cwThreshold=15, algorithmBits=1487190, enable=0, generalBits=798, antSetting=0, enable2=0)>",
            "<UBX(CFG-LOGFILTER, version=1, recordEnabled=0, psmOncePerWakupEnabled=0, applyAllFilterSettings=0, minInterval=0, timeThreshold=0, speedThreshold=0, positionThreshold=0)>",
            "<UBX(CFG-NAV5, dyn=1, minEl=1, posFixMode=1, drLim=1, posMask=1, timeMask=1, staticHoldMask=1, dgpsMask=1, cnoThreshold=1, utc=1, dynModel=0, fixMode=3, fixedAlt=0.0, fixedAltVar=1.0, minElev=5, drLimit=0, pDop=25.0, tDop=25.0, pAcc=100, tAcc=350, staticHoldThresh=0, dgnssTimeOut=60, cnoThreshNumSVs=0, cnoThresh=0, reserved0=0, staticHoldMaxDist=0, utcStandard=0, reserved1=0)>",
            "<UBX(CFG-NAVX5, version=2, minMax=1, minCno=1, initial3dfix=1, wknRoll=1, ackAid=1, ppp=1, aop=1, adr=0, sigAttenComp=1, reserved0=512, minSVs=3, maxSVs=32, minCNO=6, reserved1=0, iniFix3D=0, reserved2=1, ackAiding=0, wknRollover=2117, sigAttenCompMode=0, reserved3=1, reserved4=0, reserved5=257, usePPP=0, aopCfg=1, reserved6=25600, aopOrbMaxErr=100, reserved7=1179904, reserved8=0, useAdr=0)>",
            "<UBX(CFG-NMEA, posFilt=0, mskPosFilt=0, timeFilt=0, dateFilt=0, gpsOnlyFilter=0, trackFilt=0, nmeaVersion=65, numSV=0, compat=0, consider=1, limit82=0, highPrec=0, gps=0, sbas=0, galileo=0, qzss=0, glonass=0, beidou=0, svNumbering=0, mainTalkerId=0, gsvTalkerId=0, version=1, bdsTalkerId=b'\\x00\\x00', reserved1=0)>",
            "<UBX(CFG-ODO, version=0, reserved0=0, useODO=0, useCOG=0, outLPVel=0, outLPCog=0, profile=0, reserved1=112258400714752, cogMaxSpeed=1.0, cogMaxPosAcc=50, reserved2=0, velLpGain=153, cogLpGain=76, reserved3=0)>",
            "<UBX(CFG-PM2, version=2, reserved0=6, maxStartupStateDur=0, reserved1=0, optTarget=0, extintSel=0, extintWake=0, extintBackup=0, extintInactive=0, limitPeakCurr=0, waitTimeFix=0, updateRTC=0, updateEPH=1, doNotEnterOff=0, operationMode=1, updatePeriod=10000, searchPeriod=10000, gridOffset=0, onTime=0, minAcqTime=0, reserved2=27909959734875496726556525041030975214059820, extintInactivityMs=0)>",
            "<UBX(CFG-PMS, version=0, powerSetupValue=1, period=0, onTime=0, reserved0=0)>",
            "<UBX(CFG-PRT, portID=3, reserved0=0, enable=0, pol=0, pin=0, thres=0, charLen=0, parity=0, nStopBits=0, baudRate=0, inUBX=1, inNMEA=1, inRTCM=0, inRTCM3=1, outUBX=1, outNMEA=1, outRTCM3=0, extendedTxTimeout=0, reserved1=0)>",
            "<UBX(CFG-PWR, version=247, reserved1=16548860, state=38273024)>",
            "<UBX(CFG-RATE, measRate=1000, navRate=1, timeRef=1)>",
            "<UBX(CFG-RINV, dump=0, binary=0, data_01=78, data_02=111, data_03=116, data_04=105, data_05=99, data_06=101, data_07=58, data_08=32, data_09=110, data_10=111, data_11=32, data_12=100, data_13=97, data_14=116, data_15=97, data_16=32, data_17=115, data_18=97, data_19=118, data_20=101, data_21=100, data_22=33)>",
            "<UBX(CFG-RXM, reserved0=72, lpMode=0)>",
            "<UBX(CFG-SBAS, enabled=1, test=0, range=1, diffCorr=1, integrity=0, maxSBAS=0, PRN152=0, PRN153=0, PRN154=0, PRN155=0, PRN156=0, PRN157=0, PRN158=0, PRN120=0, PRN121=0, PRN122=0, PRN123=1, PRN124=0, PRN125=0, PRN126=0, PRN127=1, PRN128=1, PRN129=1, PRN130=0, PRN131=1, PRN132=0, PRN133=1, PRN134=0, PRN135=0, PRN136=1, PRN137=1, PRN138=1, PRN139=0, PRN140=0, PRN141=0, PRN142=0, PRN143=0, PRN144=0, PRN145=0, PRN146=0, PRN147=0, PRN148=0, PRN149=0, PRN150=0, PRN151=0)>",
            "<UBX(CFG-TP5, tpIdx=0, version=1, reserved0=0, antCableDelay=50, rfGroupDelay=0, freqPeriod=1000000, freqPeriodLock=1000000, pulseLenRatio=0, pulseLenRatioLock=100000, userConfigDelay=0, active=1, lockGnssFreq=1, lockedOtherSet=1, isFreq=0, isLength=1, alignToTow=1, polarity=1, gridUtcGnss=0, syncMode=0)>",
            "<UBX(CFG-USB, vendorID=5446, productID=425, reserved0=0, reserved1=0, powerConsumption=0, reEnum=0, powerMode=1, vendorString=b'\\x75\\x2d\\x62\\x6c\\x6f\\x78\\x20\\x41\\x47\\x20\\x2d\\x20\\x77\\x77\\x77\\x2e\\x75\\x2d\\x62\\x6c\\x6f\\x78\\x2e\\x63\\x6f\\x6d\\x00\\x00\\x00\\x00\\x00\\x00', productString=b'\\x75\\x2d\\x62\\x6c\\x6f\\x78\\x20\\x47\\x4e\\x53\\x53\\x20\\x72\\x65\\x63\\x65\\x69\\x76\\x65\\x72\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00', serialNumber=b'\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00\\x00')>",
        )

        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamCFG)
        while raw is not None:
            (raw, parsed) = ubxreader.read()
            if raw is not None:
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1

    def testMIXED(self):  # TODO test mixed UBX/NMEA stream with no protfilter
        dirname = os.path.dirname(__file__)
        stream = open(os.path.join(dirname, "pygpsdata-MIXED3.log"), "rb")
        EXPECTED_RESULTS = (
            "<UBX(NAV-PVT, iTOW=08:02:47, year=2022, month=1, day=18, hour=8, min=2, second=47, validDate=1, validTime=1, fullyResolved=1, validMag=0, tAcc=37, nano=-175377, fixType=3, gnssFixOk=1, difSoln=0, psmState=0, headVehValid=0, carrSoln=0, confirmedAvai=1, confirmedDate=1, confirmedTime=1, numSV=7, lon=-2.2402308, lat=53.4507167, height=85162, hMSL=36678, hAcc=8927, vAcc=7261, velN=-109, velE=-4, velD=7, gSpeed=109, headMot=0.0, sAcc=872, headAcc=103.82556, pDOP=2.44, invalidLlh=0, lastCorrectionAge=0, reserved0=793711598, headVeh=0.0, magDec=0.0, magAcc=0.0)>",
            "<NMEA(GPGGA, time=08:02:47, lat=53.4507166667, NS=N, lon=-2.2402308333, EW=W, quality=1, numSV=7, HDOP=1.63, alt=36.7, altUnit=M, sep=48.5, sepUnit=M, diffAge=, diffStation=)>",
            "<NMEA(GPGSA, opMode=A, navMode=3, svid_01=2, svid_02=13, svid_03=20, svid_04=7, svid_05=5, svid_06=30, svid_07=9, svid_08=, svid_09=, svid_10=, svid_11=, svid_12=, PDOP=2.44, HDOP=1.63, VDOP=1.82)>",
            "<UBX(NAV-PVT, iTOW=08:02:48, year=2022, month=1, day=18, hour=8, min=2, second=48, validDate=1, validTime=1, fullyResolved=1, validMag=0, tAcc=37, nano=-175363, fixType=3, gnssFixOk=1, difSoln=0, psmState=0, headVehValid=0, carrSoln=0, confirmedAvai=1, confirmedDate=1, confirmedTime=1, numSV=7, lon=-2.2402314, lat=53.4507186, height=85266, hMSL=36782, hAcc=8931, vAcc=7312, velN=72, velE=-12, velD=10, gSpeed=73, headMot=0.0, sAcc=861, headAcc=103.86284, pDOP=2.44, invalidLlh=0, lastCorrectionAge=0, reserved0=793711598, headVeh=0.0, magDec=0.0, magAcc=0.0)>",
            "<NMEA(GPGGA, time=08:02:48, lat=53.4507186667, NS=N, lon=-2.2402315, EW=W, quality=1, numSV=7, HDOP=1.63, alt=36.8, altUnit=M, sep=48.5, sepUnit=M, diffAge=, diffStation=)>",
            "<NMEA(GPGSA, opMode=A, navMode=3, svid_01=2, svid_02=13, svid_03=20, svid_04=7, svid_05=5, svid_06=30, svid_07=9, svid_08=, svid_09=, svid_10=, svid_11=, svid_12=, PDOP=2.44, HDOP=1.63, VDOP=1.82)>",
            "<UBX(NAV-PVT, iTOW=08:02:49, year=2022, month=1, day=18, hour=8, min=2, second=49, validDate=1, validTime=1, fullyResolved=1, validMag=0, tAcc=38, nano=-175349, fixType=3, gnssFixOk=1, difSoln=0, psmState=0, headVehValid=0, carrSoln=0, confirmedAvai=1, confirmedDate=1, confirmedTime=1, numSV=7, lon=-2.2402319, lat=53.4507207, height=85325, hMSL=36841, hAcc=8930, vAcc=7363, velN=110, velE=-25, velD=1, gSpeed=113, headMot=0.0, sAcc=855, headAcc=103.8999, pDOP=2.44, invalidLlh=0, lastCorrectionAge=0, reserved0=793711598, headVeh=0.0, magDec=0.0, magAcc=0.0)>",
        )
        i = 0
        raw = 0
        ubxreader = UBXReader(stream)
        while raw is not None:
            (raw, parsed) = ubxreader.read()
            if raw is not None:
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
        stream.close()

    def testMIXEDUBXFILT(self):  # TODO test mixed UBX/NMEA stream with UBX protfilter
        dirname = os.path.dirname(__file__)
        stream = open(os.path.join(dirname, "pygpsdata-MIXED3.log"), "rb")
        EXPECTED_RESULTS = (
            "<UBX(NAV-PVT, iTOW=08:02:47, year=2022, month=1, day=18, hour=8, min=2, second=47, validDate=1, validTime=1, fullyResolved=1, validMag=0, tAcc=37, nano=-175377, fixType=3, gnssFixOk=1, difSoln=0, psmState=0, headVehValid=0, carrSoln=0, confirmedAvai=1, confirmedDate=1, confirmedTime=1, numSV=7, lon=-2.2402308, lat=53.4507167, height=85162, hMSL=36678, hAcc=8927, vAcc=7261, velN=-109, velE=-4, velD=7, gSpeed=109, headMot=0.0, sAcc=872, headAcc=103.82556, pDOP=2.44, invalidLlh=0, lastCorrectionAge=0, reserved0=793711598, headVeh=0.0, magDec=0.0, magAcc=0.0)>",
            "<UBX(NAV-PVT, iTOW=08:02:48, year=2022, month=1, day=18, hour=8, min=2, second=48, validDate=1, validTime=1, fullyResolved=1, validMag=0, tAcc=37, nano=-175363, fixType=3, gnssFixOk=1, difSoln=0, psmState=0, headVehValid=0, carrSoln=0, confirmedAvai=1, confirmedDate=1, confirmedTime=1, numSV=7, lon=-2.2402314, lat=53.4507186, height=85266, hMSL=36782, hAcc=8931, vAcc=7312, velN=72, velE=-12, velD=10, gSpeed=73, headMot=0.0, sAcc=861, headAcc=103.86284, pDOP=2.44, invalidLlh=0, lastCorrectionAge=0, reserved0=793711598, headVeh=0.0, magDec=0.0, magAcc=0.0)>",
            "<UBX(NAV-PVT, iTOW=08:02:49, year=2022, month=1, day=18, hour=8, min=2, second=49, validDate=1, validTime=1, fullyResolved=1, validMag=0, tAcc=38, nano=-175349, fixType=3, gnssFixOk=1, difSoln=0, psmState=0, headVehValid=0, carrSoln=0, confirmedAvai=1, confirmedDate=1, confirmedTime=1, numSV=7, lon=-2.2402319, lat=53.4507207, height=85325, hMSL=36841, hAcc=8930, vAcc=7363, velN=110, velE=-25, velD=1, gSpeed=113, headMot=0.0, sAcc=855, headAcc=103.8999, pDOP=2.44, invalidLlh=0, lastCorrectionAge=0, reserved0=793711598, headVeh=0.0, magDec=0.0, magAcc=0.0)>",
        )
        i = 0
        raw = 0
        ubxreader = UBXReader(stream, protfilter=UBX_PROTOCOL)
        while raw is not None:
            (raw, parsed) = ubxreader.read()
            if raw is not None:
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
        stream.close()

    def testMIXEDNMEAFILT(self):  # TODO test mixed UBX/NMEA stream with NMEA protfilter
        dirname = os.path.dirname(__file__)
        stream = open(os.path.join(dirname, "pygpsdata-MIXED3.log"), "rb")
        EXPECTED_RESULTS = (
            "<NMEA(GPGGA, time=08:02:47, lat=53.4507166667, NS=N, lon=-2.2402308333, EW=W, quality=1, numSV=7, HDOP=1.63, alt=36.7, altUnit=M, sep=48.5, sepUnit=M, diffAge=, diffStation=)>",
            "<NMEA(GPGSA, opMode=A, navMode=3, svid_01=2, svid_02=13, svid_03=20, svid_04=7, svid_05=5, svid_06=30, svid_07=9, svid_08=, svid_09=, svid_10=, svid_11=, svid_12=, PDOP=2.44, HDOP=1.63, VDOP=1.82)>",
            "<NMEA(GPGGA, time=08:02:48, lat=53.4507186667, NS=N, lon=-2.2402315, EW=W, quality=1, numSV=7, HDOP=1.63, alt=36.8, altUnit=M, sep=48.5, sepUnit=M, diffAge=, diffStation=)>",
            "<NMEA(GPGSA, opMode=A, navMode=3, svid_01=2, svid_02=13, svid_03=20, svid_04=7, svid_05=5, svid_06=30, svid_07=9, svid_08=, svid_09=, svid_10=, svid_11=, svid_12=, PDOP=2.44, HDOP=1.63, VDOP=1.82)>",
        )
        i = 0
        raw = 0
        ubxreader = UBXReader(stream, protfilter=NMEA_PROTOCOL)
        while raw is not None:
            (raw, parsed) = ubxreader.read()
            if raw is not None:
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
        stream.close()

    def testMIXEDRTCM(
        self,
    ):  # test mixed stream of NMEA, UBX & RTCM messages with protfilter = 7
        EXPECTED_RESULTS = (
            "<NMEA(GNGLL, lat=32.0658325, NS=N, lon=34.773819, EW=E, time=08:41:58, status=A, posMode=D)>",
            "<RTCM(1005, DF002=1005, DF003=0, DF021=0, DF022=1, DF023=1, DF024=1, DF141=0, DF025=4444030.802800001, DF142=1, DF001_1=0, DF026=3085671.2349, DF364=0, DF027=3366658.256)>",
            "<RTCM(4072, DF002=4072, Not_Yet_Implemented)>",
            "<RTCM(1077, DF002=1077, DF003=0, GNSSEpoch=204137001, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=760738918298550272, NSat=10, DF395=1073807360, NSig=2, DF396=1044459, NCell=17, DF397_01=75, DF397_02=75, DF397_03=81, DF397_04=72, DF397_05=67, DF397_06=80, DF397_07=75, DF397_08=82, DF397_09=75, DF397_10=71, GNSSSpecific_01=0, GNSSSpecific_02=0, GNSSSpecific_03=0, GNSSSpecific_04=0, GNSSSpecific_05=0, GNSSSpecific_06=0, GNSSSpecific_07=0, GNSSSpecific_08=0, GNSSSpecific_09=0, GNSSSpecific_10=0, DF398_01=0.005859375, DF398_02=0.5341796875, DF398_03=0.7626953125, DF398_04=0.138671875, DF398_05=0.5498046875, DF398_06=0.11328125, DF398_07=0.8037109375, DF398_08=0.1025390625, DF398_09=0.521484375, DF398_10=0.345703125, DF399_01=-178, DF399_02=-304, DF399_03=-643, DF399_04=477, DF399_05=-52, DF399_06=645, DF399_07=529, DF399_08=643, DF399_09=-428, DF399_10=-181, DF405_01=0.00014309026300907135, DF405_02=0.00014183297753334045, DF405_03=0.0003883279860019684, DF405_04=0.00038741156458854675, DF405_05=-0.0004838351160287857, DF405_06=-0.00046883709728717804, DF405_07=0.0003478657454252243, DF405_08=0.0002196934074163437, DF405_09=0.00021521002054214478, DF405_10=-0.00018852390348911285, DF405_11=-0.00018319115042686462, DF405_12=-0.00010087713599205017, DF405_13=-9.844452142715454e-05, DF405_14=0.00047875382006168365, DF405_15=0.00043664872646331787, DF405_16=-0.0003105681389570236, DF405_17=-0.00030865520238876343, DF406_01=0.00014193402603268623, DF406_02=0.00014339853078126907, DF406_03=0.00039040297269821167, DF406_04=0.00038743019104003906, DF406_05=-0.0004843934439122677, DF406_06=-0.00046825408935546875, DF406_07=0.0003473707474768162, DF406_08=0.00021758908405900002, DF406_09=0.00021597417071461678, DF406_10=-0.00018658116459846497, DF406_11=-0.00018350128084421158, DF406_12=-9.993184357881546e-05, DF406_13=-9.724870324134827e-05, DF406_14=0.0004128236323595047, DF406_15=0.0004355977289378643, DF406_16=-0.0003112703561782837, DF406_17=-0.00030898721888661385, DF407_01=341, DF407_02=341, DF407_03=341, DF407_04=341, DF407_05=341, DF407_06=341, DF407_07=341, DF407_08=341, DF407_09=341, DF407_10=341, DF407_11=341, DF407_12=341, DF407_13=341, DF407_14=295, DF407_15=341, DF407_16=341, DF407_17=341, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF408_01=45.0, DF408_02=38.0, DF408_03=43.0, DF408_04=39.0, DF408_05=39.0, DF408_06=37.0, DF408_07=45.0, DF408_08=46.0, DF408_09=46.0, DF408_10=39.0, DF408_11=34.0, DF408_12=45.0, DF408_13=38.0, DF408_14=31.0, DF408_15=45.0, DF408_16=46.0, DF408_17=41.0, DF404_01=-0.9231, DF404_02=-0.9194, DF404_03=-0.8321000000000001, DF404_04=-0.8326, DF404_05=-0.4107, DF404_06=-0.4072, DF404_07=0.2451, DF404_08=-0.0693, DF404_09=-0.0684, DF404_10=0.9390000000000001, DF404_11=0.9417000000000001, DF404_12=0.2384, DF404_13=0.2416, DF404_14=0.6636000000000001, DF404_15=-0.9556, DF404_16=-0.21480000000000002, DF404_17=-0.2174)>",
            "<RTCM(1087, DF002=1087, DF003=0, GNSSEpoch=310554457, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=4039168114821169152, NSat=7, DF395=1090519040, NSig=2, DF396=16382, NCell=13, DF397_01=69, DF397_02=64, DF397_03=73, DF397_04=76, DF397_05=66, DF397_06=70, DF397_07=78, GNSSSpecific_01=12, GNSSSpecific_02=13, GNSSSpecific_03=8, GNSSSpecific_04=5, GNSSSpecific_05=0, GNSSSpecific_06=7, GNSSSpecific_07=10, DF398_01=0.6337890625, DF398_02=0.3427734375, DF398_03=0.25390625, DF398_04=0.310546875, DF398_05=0.5126953125, DF398_06=0.8271484375, DF398_07=0.8837890625, DF399_01=-665, DF399_02=29, DF399_03=672, DF399_04=-573, DF399_05=-211, DF399_06=312, DF399_07=317, DF405_01=0.00024936161935329437, DF405_02=0.0002511627972126007, DF405_03=-4.678964614868164e-05, DF405_04=-5.141831934452057e-05, DF405_05=1.1144205927848816e-05, DF405_06=2.15042382478714e-05, DF405_07=0.00047079287469387054, DF405_08=0.0004794951528310776, DF405_09=-0.0003879182040691376, DF405_10=-0.00037603825330734253, DF405_11=0.0002771839499473572, DF405_12=0.0002871435135602951, DF405_13=-0.00023611821234226227, DF406_01=0.00024937279522418976, DF406_02=0.00025077443569898605, DF406_03=-4.834495484828949e-05, DF406_04=-5.1246024668216705e-05, DF406_05=1.1149328202009201e-05, DF406_06=2.1803192794322968e-05, DF406_07=0.00047026341781020164, DF406_08=0.0004848274402320385, DF406_09=-0.0003876127302646637, DF406_10=-0.0003757951781153679, DF406_11=0.0002778824418783188, DF406_12=0.0002880701795220375, DF406_13=-0.00023698341101408005, DF407_01=341, DF407_02=341, DF407_03=340, DF407_04=340, DF407_05=341, DF407_06=341, DF407_07=340, DF407_08=341, DF407_09=341, DF407_10=341, DF407_11=341, DF407_12=341, DF407_13=340, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF408_01=47.0, DF408_02=40.0, DF408_03=47.0, DF408_04=42.0, DF408_05=47.0, DF408_06=39.0, DF408_07=36.0, DF408_08=33.0, DF408_09=48.0, DF408_10=43.0, DF408_11=48.0, DF408_12=40.0, DF408_13=41.0, DF404_01=-0.8193, DF404_02=-0.8173, DF404_03=0.8539, DF404_04=0.8501000000000001, DF404_05=0.7333000000000001, DF404_06=0.7311000000000001, DF404_07=-0.24930000000000002, DF404_08=-0.2543, DF404_09=-0.21580000000000002, DF404_10=-0.21780000000000002, DF404_11=0.3924, DF404_12=0.3947, DF404_13=0.6146)>",
            "<RTCM(1097, DF002=1097, DF003=0, GNSSEpoch=204137001, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=216181732825628672, NSat=5, DF395=1073872896, NSig=2, DF396=1023, NCell=10, DF397_01=79, DF397_02=84, DF397_03=89, DF397_04=78, DF397_05=83, GNSSSpecific_01=0, GNSSSpecific_02=0, GNSSSpecific_03=0, GNSSSpecific_04=0, GNSSSpecific_05=0, DF398_01=0.15625, DF398_02=0.2509765625, DF398_03=0.3544921875, DF398_04=0.37109375, DF398_05=0.259765625, DF399_01=-198, DF399_02=-516, DF399_03=423, DF399_04=63, DF399_05=-384, DF405_01=-4.5398250222206116e-05, DF405_02=-2.8252601623535156e-05, DF405_03=-0.00034597329795360565, DF405_04=-0.0003268253058195114, DF405_05=0.0004809703677892685, DF405_06=0.0005012489855289459, DF405_07=-0.00013696029782295227, DF405_08=-0.0001260414719581604, DF405_09=-1.8440186977386475e-05, DF405_10=-3.041699528694153e-06, DF406_01=-4.44464385509491e-05, DF406_02=-2.835458144545555e-05, DF406_03=-0.0003525479696691036, DF406_04=-0.0003263736143708229, DF406_05=0.00048203859478235245, DF406_06=0.0005008447915315628, DF406_07=-0.0001375703141093254, DF406_08=-0.00012635625898838043, DF406_09=-1.8037855625152588e-05, DF406_10=-3.2926909625530243e-06, DF407_01=341, DF407_02=341, DF407_03=341, DF407_04=341, DF407_05=341, DF407_06=341, DF407_07=341, DF407_08=341, DF407_09=341, DF407_10=341, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF408_01=46.0, DF408_02=49.0, DF408_03=41.0, DF408_04=43.0, DF408_05=43.0, DF408_06=43.0, DF408_07=45.0, DF408_08=49.0, DF408_09=43.0, DF408_10=47.0, DF404_01=-0.5806, DF404_02=-0.5831000000000001, DF404_03=-0.7947000000000001, DF404_04=-0.7943, DF404_05=0.7243, DF404_06=0.7174, DF404_07=0.5534, DF404_08=0.5545, DF404_09=-0.7726000000000001, DF404_10=-0.7733)>",
            "<RTCM(1127, DF002=1127, DF003=0, GNSSEpoch=204123001, DF393=0, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=198178247981137920, NSat=10, DF395=1074003968, NSig=2, DF396=387754, NCell=11, DF397_01=129, DF397_02=132, DF397_03=126, DF397_04=75, DF397_05=81, DF397_06=84, DF397_07=78, DF397_08=74, DF397_09=130, DF397_10=86, GNSSSpecific_01=0, GNSSSpecific_02=0, GNSSSpecific_03=0, GNSSSpecific_04=0, GNSSSpecific_05=0, GNSSSpecific_06=0, GNSSSpecific_07=0, GNSSSpecific_08=0, GNSSSpecific_09=0, GNSSSpecific_10=0, DF398_01=0.1171875, DF398_02=0.4814453125, DF398_03=0.3095703125, DF398_04=0.7255859375, DF398_05=0.41015625, DF398_06=0.5703125, DF398_07=0.5595703125, DF398_08=0.322265625, DF398_09=0.578125, DF398_10=0.673828125, DF399_01=-130, DF399_02=-58, DF399_03=-81, DF399_04=32, DF399_05=-398, DF399_06=436, DF399_07=-523, DF399_08=-65, DF399_09=-182, DF399_10=79, DF405_01=-0.0003885403275489807, DF405_02=0.00022730417549610138, DF405_03=0.0004036612808704376, DF405_04=0.00039606913924217224, DF405_05=-0.00016684085130691528, DF405_06=-4.75514680147171e-05, DF405_07=0.0003674682229757309, DF405_08=0.00026629865169525146, DF405_09=-0.0002502594143152237, DF405_10=-0.00011803768575191498, DF405_11=-0.0002937670797109604, DF406_01=-0.0003882073797285557, DF406_02=0.0002264929935336113, DF406_03=0.0004031979478895664, DF406_04=0.0003964221104979515, DF406_05=-0.00016694329679012299, DF406_06=-4.848744720220566e-05, DF406_07=0.00036971503868699074, DF406_08=0.0002654106356203556, DF406_09=-0.00025115441530942917, DF406_10=-0.00011868216097354889, DF406_11=-0.00029495684430003166, DF407_01=341, DF407_02=341, DF407_03=341, DF407_04=341, DF407_05=341, DF407_06=341, DF407_07=341, DF407_08=341, DF407_09=341, DF407_10=341, DF407_11=341, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF408_01=45.0, DF408_02=41.0, DF408_03=42.0, DF408_04=45.0, DF408_05=48.0, DF408_06=46.0, DF408_07=42.0, DF408_08=47.0, DF408_09=48.0, DF408_10=44.0, DF408_11=43.0, DF404_01=-0.5674, DF404_02=-0.612, DF404_03=-0.1384, DF404_04=-0.1332, DF404_05=0.5992000000000001, DF404_06=-0.7312000000000001, DF404_07=0.17320000000000002, DF404_08=-0.4308, DF404_09=-0.5975, DF404_10=-0.6733, DF404_11=0.6122000000000001)>",
            "<RTCM(1230, DF002=1230, DF003=0, DF421=1, DF001_3=0, DF422_1=0, DF422_2=0, DF422_3=0, DF422_4=0)>",
            "<UBX(NAV-PVT, iTOW=08:41:59, year=2022, month=2, day=8, hour=8, min=41, second=59, validDate=1, validTime=1, fullyResolved=1, validMag=0, tAcc=21, nano=360400, fixType=5, gnssFixOk=1, difSoln=1, psmState=0, headVehValid=0, carrSoln=0, confirmedAvai=1, confirmedDate=1, confirmedTime=1, numSV=31, lon=34.773819, lat=32.0658325, height=72134, hMSL=54642, hAcc=685, vAcc=484, velN=0, velE=0, velD=0, gSpeed=0, headMot=290.13822, sAcc=10, headAcc=20.15693, pDOP=99.99, invalidLlh=0, lastCorrectionAge=0, reserved0=860200482, headVeh=0.0, magDec=0.0, magAcc=0.0)>",
            "<NMEA(GNRMC, time=08:41:59, status=A, lat=32.0658325, NS=N, lon=34.773819, EW=E, spd=0.0, cog=, date=2022-02-08, mv=, mvEW=, posMode=D, navStatus=V)>",
        )
        dirname = os.path.dirname(__file__)
        stream = open(os.path.join(dirname, "pygpsdata-MIXED-RTCM3.log"), "rb")
        i = 0
        raw = 0
        ubr = UBXReader(stream, protfilter=7, labelmsm=0, quitonerror=ubt.ERR_RAISE)
        # stdout_saved = sys.stdout
        # sys.stdout = open("output.txt", "w")
        for raw, parsed in ubr:
            if raw is not None:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
        stream.close()
        # sys.stdout = stdout_saved

    def testMIXEDRTCM2(
        self,
    ):  # test mixed stream of NMEA, UBX & RTCM messages with protfilter = 3
        EXPECTED_RESULTS = (
            "<NMEA(GNGLL, lat=32.0658325, NS=N, lon=34.773819, EW=E, time=08:41:58, status=A, posMode=D)>",
            "<UBX(NAV-PVT, iTOW=08:41:59, year=2022, month=2, day=8, hour=8, min=41, second=59, validDate=1, validTime=1, fullyResolved=1, validMag=0, tAcc=21, nano=360400, fixType=5, gnssFixOk=1, difSoln=1, psmState=0, headVehValid=0, carrSoln=0, confirmedAvai=1, confirmedDate=1, confirmedTime=1, numSV=31, lon=34.773819, lat=32.0658325, height=72134, hMSL=54642, hAcc=685, vAcc=484, velN=0, velE=0, velD=0, gSpeed=0, headMot=290.13822, sAcc=10, headAcc=20.15693, pDOP=99.99, invalidLlh=0, lastCorrectionAge=0, reserved0=860200482, headVeh=0.0, magDec=0.0, magAcc=0.0)>",
            "<NMEA(GNRMC, time=08:41:59, status=A, lat=32.0658325, NS=N, lon=34.773819, EW=E, spd=0.0, cog=, date=2022-02-08, mv=, mvEW=, posMode=D, navStatus=V)>",
        )
        dirname = os.path.dirname(__file__)
        stream = open(os.path.join(dirname, "pygpsdata-MIXED-RTCM3.log"), "rb")
        i = 0
        raw = 0
        ubr = UBXReader(stream, protfilter=3, quitonerror=ubt.ERR_RAISE)
        for raw, parsed in ubr:
            if raw is not None:
                self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                i += 1
        stream.close()

    def testMIXEDRTCMBADCRC(
        self,
    ):  # test mixed stream of NMEA, UBX & RTCM messages with invalid RTCM CRC
        EXPECTED_ERROR = "RTCM3 message invalid - failed CRC: (.*)"
        dirname = os.path.dirname(__file__)
        stream = open(os.path.join(dirname, "pygpsdata-MIXED-RTCM3BADCRC.log"), "rb")
        i = 0
        raw = 0
        ubr = UBXReader(stream, protfilter=7, quitonerror=ubt.ERR_RAISE)
        with self.assertRaisesRegex(UBXParseError, EXPECTED_ERROR):
            for raw, parsed in ubr:
                if raw is not None:
                    # print(parsed)
                    i += 1
        stream.close()

    def testIterator(
        self,
    ):  # test iterator function with UBX data stream & parsebitfield = False
        EXPECTED_RESULTS = (
            "<UBX(MON-MSGPP, msg1_01=0, msg1_02=0, msg1_03=0, msg1_04=0, msg1_05=0, msg1_06=0, msg1_07=0, msg1_08=0, msg2_01=0, msg2_02=0, msg2_03=0, msg2_04=0, msg2_05=0, msg2_06=0, msg2_07=0, msg2_08=0, msg3_01=0, msg3_02=0, msg3_03=0, msg3_04=0, msg3_05=0, msg3_06=0, msg3_07=0, msg3_08=0, msg4_01=69, msg4_02=0, msg4_03=0, msg4_04=0, msg4_05=0, msg4_06=0, msg4_07=0, msg4_08=0, msg5_01=0, msg5_02=0, msg5_03=0, msg5_04=0, msg5_05=0, msg5_06=0, msg5_07=0, msg5_08=0, msg6_01=0, msg6_02=0, msg6_03=0, msg6_04=0, msg6_05=0, msg6_06=0, msg6_07=0, msg6_08=0, skipped_01=0, skipped_02=0, skipped_03=0, skipped_04=0, skipped_05=0, skipped_06=0)>",
            "<UBX(MON-TXBUF, pending_01=0, pending_02=0, pending_03=0, pending_04=0, pending_05=0, pending_06=0, usage_01=0, usage_02=2, usage_03=0, usage_04=0, usage_05=0, usage_06=0, peakUsage_01=0, peakUsage_02=12, peakUsage_03=0, peakUsage_04=25, peakUsage_05=0, peakUsage_06=0, tUsage=2, tPeakUsage=25, errors=b'\\x00', reserved0=0)>",
            "<UBX(MON-RXBUF, pending_01=0, pending_02=0, pending_03=0, pending_04=0, pending_05=0, pending_06=0, usage_01=0, usage_02=0, usage_03=0, usage_04=0, usage_05=0, usage_06=0, peakUsage_01=0, peakUsage_02=0, peakUsage_03=0, peakUsage_04=2, peakUsage_05=0, peakUsage_06=0)>",
            "<UBX(MON-IO, rxBytes=0, txBytes=0, parityErrs=0, framingErrs=0, overrunErrs=0, breakCond=0, rxBusy=0, txBusy=0, reserved1=0)>",
            "<UBX(MON-HW, pinSel=b'\\x00\\xf4\\x01\\x00', pinBank=b'\\x00\\x00\\x00\\x00', pinDir=b'\\x00\\x00\\x01\\x00', pinVal=b'\\xef\\xf7\\x00\\x00', noisePerMS=87, agcCnt=3042, aStatus=2, aPower=1, flags=b'\\x01', reserved0=132, usedMask=b'\\xff\\xeb\\x01\\x00', VP_01=b'\\x0a', VP_02=b'\\x0b', VP_03=b'\\x0c', VP_04=b'\\x0d', VP_05=b'\\x0e', VP_06=b'\\x0f', VP_07=b'\\x01', VP_08=b'\\x00', VP_09=b'\\x02', VP_10=b'\\x03', VP_11=b'\\xff', VP_12=b'\\x10', VP_13=b'\\xff', VP_14=b'\\x12', VP_15=b'\\x13', VP_16=b'\\x36', VP_17=b'\\x35', jamInd=5, reserved1=24303, pinIrq=b'\\x00\\x00\\x00\\x00', pullH=b'\\x80\\xf7\\x00\\x00', pullL=b'\\x00\\x00\\x00\\x00')>",
            "<UBX(MON-HW2, ofsI=4, magI=110, ofsQ=5, magQ=112, cfgSource=111, reserved0=1800, lowLevCfg=4294967295, reserved1=18446744073709551615, postStatus=0, reserved2=0)>",
        )
        i = 0
        ubxreader = UBXReader(self.streamITER, parsebitfield=False)
        for _, parsed in ubxreader:
            self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
            i += 1

    def testNMEAITERATE(self):  # UBXReader helper method
        EXPECTED_RESULTS = (
            "<UBX(MON-MSGPP, msg1_01=0, msg1_02=0, msg1_03=0, msg1_04=0, msg1_05=0, msg1_06=0, msg1_07=0, msg1_08=0, msg2_01=0, msg2_02=0, msg2_03=0, msg2_04=0, msg2_05=0, msg2_06=0, msg2_07=0, msg2_08=0, msg3_01=0, msg3_02=0, msg3_03=0, msg3_04=0, msg3_05=0, msg3_06=0, msg3_07=0, msg3_08=0, msg4_01=69, msg4_02=0, msg4_03=0, msg4_04=0, msg4_05=0, msg4_06=0, msg4_07=0, msg4_08=0, msg5_01=0, msg5_02=0, msg5_03=0, msg5_04=0, msg5_05=0, msg5_06=0, msg5_07=0, msg5_08=0, msg6_01=0, msg6_02=0, msg6_03=0, msg6_04=0, msg6_05=0, msg6_06=0, msg6_07=0, msg6_08=0, skipped_01=0, skipped_02=0, skipped_03=0, skipped_04=0, skipped_05=0, skipped_06=0)>",
            "<UBX(MON-TXBUF, pending_01=0, pending_02=0, pending_03=0, pending_04=0, pending_05=0, pending_06=0, usage_01=0, usage_02=2, usage_03=0, usage_04=0, usage_05=0, usage_06=0, peakUsage_01=0, peakUsage_02=12, peakUsage_03=0, peakUsage_04=25, peakUsage_05=0, peakUsage_06=0, tUsage=2, tPeakUsage=25, errors=b'\\x00', reserved0=0)>",
            "<UBX(MON-RXBUF, pending_01=0, pending_02=0, pending_03=0, pending_04=0, pending_05=0, pending_06=0, usage_01=0, usage_02=0, usage_03=0, usage_04=0, usage_05=0, usage_06=0, peakUsage_01=0, peakUsage_02=0, peakUsage_03=0, peakUsage_04=2, peakUsage_05=0, peakUsage_06=0)>",
            "<UBX(MON-IO, rxBytes=0, txBytes=0, parityErrs=0, framingErrs=0, overrunErrs=0, breakCond=0, rxBusy=0, txBusy=0, reserved1=0)>",
            "<UBX(MON-HW, pinSel=b'\\x00\\xf4\\x01\\x00', pinBank=b'\\x00\\x00\\x00\\x00', pinDir=b'\\x00\\x00\\x01\\x00', pinVal=b'\\xef\\xf7\\x00\\x00', noisePerMS=87, agcCnt=3042, aStatus=2, aPower=1, flags=b'\\x01', reserved0=132, usedMask=b'\\xff\\xeb\\x01\\x00', VP_01=b'\\x0a', VP_02=b'\\x0b', VP_03=b'\\x0c', VP_04=b'\\x0d', VP_05=b'\\x0e', VP_06=b'\\x0f', VP_07=b'\\x01', VP_08=b'\\x00', VP_09=b'\\x02', VP_10=b'\\x03', VP_11=b'\\xff', VP_12=b'\\x10', VP_13=b'\\xff', VP_14=b'\\x12', VP_15=b'\\x13', VP_16=b'\\x36', VP_17=b'\\x35', jamInd=5, reserved1=24303, pinIrq=b'\\x00\\x00\\x00\\x00', pullH=b'\\x80\\xf7\\x00\\x00', pullL=b'\\x00\\x00\\x00\\x00')>",
            "<UBX(MON-HW2, ofsI=4, magI=110, ofsQ=5, magQ=112, cfgSource=111, reserved0=1800, lowLevCfg=4294967295, reserved1=18446744073709551615, postStatus=0, reserved2=0)>",
        )
        i = 0
        ubr = UBXReader(self.streamITER, protfilter=3, parsebitfield=False)
        for _, parsed in ubr:
            self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
            i += 1

    def testUBXITERATE_ERR1(
        self,
    ):  # UBXReader helper method with bad checksum
        dirname = os.path.dirname(__file__)
        stream = open(os.path.join(dirname, "pygpsdata-MIXED3BADCK.log"), "rb")
        EXPECTED_ERROR = "Message checksum b's\\x8e' invalid - should be b's\\x8d'"
        with self.assertRaises(UBXParseError) as context:
            ubr = UBXReader(
                stream,
                protfilter=3,
                validate=VALCKSUM,
                msgmode=0,
                parsebitfield=True,
                quitonerror=ERR_RAISE,
            )
            for raw, parsed in ubr:
                pass
        self.assertTrue(EXPECTED_ERROR in str(context.exception))
        stream.close()

    def testUBXITERATE_ERR2(
        self,
    ):  # UBXReader helper method ignoring bad checksum and passing error handler
        dirname = os.path.dirname(__file__)
        stream = open(os.path.join(dirname, "pygpsdata-MIXED3BADCK.log"), "rb")
        EXPECTED_RESULTS = [
            "<UBX(NAV-PVT, iTOW=08:02:47, year=2022, month=1, day=18, hour=8, min=2, second=47, validDate=1, validTime=1, fullyResolved=1, validMag=0, tAcc=37, nano=-175377, fixType=3, gnssFixOk=1, difSoln=0, psmState=0, headVehValid=0, carrSoln=0, confirmedAvai=1, confirmedDate=1, confirmedTime=1, numSV=7, lon=-2.2402308, lat=53.4507167, height=85162, hMSL=36678, hAcc=8927, vAcc=7261, velN=-109, velE=-4, velD=7, gSpeed=109, headMot=0.0, sAcc=872, headAcc=103.82556, pDOP=2.44, invalidLlh=0, lastCorrectionAge=0, reserved0=793711598, headVeh=0.0, magDec=0.0, magAcc=0.0)>",
            "<NMEA(GPGGA, time=08:02:47, lat=53.4507166667, NS=N, lon=-2.2402308333, EW=W, quality=1, numSV=7, HDOP=1.63, alt=36.7, altUnit=M, sep=48.5, sepUnit=M, diffAge=, diffStation=)>",
            "<NMEA(GPGSA, opMode=A, navMode=3, svid_01=2, svid_02=13, svid_03=20, svid_04=7, svid_05=5, svid_06=30, svid_07=9, svid_08=, svid_09=, svid_10=, svid_11=, svid_12=, PDOP=2.44, HDOP=1.63, VDOP=1.82)>",
            "I ignored the following error: Message checksum b's\\x8e' invalid - should be b's\\x8d'",
            "<NMEA(GPGGA, time=08:02:48, lat=53.4507186667, NS=N, lon=-2.2402315, EW=W, quality=1, numSV=7, HDOP=1.63, alt=36.8, altUnit=M, sep=48.5, sepUnit=M, diffAge=, diffStation=)>",
        ]
        self.catchio()
        ubr = UBXReader(
            stream,
            protfilter=3,
            validate=VALCKSUM,
            msgmode=0,
            parsebitfield=True,
            quitonerror=ERR_LOG,
            errorhandler=lambda e: print(f"I ignored the following error: {e}"),
        )
        for _, parsed in ubr:
            if not isinstance(parsed, str):
                print(parsed)
        output = self.restoreio().split("\n")
        # print(output)
        self.assertEqual(EXPECTED_RESULTS, output)

    def testUBXITERATE_ERR3(
        self,
    ):  # UBXReader helper method ignoring bad checksum and continuing
        dirname = os.path.dirname(__file__)
        stream = open(os.path.join(dirname, "pygpsdata-MIXED3BADCK.log"), "rb")
        EXPECTED_RESULT = "<NMEA(GPGGA, time=08:02:48, lat=53.4507186667, NS=N, lon=-2.2402315, EW=W, quality=1, numSV=7, HDOP=1.63, alt=36.8, altUnit=M, sep=48.5, sepUnit=M, diffAge=, diffStation=)>"
        ubr = UBXReader(
            stream,
            validate=VALCKSUM,
            msgmode=0,
            parsebitfield=True,
            quitonerror=ERR_LOG,
        )
        res = ""
        for raw, parsed in ubr:
            res = str(parsed)
        self.assertEqual(EXPECTED_RESULT, res)
        stream.close()

    def testUBXITERATE_ERR4(
        self,
    ):  # UBXReader helper method ignoring bad checksum and continuing
        dirname = os.path.dirname(__file__)
        stream = open(os.path.join(dirname, "pygpsdata-MIXED3BADCK.log"), "rb")
        EXPECTED_RESULT = "<NMEA(GPGGA, time=08:02:48, lat=53.4507186667, NS=N, lon=-2.2402315, EW=W, quality=1, numSV=7, HDOP=1.63, alt=36.8, altUnit=M, sep=48.5, sepUnit=M, diffAge=, diffStation=)>"
        ubr = UBXReader(
            stream,
            validate=VALCKSUM,
            msgmode=0,
            parsebitfield=True,
            quitonerror=ERR_LOG,
        )
        res = ""
        for raw, parsed in ubr:
            res = str(parsed)
        self.assertEqual(EXPECTED_RESULT, res)
        stream.close()

    def testBADHDR_FAIL(self):  # invalid header in data with quitonerror = 2
        EXPECTED_ERROR = "Unknown protocol b'\\xb5w'"
        with self.assertRaises(UBXParseError) as context:
            i = 0
            ubxreader = UBXReader(self.streamBADHDR, quitonerror=ERR_RAISE)
            for _, _ in ubxreader:
                i += 1
        self.assertTrue(EXPECTED_ERROR in str(context.exception))

    def testBADHDR_LOG(self):  # invalid header in data with quitonerror = 1
        i = 0
        ubxreader = UBXReader(self.streamBADHDR, quitonerror=ERR_LOG)
        for raw, parsed in ubxreader:
            i += 1
        self.assertEqual(parsed, "<UNKNOWN PROTOCOL(header=b'\\xb5w')>")

    def testBADHDR_IGNORE(self):  # invalid header in data with quitonerror = 0
        i = 0
        ubxreader = UBXReader(self.streamBADHDR, quitonerror=ERR_IGNORE)
        for raw, parsed in ubxreader:
            i += 1
        self.assertEqual(i, 2)

    def testBADEOF1(self):  # premature EOF after header
        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamBADEOF1)
        while raw is not None:
            (raw, _) = ubxreader.read()
            i += 1
        self.assertEqual(i, 4)

    def testBADEOF2(self):  # premature EOF after message class and length
        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamBADEOF2)
        while raw is not None:
            (raw, _) = ubxreader.read()
            i += 1
        self.assertEqual(i, 3)

    def testBADEOF3(self):  # premature EOF after first byte of header
        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamBADEOF3)
        while raw is not None:
            (raw, _) = ubxreader.read()
            i += 1
        self.assertEqual(i, 3)

    def testBADNMEAEOF(self):  # premature EOF of NMEA stream
        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamBADNMEAEOF)
        while raw is not None:
            (raw, parsed) = ubxreader.read()
            i += 1
        self.assertEqual(i, 6)

    def testDEBUG(
        self,
    ):  # test stream of UBX DEBUG messages
        EXPECTED_RESULTS = [
            "<UBX(CFG-VALGET, version=1, layer=0, position=0, CFG_0x10010001=b'\\x00', CFG_0x10010101=b'\\x00', CFG_0x10040009=b'\\x00', CFG_TP_TP1_ENA=1, CFG_TP_SYNC_GNSS_TP1=1, CFG_TP_USE_LOCKED_TP1=1, CFG_TP_ALIGN_TO_TOW_TP1=1, CFG_TP_POL_TP1=1, CFG_0x10110012=b'\\x00', CFG_NAVSPG_INIFIX3D=0, CFG_0x10110014=b'\\x01', CFG_0x10110015=b'\\x01', CFG_0x10110016=b'\\x01', CFG_0x10110018=b'\\x01', CFG_0x1011001b=b'\\x00', CFG_NAVSPG_ACKAIDING=0, CFG_0x10110046=b'\\x01', CFG_0x10110052=b'\\x00', CFG_0x10110053=b'\\x00', CFG_NAVSPG_USE_USRDAT=0, CFG_0x10110081=b'\\x00', CFG_0x10110082=b'\\x00', CFG_0x10110083=b'\\x00', CFG_NAVSPG_PL_ENA=1, CFG_0x10140051=b'\\x00', CFG_0x10140052=b'\\x00', CFG_0x10140053=b'\\x00', CFG_0x10140061=b'\\x01', CFG_NAV2_OUT_ENABLED=0, CFG_NAV2_SBAS_USE_INTEGRITY=0, CFG_0x10210005=b'\\x00', CFG_ODO_USE_ODO=0, CFG_ODO_USE_COG=0, CFG_ODO_OUTLPVEL=0, CFG_ODO_OUTLPCOG=0, CFG_GEOFENCE_USE_PIO=0, CFG_GEOFENCE_USE_FENCE1=0, CFG_GEOFENCE_USE_FENCE2=0, CFG_GEOFENCE_USE_FENCE3=0, CFG_GEOFENCE_USE_FENCE4=0, CFG_0x10250001=b'\\x01', CFG_SIGNAL_GPS_L1CA_ENA=1, CFG_SIGNAL_GPS_L2C_ENA=1, CFG_SIGNAL_SBAS_L1CA_ENA=1, CFG_SIGNAL_GAL_E1_ENA=1, CFG_SIGNAL_GAL_E5B_ENA=1, CFG_SIGNAL_BDS_B1_ENA=1, CFG_SIGNAL_BDS_B2_ENA=1, CFG_SIGNAL_QZSS_L1CA_ENA=1, CFG_SIGNAL_QZSS_L1S_ENA=0, CFG_SIGNAL_QZSS_L2C_ENA=1, CFG_SIGNAL_GLO_L1_ENA=1, CFG_SIGNAL_GLO_L2_ENA=1, CFG_SIGNAL_GPS_ENA=1, CFG_SIGNAL_SBAS_ENA=1, CFG_SIGNAL_GAL_ENA=1, CFG_SIGNAL_BDS_ENA=1, CFG_SIGNAL_QZSS_ENA=1, CFG_SIGNAL_GLO_ENA=1, CFG_0x10310027=b'\\x01', CFG_0x10330001=b'\\x00', CFG_0x10330002=b'\\x00', CFG_0x10330003=b'\\x00', CFG_0x10330011=b'\\x00')>",
            "<UBX(DBG-0c4b-NOMINAL, payload=b'\\x00\\x00\\x64\\x00\\x26\\x00\\x01\\x00\\x33\\x10\\x00\\x02\\x00\\x33\\x10\\x00\\x03\\x00\\x33\\x10\\x00\\x11\\x00\\x33\\x10\\x00\\x21\\x00\\x33\\x10\\x01\\x01\\x00\\x34\\x10\\x00\\x02\\x00\\x34\\x10\\x00\\x03\\x00\\x34\\x10\\x01\\x04\\x00\\x34\\x10\\x00\\x11\\x00\\x34\\x10\\x00\\x14\\x00\\x34\\x10\\x00\\x01\\x00\\x35\\x10\\x00\\x02\\x00\\x35\\x10\\x00\\x03\\x00\\x35\\x10\\x00\\x04\\x00\\x35\\x10\\x00\\x02\\x00\\x36\\x10\\x00\\x03\\x00\\x36\\x10\\x01\\x04\\x00\\x36\\x10\\x01\\x05\\x00\\x36\\x10\\x00\\x07\\x00\\x36\\x10\\x00')>",
        ]
        i = 0
        raw = 0
        ubxreader = UBXReader(self.streamDEBUG)
        while raw is not None:
            i += 1
            (raw, parsed) = ubxreader.read()
            if raw is not None:
                res = str(parsed)
                if i == 5:
                    # print(res)
                    self.assertEqual(res, EXPECTED_RESULTS[0])
                if i == 188:
                    # print(res.replace("\\", "\\\\"))
                    self.assertEqual(res, EXPECTED_RESULTS[1])
        self.assertEqual(i, 189)


if __name__ == "__main__":
    # import sys;sys.argv = ['', 'Test.testName']
    unittest.main()
