"use strict";
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
Object.defineProperty(exports, "__esModule", { value: true });
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const synth_1 = require("projen/lib/util/synth");
const src_1 = require("../src");
describe("NX Monorepo Unit Tests", () => {
    it("Empty Monorepo", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "Defaults",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Ignore Patterns", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "IgnorePatterns",
            nxConfig: {
                nxIgnore: ["pattern1.txt", "**/foo.ts"],
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Target Dependencies", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "TargetDependencies",
            nxConfig: {
                targetDependencies: {
                    test: [
                        { target: "test", projects: src_1.TargetDependencyProject.DEPENDENCIES },
                    ],
                    eslint: [
                        { target: "eslint", projects: src_1.TargetDependencyProject.SELF },
                    ],
                },
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Affected Branch", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "main",
            name: "AffectedBranch",
            nxConfig: {
                affectedBranch: "main",
            },
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it.each([javascript_1.NodePackageManager.PNPM, javascript_1.NodePackageManager.YARN])("Additional Workspace Packages", (packageManager) => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            packageManager,
            name: "AdditionalWorkspacePackages",
            workspaceConfig: {
                additionalPackages: ["my/custom/package"],
            },
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            packageManager,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("another/custom/package", "yet/another/package");
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Workspace Package Order", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "WorkspacePackageOrder",
            workspaceConfig: {
                additionalPackages: ["packages/one"],
            },
        });
        new typescript_1.TypeScriptProject({
            name: "two",
            outdir: "packages/two",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("packages/three", "packages/four");
        new typescript_1.TypeScriptProject({
            name: "five",
            outdir: "packages/five",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        project.addWorkspacePackages("packages/six");
        new typescript_1.TypeScriptProject({
            name: "seven",
            outdir: "packages/seven",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("PNPM", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "PNPM",
            packageManager: javascript_1.NodePackageManager.PNPM,
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            packageManager: javascript_1.NodePackageManager.PNPM,
            defaultReleaseBranch: "mainline",
        });
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
    it("Validate consistent Package Managers", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "PNPM",
            packageManager: javascript_1.NodePackageManager.PNPM,
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        expect(() => synth_1.synthSnapshot(project)).toThrow("ts-subproject packageManager does not match the monorepo packageManager: pnpm.");
    });
    it("Composite", () => {
        const project = new src_1.NxMonorepoProject({
            defaultReleaseBranch: "mainline",
            name: "Composite",
        });
        const tsProject = new typescript_1.TypeScriptProject({
            name: "ts-subproject",
            outdir: "packages/ts-subproject",
            parent: project,
            defaultReleaseBranch: "mainline",
        });
        new typescript_1.TypeScriptProject({
            name: "ts-subproject2",
            outdir: "packages/ts-subproject2",
            parent: project,
            defaultReleaseBranch: "mainline",
            devDeps: ["ts-subproject"],
        });
        const pyProject = new python_1.PythonProject({
            parent: project,
            outdir: "packages/py-subproject",
            authorEmail: "test@test.com",
            authorName: "test",
            moduleName: "py_subproject",
            name: "py-subproject",
            version: "0.0.0",
        });
        project.addImplicitDependency(tsProject, pyProject);
        expect(synth_1.synthSnapshot(project)).toMatchSnapshot();
    });
});
//# sourceMappingURL=data:application/json;base64,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