"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NxMonorepoProject = exports.TargetDependencyProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const fs = require("fs");
const path = require("path");
const projen_1 = require("projen");
const javascript_1 = require("projen/lib/javascript");
const python_1 = require("projen/lib/python");
const typescript_1 = require("projen/lib/typescript");
const syncpack_options_1 = require("./syncpack-options");
const NX_MONOREPO_PLUGIN_PATH = ".nx/plugins/nx-monorepo-plugin.js";
/**
 * Supported enums for a TargetDependency.
 */
var TargetDependencyProject;
(function (TargetDependencyProject) {
    /**
     * Only rely on the package where the target is called.
     *
     * This is usually done for test like targets where you only want to run unit
     * tests on the target packages without testing all dependent packages.
     */
    TargetDependencyProject["SELF"] = "self";
    /**
     * Target relies on executing the target against all dependencies first.
     *
     * This is usually done for build like targets where you want to build all
     * dependant projects first.
     */
    TargetDependencyProject["DEPENDENCIES"] = "dependencies";
})(TargetDependencyProject = exports.TargetDependencyProject || (exports.TargetDependencyProject = {}));
/**
 * This project type will bootstrap a NX based monorepo with support for polygot
 * builds, build caching, dependency graph visualization and much more.
 *
 * @pjid nx-monorepo
 */
class NxMonorepoProject extends typescript_1.TypeScriptProject {
    constructor(options) {
        super({
            ...options,
            github: options.github ?? false,
            package: options.package ?? false,
            prettier: options.prettier ?? true,
            projenrcTs: true,
            release: options.release ?? false,
            jest: options.jest ?? false,
            defaultReleaseBranch: options.defaultReleaseBranch ?? "mainline",
            sampleCode: false,
        });
        this.targetOverrides = {};
        this.nxConfig = options.nxConfig;
        this.workspaceConfig = options.workspaceConfig;
        this.workspacePackages = options.workspaceConfig?.additionalPackages ?? [];
        this.implicitDependencies = this.nxConfig?.implicitDependencies || {};
        // Never publish a monorepo root package.
        this.package.addField("private", true);
        // No need to compile or test a monorepo root package.
        this.compileTask.reset();
        this.testTask.reset();
        this.addDevDeps("@nrwl/cli", "@nrwl/workspace");
        this.addDeps("aws-cdk-lib", "constructs", "cdk-nag"); // Needed as this can be bundled in aws-prototyping-sdk
        if (options.monorepoUpgradeDeps !== false) {
            this.addDevDeps("npm-check-updates", "syncpack");
            const upgradeDepsTask = this.addTask(options.monorepoUpgradeDepsOptions?.taskName || "upgrade-deps");
            upgradeDepsTask.exec("npx npm-check-updates --deep --rejectVersion 0.0.0 -u");
            upgradeDepsTask.exec("npx syncpack fix-mismatches");
            upgradeDepsTask.exec(`${this.package.packageManager} install`);
            upgradeDepsTask.exec("npx projen");
            new projen_1.JsonFile(this, ".syncpackrc.json", {
                obj: options.monorepoUpgradeDepsOptions?.syncpackConfig || syncpack_options_1.DEFAULT_CONFIG,
                readonly: true,
            });
        }
        options.nxConfig?.nxCloudReadOnlyAccessToken &&
            this.addDevDeps("@nrwl/nx-cloud");
        new projen_1.IgnoreFile(this, ".nxignore").exclude("test-reports", "target", ".env", ".pytest_cache", ...(this.nxConfig?.nxIgnore || []));
        new projen_1.TextFile(this, NX_MONOREPO_PLUGIN_PATH, {
            readonly: true,
            lines: fs.readFileSync(getPluginPath()).toString("utf-8").split("\n"),
        });
        this.nxJson = new projen_1.JsonFile(this, "nx.json", {
            obj: {
                extends: "@nrwl/workspace/presets/npm.json",
                plugins: [`./${NX_MONOREPO_PLUGIN_PATH}`],
                npmScope: "monorepo",
                tasksRunnerOptions: {
                    default: {
                        runner: options.nxConfig?.nxCloudReadOnlyAccessToken
                            ? "@nrwl/nx-cloud"
                            : "@nrwl/workspace/tasks-runners/default",
                        options: {
                            useDaemonProcess: false,
                            cacheableOperations: ["build", "test"],
                            accessToken: options.nxConfig?.nxCloudReadOnlyAccessToken,
                        },
                    },
                },
                implicitDependencies: this.implicitDependencies,
                targetDependencies: {
                    build: [
                        {
                            target: "build",
                            projects: "dependencies",
                        },
                    ],
                    ...(this.nxConfig?.targetDependencies || {}),
                },
                affected: {
                    defaultBase: this.nxConfig?.affectedBranch || "mainline",
                },
            },
        });
    }
    /**
     * Create an implicit dependency between two Project's. This is typically
     * used in polygot repos where a Typescript project wants a build dependency
     * on a Python project as an example.
     *
     * @param dependent project you want to have the dependency.
     * @param dependee project you wish to depend on.
     */
    addImplicitDependency(dependent, dependee) {
        if (this.implicitDependencies[dependent.name]) {
            this.implicitDependencies[dependent.name].push(dependee.name);
        }
        else {
            this.implicitDependencies[dependent.name] = [dependee.name];
        }
    }
    /**
     * Add one or more additional package globs to the workspace.
     * @param packageGlobs paths to the package to include in the workspace (for example packages/my-package)
     */
    addWorkspacePackages(...packageGlobs) {
        // Any subprojects that were added since the last call to this method need to be added first, in order to ensure
        // we add the workspace packages in a sane order.
        const relativeSubProjectWorkspacePackages = this.instantiationOrderSubProjects.map((project) => path.relative(this.outdir, project.outdir));
        const existingWorkspacePackages = new Set(this.workspacePackages);
        this.workspacePackages.push(...relativeSubProjectWorkspacePackages.filter((pkg) => !existingWorkspacePackages.has(pkg)));
        // Add the additional packages next
        this.workspacePackages.push(...packageGlobs);
    }
    /**
     * Allow project specific target overrides.
     */
    overrideProjectTargets(project, targets) {
        const _package = project.tryFindObjectFile("package.json");
        _package?.addOverride("nx", {
            targets: targets,
        });
        !_package && (this.targetOverrides[project.outdir] = targets);
    }
    // Remove this hack once subProjects is made public in Projen
    get instantiationOrderSubProjects() {
        // @ts-ignore
        const subProjects = this.subprojects || [];
        return subProjects;
    }
    get subProjects() {
        return this.instantiationOrderSubProjects.sort((a, b) => a.name.localeCompare(b.name));
    }
    /**
     * @inheritDoc
     */
    synth() {
        this.validateSubProjects();
        this.updateWorkspace();
        this.wirePythonDependencies();
        this.synthesizeNonNodePackageJson();
        super.synth();
    }
    /**
     * Ensures subprojects don't have a default task and that all packages use the same package manager.
     */
    validateSubProjects() {
        this.subProjects.forEach((subProject) => {
            // Disable default task on subprojects as this isn't supported in a monorepo
            subProject.defaultTask?.reset();
            if (isNodeProject(subProject) &&
                subProject.package.packageManager !== this.package.packageManager) {
                throw new Error(`${subProject.name} packageManager does not match the monorepo packageManager: ${this.package.packageManager}.`);
            }
        });
    }
    /**
     * For non-node projects, a package.json is required in order to be discovered by NX.
     */
    synthesizeNonNodePackageJson() {
        this.subProjects
            .filter((subProject) => !isNodeProject(subProject))
            .filter((subProject) => !subProject.tryFindFile("package.json"))
            .forEach((subProject) => {
            // generate a package.json if not found
            const manifest = {};
            (manifest.nx = this.targetOverrides[subProject.outdir]
                ? { targets: this.targetOverrides[subProject.outdir] }
                : undefined),
                (manifest.name = subProject.name);
            manifest.private = true;
            manifest.__pdk__ = true;
            manifest.scripts = subProject.tasks.all.reduce((p, c) => ({
                [c.name]: `npx projen ${c.name}`,
                ...p,
            }), {});
            manifest.version = "0.0.0";
            new projen_1.JsonFile(subProject, "package.json", {
                obj: manifest,
                readonly: true,
            });
        });
    }
    /**
     * Add a submodule entry to the appropriate workspace file.
     */
    updateWorkspace() {
        // A final call to addWorkspacePackages will update the list of workspace packages with any subprojects that have
        // not yet been added, in the correct order
        this.addWorkspacePackages();
        // Add workspaces for each subproject
        if (this.package.packageManager === javascript_1.NodePackageManager.PNPM) {
            new projen_1.YamlFile(this, "pnpm-workspace.yaml", {
                readonly: true,
                obj: {
                    packages: this.workspacePackages,
                },
            });
        }
        else {
            this.package.addField("workspaces", {
                packages: this.workspacePackages,
                nohoist: this.workspaceConfig?.noHoist,
            });
        }
    }
    /**
     * Updates the install task for python projects so that they are run serially and in dependency order such that python
     * projects within the monorepo can declare dependencies on one another.
     * @private
     */
    wirePythonDependencies() {
        // Find any python projects
        const pythonProjects = this.subProjects.filter((project) => project instanceof python_1.PythonProject);
        if (pythonProjects.length === 0) {
            // Nothing to do for no python projects
            return;
        }
        // Move all install commands to install-py so that they are not installed in parallel by the monorepo package manager.
        // eg yarn install will run the install task for all packages in parallel which can lead to conflicts for python.
        pythonProjects.forEach((pythonProject) => {
            const installPyTask = pythonProject.tasks.tryFind("install-py") ??
                pythonProject.addTask("install-py");
            const installTask = pythonProject.tasks.tryFind("install");
            (installTask?.steps || []).forEach((step) => {
                this.copyStepIntoTask(step, installPyTask, pythonProject);
            });
            installTask?.reset();
        });
        // Add an install task to the monorepo to include running the install-py command serially to avoid conflicting writes
        // to a shared virtual env. This is also managed by nx so that installs occur in dependency order.
        const monorepoInstallTask = this.tasks.tryFind("install") ?? this.addTask("install");
        monorepoInstallTask.exec(`npx nx run-many --target install-py --projects ${pythonProjects
            .map((project) => project.name)
            .join(",")} --parallel=1`);
        // Update the nx.json to ensure that install-py follows dependency order
        this.nxJson.addOverride("targetDependencies.install-py", [
            {
                target: "install-py",
                projects: "dependencies",
            },
        ]);
    }
    /**
     * Copies the given step into the given task. Step and Task must be from the given Project
     * @private
     */
    copyStepIntoTask(step, task, project) {
        if (step.exec) {
            task.exec(step.exec, { name: step.name, cwd: step.cwd });
        }
        else if (step.say) {
            task.say(step.say, { name: step.name, cwd: step.cwd });
        }
        else if (step.spawn) {
            const stepToSpawn = project.tasks.tryFind(step.spawn);
            if (stepToSpawn) {
                task.spawn(stepToSpawn, { name: step.name, cwd: step.cwd });
            }
        }
        else if (step.builtin) {
            task.builtin(step.builtin);
        }
    }
}
exports.NxMonorepoProject = NxMonorepoProject;
_a = JSII_RTTI_SYMBOL_1;
NxMonorepoProject[_a] = { fqn: "aws-prototyping-sdk.nx_monorepo.NxMonorepoProject", version: "0.12.4" };
/**
 * Determines if the passed in project is of type NodeProject.
 *
 * @param project Project instance.
 * @returns true if the project instance is of type NodeProject.
 */
function isNodeProject(project) {
    return project instanceof javascript_1.NodeProject || project.package;
}
function getPluginPath() {
    return path.join(__dirname, "plugin/nx-monorepo-plugin.js");
}
//# sourceMappingURL=data:application/json;base64,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