# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['minos', 'minos.plugins.graphql', 'minos.plugins.graphql.builders']

package_data = \
{'': ['*']}

install_requires = \
['graphql-core>=3.2.0,<4.0.0',
 'minos-microservice-common',
 'minos-microservice-networks']

setup_kwargs = {
    'name': 'minos-router-graphql',
    'version': '0.8.0.dev1',
    'description': 'The graphql plugin of the Minos Framework',
    'long_description': '<p align="center">\n  <a href="https://minos.run" target="_blank"><img src="https://raw.githubusercontent.com/minos-framework/.github/main/images/logo.png" alt="Minos logo"></a>\n</p>\n\n## minos-router-graphql\n\n[![PyPI Latest Release](https://img.shields.io/pypi/v/minos-broker-kafka.svg)](https://pypi.org/project/minos-broker-kafka/)\n[![GitHub Workflow Status](https://img.shields.io/github/workflow/status/minos-framework/minos-python/pages%20build%20and%20deployment?label=docs)](https://minos-framework.github.io/minos-python)\n[![License](https://img.shields.io/github/license/minos-framework/minos-python.svg)](https://github.com/minos-framework/minos-python/blob/main/LICENSE)\n[![Coverage](https://codecov.io/github/minos-framework/minos-python/coverage.svg?branch=main)](https://codecov.io/gh/minos-framework/minos-python)\n[![Stack Overflow](https://img.shields.io/badge/Stack%20Overflow-Ask%20a%20question-green)](https://stackoverflow.com/questions/tagged/minos)\n\n## Summary\n\nThis is graphQL plugin for Minos framework. This plugin integrates the official [graphql-core](https://github.com/graphql-python/graphql-core) library. It is oriented to facilitate the development and better organize the graphql code.\n\n## Installation\n\nInstall the dependency:\n\n```shell\npip install minos-router-graphql\n```\n\nModify `config.yml` file:\n\n```yaml\n...\nrouters:\n  - minos.plugins.graphql.GraphQlRouter\n...\n```\n\n## How to use\n\n### Define your business operation\n\nWe will use simple query for this demonstration:\n\n```python\nfrom graphql import (\n    GraphQLString,\n)\n\nfrom minos.networks import (\n    Request,\n    Response,\n    enroute,\n)\n\n\nclass QueryService:\n    @enroute.graphql.query(name="SimpleQuery", output=GraphQLString)\n    def simple_query(self, request: Request):\n        ...\n\n        return Response("ABCD")\n```\n\n### Execute query\n\nSend `post` request to `http://your_ip_address:port/service_name/graphql` endpoint:\n\n```json\n{\n  "query": "{ SimpleQuery }"\n}\n```\n\nYou will receive:\n\n```json\n{\n  "data": {\n    "SimpleQuery": "ABCD"\n  },\n  "errors": []\n}\n```\n\nThat\'s all you need to make it work !\n\nFor more information about graphql and how to define fields or structures, please see the official [graphql-core](https://github.com/graphql-python/graphql-core). library.\n\n## Decorators\n\nThere are 2 types of decorators, one for `queries` and one for `mutations` (commands).\n\n```python\n@enroute.graphql.query(name="TestQuery", argument=GraphQLString, output=GraphQLString)\ndef test_query(self, request: Request):\n    ...\n    return Responnse(...)\n\n\n@enroute.graphql.command(name="TestCommand", argument=GraphQLString, output=GraphQLString)\ndef test_command(self, request: Request):\n    ...\n    return Responnse(...)\n```\n\nBoth decorators have the following arguments:\n\n- `name`: The name of the query or command\n- `argument` [Optional]: The arguments it receives, if any.\n- `output`: The expected output.\n\n### Resolvers\n\nAs you have seen above, the decorator does not specify the function that will resolve it.\n\nThis is because it automatically takes the decorated function.\n\nIn the following example:\n\n```python\n@enroute.graphql.query(name="TestQuery", argument=GraphQLString, output=GraphQLString)\ndef test_query(self, request: Request):\n    ...\n    return Responnse(...)\n```\n\nThe function in charge of resolving the query is the decorated function `test_query`.\n\n### Queries (Query Service)\n\nQueries are used for a single purpose as their name indicates and that is to obtain information, that is, for queries.\n\nBase structure example:\n\n```python\nclass QueryService:\n    @enroute.graphql.query(name="TestQuery", argument=GraphQLString, output=GraphQLString)\n    def test_query(self, request: Request):\n        ...\n        return Responnse(...)\n```\n\nMore complex example:\n\n```python\nfrom graphql import (\n    GraphQLBoolean,\n    GraphQLField,\n    GraphQLID,\n    GraphQLInt,\n    GraphQLNonNull,\n    GraphQLObjectType,\n    GraphQLString,\n)\nfrom typing import (\n    NamedTuple,\n    Optional,\n)\nfrom minos.networks import (\n    Request,\n    Response,\n    enroute,\n)\n\nuser_type = GraphQLObjectType(\n    "UserType",\n    {\n        "id": GraphQLField(GraphQLNonNull(GraphQLID)),\n        "firstName": GraphQLField(GraphQLNonNull(GraphQLString)),\n        "lastName": GraphQLField(GraphQLNonNull(GraphQLString)),\n        "tweets": GraphQLField(GraphQLInt),\n        "verified": GraphQLField(GraphQLNonNull(GraphQLBoolean)),\n    },\n)\n\n\nclass User(NamedTuple):\n    """A simple user object class."""\n\n    firstName: str\n    lastName: str\n    tweets: Optional[int]\n    id: Optional[str] = None\n    verified: bool = False\n\n\nclass QueryService:\n    @enroute.graphql.query(name="GetUser", argument=GraphQLInt, output=user_type)\n    def test_query(self, request: Request):\n        id = await request.content()\n        return Response(User(firstName="Jack", lastName="Johnson", tweets=563, id=str(id), verified=True))\n```\n\nIf you POST `{service_name}/graphql` endpoint passing the query and variables:\n\n```json\n{\n  "query": "query ($userId: Int!) { GetUser(request: $userId) {id firstName lastName tweets verified}}",\n  "variables": {\n    "userId": 3\n  }\n}\n```\n\nYoy will receive:\n\n```json\n{\n  "data": {\n    "GetUser": {\n      "id": "3",\n      "firstName": "Jack",\n      "lastName": "Johnson",\n      "tweets": 563,\n      "verified": true\n    }\n  },\n  "errors": []\n}\n```\n\n### Mutations (Command Service)\n\nMutations are used to create, update or delete information.\n\nBase structure example:\n\n```python\nclass CommandService:\n    @enroute.graphql.command(name="TestQuery", argument=GraphQLString, output=GraphQLString)\n    def test_command(self, request: Request):\n        ...\n        return Responnse(...)\n```\n\nMore complex example:\n\n```python\nfrom graphql import (\n    GraphQLBoolean,\n    GraphQLField,\n    GraphQLID,\n    GraphQLInputField,\n    GraphQLInputObjectType,\n    GraphQLInt,\n    GraphQLNonNull,\n    GraphQLObjectType,\n    GraphQLString,\n)\nfrom typing import (\n    NamedTuple,\n    Optional,\n)\nfrom minos.networks import (\n    Request,\n    Response,\n    enroute,\n)\n\nuser_type = GraphQLObjectType(\n    "UserType",\n    {\n        "id": GraphQLField(GraphQLNonNull(GraphQLID)),\n        "firstName": GraphQLField(GraphQLNonNull(GraphQLString)),\n        "lastName": GraphQLField(GraphQLNonNull(GraphQLString)),\n        "tweets": GraphQLField(GraphQLInt),\n        "verified": GraphQLField(GraphQLNonNull(GraphQLBoolean)),\n    },\n)\n\nuser_input_type = GraphQLInputObjectType(\n    "UserInputType",\n    {\n        "firstName": GraphQLInputField(GraphQLNonNull(GraphQLString)),\n        "lastName": GraphQLInputField(GraphQLNonNull(GraphQLString)),\n        "tweets": GraphQLInputField(GraphQLInt),\n        "verified": GraphQLInputField(GraphQLBoolean),\n    },\n)\n\n\nclass User(NamedTuple):\n    """A simple user object class."""\n\n    firstName: str\n    lastName: str\n    tweets: Optional[int]\n    id: Optional[str] = None\n    verified: bool = False\n\n\nclass CommandService:\n    @enroute.graphql.command(name="CreateUser", argument=GraphQLNonNull(user_input_type), output=user_type)\n    def test_command(self, request: Request):\n        params = await request.content()\n        return Response(\n            User(\n                firstName=params["firstName"],\n                lastName=params["lastName"],\n                tweets=params["tweets"],\n                id="4kjjj43-l23k4l3-325kgaa2",\n                verified=params["verified"],\n            )\n        )\n```\n\nIf you POST `{service_name}/graphql` endpoint passing the query and variables:\n\n```json\n{\n  "query": "mutation ($userData: UserInputType!) { CreateUser(request: $userData) {id, firstName, lastName, tweets, verified}}",\n  "variables": {\n    "userData": {\n      "firstName": "John",\n      "lastName": "Doe",\n      "tweets": 42,\n      "verified": true\n    }\n  }\n}\n```\n\nYoy will receive:\n\n```json\n{\n  "data": {\n    "CreateUser": {\n      "id": "4kjjj43-l23k4l3-325kgaa2",\n      "firstName": "John",\n      "lastName": "Doe",\n      "tweets": 42,\n      "verified": true\n    }\n  },\n  "errors": []\n}\n```\n\n### Get Schema\n\nBy calling `{service_name}/graphql/schema` with `GET` method, you will receive the schema:\n\n```text\n"type Query {\\n  GetUser(request: Int): UserType\\n}\\n\\ntype UserType {\\n  id: ID!\\n  firstName: String!\\n  lastName: String!\\n  tweets: Int\\n  verified: Boolean!\\n}\\n\\ntype Mutation {\\n  CreateUser(request: UserInputType!): UserType\\n}\\n\\ninput UserInputType {\\n  firstName: String!\\n  lastName: String!\\n  tweets: Int\\n  verified: Boolean\\n}"\n```\n\n## Documentation\n\nThe official API Reference is publicly available at the [GitHub Pages](https://minos-framework.github.io/minos-python).\n\n## Source Code\n\nThe source code of this project is hosted at the [GitHub Repository](https://github.com/minos-framework/minos-python).\n\n## Getting Help\n\nFor usage questions, the best place to go to is [StackOverflow](https://stackoverflow.com/questions/tagged/minos).\n\n## Discussion and Development\n\nMost development discussions take place over the [GitHub Issues](https://github.com/minos-framework/minos-python/issues). In addition, a [Gitter channel](https://gitter.im/minos-framework/community) is available for development-related questions.\n\n## License\n\nThis project is distributed under the [MIT](https://raw.githubusercontent.com/minos-framework/minos-python/main/LICENSE) license.\n',
    'author': 'Minos Framework Devs',
    'author_email': 'hey@minos.run',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://www.minos.run/',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4.0',
}


setup(**setup_kwargs)
