# -*- coding: utf-8; -*-
################################################################################
#
#  Rattail -- Retail Software Framework
#  Copyright © 2010-2022 Lance Edgar
#
#  This file is part of Rattail.
#
#  Rattail is free software: you can redistribute it and/or modify it under the
#  terms of the GNU General Public License as published by the Free Software
#  Foundation, either version 3 of the License, or (at your option) any later
#  version.
#
#  Rattail is distributed in the hope that it will be useful, but WITHOUT ANY
#  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
#  FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
#  details.
#
#  You should have received a copy of the GNU General Public License along with
#  Rattail.  If not, see <http://www.gnu.org/licenses/>.
#
################################################################################
"""
Poser Views for Views...
"""

from __future__ import unicode_literals, absolute_import

import six

import colander

from .master import PoserMasterView


class PoserViewView(PoserMasterView):
    """
    Master view for Poser views
    """
    normalized_model_name = 'poser_view'
    model_title = "Poser View"
    route_prefix = 'poser_views'
    url_prefix = '/poser/views'
    configurable = True
    config_title = "Included Views"

    # TODO
    creatable = False
    editable = False
    deletable = False
    # downloadable = True

    grid_columns = [
        'key',
        'class_name',
        'description',
        'error',
    ]

    def get_poser_data(self, session=None):
        return self.poser_handler.get_all_tailbone_views()

    def make_form_schema(self):
        return PoserViewSchema()

    def make_create_form(self):
        return self.make_form({})

    def configure_form(self, f):
        super(PoserViewView, self).configure_form(f)
        view = f.model_instance

        # key
        f.set_default('key', 'cool_widget')
        f.set_helptext('key', "Unique key for the view; used as basis for filename.")
        if self.creating:
            f.set_validator('view_key', self.unique_view_key)

        # class_name
        f.set_default('class_name', "CoolWidget")
        f.set_helptext('class_name', "Python-friendly basis for view class name.")

        # description
        f.set_default('description', "Master view for Cool Widgets")
        f.set_helptext('description', "Brief description of the view.")

    def unique_view_key(self, node, value):
        for view in self.get_data():
            if view['key'] == value:
                raise node.raise_invalid("Poser view key must be unique")

    def collect_available_view_settings(self):

        # TODO: this probably should be more dynamic?  definitely need
        # to let integration packages register some more options...

        return {

            'people': {

                # TODO: need some way for integration / extension
                # packages to register alternate view options for some
                # of these.  that is the main reason these are dicts
                # even though at the moment it's a bit overkill.

                'tailbone.views.customers': {
                    # 'spec': 'tailbone.views.customers',
                    'label': "Customers",
                },
                'tailbone.views.customergroups': {
                    # 'spec': 'tailbone.views.customergroups',
                    'label': "Customer Groups",
                },
                'tailbone.views.employees': {
                    # 'spec': 'tailbone.views.employees',
                    'label': "Employees",
                },
                'tailbone.views.members': {
                    # 'spec': 'tailbone.views.members',
                    'label': "Members",
                },
            },

            'products': {

                'tailbone.views.departments': {
                    # 'spec': 'tailbone.views.departments',
                    'label': "Departments",
                },

                'tailbone.views.subdepartments': {
                    # 'spec': 'tailbone.views.subdepartments',
                    'label': "Subdepartments",
                },

                'tailbone.views.vendors': {
                    # 'spec': 'tailbone.views.vendors',
                    'label': "Vendors",
                },

                'tailbone.views.products': {
                    # 'spec': 'tailbone.views.products',
                    'label': "Products",
                },

                'tailbone.views.brands': {
                    # 'spec': 'tailbone.views.brands',
                    'label': "Brands",
                },

                'tailbone.views.categories': {
                    # 'spec': 'tailbone.views.categories',
                    'label': "Categories",
                },

                'tailbone.views.depositlinks': {
                    # 'spec': 'tailbone.views.depositlinks',
                    'label': "Deposit Links",
                },

                'tailbone.views.families': {
                    # 'spec': 'tailbone.views.families',
                    'label': "Families",
                },

                'tailbone.views.reportcodes': {
                    # 'spec': 'tailbone.views.reportcodes',
                    'label': "Report Codes",
                },
            },

            'other': {

                'tailbone.views.stores': {
                    # 'spec': 'tailbone.views.stores',
                    'label': "Stores",
                },

                'tailbone.views.taxes': {
                    # 'spec': 'tailbone.views.taxes',
                    'label': "Taxes",
                },
            },
        }

    def configure_get_simple_settings(self):
        settings = []

        view_settings = self.collect_available_view_settings()
        for view_section, section_settings in six.iteritems(view_settings):
            for key in section_settings:
                settings.append({'section': 'tailbone.includes',
                                 'option': key})

        return settings

    def configure_get_context(self, simple_settings=None,
                              input_file_templates=True):

        # first get normal context
        context = super(PoserViewView, self).configure_get_context(
            simple_settings=simple_settings,
            input_file_templates=input_file_templates)

        # add available settings as sorted (key, label) options
        view_settings = self.collect_available_view_settings()
        for key in list(view_settings):
            settings = view_settings[key]
            settings = [(key, setting['label'])
                        for key, setting in six.iteritems(settings)]
            settings.sort(key=lambda itm: itm[1])
            view_settings[key] = settings
        context['view_settings'] = view_settings

        return context

    def configure_flash_settings_saved(self):
        super(PoserViewView, self).configure_flash_settings_saved()
        self.request.session.flash("Please restart the web app!", 'warning')


class PoserViewSchema(colander.MappingSchema):

    key = colander.SchemaNode(colander.String())

    class_name = colander.SchemaNode(colander.String())

    description = colander.SchemaNode(colander.String())

    # include_comments = colander.SchemaNode(colander.Bool())


def defaults(config, **kwargs):
    base = globals()

    PoserViewView = kwargs.get('PoserViewView', base['PoserViewView'])
    PoserViewView.defaults(config)


def includeme(config):
    defaults(config)
