#!/usr/bin/env python
"""
NVMe System Bus (PCI Express) Registers

@author: Liangmin Huang
"""
from ctypes import Structure, c_uint8, c_uint16, c_uint32, c_uint64


class PCIHeader(Structure):
    """System Bus (PCI Express) Registers - PCI Header"""
    _pack_ = 1
    _fields_ = [
        ('ID_VID', c_uint32, 16),
        ('ID_DID', c_uint32, 16),
        ('CMD_IOSE', c_uint32, 1),
        ('CMD_MSE', c_uint32, 1),
        ('CMD_BME', c_uint32, 1),
        ('CMD_SCE', c_uint32, 1),
        ('CMD_MWIE', c_uint32, 1),
        ('CMD_VGA', c_uint32, 1),
        ('CMD_PEE', c_uint32, 1),
        ('CMD_RSV0', c_uint32, 1),
        ('CMD_SEE', c_uint32, 1),
        ('CMD_FBE', c_uint32, 1),
        ('CMD_ID', c_uint32, 1),
        ('CMD_RSV1', c_uint32, 5),
        ('STS_RSV0', c_uint32, 3),
        ('STS_IS', c_uint32, 1),
        ('STS_CL', c_uint32, 1),
        ('STS_C66', c_uint32, 1),
        ('STS_RSV1', c_uint32, 1),
        ('STS_FBC', c_uint32, 1),
        ('STS_DPD', c_uint32, 1),
        ('STS_DEVT', c_uint32, 2),
        ('STS_STA', c_uint32, 1),
        ('STS_RTA', c_uint32, 1),
        ('STS_RMA', c_uint32, 1),
        ('STS_SSE', c_uint32, 1),
        ('STS_DPE', c_uint32, 1),
        ('RID', c_uint32, 8),
        ('CC_PI', c_uint32, 8),
        ('CC_SCC', c_uint32, 8),
        ('CC_BCC', c_uint32, 8),
        ('CLS', c_uint32, 8),
        ('MLT', c_uint32, 8),
        ('HTYPE_HL', c_uint32, 7),
        ('HTYPE_MFD', c_uint32, 1),
        ('BIST_CC', c_uint32, 4),
        ('BIST_RSV', c_uint32, 2),
        ('BIST_SB', c_uint32, 1),
        ('BIST_BC', c_uint32, 1),
        ('MLBAR_RTE', c_uint32, 1),
        ('MLBAR_TP', c_uint32, 2),
        ('MLBAR_PF', c_uint32, 1),
        ('MLBAR_RSV', c_uint32, 10),
        ('MLBAR_BA', c_uint32, 18),
        ('MUBAR_BA', c_uint32),
        ('IDBAR_RTE', c_uint32, 1),
        ('IDBAR_RSV', c_uint32, 2),
        ('IDBAR_BA', c_uint32, 29),
        ('BAR3', c_uint32),
        ('BAR4', c_uint32),
        ('BAR5', c_uint32),
        ('CCPTR', c_uint32),
        ('SS_SSVID', c_uint32, 16),
        ('SS_SSID', c_uint32, 16),
        ('EROM', c_uint32),
        ('CAP_CP', c_uint64, 8),
        ('RSV', c_uint64, 56),
        ('INTR_ILINE', c_uint32, 8),
        ('INTR_IPIN', c_uint32, 8),
        ('MGNT', c_uint32, 8),
        ('MLAT', c_uint32, 8)
    ]


class PMCapability(Structure):
    """System Bus (PCI Express) Registers - PCI Power Management Capability"""
    _pack_ = 1
    _fields_ = [
        ('CID', c_uint16, 8),
        ('NEXT', c_uint16, 8),
        ('VS', c_uint16, 3),
        ('PMEC', c_uint16, 1),
        ('RSV0', c_uint16, 1),
        ('DSI', c_uint16, 1),
        ('AUXC', c_uint16, 3),
        ('D1S', c_uint16, 1),
        ('D2S', c_uint16, 1),
        ('PSUP', c_uint16, 5),
        ('PS', c_uint16, 2),
        ('RSV1', c_uint16, 1),
        ('NSFRST', c_uint16, 1),
        ('RSV2', c_uint16, 4),
        ('PMEE', c_uint16, 1),
        ('DSE', c_uint16, 4),
        ('DSC', c_uint16, 2),
        ('PMES', c_uint16, 1),
        ('RSV3', c_uint16)
    ]


class MSICapability(Structure):
    """System Bus (PCI Express) Registers - Message Signaled Interrupt Capability"""
    _pack_ = 1
    _fields_ = [
        ('CID', c_uint32, 8),
        ('NEXT', c_uint32, 8),
        ('MSIE', c_uint32, 1),
        ('MMC', c_uint32, 3),
        ('MME', c_uint32, 3),
        ('C64', c_uint32, 1),
        ('PVM', c_uint32, 1),
        ('RSV0', c_uint32, 7),
        ('RSV1', c_uint32, 2),
        ('ADDR', c_uint32, 30),
        ('UADDR', c_uint32),
        ('DATA', c_uint16),
        ('RSV2', c_uint16),
        ('MASK', c_uint32),
        ('PEND', c_uint32)
    ]


class MSIXCapability(Structure):
    """System Bus (PCI Express) Registers - MSI-X Capability"""
    _pack_ = 1
    _fields_ = [
        ('CID', c_uint32, 8),
        ('NEXT', c_uint32, 8),
        ('TS', c_uint32, 11),
        ('RSV0', c_uint32, 3),
        ('FM', c_uint32, 1),
        ('MXE', c_uint32, 1),
        ('TBIR', c_uint32, 3),
        ('TO', c_uint32, 29),
        ('PBIR', c_uint32, 3),
        ('PBAO', c_uint32, 29),
    ]


class PXCapability(Structure):
    """System Bus (PCI Express) Registers -  PCI Express Capability"""
    _pack_ = 1
    _fields_ = [
        ('CID', c_uint32, 8),
        ('NEXT', c_uint32, 8),
        ('VER', c_uint32, 4),
        ('DPT', c_uint32, 4),
        ('SI', c_uint32, 1),
        ('IMN', c_uint32, 5),
        ('RSV0', c_uint32, 2),
        ('MPSS', c_uint32, 3),
        ('PFS', c_uint32, 2),
        ('ETFS', c_uint32, 1),
        ('L0SL', c_uint32, 3),
        ('L1L', c_uint32, 3),
        ('RSB1', c_uint32, 3),
        ('RER', c_uint32, 1),
        ('RSV1', c_uint32, 2),
        ('CSPLV', c_uint32, 8),
        ('CSPLS', c_uint32, 2),
        ('FLRC', c_uint32, 1),
        ('RSV2', c_uint32, 3),
        ('CERE', c_uint32, 1),
        ('NFERE', c_uint32, 1),
        ('FERE', c_uint32, 1),
        ('URRE', c_uint32, 1),
        ('ERO', c_uint32, 1),
        ('MPS', c_uint32, 3),
        ('ETE', c_uint32, 1),
        ('PFE', c_uint32, 1),
        ('APPME', c_uint32, 1),
        ('ENS', c_uint32, 1),
        ('MRRS', c_uint32, 3),
        ('IFLR', c_uint32, 1),
        ('CED', c_uint32, 1),
        ('NFED', c_uint32, 1),
        ('FED', c_uint32, 1),
        ('URD', c_uint32, 1),
        ('APD', c_uint32, 1),
        ('TP', c_uint32, 1),
        ('RSV3', c_uint32, 10),
        ('SLS', c_uint32, 4),
        ('MLW', c_uint32, 6),
        ('ASPMS', c_uint32, 2),
        ('L0SEL', c_uint32, 3),
        ('L1EL', c_uint32, 3),
        ('CPM', c_uint32, 1),
        ('SDERC', c_uint32, 1),
        ('DLLLA', c_uint32, 1),
        ('LBNC', c_uint32, 1),
        ('AOC', c_uint32, 1),
        ('RSV4', c_uint32, 1),
        ('PN', c_uint32, 8),
        ('ASPMC', c_uint32, 2),
        ('RSV5', c_uint32, 1),
        ('RCB', c_uint32, 1),
        ('RSV6', c_uint32, 2),
        ('CCC', c_uint32, 1),
        ('ES', c_uint32, 1),
        ('ECPM', c_uint32, 1),
        ('HAWD', c_uint32, 1),
        ('RSV7', c_uint32, 6),
        ('CLS', c_uint32, 4),
        ('NLW', c_uint32, 6),
        ('RSV8', c_uint32, 2),
        ('SCC', c_uint32, 1),
        ('RSV9', c_uint32, 3),
        ('RSV10', c_uint32 * 4),
        ('CTRS', c_uint32, 4),
        ('CTDS', c_uint32, 1),
        ('ARIFS', c_uint32, 1),
        ('AORS', c_uint32, 1),
        ('32AOCS', c_uint32, 1),
        ('64AOCS', c_uint32, 1),
        ('128CCS', c_uint32, 1),
        ('NPRPR', c_uint32, 1),
        ('LTRS', c_uint32, 1),
        ('TPHCS', c_uint32, 2),
        ('RSV11', c_uint32, 4),
        ('OBFFS', c_uint32, 2),
        ('EFFS', c_uint32, 1),
        ('EETPS', c_uint32, 1),
        ('MEETP', c_uint32, 2),
        ('RSV12', c_uint32, 8),
        ('CTV', c_uint32, 4),
        ('CTD', c_uint32, 1),
        ('RSV13', c_uint32, 5),
        ('LTRME', c_uint32, 1),
        ('RSV14', c_uint32, 2),
        ('OBFFE', c_uint32, 2),
        ('RSV15', c_uint32, 17),
    ]


class AERCapability(Structure):
    """System Bus (PCI Express) Registers - Advanced Error Reporting Capability"""
    _pack_ = 1
    _fields_ = [
        ('CID', c_uint32, 16),
        ('CVER', c_uint32, 4),
        ('NEXT', c_uint32, 12),
        ('RSV0', c_uint32, 4),
        ('DLPES', c_uint32, 1),
        ('RSV1', c_uint32, 7),
        ('PTS', c_uint32, 1),
        ('FCPES', c_uint32, 1),
        ('CTS', c_uint32, 1),
        ('CAS', c_uint32, 1),
        ('UCS', c_uint32, 1),
        ('ROS', c_uint32, 1),
        ('MTS', c_uint32, 1),
        ('ECRCES', c_uint32, 1),
        ('URES', c_uint32, 1),
        ('ACSVS', c_uint32, 1),
        ('UIES', c_uint32, 1),
        ('MCBTS', c_uint32, 1),
        ('AOEBS', c_uint32, 1),
        ('TPBES', c_uint32, 1),
        ('RSV2', c_uint32, 6),
        ('RSV3', c_uint32, 4),
        ('DLPEM', c_uint32, 1),
        ('RSV4', c_uint32, 7),
        ('PTM', c_uint32, 1),
        ('FCPEM', c_uint32, 1),
        ('CTM', c_uint32, 1),
        ('CAM', c_uint32, 1),
        ('UCM', c_uint32, 1),
        ('ROM', c_uint32, 1),
        ('MTM', c_uint32, 1),
        ('ECRCEM', c_uint32, 1),
        ('UREM', c_uint32, 1),
        ('ACSVM', c_uint32, 1),
        ('UIEM', c_uint32, 1),
        ('MCBTM', c_uint32, 1),
        ('AOEBM', c_uint32, 1),
        ('TPBEM', c_uint32, 1),
        ('RSV5', c_uint32, 6),
        ('RSV6', c_uint32, 4),
        ('DLPESEV', c_uint32, 1),
        ('RSV7', c_uint32, 7),
        ('PTSEV', c_uint32, 1),
        ('FCPESEV', c_uint32, 1),
        ('CTSEV', c_uint32, 1),
        ('CASEV', c_uint32, 1),
        ('UCSEV', c_uint32, 1),
        ('ROSEV', c_uint32, 1),
        ('MTSEV', c_uint32, 1),
        ('ECRCESEV', c_uint32, 1),
        ('URESEV', c_uint32, 1),
        ('ACSVSEV', c_uint32, 1),
        ('UIESEV', c_uint32, 1),
        ('MCBTSEV', c_uint32, 1),
        ('AOEBSEV', c_uint32, 1),
        ('TPBESEV', c_uint32, 1),
        ('RSV8', c_uint32, 6),
        ('RES', c_uint32, 1),
        ('RSV9', c_uint32, 5),
        ('BTS', c_uint32, 1),
        ('BDS', c_uint32, 1),
        ('RRS', c_uint32, 1),
        ('RSV10', c_uint32, 3),
        ('RTS', c_uint32, 1),
        ('ANFES', c_uint32, 1),
        ('CIES', c_uint32, 1),
        ('HLOS', c_uint32, 1),
        ('RSV11', c_uint32, 16),
        ('REM', c_uint32, 1),
        ('RSV12', c_uint32, 5),
        ('BTM', c_uint32, 1),
        ('BDM', c_uint32, 1),
        ('RRM', c_uint32, 1),
        ('RSV13', c_uint32, 3),
        ('RTM', c_uint32, 1),
        ('ANFEM', c_uint32, 1),
        ('CIEM', c_uint32, 1),
        ('HLOM', c_uint32, 1),
        ('RSV14', c_uint32, 16),
        ('FEP', c_uint32, 5),
        ('EGC', c_uint32, 1),
        ('EGE', c_uint32, 1),
        ('ECC', c_uint32, 1),
        ('ECE', c_uint32, 1),
        ('MHRC', c_uint32, 1),
        ('MHRE', c_uint32, 1),
        ('TPLP', c_uint32, 1),
        ('RSV15', c_uint32, 20),
        ('AERHL', c_uint8 * 16),
        ('RSV16', c_uint8 * 12),
        ('AERTLP', c_uint8 * 16),
    ]


class CapabilityID(object):
    """Capability ID"""
    PMCapability = 0x01
    MSIXCapability = 0x11
    MSICapability = 0x05
    PXCapability = 0x10
    AERCapability = 0x1
