# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.base.exchange import Exchange
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import AuthenticationError
from ccxt.base.errors import ArgumentsRequired
from ccxt.base.errors import BadSymbol
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import RateLimitExceeded
from ccxt.base.errors import ExchangeNotAvailable
from ccxt.base.decimal_to_precision import TICK_SIZE


class bw(Exchange):

    def describe(self):
        return self.deep_extend(super(bw, self).describe(), {
            'id': 'bw',
            'name': 'BW',
            'countries': ['CN'],
            'rateLimit': 1500,
            'version': 'v1',
            'has': {
                'CORS': None,
                'spot': True,
                'margin': None,  # has but unimplemented
                'swap': None,  # has but unimplemented
                'future': None,
                'option': None,
                'cancelAllOrders': None,
                'cancelOrder': True,
                'cancelOrders': None,
                'createDepositAddress': None,
                'createLimitOrder': True,
                'createMarketOrder': None,
                'createOrder': True,
                'createStopLimitOrder': False,
                'createStopMarketOrder': False,
                'createStopOrder': False,
                'editOrder': None,
                'fetchBalance': True,
                'fetchBidsAsks': None,
                'fetchClosedOrders': True,
                'fetchCurrencies': True,
                'fetchDepositAddress': True,
                'fetchDeposits': True,
                'fetchL2OrderBook': None,
                'fetchLedger': None,
                'fetchMarginMode': False,
                'fetchMarkets': True,
                'fetchMyTrades': None,
                'fetchOHLCV': True,
                'fetchOpenOrders': True,
                'fetchOrder': True,
                'fetchOrderBook': True,
                'fetchOrderBooks': None,
                'fetchOrders': True,
                'fetchPositionMode': False,
                'fetchTicker': True,
                'fetchTickers': True,
                'fetchTrades': True,
                'fetchTradingFee': False,
                'fetchTradingFees': True,
                'fetchTradingLimits': None,
                'fetchTransactionFees': None,
                'fetchTransactions': None,
                'fetchWithdrawals': True,
                'withdraw': None,
            },
            'timeframes': {
                '1m': '1M',
                '5m': '5M',
                '15m': '15M',
                '30m': '30M',
                '1h': '1H',
                '1w': '1W',
            },
            'hostname': 'bw.com',  # set to 'bw.io' for China mainland
            'urls': {
                'logo': 'https://user-images.githubusercontent.com/1294454/69436317-31128c80-0d52-11ea-91d1-eb7bb5818812.jpg',
                'api': 'https://www.{hostname}',
                'www': 'https://www.bw.com',
                'doc': 'https://github.com/bw-exchange/api_docs_en/wiki',
                'fees': 'https://www.bw.com/feesRate',
                'referral': 'https://www.bw.com/regGetCommission/N3JuT1R3bWxKTE0',
            },
            'requiredCredentials': {
                'apiKey': True,
                'secret': True,
            },
            'fees': {
                'trading': {
                    'tierBased': True,
                    'percentage': True,
                    'taker': self.parse_number('0.002'),
                    'maker': self.parse_number('0.002'),
                },
                'funding': {
                },
            },
            'precisionMode': TICK_SIZE,
            'exceptions': {
                'exact': {
                    '999': AuthenticationError,
                    '1000': ExchangeNotAvailable,  # {"datas":null,"resMsg":{"message":"getKlines error:data not exitsts\uff0cplease wait ,dataType=4002_KLINE_1M","method":null,"code":"1000"}}
                    '2012': OrderNotFound,  # {"datas":null,"resMsg":{"message":"entrust not exists or on dealing with system","method":null,"code":"2012"}}
                    '5017': BadSymbol,  # {"datas":null,"resMsg":{"message":"market not exist","method":null,"code":"5017"}}
                    '10001': RateLimitExceeded,  # {"resMsg":{"code":"10001","message":"API frequency limit"}}
                },
            },
            'api': {
                'public': {
                    'get': [
                        'api/data/v1/klines',
                        'api/data/v1/ticker',
                        'api/data/v1/tickers',
                        'api/data/v1/trades',
                        'api/data/v1/entrusts',
                        'exchange/config/controller/website/marketcontroller/getByWebId',
                        'exchange/config/controller/website/currencycontroller/getCurrencyList',
                    ],
                },
                'private': {
                    'get': [
                        'exchange/entrust/controller/website/EntrustController/getEntrustById',
                        'exchange/entrust/controller/website/EntrustController/getUserEntrustRecordFromCacheWithPage',
                        'exchange/entrust/controller/website/EntrustController/getUserEntrustList',
                        'exchange/fund/controller/website/fundwebsitecontroller/getwithdrawaddress',
                        'exchange/fund/controller/website/fundwebsitecontroller/getpayoutcoinrecord',
                        'exchange/entrust/controller/website/EntrustController/getUserEntrustList',
                        # the docs say that the following URLs are HTTP POST
                        # in the docs header and HTTP GET in the docs body
                        # the docs contradict themselves, a typo most likely
                        # the actual HTTP method is POST for self endpoint
                        # 'exchange/fund/controller/website/fundcontroller/getPayinAddress',
                        # 'exchange/fund/controller/website/fundcontroller/getPayinCoinRecord',
                    ],
                    'post': [
                        'exchange/fund/controller/website/fundcontroller/getPayinAddress',  # see the comment above
                        'exchange/fund/controller/website/fundcontroller/getPayinCoinRecord',  # see the comment above
                        'exchange/fund/controller/website/fundcontroller/findbypage',
                        'exchange/entrust/controller/website/EntrustController/addEntrust',
                        'exchange/entrust/controller/website/EntrustController/cancelEntrust',
                    ],
                },
            },
        })

    def fetch_markets(self, params={}):
        """
        retrieves data on all markets for bw
        :param dict params: extra parameters specific to the exchange api endpoint
        :returns [dict]: an array of objects representing market data
        """
        response = self.publicGetExchangeConfigControllerWebsiteMarketcontrollerGetByWebId(params)
        #
        #    {
        #        resMsg: {
        #            method: null,
        #            code: '1',
        #            message: 'success !'
        #        },
        #        datas: [
        #            {
        #                leverMultiple: '10',
        #                amountDecimal: '4',
        #                minAmount: '0.0100000000',
        #                modifyUid: null,
        #                buyerCurrencyId: '11',
        #                isCombine: '0',
        #                priceDecimal: '3',
        #                combineMarketId: '',
        #                openPrice: '0',
        #                leverEnable: True,
        #                marketId: '291',
        #                serverId: 'entrust_bw_2',
        #                isMining: '0',
        #                webId: '102',
        #                modifyTime: '1581595375498',
        #                defaultFee: '0.00200000',
        #                sellerCurrencyId: '7',
        #                createTime: '0',
        #                state: '1',
        #                name: 'eos_usdt',
        #                leverType: '2',
        #                createUid: null,
        #                orderNum: null,
        #                openTime: '1574956800000'
        #            },
        #        ]
        #    }
        #
        markets = self.safe_value(response, 'datas', [])
        result = []
        for i in range(0, len(markets)):
            market = markets[i]
            id = self.safe_string(market, 'marketId')
            numericId = int(id)
            name = self.safe_string_upper(market, 'name')
            base, quote = name.split('_')
            base = self.safe_currency_code(base)
            quote = self.safe_currency_code(quote)
            baseId = self.safe_string(market, 'sellerCurrencyId')
            quoteId = self.safe_string(market, 'buyerCurrencyId')
            state = self.safe_integer(market, 'state')
            fee = self.safe_number(market, 'defaultFee')
            result.append({
                'id': id,
                'numericId': numericId,
                'symbol': base + '/' + quote,
                'base': base,
                'quote': quote,
                'settle': None,
                'baseId': baseId,
                'quoteId': quoteId,
                'settleId': None,
                'baseNumericId': int(baseId),
                'quoteNumericId': int(quoteId),
                'type': 'spot',
                'spot': True,
                'margin': False,
                'swap': False,
                'future': False,
                'option': False,
                'active': (state == 1),
                'contract': False,
                'linear': None,
                'inverse': None,
                'taker': fee,
                'maker': fee,
                'contractSize': None,
                'expiry': None,
                'expiryDatetime': None,
                'strike': None,
                'optionType': None,
                'precision': {
                    'amount': self.parse_number(self.parse_precision(self.safe_string(market, 'amountDecimal'))),
                    'price': self.parse_number(self.parse_precision(self.safe_string(market, 'priceDecimal'))),
                },
                'limits': {
                    'leverage': {
                        'min': None,
                        'max': None,
                    },
                    'amount': {
                        'min': self.safe_number(market, 'minAmount'),
                        'max': None,
                    },
                    'price': {
                        'min': self.parse_number('0'),
                        'max': None,
                    },
                    'cost': {
                        'min': self.parse_number('0'),
                        'max': None,
                    },
                },
                'info': market,
            })
        return result

    def fetch_currencies(self, params={}):
        """
        fetches all available currencies on an exchange
        :param dict params: extra parameters specific to the bw api endpoint
        :returns dict: an associative dictionary of currencies
        """
        response = self.publicGetExchangeConfigControllerWebsiteCurrencycontrollerGetCurrencyList(params)
        #
        #     {
        #         "datas":[
        #             {
        #                 "currencyId":"456",
        #                 "name":"pan",
        #                 "alias":"pan",
        #                 "logo":"pan.svg",
        #                 "description":"pan",
        #                 "descriptionEnglish":"pan",
        #                 "defaultDecimal":2,
        #                 "createUid":null,
        #                 "createTime":1574068133762,
        #                 "modifyUid":null,
        #                 "modifyTime":0,
        #                 "state":1,
        #                 "mark":"pan",
        #                 "totalNumber":"0",
        #                 "publishNumber":"0",
        #                 "marketValue":"0",
        #                 "isLegalCoin":0,
        #                 "needBlockUrl":1,
        #                 "blockChainUrl":"https://etherscan.io/tx/",
        #                 "tradeSearchUrl":null,
        #                 "tokenCoinsId":0,
        #                 "isMining":"0",
        #                 "arithmetic":null,
        #                 "founder":"bw_nxwal",
        #                 "teamAddress":null,
        #                 "remark":null,
        #                 "tokenName":"ethw2",
        #                 "isMemo":0,
        #                 "websiteCurrencyId":"7rhqoHLohkG",
        #                 "drawFlag":0,
        #                 "rechargeFlag":1,
        #                 "drawFee":"0.03000000",
        #                 "onceDrawLimit":100,
        #                 "dailyDrawLimit":500,
        #                 "timesFreetrial":"0",
        #                 "hourFreetrial":"0",
        #                 "dayFreetrial":"0",
        #                 "minFee":"0",
        #                 "inConfigTimes":7,
        #                 "outConfigTimes":7,
        #                 "minCash":"0.06000000",
        #                 "limitAmount":"0",
        #                 "zbExist":false,
        #                 "zone":1
        #             },
        #         ],
        #         "resMsg": {"message":"success !", "method":null, "code":"1"}
        #     }
        #
        currencies = self.safe_value(response, 'datas', [])
        result = {}
        for i in range(0, len(currencies)):
            currency = currencies[i]
            id = self.safe_string(currency, 'currencyId')
            code = self.safe_currency_code(self.safe_string_upper(currency, 'name'))
            state = self.safe_integer(currency, 'state')
            rechargeFlag = self.safe_integer(currency, 'rechargeFlag')
            drawFlag = self.safe_integer(currency, 'drawFlag')
            deposit = rechargeFlag == 1
            withdraw = drawFlag == 1
            active = state == 1
            result[code] = {
                'id': id,
                'code': code,
                'info': currency,
                'name': code,
                'active': active,
                'deposit': deposit,
                'withdraw': withdraw,
                'fee': self.safe_number(currency, 'drawFee'),
                'precision': None,
                'limits': {
                    'amount': {
                        'min': self.safe_number(currency, 'limitAmount', 0),
                        'max': None,
                    },
                    'withdraw': {
                        'min': None,
                        'max': self.safe_number(currency, 'onceDrawLimit'),
                    },
                },
            }
        return result

    def parse_ticker(self, ticker, market=None):
        #
        #     [
        #         "281",            # market id
        #         "9754.4",         # last
        #         "9968.8",         # high
        #         "9631.5",         # low
        #         "47865.6432",     # base volume
        #         "-2.28",          # change
        #         # closing price for last 6 hours
        #         "[[1, 9750.1], [2, 9737.1], [3, 9727.5], [4, 9722], [5, 9722.1], [6, 9754.4]]",
        #         "9752.12",        # bid
        #         "9756.69",        # ask
        #         "469849357.2364"  # quote volume
        #     ]
        #
        marketId = self.safe_string(ticker, 0)
        market = self.safe_market(marketId, market)
        symbol = market['symbol']
        timestamp = self.milliseconds()
        close = self.safe_string(ticker, 1)
        bid = self.safe_value(ticker, 'bid', {})
        ask = self.safe_value(ticker, 'ask', {})
        return self.safe_ticker({
            'symbol': symbol,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'high': self.safe_string(ticker, 2),
            'low': self.safe_string(ticker, 3),
            'bid': self.safe_string(ticker, 7),
            'bidVolume': self.safe_string(bid, 'quantity'),
            'ask': self.safe_string(ticker, 8),
            'askVolume': self.safe_string(ask, 'quantity'),
            'vwap': None,
            'open': None,
            'close': close,
            'last': close,
            'previousClose': None,
            'change': self.safe_string(ticker, 5),
            'percentage': None,
            'average': None,
            'baseVolume': self.safe_string(ticker, 4),
            'quoteVolume': self.safe_string(ticker, 9),
            'info': ticker,
        }, market)

    def fetch_ticker(self, symbol, params={}):
        """
        fetches a price ticker, a statistical calculation with the information calculated over the past 24 hours for a specific market
        :param str symbol: unified symbol of the market to fetch the ticker for
        :param dict params: extra parameters specific to the bw api endpoint
        :returns dict: a `ticker structure <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'marketId': market['id'],
        }
        response = self.publicGetApiDataV1Ticker(self.extend(request, params))
        #
        #     {
        #         "datas": [
        #             "281",
        #             "7601.99",
        #             "8126.5",
        #             "7474.68",
        #             "47004.8708",
        #             "-6.18",
        #             "[[1, 7800.34], [2, 7626.41], [3, 7609.97], [4, 7569.04], [5, 7577.93], [6, 7601.99]]",
        #             "7600.24",
        #             "7603.69",
        #             "371968300.0119",
        #         ],
        #         "resMsg": {"message": "success !", "method": null, "code": "1"}
        #     }
        #
        ticker = self.safe_value(response, 'datas', [])
        return self.parse_ticker(ticker, market)

    def fetch_tickers(self, symbols=None, params={}):
        """
        fetches price tickers for multiple markets, statistical calculations with the information calculated over the past 24 hours each market
        :param [str]|None symbols: unified symbols of the markets to fetch the ticker for, all market tickers are returned if not assigned
        :param dict params: extra parameters specific to the bw api endpoint
        :returns dict: an array of `ticker structures <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        self.load_markets()
        response = self.publicGetApiDataV1Tickers(params)
        #
        #     {
        #         "datas": [
        #             [
        #                 "4051",
        #                 "0.00194",
        #                 "0.00863",
        #                 "0.0012",
        #                 "1519020",
        #                 "-38.22",
        #                 "[[1, 0.0023], [2, 0.00198], [3, 0.00199], [4, 0.00195], [5, 0.00199], [6, 0.00194]]",
        #                 "0.00123",
        #                 "0.0045",
        #                 "4466.8104",
        #             ],
        #         ],
        #         "resMsg": {"message": "success !", "method": null, "code": "1"},
        #     }
        #
        datas = self.safe_value(response, 'datas', [])
        result = {}
        for i in range(0, len(datas)):
            ticker = self.parse_ticker(datas[i])
            symbol = ticker['symbol']
            if (symbols is None) or self.in_array(symbol, symbols):
                result[symbol] = ticker
        return self.filter_by_array(result, 'symbol', symbols)

    def fetch_order_book(self, symbol, limit=None, params={}):
        """
        fetches information on open orders with bid(buy) and ask(sell) prices, volumes and other data
        :param str symbol: unified symbol of the market to fetch the order book for
        :param int|None limit: the maximum amount of order book entries to return
        :param dict params: extra parameters specific to the bw api endpoint
        :returns dict: A dictionary of `order book structures <https://docs.ccxt.com/en/latest/manual.html#order-book-structure>` indexed by market symbols
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'marketId': market['id'],
        }
        if limit is not None:
            request['dataSize'] = limit
        response = self.publicGetApiDataV1Entrusts(self.extend(request, params))
        #
        #     {
        #         "datas": {
        #             "asks": [
        #                 ["9740.43", "0.0083"],
        #             ],
        #             "bids": [
        #                 ["9734.33", "0.0133"],
        #             ],
        #             "timestamp": "1569303520",
        #         },
        #         "resMsg": {
        #             "message": "success !",
        #             "method": null,
        #             "code": "1",
        #         },
        #     }
        #
        orderbook = self.safe_value(response, 'datas', [])
        timestamp = self.safe_timestamp(orderbook, 'timestamp')
        return self.parse_order_book(orderbook, market['symbol'], timestamp)

    def parse_trade(self, trade, market=None):
        #
        # fetchTrades(public)
        #
        #     [
        #         "T",          # trade
        #         "281",        # market id
        #         "1569303302",  # timestamp
        #         "BTC_USDT",   # market name
        #         "ask",        # side
        #         "9745.08",    # price
        #         "0.0026"      # amount
        #     ]
        #
        # fetchMyTrades(private)
        #
        #     ...
        #
        timestamp = self.safe_timestamp(trade, 2)
        priceString = self.safe_string(trade, 5)
        amountString = self.safe_string(trade, 6)
        marketId = self.safe_string(trade, 1)
        delimiter = None
        if marketId is not None:
            if not (marketId in self.markets_by_id):
                delimiter = '_'
                marketId = self.safe_string(trade, 3)
        market = self.safe_market(marketId, market, delimiter)
        sideString = self.safe_string(trade, 4)
        side = 'sell' if (sideString == 'ask') else 'buy'
        return self.safe_trade({
            'id': None,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'symbol': market['symbol'],
            'order': None,
            'type': 'limit',
            'side': side,
            'takerOrMaker': None,
            'price': priceString,
            'amount': amountString,
            'cost': None,
            'fee': None,
            'info': trade,
        }, market)

    def fetch_trades(self, symbol, since=None, limit=None, params={}):
        """
        get the list of most recent trades for a particular symbol
        :param str symbol: unified symbol of the market to fetch trades for
        :param int|None since: timestamp in ms of the earliest trade to fetch
        :param int|None limit: the maximum amount of trades to fetch
        :param dict params: extra parameters specific to the bw api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html?#public-trades>`
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'marketId': market['id'],
        }
        if limit is not None:
            request['dataSize'] = limit  # max 20
        response = self.publicGetApiDataV1Trades(self.extend(request, params))
        #
        #     {
        #         "datas": [
        #             [
        #                 "T",          # trade
        #                 "281",        # market id
        #                 "1569303302",  # timestamp
        #                 "BTC_USDT",   # market name
        #                 "ask",        # side
        #                 "9745.08",    # price
        #                 "0.0026"      # amount
        #             ],
        #         ],
        #         "resMsg": {"code": "1", "method": null, "message": "success !"},
        #     }
        #
        trades = self.safe_value(response, 'datas', [])
        return self.parse_trades(trades, market, since, limit)

    def fetch_trading_fees(self, params={}):
        """
        fetch the trading fees for multiple markets
        :param dict params: extra parameters specific to the bw api endpoint
        :returns dict: a dictionary of `fee structures <https://docs.ccxt.com/en/latest/manual.html#fee-structure>` indexed by market symbols
        """
        self.load_markets()
        response = self.publicGetExchangeConfigControllerWebsiteMarketcontrollerGetByWebId()
        #
        #    {
        #        resMsg: {method: null, code: '1', message: 'success !'},
        #        datas: [
        #            {
        #                leverMultiple: '10',
        #                amountDecimal: '4',
        #                minAmount: '0.0100000000',
        #                modifyUid: null,
        #                buyerCurrencyId: '11',
        #                isCombine: '0',
        #                priceDecimal: '3',
        #                combineMarketId: '',
        #                openPrice: '0',
        #                leverEnable: True,
        #                marketId: '291',
        #                serverId: 'entrust_bw_2',
        #                isMining: '0',
        #                webId: '102',
        #                modifyTime: '1581595375498',
        #                defaultFee: '0.00200000',
        #                sellerCurrencyId: '7',
        #                createTime: '0',
        #                state: '1',
        #                name: 'eos_usdt',
        #                leverType: '2',
        #                createUid: null,
        #                orderNum: null,
        #                openTime: '1574956800000'
        #            },
        #            ...
        #        ]
        #    }
        #
        datas = self.safe_value(response, 'datas', [])
        result = {}
        for i in range(0, len(datas)):
            data = datas[i]
            marketId = self.safe_string(data, 'name')
            symbol = self.safe_symbol(marketId, None, '_')
            fee = self.safe_number(data, 'defaultFee')
            result[symbol] = {
                'info': data,
                'symbol': symbol,
                'maker': fee,
                'taker': fee,
                'percentage': True,
                'tierBased': True,
            }
        return result

    def parse_ohlcv(self, ohlcv, market=None):
        #
        #     [
        #         "K",
        #         "305",
        #         "eth_btc",
        #         "1591511280",
        #         "0.02504",
        #         "0.02504",
        #         "0.02504",
        #         "0.02504",
        #         "0.0123",
        #         "0",
        #         "285740.17",
        #         "1M",
        #         "false",
        #         "0.000308"
        #     ]
        #
        return [
            self.safe_timestamp(ohlcv, 3),
            self.safe_number(ohlcv, 4),
            self.safe_number(ohlcv, 5),
            self.safe_number(ohlcv, 6),
            self.safe_number(ohlcv, 7),
            self.safe_number(ohlcv, 8),
        ]

    def fetch_ohlcv(self, symbol, timeframe='1m', since=None, limit=None, params={}):
        """
        fetches historical candlestick data containing the open, high, low, and close price, and the volume of a market
        :param str symbol: unified symbol of the market to fetch OHLCV data for
        :param str timeframe: the length of time each candle represents
        :param int|None since: timestamp in ms of the earliest candle to fetch
        :param int|None limit: the maximum amount of candles to fetch
        :param dict params: extra parameters specific to the bw api endpoint
        :returns [[int]]: A list of candles ordered as timestamp, open, high, low, close, volume
        """
        self.load_markets()
        market = self.market(symbol)
        request = {
            'marketId': market['id'],
            'type': self.timeframes[timeframe],
            'dataSize': 500,
        }
        if limit is not None:
            request['dataSize'] = limit
        response = self.publicGetApiDataV1Klines(self.extend(request, params))
        #
        #     {
        #         "datas":[
        #             ["K","305","eth_btc","1591511280","0.02504","0.02504","0.02504","0.02504","0.0123","0","285740.17","1M","false","0.000308"],
        #             ["K","305","eth_btc","1591511220","0.02504","0.02504","0.02504","0.02504","0.0006","0","285740.17","1M","false","0.00001502"],
        #             ["K","305","eth_btc","1591511100","0.02505","0.02505","0.02504","0.02504","0.0012","-0.0399","285740.17","1M","false","0.00003005"],
        #         ],
        #         "resMsg":{"code":"1","method":null,"message":"success !"}
        #     }
        #
        data = self.safe_value(response, 'datas', [])
        return self.parse_ohlcvs(data, market, timeframe, since, limit)

    def parse_balance(self, response):
        data = self.safe_value(response, 'datas', {})
        balances = self.safe_value(data, 'list', [])
        result = {'info': response}
        for i in range(0, len(balances)):
            balance = balances[i]
            currencyId = self.safe_string(balance, 'currencyTypeId')
            code = self.safe_currency_code(currencyId)
            account = self.account()
            account['free'] = self.safe_string(balance, 'amount')
            account['used'] = self.safe_string(balance, 'freeze')
            result[code] = account
        return self.safe_balance(result)

    def fetch_balance(self, params={}):
        """
        query for balance and get the amount of funds available for trading or funds locked in orders
        :param dict params: extra parameters specific to the bw api endpoint
        :returns dict: a `balance structure <https://docs.ccxt.com/en/latest/manual.html?#balance-structure>`
        """
        self.load_markets()
        response = self.privatePostExchangeFundControllerWebsiteFundcontrollerFindbypage(params)
        #
        #     {
        #         "datas": {
        #             "totalRow": 6,
        #             "pageSize": 99,
        #             "list": [
        #                 {
        #                     "amount": "0.000090000000000000",  # The current number of tokens available
        #                     "currencyTypeId": 2,              # Token ID
        #                     "freeze": "0.009900000000000000",  # Current token freezing quantity
        #                 },
        #             ],
        #             "pageNum": 1,
        #         },
        #         "resMsg": {"code": "1", "message": "success !"}
        #     }
        #
        return self.parse_balance(response)

    def create_order(self, symbol, type, side, amount, price=None, params={}):
        """
        create a trade order
        :param str symbol: unified symbol of the market to create an order in
        :param str type: 'market' or 'limit'
        :param str side: 'buy' or 'sell'
        :param float amount: how much of currency you want to trade in units of base currency
        :param float|None price: the price at which the order is to be fullfilled, in units of the quote currency, ignored in market orders
        :param dict params: extra parameters specific to the bw api endpoint
        :returns dict: an `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        if price is None:
            raise ExchangeError(self.id + ' createOrder() allows limit orders only')
        self.load_markets()
        market = self.market(symbol)
        request = {
            'amount': self.amount_to_precision(symbol, amount),
            'price': self.price_to_precision(symbol, price),
            'type': 1 if (side == 'buy') else 0,
            'rangeType': 0,  # limit order
            'marketId': market['id'],
        }
        response = self.privatePostExchangeEntrustControllerWebsiteEntrustControllerAddEntrust(self.extend(request, params))
        #
        #     {
        #         "datas": {
        #             "entrustId": "E6581105708337483776",
        #         },
        #         "resMsg": {
        #             "message": "success !",
        #             "method": null,
        #             "code": "1"
        #         }
        #     }
        #
        data = self.safe_value(response, 'datas')
        id = self.safe_string(data, 'entrustId')
        return {
            'id': id,
            'info': response,
            'timestamp': None,
            'datetime': None,
            'lastTradeTimestamp': None,
            'symbol': symbol,
            'type': type,
            'side': side,
            'price': price,
            'amount': amount,
            'cost': None,
            'average': None,
            'filled': None,
            'remaining': None,
            'status': 'open',
            'fee': None,
            'trades': None,
            'clientOrderId': None,
        }

    def parse_order_status(self, status):
        statuses = {
            '-3': 'canceled',
            '-2': 'canceled',
            '-1': 'canceled',
            '0': 'open',
            '1': 'canceled',
            '2': 'closed',
            '3': 'open',
            '4': 'canceled',
        }
        return self.safe_string(statuses, status, status)

    def parse_order(self, order, market=None):
        #
        # fetchOrder, fetchOpenOrders, fetchClosedOrders
        #
        #     {
        #         "entrustId": "E6581108027628212224",  # Order id
        #         "price": "1450",                     # price
        #         "rangeType": 0,                      # Commission type 0: limit price commission 1: interval commission
        #         "amount": "14.05",                   # Order quantity
        #         "totalMoney": "20372.50",            # Total order amount
        #         "completeAmount": "0",               # Quantity sold
        #         "completeTotalMoney": "0",           # Total dealt amount
        #         "type": 1,                           # 0 = sell, 1 = buy, -1 = cancel
        #         "entrustType": 0,                    # 0 = ordinary current price commission, 1 = lever commission
        #         "status": 0,                         #
        #         "marketId": "318",                   # The market id
        #         "createTime": 1569058424861,         # Create time
        #         "availabelAmount": "14.05"           # Outstanding quantity, typo in the docs or in the API, availabel vs available
        #     }
        #
        marketId = self.safe_string(order, 'marketId')
        symbol = self.safe_symbol(marketId, market)
        timestamp = self.safe_integer(order, 'createTime')
        side = self.safe_string(order, 'type')
        if side == '0':
            side = 'sell'
        elif side == '1':
            side = 'buy'
        amount = self.safe_string(order, 'amount')
        price = self.safe_string(order, 'price')
        filled = self.safe_string(order, 'completeAmount')
        remaining = self.safe_string_2(order, 'availabelAmount', 'availableAmount')  # typo in the docs or in the API, availabel vs available
        cost = self.safe_string(order, 'totalMoney')
        status = self.parse_order_status(self.safe_string(order, 'status'))
        return self.safe_order({
            'info': order,
            'id': self.safe_string(order, 'entrustId'),
            'clientOrderId': None,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'lastTradeTimestamp': None,
            'symbol': symbol,
            'type': 'limit',
            'timeInForce': None,
            'postOnly': None,
            'side': side,
            'price': price,
            'stopPrice': None,
            'amount': amount,
            'cost': cost,
            'average': None,
            'filled': filled,
            'remaining': remaining,
            'status': status,
            'fee': None,
            'trades': None,
        }, market)

    def fetch_order(self, id, symbol=None, params={}):
        """
        fetches information on an order made by the user
        :param str symbol: unified symbol of the market the order was made in
        :param dict params: extra parameters specific to the bw api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' fetchOrder() requires a symbol argument')
        self.load_markets()
        market = self.market(symbol)
        request = {
            'marketId': market['id'],
            'entrustId': id,
        }
        response = self.privateGetExchangeEntrustControllerWebsiteEntrustControllerGetEntrustById(self.extend(request, params))
        #
        #     {
        #         "datas": {
        #             "entrustId": "E6581108027628212224",  # Order id
        #             "price": "1450",                     # price
        #             "rangeType": 0,                      # Commission type 0: limit price commission 1: interval commission
        #             "amount": "14.05",                   # Order quantity
        #             "totalMoney": "20372.50",            # Total order amount
        #             "completeAmount": "0",               # Quantity sold
        #             "completeTotalMoney": "0",           # Total dealt amount
        #             "type": 1,                           # Trade direction, 0: sell, 1: buy, -1: cancel
        #             "entrustType": 0,                    # Commission type, 0: ordinary current price commission, 1: lever commission
        #             "status": 0,                         # Order status,-3:fund Freeze exception,Order status to be confirmed  -2: fund freeze failure, order failure, -1: insufficient funds, order failure, 0: pending order, 1: cancelled, 2: dealt, 3: partially dealt
        #             "marketId": "318",                   # The market id
        #             "createTime": 1569058424861,         # Create time
        #             "availabelAmount": "14.05"           # Outstanding quantity
        #         },
        #         "resMsg": {"message": "success !", "method": null, "code": "1"}
        #     }
        #
        order = self.safe_value(response, 'datas', {})
        return self.parse_order(order, market)

    def cancel_order(self, id, symbol=None, params={}):
        """
        cancels an open order
        :param str id: order id
        :param str symbol: unified symbol of the market the order was made in
        :param dict params: extra parameters specific to the bw api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' cancelOrder() requires a symbol argument')
        self.load_markets()
        market = self.market(symbol)
        request = {
            'marketId': market['id'],
            'entrustId': id,
        }
        response = self.privatePostExchangeEntrustControllerWebsiteEntrustControllerCancelEntrust(self.extend(request, params))
        #
        #     {
        #         "datas": null,
        #         "resMsg": {"message": "success !", "method": null, "code": "1"}
        #     }
        #
        return {
            'info': response,
            'id': id,
        }

    def fetch_open_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all unfilled currently open orders
        :param str symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch open orders for
        :param int|None limit: the maximum number of  open orders structures to retrieve
        :param dict params: extra parameters specific to the bw api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' fetchOpenOrders() requires a symbol argument')
        self.load_markets()
        market = self.market(symbol)
        request = {
            'marketId': market['id'],
            # 'pageSize': limit,  # documented as required, but it works without it
            # 'pageIndex': 0,  # also works without it, most likely a typo in the docs
        }
        if limit is not None:
            request['pageSize'] = limit  # default limit is 20
        response = self.privateGetExchangeEntrustControllerWebsiteEntrustControllerGetUserEntrustRecordFromCacheWithPage(self.extend(request, params))
        #
        #     {
        #         "datas": {
        #             "pageNum": 1,
        #             "pageSize": 2,
        #             "totalPage": 20,
        #             "totalRow": 40,
        #             "entrustList": [
        #                 {
        #                     "amount": "14.050000000000000000",        # Order quantity
        #                     "rangeType": 0,                           # Commission type 0: limit price commission 1: interval commission
        #                     "totalMoney": "20372.500000000000000000",  # Total order amount
        #                     "entrustId": "E6581108027628212224",      # Order id
        #                     "type": 1,                                # Trade direction, 0: sell, 1: buy, -1: cancel
        #                     "completeAmount": "0",                    # Quantity sold
        #                     "marketId": "318",                        # The market id
        #                     "createTime": 1569058424861,              # Create time
        #                     "price": "1450.000000000",                # price
        #                     "completeTotalMoney": "0",                # Quantity sold
        #                     "entrustType": 0,                         # Commission type, 0: ordinary current price commission, 1: lever commission
        #                     "status": 0                               # Order status,-3:fund Freeze exception,Order status to be confirmed  -2: fund freeze failure, order failure, -1: insufficient funds, order failure, 0: pending order, 1: cancelled, 2: dealt, 3: partially dealt
        #                 },
        #             ],
        #         },
        #         "resMsg": {"message": "success !", "method": null, "code": "1"},
        #     }
        #
        data = self.safe_value(response, 'datas', {})
        orders = self.safe_value(data, 'entrustList', [])
        return self.parse_orders(orders, market, since, limit)

    def fetch_closed_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetches information on multiple closed orders made by the user
        :param str symbol: unified market symbol of the market orders were made in
        :param int|None since: the earliest time in ms to fetch orders for
        :param int|None limit: the maximum number of  orde structures to retrieve
        :param dict params: extra parameters specific to the bw api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' fetchClosedOrders() requires a symbol argument')
        self.load_markets()
        market = self.market(symbol)
        request = {
            'marketId': market['id'],
        }
        if limit is not None:
            request['pageSize'] = limit  # default limit is 20
        if since is not None:
            request['startDateTime'] = since
        response = self.privateGetExchangeEntrustControllerWebsiteEntrustControllerGetUserEntrustList(self.extend(request, params))
        data = self.safe_value(response, 'datas', {})
        orders = self.safe_value(data, 'entrustList', [])
        return self.parse_orders(orders, market, since, limit)

    def fetch_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetches information on multiple orders made by the user
        :param str symbol: unified market symbol of the market orders were made in
        :param int|None since: the earliest time in ms to fetch orders for
        :param int|None limit: the maximum number of  orde structures to retrieve
        :param dict params: extra parameters specific to the bw api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        if symbol is None:
            raise ArgumentsRequired(self.id + ' fetchOrders() requires a symbol argument')
        self.load_markets()
        market = self.market(symbol)
        request = {
            'marketId': market['id'],
            # 'pageSize': limit,  # documented as required, but it works without it
            # 'pageIndex': 0,  # also works without it, most likely a typo in the docs
            # 'type': 0,  # 0 = sell, 1 = buy, -1 = cancel
            # 'status': -1,  # -1 = insufficient funds, failed orders, 0 = pending orders, 1 = canceled, 2 = closed, 3 = partial
            # 'startDateTime': since,
            # 'endDateTime': self.milliseconds(),
        }
        if since is not None:
            request['startDateTime'] = since
        if limit is not None:
            request['pageSize'] = limit  # default limit is 20
        response = self.privateGetExchangeEntrustControllerWebsiteEntrustControllerGetUserEntrustList(self.extend(request, params))
        #
        #     {
        #         "datas": {
        #             "pageNum": 1,
        #             "pageSize": 2,
        #             "totalPage": 20,
        #             "totalRow": 40,
        #             "entrustList": [
        #                 {
        #                     "amount": "14.050000000000000000",        # Order quantity
        #                     "rangeType": 0,                           # Commission type 0: limit price commission 1: interval commission
        #                     "totalMoney": "20372.500000000000000000",  # Total order amount
        #                     "entrustId": "E6581108027628212224",      # Order id
        #                     "type": 1,                                # Trade direction, 0: sell, 1: buy, -1: cancel
        #                     "completeAmount": "0",                    # Quantity sold
        #                     "marketId": "318",                        # The market id
        #                     "createTime": 1569058424861,              # Create time
        #                     "price": "1450.000000000",                # price
        #                     "completeTotalMoney": "0",                # Quantity sold
        #                     "entrustType": 0,                         # Commission type, 0: ordinary current price commission, 1: lever commission
        #                     "status": 0                               # Order status,-3:fund Freeze exception,Order status to be confirmed  -2: fund freeze failure, order failure, -1: insufficient funds, order failure, 0: pending order, 1: cancelled, 2: dealt, 3: partially dealt
        #                 },
        #             ],
        #         },
        #         "resMsg": {"message": "success !", "method": null, "code": "1"},
        #     }
        #
        data = self.safe_value(response, 'datas', {})
        orders = self.safe_value(data, 'entrustList', [])
        return self.parse_orders(orders, market, since, limit)

    def sign(self, path, api='public', method='GET', params={}, headers=None, body=None):
        url = self.implode_hostname(self.urls['api']) + '/' + path
        if method == 'GET':
            if params:
                url += '?' + self.urlencode(params)
        else:
            body = self.json(params)
        if api == 'private':
            ms = str(self.milliseconds())
            content = ''
            if method == 'GET':
                sortedParams = self.keysort(params)
                keys = list(sortedParams.keys())
                for i in range(0, len(keys)):
                    key = keys[i]
                    content += key + str(sortedParams[key])
            else:
                content = body
            signature = self.apiKey + ms + content + self.secret
            hash = self.hash(self.encode(signature), 'md5')
            if not headers:
                headers = {}
            headers['Apiid'] = self.apiKey
            headers['Timestamp'] = ms
            headers['Sign'] = hash
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def fetch_deposit_address(self, code, params={}):
        """
        fetch the deposit address for a currency associated with self account
        :param str code: unified currency code
        :param dict params: extra parameters specific to the bw api endpoint
        :returns dict: an `address structure <https://docs.ccxt.com/en/latest/manual.html#address-structure>`
        """
        self.load_markets()
        currency = self.currency(code)
        request = {
            'currencyTypeName': currency['name'],
        }
        response = self.privatePostExchangeFundControllerWebsiteFundcontrollerGetPayinAddress(self.extend(request, params))
        #
        #     {
        #         "datas": {
        #             "isMemo": True,                                # 是否为memo 格式，false：否，true ：是
        #             "address": "bweosdeposit_787928102918558272",  # 充币地址
        #             "memo": "787928102918558272",                  # 币种memo
        #             "account": "bweosdeposit"                      # 币种账户
        #         },
        #         "resMsg": {"message": "success !", "method": null, "code": "1"}
        #     }
        #
        data = self.safe_value(response, 'datas', {})
        address = self.safe_string(data, 'address')
        tag = self.safe_string(data, 'memo')
        self.check_address(address)
        return {
            'currency': code,
            'address': self.check_address(address),
            'tag': tag,
            'network': None,
            'info': response,
        }

    def parse_transaction_status(self, status):
        statuses = {
            '-1': 'canceled',  # or auditing failed
            '0': 'pending',
            '1': 'ok',
        }
        return self.safe_string(statuses, status, status)

    def parse_transaction(self, transaction, currency=None):
        #
        # fetchDeposits
        #
        #     {
        #         "depositId": "D6574268549744189441",                  # Deposit ID
        #         "amount": "54.753589700000000000",                    # Deposit amount
        #         "txId": "INNER_SYSTEM_TRANSFER_1198941",              # Trading ID
        #         "confirmTimes": 0,                                    # Confirmation number
        #         "depositAddress": "bweosdeposit_787928102918558272",  # Deposit address
        #         "createTime": "2019-09-02 20:36:08.0",                # Deposit time
        #         "status": 1,                                          # Deposit status, 0: not received, 1: received
        #         "currencyTypeId": 7,                                  # Token ID
        #     }
        #
        # fetchWithdrawals
        #
        #     {
        #         "withdrawalId": "W6527498439872634880",      # Withdrawal ID
        #         "fees": "0.500000000000000000",              # Withdrawal fee
        #         "withdrawalAddress": "okbtothemoon_941657",  # Withdrawal address
        #         "currencyId": "7",                           # Token ID
        #         "amount": "10.000000000000000000",           # Withdrawal amount
        #         "state": 1,                                  # Status, 1: normal, -1: del         #         "verifyStatus": 1,                           # Audit status, 0: to be audited, 1: auditing passed, -1: auditing failed
        #         "createTime": 1556276903656,                 # WIthdrawal time
        #         "actuallyAmount": "9.500000000000000000",    # Actual amount received
        #     }
        #
        id = self.safe_string(transaction, 'depositId', 'withdrawalId')
        address = self.safe_string_2(transaction, 'depositAddress', 'withdrawalAddress')
        currencyId = self.safe_string_2(transaction, 'currencyId', 'currencyTypeId')
        code = None
        if currencyId in self.currencies_by_id:
            currency = self.currencies_by_id[currencyId]
        if (code is None) and (currency is not None):
            code = currency['code']
        type = 'deposit' if ('depositId' in transaction) else 'withdrawal'
        amount = self.safe_number_2(transaction, 'actuallyAmount', 'amount')
        status = self.parse_transaction_status(self.safe_string_2(transaction, 'verifyStatus', 'state'))
        timestamp = self.safe_integer(transaction, 'createTime')
        txid = self.safe_string(transaction, 'txId')
        fee = None
        feeCost = self.safe_number(transaction, 'fees')
        if feeCost is not None:
            fee = {
                'cost': feeCost,
                'currency': code,
            }
        return {
            'info': transaction,
            'id': id,
            'txid': txid,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'network': None,
            'addressFrom': None,
            'address': address,
            'addressTo': None,
            'tagFrom': None,
            'tag': None,
            'tagTo': None,
            'type': type,
            'amount': amount,
            'currency': code,
            'status': status,
            'updated': None,
            'fee': fee,
        }

    def fetch_deposits(self, code=None, since=None, limit=None, params={}):
        """
        fetch all deposits made to an account
        :param str code: unified currency code
        :param int|None since: the earliest time in ms to fetch deposits for
        :param int|None limit: the maximum number of deposits structures to retrieve
        :param dict params: extra parameters specific to the bw api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        if code is None:
            raise ArgumentsRequired(self.id + ' fetchDeposits() requires a currency code argument')
        self.load_markets()
        currency = self.currency(code)
        request = {
            'currencyTypeName': currency['name'],
            # 'pageSize': limit,  # documented as required, but it works without it
            # 'pageNum': 0,  # also works without it, most likely a typo in the docs
            # 'sort': 1,  # 1 = asc, 0 = desc
        }
        if limit is not None:
            request['pageSize'] = limit  # default 50
        response = self.privatePostExchangeFundControllerWebsiteFundcontrollerGetPayinCoinRecord(self.extend(request, params))
        #
        #     {
        #         "datas": {
        #             "totalRow":2,
        #             "totalPage": 1,
        #             "pageSize": 2,
        #             "pageNum": 1,
        #             "list": [
        #                 {
        #                     "depositId": "D6574268549744189441",                  # Deposit ID
        #                     "amount": "54.753589700000000000",                    # Deposit amount
        #                     "txId": "INNER_SYSTEM_TRANSFER_1198941",              # Trading ID
        #                     "confirmTimes": 0,                                    # Confirmation number
        #                     "depositAddress": "bweosdeposit_787928102918558272",  # Deposit address
        #                     "createTime": "2019-09-02 20:36:08.0",                # Deposit time
        #                     "status": 1,                                          # Deposit status, 0: not received, 1: received
        #                     "currencyTypeId": 7,                                  # Token ID
        #                 },
        #             ]
        #         },
        #         "resMsg": {"message": "success !", "method": null, "code": "1"},
        #     }
        #
        data = self.safe_value(response, 'datas', {})
        deposits = self.safe_value(data, 'list', [])
        return self.parse_transactions(deposits, code, since, limit)

    def fetch_withdrawals(self, code=None, since=None, limit=None, params={}):
        """
        fetch all withdrawals made from an account
        :param str code: unified currency code
        :param int|None since: the earliest time in ms to fetch withdrawals for
        :param int|None limit: the maximum number of withdrawals structures to retrieve
        :param dict params: extra parameters specific to the bw api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        if code is None:
            raise ArgumentsRequired(self.id + ' fetchWithdrawals() requires a currency code argument')
        self.load_markets()
        currency = self.currency(code)
        request = {
            'currencyId': currency['id'],
            # 'pageSize': limit,  # documented as required, but it works without it
            # 'pageIndex': 0,  # also works without it, most likely a typo in the docs
            # 'tab': 'all',  # all, wait(submitted, not audited), success(auditing passed), fail(auditing failed), cancel(canceled by user)
        }
        if limit is not None:
            request['pageSize'] = limit  # default 50
        response = self.privateGetExchangeFundControllerWebsiteFundwebsitecontrollerGetpayoutcoinrecord(self.extend(request, params))
        #
        #     {
        #         "datas": {
        #             "totalRow": 1,
        #             "totalPage": 1,
        #             "pageSize": 2,
        #             "pageNum": 1,
        #             "list": [
        #                 {
        #                     "withdrawalId": "W6527498439872634880",      # Withdrawal ID
        #                     "fees": "0.500000000000000000",              # Withdrawal fee
        #                     "withdrawalAddress": "okbtothemoon_941657",  # Withdrawal address
        #                     "currencyId": "7",                           # Token ID
        #                     "amount": "10.000000000000000000",           # Withdrawal amount
        #                     "state": 1,                                  # Status, 1: normal, -1: del         #                     "verifyStatus": 1,                           # Audit status, 0: to be audited, 1: auditing passed, -1: auditing failed
        #                     "createTime": 1556276903656,                 # WIthdrawal time
        #                     "actuallyAmount": "9.500000000000000000",    # Actual amount received
        #                 },
        #             ],
        #         },
        #         "resMsg": {"message": "success !", "method": null, "code": "1"},
        #     }
        #
        data = self.safe_value(response, 'datas', {})
        withdrawals = self.safe_value(data, 'list', [])
        return self.parse_transactions(withdrawals, code, since, limit)

    def handle_errors(self, httpCode, reason, url, method, headers, body, response, requestHeaders, requestBody):
        if not response:
            return  # default error handler
        resMsg = self.safe_value(response, 'resMsg')
        errorCode = self.safe_string(resMsg, 'code')
        if errorCode != '1':
            feedback = self.id + ' ' + self.json(response)
            self.throw_exactly_matched_exception(self.exceptions['exact'], errorCode, feedback)
            raise ExchangeError(feedback)  # unknown error
