# -*- coding: utf-8 -*-

# PLEASE DO NOT EDIT THIS FILE, IT IS GENERATED AND WILL BE OVERWRITTEN:
# https://github.com/ccxt/ccxt/blob/master/CONTRIBUTING.md#how-to-contribute-code

from ccxt.async_support.base.exchange import Exchange
import hashlib
from ccxt.base.errors import ExchangeError
from ccxt.base.errors import AuthenticationError
from ccxt.base.errors import PermissionDenied
from ccxt.base.errors import AccountSuspended
from ccxt.base.errors import BadSymbol
from ccxt.base.errors import InsufficientFunds
from ccxt.base.errors import InvalidOrder
from ccxt.base.errors import OrderNotFound
from ccxt.base.errors import CancelPending
from ccxt.base.errors import RateLimitExceeded
from ccxt.base.errors import ExchangeNotAvailable
from ccxt.base.errors import OnMaintenance
from ccxt.base.errors import InvalidNonce
from ccxt.base.errors import RequestTimeout
from ccxt.base.decimal_to_precision import TICK_SIZE
from ccxt.base.precise import Precise


class poloniex(Exchange):

    def describe(self):
        return self.deep_extend(super(poloniex, self).describe(), {
            'id': 'poloniex',
            'name': 'Poloniex',
            'countries': ['US'],
            'rateLimit': 166.667,  # 6 calls per second,  1000ms / 6 = 166.667ms between requests
            'certified': False,
            'pro': True,
            'has': {
                'CORS': None,
                'spot': True,
                'margin': None,  # has but not fully implemented
                'swap': None,  # has but not fully implemented
                'future': None,  # has but not fully implemented
                'option': None,
                'cancelAllOrders': True,
                'cancelOrder': True,
                'createDepositAddress': True,
                'createMarketOrder': None,
                'createOrder': True,
                'editOrder': True,
                'fetchBalance': True,
                'fetchClosedOrder': 'emulated',
                'fetchCurrencies': True,
                'fetchDepositAddress': True,
                'fetchDeposits': True,
                'fetchMarginMode': False,
                'fetchMarkets': True,
                'fetchMyTrades': True,
                'fetchOHLCV': True,
                'fetchOpenInterestHistory': False,
                'fetchOpenOrder': True,  # True endpoint for a single open order
                'fetchOpenOrders': True,  # True endpoint for open orders
                'fetchOrderBook': True,
                'fetchOrderBooks': True,
                'fetchOrderTrades': True,  # True endpoint for trades of a single open or closed order
                'fetchPosition': True,
                'fetchPositionMode': False,
                'fetchTicker': True,
                'fetchTickers': True,
                'fetchTrades': True,
                'fetchTradingFee': False,
                'fetchTradingFees': True,
                'fetchTransactions': True,
                'fetchTransfer': False,
                'fetchTransfers': False,
                'fetchWithdrawals': True,
                'transfer': True,
                'withdraw': True,
            },
            'timeframes': {
                '5m': 300,
                '15m': 900,
                '30m': 1800,
                '2h': 7200,
                '4h': 14400,
                '1d': 86400,
            },
            'urls': {
                'logo': 'https://user-images.githubusercontent.com/1294454/27766817-e9456312-5ee6-11e7-9b3c-b628ca5626a5.jpg',
                'api': {
                    'public': 'https://poloniex.com/public',
                    'private': 'https://poloniex.com/tradingApi',
                },
                'www': 'https://www.poloniex.com',
                'doc': 'https://docs.poloniex.com',
                'fees': 'https://poloniex.com/fees',
                'referral': 'https://poloniex.com/signup?c=UBFZJRPJ',
            },
            'api': {
                'public': {
                    'get': {
                        'return24hVolume': 1,
                        'returnChartData': 1,
                        'returnCurrencies': 1,
                        'returnLoanOrders': 1,
                        'returnOrderBook': 1,
                        'returnTicker': 1,
                        'returnTradeHistory': 1,
                    },
                },
                'private': {
                    'post': {
                        'buy': 1,
                        'cancelLoanOffer': 1,
                        'cancelOrder': 1,
                        'cancelAllOrders': 1,
                        'closeMarginPosition': 1,
                        'createLoanOffer': 1,
                        'generateNewAddress': 1,
                        'getMarginPosition': 1,
                        'marginBuy': 1,
                        'marginSell': 1,
                        'moveOrder': 1,
                        'returnActiveLoans': 1,
                        'returnAvailableAccountBalances': 1,
                        'returnBalances': 1,
                        'returnCompleteBalances': 1,
                        'returnDepositAddresses': 1,
                        'returnDepositsWithdrawals': 1,
                        'returnFeeInfo': 1,
                        'returnLendingHistory': 1,
                        'returnMarginAccountSummary': 1,
                        'returnOpenLoanOffers': 1,
                        'returnOpenOrders': 1,
                        'returnOrderTrades': 1,
                        'returnOrderStatus': 1,
                        'returnTradableBalances': 1,
                        'returnTradeHistory': 1,
                        'sell': 1,
                        'toggleAutoRenew': 1,
                        'transferBalance': 1,
                        'withdraw': 1,
                    },
                },
            },
            'fees': {
                'trading': {
                    'feeSide': 'get',
                    # starting from Jan 8 2020
                    'maker': self.parse_number('0.0009'),
                    'taker': self.parse_number('0.0009'),
                },
                'funding': {},
            },
            'limits': {
                'amount': {
                    'min': 0.000001,
                    'max': None,
                },
                'price': {
                    'min': 0.00000001,
                    'max': None,
                },
                'cost': {
                    'min': None,
                    'max': None,
                },
            },
            'commonCurrencies': {
                'AIR': 'AirCoin',
                'APH': 'AphroditeCoin',
                'BCC': 'BTCtalkcoin',
                'BCHABC': 'BCHABC',
                'BDG': 'Badgercoin',
                'BTM': 'Bitmark',
                'CON': 'Coino',
                'GOLD': 'GoldEagles',
                'GPUC': 'GPU',
                'HOT': 'Hotcoin',
                'ITC': 'Information Coin',
                'KEY': 'KEYCoin',
                'MASK': 'NFTX Hashmasks Index',  # conflict with Mask Network
                'MEME': 'Degenerator Meme',  # Degenerator Meme migrated to Meme Inu, self exchange still has the old price
                'PLX': 'ParallaxCoin',
                'REPV2': 'REP',
                'STR': 'XLM',
                'SOC': 'SOCC',
                'TRADE': 'Unitrade',
                'XAP': 'API Coin',
                # self is not documented in the API docs for Poloniex
                # https://github.com/ccxt/ccxt/issues/7084
                # when the user calls withdraw('USDT', amount, address, tag, params)
                # with params = {'currencyToWithdrawAs': 'USDTTRON'}
                # or params = {'currencyToWithdrawAs': 'USDTETH'}
                # fetchWithdrawals('USDT') returns the corresponding withdrawals
                # with a USDTTRON or a USDTETH currency id, respectfully
                # therefore we have map them back to the original code USDT
                # otherwise the returned withdrawals are filtered out
                'USDTTRON': 'USDT',
                'USDTETH': 'USDT',
                'UST': 'USTC',
            },
            'options': {
                'networks': {
                    'ERC20': 'ETH',
                    'TRX': 'TRON',
                    'TRC20': 'TRON',
                },
                'limits': {
                    'cost': {
                        'min': {
                            'BTC': 0.0001,
                            'ETH': 0.0001,
                            'USDT': 1.0,
                            'TRX': 100,
                            'BNB': 0.06,
                            'USDC': 1.0,
                            'USDJ': 1.0,
                            'TUSD': 0.0001,
                            'DAI': 1.0,
                            'PAX': 1.0,
                            'BUSD': 1.0,
                        },
                    },
                },
                'accountsByType': {
                    'spot': 'exchange',
                    'margin': 'margin',
                    'future': 'futures',
                    'lending': 'lending',
                },
                'accountsById': {
                    'exchange': 'spot',
                    'margin': 'margin',
                    'futures': 'future',
                    'lending': 'lending',
                },
            },
            'precisionMode': TICK_SIZE,
            'exceptions': {
                'exact': {
                    'You may only place orders that reduce your position.': InvalidOrder,
                    'Invalid order number, or you are not the person who placed the order.': OrderNotFound,
                    'Permission denied': PermissionDenied,
                    'Permission denied.': PermissionDenied,
                    'Connection timed out. Please try again.': RequestTimeout,
                    'Internal error. Please try again.': ExchangeNotAvailable,
                    'Currently in maintenance mode.': OnMaintenance,
                    'Order not found, or you are not the person who placed it.': OrderNotFound,
                    'Invalid API key/secret pair.': AuthenticationError,
                    'Please do not make more than 8 API calls per second.': RateLimitExceeded,
                    'This IP has been temporarily throttled. Please ensure your requests are valid and try again in one minute.': RateLimitExceeded,
                    'Rate must be greater than zero.': InvalidOrder,  # {"error":"Rate must be greater than zero."}
                    'Invalid currency pair.': BadSymbol,  # {"error":"Invalid currency pair."}
                    'Invalid currencyPair parameter.': BadSymbol,  # {"error":"Invalid currencyPair parameter."}
                    'Trading is disabled in self market.': BadSymbol,  # {"error":"Trading is disabled in self market."}
                    'Invalid orderNumber parameter.': OrderNotFound,
                    'Order is beyond acceptable bounds.': InvalidOrder,  # {"error":"Order is beyond acceptable bounds.","fee":"0.00155000","currencyPair":"USDT_BOBA"}
                    'This account is closed.': AccountSuspended,  # {"error":"This account is closed."}
                },
                'broad': {
                    'Total must be at least': InvalidOrder,  # {"error":"Total must be at least 0.0001."}
                    'This account is frozen': AccountSuspended,  # {"error":"This account is frozen for trading."} or {"error":"This account is frozen."}
                    'This account is locked.': AccountSuspended,  # {"error":"This account is locked."}
                    'Not enough': InsufficientFunds,
                    'Nonce must be greater': InvalidNonce,
                    'You have already called cancelOrder': CancelPending,  # {"error":"You have already called cancelOrder, moveOrder, or cancelReplace on self order. Please wait for that call's response."}
                    'Amount must be at least': InvalidOrder,  # {"error":"Amount must be at least 0.000001."}
                    'is either completed or does not exist': OrderNotFound,  # {"error":"Order 587957810791 is either completed or does not exist."}
                    'Error pulling ': ExchangeError,  # {"error":"Error pulling order book"}
                },
            },
        })

    def parse_ohlcv(self, ohlcv, market=None):
        #
        #     {
        #         "date":1590913773,
        #         "high":0.02491611,
        #         "low":0.02491611,
        #         "open":0.02491611,
        #         "close":0.02491611,
        #         "volume":0,
        #         "quoteVolume":0,
        #         "weightedAverage":0.02491611
        #     }
        #
        return [
            self.safe_timestamp(ohlcv, 'date'),
            self.safe_number(ohlcv, 'open'),
            self.safe_number(ohlcv, 'high'),
            self.safe_number(ohlcv, 'low'),
            self.safe_number(ohlcv, 'close'),
            self.safe_number(ohlcv, 'quoteVolume'),
        ]

    async def fetch_ohlcv(self, symbol, timeframe='5m', since=None, limit=None, params={}):
        """
        fetches historical candlestick data containing the open, high, low, and close price, and the volume of a market
        :param str symbol: unified symbol of the market to fetch OHLCV data for
        :param str timeframe: the length of time each candle represents
        :param int|None since: timestamp in ms of the earliest candle to fetch
        :param int|None limit: the maximum amount of candles to fetch
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns [[int]]: A list of candles ordered as timestamp, open, high, low, close, volume
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'currencyPair': market['id'],
            'period': self.timeframes[timeframe],
        }
        if since is None:
            request['end'] = self.seconds()
            if limit is None:
                request['start'] = request['end'] - self.parse_timeframe('1w')  # max range = 1 week
            else:
                request['start'] = request['end'] - limit * self.parse_timeframe(timeframe)
        else:
            request['start'] = int(since / 1000)
            if limit is not None:
                end = self.sum(request['start'], limit * self.parse_timeframe(timeframe))
                request['end'] = end
        response = await self.publicGetReturnChartData(self.extend(request, params))
        #
        #     [
        #         {"date":1590913773,"high":0.02491611,"low":0.02491611,"open":0.02491611,"close":0.02491611,"volume":0,"quoteVolume":0,"weightedAverage":0.02491611},
        #         {"date":1590913800,"high":0.02495324,"low":0.02489501,"open":0.02491797,"close":0.02493693,"volume":0.0927415,"quoteVolume":3.7227869,"weightedAverage":0.02491185},
        #         {"date":1590914100,"high":0.02498596,"low":0.02488503,"open":0.02493033,"close":0.02497896,"volume":0.21196348,"quoteVolume":8.50291888,"weightedAverage":0.02492832},
        #     ]
        #
        return self.parse_ohlcvs(response, market, timeframe, since, limit)

    async def load_markets(self, reload=False, params={}):
        markets = await super(poloniex, self).load_markets(reload, params)
        currenciesByNumericId = self.safe_value(self.options, 'currenciesByNumericId')
        if (currenciesByNumericId is None) or reload:
            self.options['currenciesByNumericId'] = self.index_by(self.currencies, 'numericId')
        return markets

    async def fetch_markets(self, params={}):
        """
        retrieves data on all markets for poloniex
        :param dict params: extra parameters specific to the exchange api endpoint
        :returns [dict]: an array of objects representing market data
        """
        markets = await self.publicGetReturnTicker(params)
        keys = list(markets.keys())
        result = []
        for i in range(0, len(keys)):
            id = keys[i]
            market = markets[id]
            quoteId, baseId = id.split('_')
            base = self.safe_currency_code(baseId)
            quote = self.safe_currency_code(quoteId)
            isFrozen = self.safe_string(market, 'isFrozen')
            marginEnabled = self.safe_integer(market, 'marginTradingEnabled')
            # these are known defaults
            result.append({
                'id': id,
                'numericId': self.safe_integer(market, 'id'),
                'symbol': base + '/' + quote,
                'base': base,
                'quote': quote,
                'settle': None,
                'baseId': baseId,
                'quoteId': quoteId,
                'settleId': None,
                'type': 'spot',
                'spot': True,
                'margin': (marginEnabled == 1),
                'swap': False,
                'future': False,
                'option': False,
                'active': (isFrozen != '1'),
                'contract': False,
                'linear': None,
                'inverse': None,
                'contractSize': None,
                'expiry': None,
                'expiryDatetime': None,
                'strike': None,
                'optionType': None,
                'precision': {
                    'amount': self.parse_number('0.00000001'),
                    'price': self.parse_number('0.00000001'),
                },
                'limits': self.extend(self.limits, {
                    'leverage': {
                        'min': None,
                        'max': None,
                    },
                    'cost': {
                        'min': self.safe_value(self.options['limits']['cost']['min'], quote),
                        'max': None,
                    },
                }),
                'info': market,
            })
        return result

    def parse_balance(self, response):
        result = {
            'info': response,
            'timestamp': None,
            'datetime': None,
        }
        currencyIds = list(response.keys())
        for i in range(0, len(currencyIds)):
            currencyId = currencyIds[i]
            balance = self.safe_value(response, currencyId, {})
            code = self.safe_currency_code(currencyId)
            account = self.account()
            account['free'] = self.safe_string(balance, 'available')
            account['used'] = self.safe_string(balance, 'onOrders')
            result[code] = account
        return self.safe_balance(result)

    async def fetch_balance(self, params={}):
        """
        query for balance and get the amount of funds available for trading or funds locked in orders
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: a `balance structure <https://docs.ccxt.com/en/latest/manual.html?#balance-structure>`
        """
        await self.load_markets()
        request = {
            'account': 'all',
        }
        response = await self.privatePostReturnCompleteBalances(self.extend(request, params))
        #
        #     {
        #         "1CR":{"available":"0.00000000","onOrders":"0.00000000","btcValue":"0.00000000"},
        #         "ABY":{"available":"0.00000000","onOrders":"0.00000000","btcValue":"0.00000000"},
        #         "AC":{"available":"0.00000000","onOrders":"0.00000000","btcValue":"0.00000000"},
        #     }
        #
        return self.parse_balance(response)

    async def fetch_trading_fees(self, params={}):
        """
        fetch the trading fees for multiple markets
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: a dictionary of `fee structures <https://docs.ccxt.com/en/latest/manual.html#fee-structure>` indexed by market symbols
        """
        await self.load_markets()
        response = await self.privatePostReturnFeeInfo(params)
        #
        #     {
        #         makerFee: '0.00100000',
        #         takerFee: '0.00200000',
        #         marginMakerFee: '0.00100000',
        #         marginTakerFee: '0.00200000',
        #         thirtyDayVolume: '106.08463302',
        #         nextTier: 500000,
        #     }
        #
        result = {}
        for i in range(0, len(self.symbols)):
            symbol = self.symbols[i]
            result[symbol] = {
                'info': response,
                'symbol': symbol,
                'maker': self.safe_number(response, 'makerFee'),
                'taker': self.safe_number(response, 'takerFee'),
                'percentage': True,
                'tierBased': True,
            }
        return result

    async def fetch_order_book(self, symbol, limit=None, params={}):
        """
        fetches information on open orders with bid(buy) and ask(sell) prices, volumes and other data
        :param str symbol: unified symbol of the market to fetch the order book for
        :param int|None limit: the maximum amount of order book entries to return
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: A dictionary of `order book structures <https://docs.ccxt.com/en/latest/manual.html#order-book-structure>` indexed by market symbols
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'currencyPair': market['id'],
        }
        if limit is not None:
            request['depth'] = limit  # 100
        response = await self.publicGetReturnOrderBook(self.extend(request, params))
        orderbook = self.parse_order_book(response, market['symbol'])
        orderbook['nonce'] = self.safe_integer(response, 'seq')
        return orderbook

    async def fetch_order_books(self, symbols=None, limit=None, params={}):
        """
        fetches information on open orders with bid(buy) and ask(sell) prices, volumes and other data for multiple markets
        :param [str]|None symbols: not used by poloniex fetchOrderBooks()
        :param int|None limit: max number of entries per orderbook to return, default is None
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: a dictionary of `order book structures <https://docs.ccxt.com/en/latest/manual.html#order-book-structure>` indexed by market symbol
        """
        await self.load_markets()
        request = {
            'currencyPair': 'all',
        }
        if limit is not None:
            request['depth'] = limit  # 100
        response = await self.publicGetReturnOrderBook(self.extend(request, params))
        marketIds = list(response.keys())
        result = {}
        for i in range(0, len(marketIds)):
            marketId = marketIds[i]
            symbol = None
            if marketId in self.markets_by_id:
                symbol = self.markets_by_id[marketId]['symbol']
            else:
                quoteId, baseId = marketId.split('_')
                base = self.safe_currency_code(baseId)
                quote = self.safe_currency_code(quoteId)
                symbol = base + '/' + quote
            orderbook = self.parse_order_book(response[marketId], symbol)
            orderbook['nonce'] = self.safe_integer(response[marketId], 'seq')
            result[symbol] = orderbook
        return result

    def parse_ticker(self, ticker, market=None):
        # {
        #     id: '121',
        #     last: '43196.31469670',
        #     lowestAsk: '43209.61843169',
        #     highestBid: '43162.41965234',
        #     percentChange: '0.00963340',
        #     baseVolume: '13444643.33799658',
        #     quoteVolume: '315.84780115',
        #     isFrozen: '0',
        #     postOnly: '0',
        #     marginTradingEnabled: '1',
        #     high24hr: '43451.84481934',
        #     low24hr: '41749.89529736'
        # }
        timestamp = self.milliseconds()
        symbol = self.safe_symbol(None, market)
        last = self.safe_string(ticker, 'last')
        relativeChange = self.safe_string(ticker, 'percentChange')
        percentage = Precise.string_mul(relativeChange, '100')
        return self.safe_ticker({
            'symbol': symbol,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'high': self.safe_string(ticker, 'high24hr'),
            'low': self.safe_string(ticker, 'low24hr'),
            'bid': self.safe_string(ticker, 'highestBid'),
            'bidVolume': None,
            'ask': self.safe_string(ticker, 'lowestAsk'),
            'askVolume': None,
            'vwap': None,
            'open': None,
            'close': last,
            'last': last,
            'previousClose': None,
            'change': None,
            'percentage': percentage,
            'average': None,
            'baseVolume': self.safe_string(ticker, 'quoteVolume'),
            'quoteVolume': self.safe_string(ticker, 'baseVolume'),
            'info': ticker,
        }, market)

    async def fetch_tickers(self, symbols=None, params={}):
        """
        fetches price tickers for multiple markets, statistical calculations with the information calculated over the past 24 hours each market
        :param [str]|None symbols: unified symbols of the markets to fetch the ticker for, all market tickers are returned if not assigned
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: an array of `ticker structures <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        await self.load_markets()
        response = await self.publicGetReturnTicker(params)
        ids = list(response.keys())
        result = {}
        for i in range(0, len(ids)):
            id = ids[i]
            symbol = None
            market = None
            if id in self.markets_by_id:
                market = self.markets_by_id[id]
                symbol = market['symbol']
            else:
                quoteId, baseId = id.split('_')
                base = self.safe_currency_code(baseId)
                quote = self.safe_currency_code(quoteId)
                symbol = base + '/' + quote
                market = {'symbol': symbol}
            ticker = response[id]
            result[symbol] = self.parse_ticker(ticker, market)
        return self.filter_by_array(result, 'symbol', symbols)

    async def fetch_currencies(self, params={}):
        """
        fetches all available currencies on an exchange
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: an associative dictionary of currencies
        """
        response = await self.publicGetReturnCurrencies(params)
        #
        #     {
        #       "id": "293",
        #       "name": "0x",
        #       "humanType": "Sweep to Main Account",
        #       "currencyType": "address",
        #       "txFee": "17.21877546",
        #       "minConf": "12",
        #       "depositAddress": null,
        #       "disabled": "0",
        #       "frozen": "0",
        #       "hexColor": "003831",
        #       "blockchain": "ETH",
        #       "delisted": "0",
        #       "isGeofenced": 0
        #     }
        #
        ids = list(response.keys())
        result = {}
        for i in range(0, len(ids)):
            id = ids[i]
            currency = response[id]
            code = self.safe_currency_code(id)
            delisted = self.safe_integer(currency, 'delisted', 0)
            disabled = self.safe_integer(currency, 'disabled', 0)
            listed = not delisted
            enabled = not disabled
            active = enabled and listed
            numericId = self.safe_integer(currency, 'id')
            fee = self.safe_number(currency, 'txFee')
            result[code] = {
                'id': id,
                'numericId': numericId,
                'code': code,
                'info': currency,
                'name': currency['name'],
                'active': active,
                'deposit': None,
                'withdraw': None,
                'fee': fee,
                'precision': self.parse_number('0.00000001'),
                'limits': {
                    'amount': {
                        'min': self.parse_number('0.00000001'),
                        'max': None,
                    },
                    'withdraw': {
                        'min': fee,
                        'max': None,
                    },
                },
            }
        return result

    async def fetch_ticker(self, symbol, params={}):
        """
        fetches a price ticker, a statistical calculation with the information calculated over the past 24 hours for a specific market
        :param str symbol: unified symbol of the market to fetch the ticker for
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: a `ticker structure <https://docs.ccxt.com/en/latest/manual.html#ticker-structure>`
        """
        await self.load_markets()
        market = self.market(symbol)
        response = await self.publicGetReturnTicker(params)
        # {
        #     "BTC_BTS":{
        #        "id":14,
        #        "last":"0.00000073",
        #        "lowestAsk":"0.00000075",
        #        "highestBid":"0.00000073",
        #        "percentChange":"0.01388888",
        #        "baseVolume":"0.01413528",
        #        "quoteVolume":"19431.16872167",
        #        "isFrozen":"0",
        #        "postOnly":"0",
        #        "marginTradingEnabled":"0",
        #        "high24hr":"0.00000074",
        #        "low24hr":"0.00000071"
        #     },
        #     ...
        # }
        ticker = response[market['id']]
        return self.parse_ticker(ticker, market)

    def parse_trade(self, trade, market=None):
        #
        # fetchTrades
        #
        #      {
        #          globalTradeID: "667563407",
        #          tradeID: "1984256",
        #          date: "2022-03-01 20:06:06",
        #          type: "buy",
        #          rate: "0.13361871",
        #          amount: "28.40841257",
        #          total: "3.79589544",
        #          orderNumber: "159992152911"
        #      }
        #
        # fetchMyTrades
        #
        #     {
        #       globalTradeID: 471030550,
        #       tradeID: '42582',
        #       date: '2020-06-16 09:47:50',
        #       rate: '0.000079980000',
        #       amount: '75215.00000000',
        #       total: '6.01569570',
        #       fee: '0.00095000',
        #       feeDisplay: '0.26636100 TRX(0.07125%)',
        #       orderNumber: '5963454848',
        #       type: 'sell',
        #       category: 'exchange'
        #     }
        #
        # createOrder(taker trades)
        #
        #     {
        #         'amount': '200.00000000',
        #         'date': '2019-12-15 16:04:10',
        #         'rate': '0.00000355',
        #         'total': '0.00071000',
        #         'tradeID': '119871',
        #         'type': 'buy',
        #         'takerAdjustment': '200.00000000'
        #     }
        #
        id = self.safe_string_2(trade, 'globalTradeID', 'tradeID')
        orderId = self.safe_string(trade, 'orderNumber')
        timestamp = self.parse8601(self.safe_string(trade, 'date'))
        marketId = self.safe_string(trade, 'currencyPair')
        market = self.safe_market(marketId, market, '_')
        symbol = market['symbol']
        side = self.safe_string(trade, 'type')
        fee = None
        priceString = self.safe_string(trade, 'rate')
        amountString = self.safe_string(trade, 'amount')
        costString = self.safe_string(trade, 'total')
        feeDisplay = self.safe_string(trade, 'feeDisplay')
        if feeDisplay is not None:
            parts = feeDisplay.split(' ')
            feeCostString = self.safe_string(parts, 0)
            if feeCostString is not None:
                feeCurrencyId = self.safe_string(parts, 1)
                feeCurrencyCode = self.safe_currency_code(feeCurrencyId)
                feeRateString = self.safe_string(parts, 2)
                if feeRateString is not None:
                    feeRateString = feeRateString.replace('(', '')
                    feeRateParts = feeRateString.split('%')
                    feeRateString = self.safe_string(feeRateParts, 0)
                    feeRateString = Precise.string_div(feeRateString, '100')
                fee = {
                    'cost': feeCostString,
                    'currency': feeCurrencyCode,
                    'rate': feeRateString,
                }
        else:
            feeCostString = self.safe_string(trade, 'fee')
            if feeCostString is not None and market is not None:
                feeCurrencyCode = market['base'] if (side == 'buy') else market['quote']
                feeBase = amountString if (side == 'buy') else costString
                feeRateString = Precise.string_div(feeCostString, feeBase)
                fee = {
                    'cost': feeCostString,
                    'currency': feeCurrencyCode,
                    'rate': feeRateString,
                }
        takerOrMaker = None
        takerAdjustment = self.safe_number(trade, 'takerAdjustment')
        if takerAdjustment is not None:
            takerOrMaker = 'taker'
        return self.safe_trade({
            'id': id,
            'info': trade,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'symbol': symbol,
            'order': orderId,
            'type': 'limit',
            'side': side,
            'takerOrMaker': takerOrMaker,
            'price': priceString,
            'amount': amountString,
            'cost': costString,
            'fee': fee,
        }, market)

    async def fetch_trades(self, symbol, since=None, limit=None, params={}):
        """
        get the list of most recent trades for a particular symbol
        :param str symbol: unified symbol of the market to fetch trades for
        :param int|None since: timestamp in ms of the earliest trade to fetch
        :param int|None limit: the maximum amount of trades to fetch
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html?#public-trades>`
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'currencyPair': market['id'],
        }
        if since is not None:
            request['start'] = int(since / 1000)
            request['end'] = self.seconds()  # last 50000 trades by default
        trades = await self.publicGetReturnTradeHistory(self.extend(request, params))
        return self.parse_trades(trades, market, since, limit)

    async def fetch_my_trades(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all trades made by the user
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch trades for
        :param int|None limit: the maximum number of trades structures to retrieve
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html#trade-structure>`
        """
        await self.load_markets()
        market = None
        if symbol is not None:
            market = self.market(symbol)
        pair = market['id'] if market else 'all'
        request = {'currencyPair': pair}
        if since is not None:
            request['start'] = int(since / 1000)
            request['end'] = self.sum(self.seconds(), 1)  # adding 1 is a fix for  #3411
        # limit is disabled(does not really work as expected)
        if limit is not None:
            request['limit'] = int(limit)
        response = await self.privatePostReturnTradeHistory(self.extend(request, params))
        #
        # specific market(symbol defined)
        #
        #     [
        #         {
        #             globalTradeID: 470912587,
        #             tradeID: '42543',
        #             date: '2020-06-15 17:31:22',
        #             rate: '0.000083840000',
        #             amount: '95237.60321429',
        #             total: '7.98472065',
        #             fee: '0.00095000',
        #             feeDisplay: '0.36137761 TRX(0.07125%)',
        #             orderNumber: '5926344995',
        #             type: 'sell',
        #             category: 'exchange'
        #         },
        #         {
        #             globalTradeID: 470974497,
        #             tradeID: '42560',
        #             date: '2020-06-16 00:41:23',
        #             rate: '0.000078220000',
        #             amount: '1000000.00000000',
        #             total: '78.22000000',
        #             fee: '0.00095000',
        #             feeDisplay: '3.48189819 TRX(0.07125%)',
        #             orderNumber: '5945490830',
        #             type: 'sell',
        #             category: 'exchange'
        #         }
        #     ]
        #
        # all markets(symbol None)
        #
        #     {
        #        BTC_GNT: [{
        #             globalTradeID: 470839947,
        #             tradeID: '4322347',
        #             date: '2020-06-15 12:25:24',
        #             rate: '0.000005810000',
        #             amount: '1702.04429303',
        #             total: '0.00988887',
        #             fee: '0.00095000',
        #             feeDisplay: '4.18235294 TRX(0.07125%)',
        #             orderNumber: '102290272520',
        #             type: 'buy',
        #             category: 'exchange'
        #     }, {
        #             globalTradeID: 470895902,
        #             tradeID: '4322413',
        #             date: '2020-06-15 16:19:00',
        #             rate: '0.000005980000',
        #             amount: '18.66879219',
        #             total: '0.00011163',
        #             fee: '0.00095000',
        #             feeDisplay: '0.04733727 TRX(0.07125%)',
        #             orderNumber: '102298304480',
        #             type: 'buy',
        #             category: 'exchange'
        #         }],
        #     }
        #
        result = []
        if market is not None:
            result = self.parse_trades(response, market)
        else:
            if response:
                ids = list(response.keys())
                for i in range(0, len(ids)):
                    id = ids[i]
                    market = None
                    if id in self.markets_by_id:
                        market = self.markets_by_id[id]
                        trades = self.parse_trades(response[id], market)
                        for j in range(0, len(trades)):
                            result.append(trades[j])
                    else:
                        quoteId, baseId = id.split('_')
                        base = self.safe_currency_code(baseId)
                        quote = self.safe_currency_code(quoteId)
                        symbol = base + '/' + quote
                        trades = response[id]
                        for j in range(0, len(trades)):
                            market = {
                                'symbol': symbol,
                                'base': base,
                                'quote': quote,
                            }
                            result.append(self.parse_trade(trades[j], market))
        return self.filter_by_since_limit(result, since, limit)

    def parse_order_status(self, status):
        statuses = {
            'Open': 'open',
            'Partially filled': 'open',
        }
        return self.safe_string(statuses, status, status)

    def parse_order(self, order, market=None):
        #
        # fetchOpenOrder
        #
        #     {
        #         status: 'Open',
        #         rate: '0.40000000',
        #         amount: '1.00000000',
        #         currencyPair: 'BTC_ETH',
        #         date: '2018-10-17 17:04:50',
        #         total: '0.40000000',
        #         type: 'buy',
        #         startingAmount: '1.00000',
        #     }
        #
        # fetchOpenOrders
        #
        #     {
        #         orderNumber: '514514894224',
        #         type: 'buy',
        #         rate: '0.00001000',
        #         startingAmount: '100.00000000',
        #         amount: '100.00000000',
        #         total: '0.00100000',
        #         date: '2018-10-23 17:38:53',
        #         margin: 0,
        #     }
        #
        # createOrder
        #
        #     {
        #         'orderNumber': '9805453960',
        #         'resultingTrades': [
        #             {
        #                 'amount': '200.00000000',
        #                 'date': '2019-12-15 16:04:10',
        #                 'rate': '0.00000355',
        #                 'total': '0.00071000',
        #                 'tradeID': '119871',
        #                 'type': 'buy',
        #                 'takerAdjustment': '200.00000000',
        #             },
        #         ],
        #         'fee': '0.00000000',
        #         'clientOrderId': '12345',
        #         'currencyPair': 'BTC_MANA',
        #         # 'resultingTrades' in editOrder
        #         'resultingTrades': {
        #             'BTC_MANA': [],
        #          }
        #     }
        #
        timestamp = self.safe_integer(order, 'timestamp')
        if timestamp is None:
            timestamp = self.parse8601(self.safe_string(order, 'date'))
        marketId = self.safe_string(order, 'currencyPair')
        market = self.safe_market(marketId, market, '_')
        symbol = market['symbol']
        resultingTrades = self.safe_value(order, 'resultingTrades')
        if not isinstance(resultingTrades, list):
            resultingTrades = self.safe_value(resultingTrades, self.safe_string(market, 'id', marketId))
        price = self.safe_string_2(order, 'price', 'rate')
        remaining = self.safe_string(order, 'amount')
        amount = self.safe_string(order, 'startingAmount')
        status = self.parse_order_status(self.safe_string(order, 'status'))
        side = self.safe_string(order, 'type')
        id = self.safe_string(order, 'orderNumber')
        fee = None
        feeCurrency = self.safe_string(order, 'tokenFeeCurrency')
        feeCost = None
        feeCurrencyCode = None
        rate = self.safe_string(order, 'fee')
        if feeCurrency is None:
            feeCurrencyCode = market['base'] if (side == 'buy') else market['quote']
        else:
            # poloniex accepts a 30% discount to pay fees in TRX
            feeCurrencyCode = self.safe_currency_code(feeCurrency)
            feeCost = self.safe_string(order, 'tokenFee')
        if feeCost is not None:
            fee = {
                'rate': rate,
                'cost': feeCost,
                'currency': feeCurrencyCode,
            }
        clientOrderId = self.safe_string(order, 'clientOrderId')
        return self.safe_order({
            'info': order,
            'id': id,
            'clientOrderId': clientOrderId,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'lastTradeTimestamp': None,
            'status': status,
            'symbol': symbol,
            'type': 'limit',
            'timeInForce': None,
            'postOnly': None,
            'side': side,
            'price': price,
            'stopPrice': None,
            'cost': None,
            'average': None,
            'amount': amount,
            'filled': None,
            'remaining': remaining,
            'trades': resultingTrades,
            'fee': fee,
        }, market)

    def parse_open_orders(self, orders, market, result):
        for i in range(0, len(orders)):
            order = orders[i]
            extended = self.extend(order, {
                'status': 'open',
                'type': 'limit',
                'side': order['type'],
                'price': order['rate'],
            })
            result.append(self.parse_order(extended, market))
        return result

    async def fetch_open_orders(self, symbol=None, since=None, limit=None, params={}):
        """
        fetch all unfilled currently open orders
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch open orders for
        :param int|None limit: the maximum number of  open orders structures to retrieve
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        await self.load_markets()
        market = None
        if symbol is not None:
            market = self.market(symbol)
        pair = market['id'] if market else 'all'
        request = {
            'currencyPair': pair,
        }
        response = await self.privatePostReturnOpenOrders(self.extend(request, params))
        extension = {'status': 'open'}
        if market is None:
            marketIds = list(response.keys())
            openOrders = []
            for i in range(0, len(marketIds)):
                marketId = marketIds[i]
                orders = response[marketId]
                m = self.markets_by_id[marketId]
                openOrders = self.array_concat(openOrders, self.parse_orders(orders, m, None, None, extension))
            return self.filter_by_since_limit(openOrders, since, limit)
        else:
            return self.parse_orders(response, market, since, limit, extension)

    async def create_order(self, symbol, type, side, amount, price=None, params={}):
        """
        create a trade order
        :param str symbol: unified symbol of the market to create an order in
        :param str type: 'market' or 'limit'
        :param str side: 'buy' or 'sell'
        :param float amount: how much of currency you want to trade in units of base currency
        :param float|None price: the price at which the order is to be fullfilled, in units of the quote currency, ignored in market orders
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: an `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        if type == 'market':
            raise ExchangeError(self.id + ' createOrder() does not accept market orders')
        await self.load_markets()
        method = 'privatePost' + self.capitalize(side)
        market = self.market(symbol)
        amount = self.amount_to_precision(symbol, amount)
        request = {
            'currencyPair': market['id'],
            'rate': self.price_to_precision(symbol, price),
            'amount': amount,
        }
        clientOrderId = self.safe_string(params, 'clientOrderId')
        if clientOrderId is not None:
            request['clientOrderId'] = clientOrderId
            params = self.omit(params, 'clientOrderId')
        # remember the timestamp before issuing the request
        response = await getattr(self, method)(self.extend(request, params))
        #
        #     {
        #         'orderNumber': '9805453960',
        #         'resultingTrades': [
        #             {
        #                 'amount': '200.00000000',
        #                 'date': '2019-12-15 16:04:10',
        #                 'rate': '0.00000355',
        #                 'total': '0.00071000',
        #                 'tradeID': '119871',
        #                 'type': 'buy',
        #                 'takerAdjustment': '200.00000000',
        #             },
        #         ],
        #         'fee': '0.00000000',
        #         'currencyPair': 'BTC_MANA',
        #     }
        #
        response = self.extend(response, {
            'type': side,
        })
        return self.parse_order(response, market)

    async def edit_order(self, id, symbol, type, side, amount, price=None, params={}):
        await self.load_markets()
        price = float(price)
        request = {
            'orderNumber': id,
            'rate': self.price_to_precision(symbol, price),
        }
        if amount is not None:
            request['amount'] = self.amount_to_precision(symbol, amount)
        response = await self.privatePostMoveOrder(self.extend(request, params))
        return self.parse_order(response)

    async def cancel_order(self, id, symbol=None, params={}):
        """
        cancels an open order
        :param str id: order id
        :param str|None symbol: unified symbol of the market the order was made in
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: An `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        await self.load_markets()
        request = {}
        clientOrderId = self.safe_value(params, 'clientOrderId')
        if clientOrderId is None:
            request['orderNumber'] = id
        else:
            request['clientOrderId'] = clientOrderId
        params = self.omit(params, 'clientOrderId')
        return await self.privatePostCancelOrder(self.extend(request, params))

    async def cancel_all_orders(self, symbol=None, params={}):
        """
        cancel all open orders
        :param str|None symbol: unified market symbol, only orders in the market of self symbol are cancelled when symbol is not None
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns [dict]: a list of `order structures <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        await self.load_markets()
        request = {}
        market = None
        if symbol is not None:
            market = self.market(symbol)
            request['currencyPair'] = market['id']
        response = await self.privatePostCancelAllOrders(self.extend(request, params))
        #
        #     {
        #         "success": 1,
        #         "message": "Orders canceled",
        #         "orderNumbers": [
        #             503749,
        #             888321,
        #             7315825,
        #             7316824
        #         ]
        #     }
        #
        return response

    async def fetch_open_order(self, id, symbol=None, params={}):
        """
        fetch an open order by it's id
        :param str id: order id
        :param str|None symbol: unified market symbol, default is None
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: an `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        await self.load_markets()
        id = str(id)
        request = {
            'orderNumber': id,
        }
        response = await self.privatePostReturnOrderStatus(self.extend(request, params))
        #
        #     {
        #         success: 1,
        #         result: {
        #             '6071071': {
        #                 status: 'Open',
        #                 rate: '0.40000000',
        #                 amount: '1.00000000',
        #                 currencyPair: 'BTC_ETH',
        #                 date: '2018-10-17 17:04:50',
        #                 total: '0.40000000',
        #                 type: 'buy',
        #                 startingAmount: '1.00000',
        #             },
        #         },
        #     }
        #
        result = self.safe_value(response['result'], id)
        if result is None:
            raise OrderNotFound(self.id + ' order id ' + id + ' not found')
        return self.extend(self.parse_order(result), {
            'id': id,
        })

    async def fetch_closed_order(self, id, symbol=None, params={}):
        """
        fetch an open order by it's id
        :param str id: order id
        :param str|None symbol: not used by poloniex fetchClosedOrder
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: an `order structure <https://docs.ccxt.com/en/latest/manual.html#order-structure>`
        """
        await self.load_markets()
        request = {
            'orderNumber': id,
        }
        response = await self.privatePostReturnOrderTrades(self.extend(request, params))
        #
        #     [
        #         {
        #             "globalTradeID":570264000,
        #             "tradeID":8026283,
        #             "currencyPair":"USDT_LTC",
        #             "type":"sell",
        #             "rate":"144.73833409",
        #             "amount":"0.18334460",
        #             "total":"26.53699196",
        #             "fee":"0.00155000",
        #             "date":"2021-07-04 15:16:20"
        #         }
        #     ]
        #
        firstTrade = self.safe_value(response, 0)
        if firstTrade is None:
            raise OrderNotFound(self.id + ' order id ' + id + ' not found')
        id = self.safe_value(firstTrade, 'globalTradeID', id)
        return self.safe_order({
            'info': response,
            'id': id,
            'clientOrderId': self.safe_value(firstTrade, 'clientOrderId'),
            'timestamp': None,
            'datetime': None,
            'lastTradeTimestamp': None,
            'status': 'closed',
            'symbol': None,
            'type': None,
            'timeInForce': None,
            'postOnly': None,
            'side': None,
            'price': None,
            'stopPrice': None,
            'cost': None,
            'average': None,
            'amount': None,
            'filled': None,
            'remaining': None,
            'trades': response,
            'fee': None,
        })

    async def fetch_order_status(self, id, symbol=None, params={}):
        await self.load_markets()
        orders = await self.fetch_open_orders(symbol, None, None, params)
        indexed = self.index_by(orders, 'id')
        return 'open' if (id in indexed) else 'closed'

    async def fetch_order_trades(self, id, symbol=None, since=None, limit=None, params={}):
        """
        fetch all the trades made from a single order
        :param str id: order id
        :param str|None symbol: unified market symbol
        :param int|None since: the earliest time in ms to fetch trades for
        :param int|None limit: the maximum number of trades to retrieve
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns [dict]: a list of `trade structures <https://docs.ccxt.com/en/latest/manual.html#trade-structure>`
        """
        await self.load_markets()
        request = {
            'orderNumber': id,
        }
        trades = await self.privatePostReturnOrderTrades(self.extend(request, params))
        return self.parse_trades(trades)

    async def create_deposit_address(self, code, params={}):
        """
        create a currency deposit address
        :param str code: unified currency code of the currency for the deposit address
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: an `address structure <https://docs.ccxt.com/en/latest/manual.html#address-structure>`
        """
        await self.load_markets()
        # USDT, USDTETH, USDTTRON
        currencyId = None
        currency = None
        if code in self.currencies:
            currency = self.currency(code)
            currencyId = currency['id']
        else:
            currencyId = code
        request = {
            'currency': currencyId,
        }
        response = await self.privatePostGenerateNewAddress(self.extend(request, params))
        address = None
        tag = None
        success = self.safe_string(response, 'success')
        if success == '1':
            address = self.safe_string(response, 'response')
        self.check_address(address)
        if currency is not None:
            depositAddress = self.safe_string(currency['info'], 'depositAddress')
            if depositAddress is not None:
                tag = address
                address = depositAddress
        return {
            'currency': code,
            'address': address,
            'tag': tag,
            'info': response,
        }

    async def fetch_deposit_address(self, code, params={}):
        """
        fetch the deposit address for a currency associated with self account
        :param str code: unified currency code
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: an `address structure <https://docs.ccxt.com/en/latest/manual.html#address-structure>`
        """
        await self.load_markets()
        response = await self.privatePostReturnDepositAddresses(params)
        # USDT, USDTETH, USDTTRON
        currencyId = None
        currency = None
        if code in self.currencies:
            currency = self.currency(code)
            currencyId = currency['id']
        else:
            currencyId = code
        address = self.safe_string(response, currencyId)
        tag = None
        self.check_address(address)
        if currency is not None:
            depositAddress = self.safe_string(currency['info'], 'depositAddress')
            if depositAddress is not None:
                tag = address
                address = depositAddress
        return {
            'currency': code,
            'address': address,
            'tag': tag,
            'network': None,
            'info': response,
        }

    async def transfer(self, code, amount, fromAccount, toAccount, params={}):
        """
        transfer currency internally between wallets on the same account
        :param str code: unified currency code
        :param float amount: amount to transfer
        :param str fromAccount: account to transfer from
        :param str toAccount: account to transfer to
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: a `transfer structure <https://docs.ccxt.com/en/latest/manual.html#transfer-structure>`
        """
        await self.load_markets()
        currency = self.currency(code)
        amount = self.currency_to_precision(code, amount)
        accountsByType = self.safe_value(self.options, 'accountsByType', {})
        fromId = self.safe_string(accountsByType, fromAccount, fromAccount)
        toId = self.safe_string(accountsByType, toAccount, fromAccount)
        request = {
            'amount': amount,
            'currency': currency['id'],
            'fromAccount': fromId,
            'toAccount': toId,
        }
        response = await self.privatePostTransferBalance(self.extend(request, params))
        #
        #    {
        #        success: '1',
        #        message: 'Transferred 1.00000000 USDT from exchange to lending account.'
        #    }
        #
        return self.parse_transfer(response, currency)

    def parse_transfer_status(self, status):
        statuses = {
            '1': 'ok',
        }
        return self.safe_string(statuses, status, status)

    def parse_transfer(self, transfer, currency=None):
        #
        #    {
        #        success: '1',
        #        message: 'Transferred 1.00000000 USDT from exchange to lending account.'
        #    }
        #
        message = self.safe_string(transfer, 'message')
        words = message.split(' ')
        amount = self.safe_number(words, 1)
        currencyId = self.safe_string(words, 2)
        fromAccountId = self.safe_string(words, 4)
        toAccountId = self.safe_string(words, 6)
        accountsById = self.safe_value(self.options, 'accountsById', {})
        return {
            'info': transfer,
            'id': None,
            'timestamp': None,
            'datetime': None,
            'currency': self.safe_currency_code(currencyId, currency),
            'amount': amount,
            'fromAccount': self.safe_string(accountsById, fromAccountId),
            'toAccount': self.safe_string(accountsById, toAccountId),
            'status': self.parse_order_status(self.safe_string(transfer, 'success', 'failed')),
        }

    async def withdraw(self, code, amount, address, tag=None, params={}):
        """
        make a withdrawal
        :param str code: unified currency code
        :param float amount: the amount to withdraw
        :param str address: the address to withdraw to
        :param str|None tag:
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: a `transaction structure <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        tag, params = self.handle_withdraw_tag_and_params(tag, params)
        self.check_address(address)
        await self.load_markets()
        currency = self.currency(code)
        request = {
            'currency': currency['id'],
            'amount': amount,
            'address': address,
        }
        if tag is not None:
            request['paymentId'] = tag
        networks = self.safe_value(self.options, 'networks', {})
        network = self.safe_string_upper(params, 'network')  # self line allows the user to specify either ERC20 or ETH
        network = self.safe_string(networks, network, network)  # handle ERC20>ETH alias
        if network is not None:
            request['currency'] += network  # when network the currency need to be changed to currency+network https://docs.poloniex.com/#withdraw on MultiChain Currencies section
            params = self.omit(params, 'network')
        response = await self.privatePostWithdraw(self.extend(request, params))
        #
        #     {
        #         response: 'Withdrew 1.00000000 USDT.',
        #         email2FA: False,
        #         withdrawalNumber: 13449869
        #     }
        #
        return self.parse_transaction(response, currency)

    async def fetch_transactions_helper(self, code=None, since=None, limit=None, params={}):
        await self.load_markets()
        year = 31104000  # 60 * 60 * 24 * 30 * 12 = one year of history, why not
        now = self.seconds()
        start = int(since / 1000) if (since is not None) else now - 10 * year
        request = {
            'start': start,  # UNIX timestamp, required
            'end': now,  # UNIX timestamp, required
        }
        if limit is not None:
            request['limit'] = limit
        response = await self.privatePostReturnDepositsWithdrawals(self.extend(request, params))
        #
        #     {
        #         "adjustments":[],
        #         "deposits":[
        #             {
        #                 currency: "BTC",
        #                 address: "1MEtiqJWru53FhhHrfJPPvd2tC3TPDVcmW",
        #                 amount: "0.01063000",
        #                 confirmations:  1,
        #                 txid: "952b0e1888d6d491591facc0d37b5ebec540ac1efb241fdbc22bcc20d1822fb6",
        #                 timestamp:  1507916888,
        #                 status: "COMPLETE"
        #             },
        #             {
        #                 currency: "ETH",
        #                 address: "0x20108ba20b65c04d82909e91df06618107460197",
        #                 amount: "4.00000000",
        #                 confirmations: 38,
        #                 txid: "0x4be260073491fe63935e9e0da42bd71138fdeb803732f41501015a2d46eb479d",
        #                 timestamp: 1525060430,
        #                 status: "COMPLETE"
        #             }
        #         ],
        #         "withdrawals":[
        #             {
        #                 "withdrawalNumber":13449869,
        #                 "currency":"USDTTRON",  # not documented in API docs, see commonCurrencies in describe()
        #                 "address":"TXGaqPW23JdRWhsVwS2mRsGsegbdnAd3Rw",
        #                 "amount":"1.00000000",
        #                 "fee":"0.00000000",
        #                 "timestamp":1591573420,
        #                 "status":"COMPLETE: dadf427224b3d44b38a2c13caa4395e4666152556ca0b2f67dbd86a95655150f",
        #                 "ipAddress":"x.x.x.x",
        #                 "canCancel":0,
        #                 "canResendEmail":0,
        #                 "paymentID":null,
        #                 "scope":"crypto"
        #             },
        #             {
        #                 withdrawalNumber: 8224394,
        #                 currency: "EMC2",
        #                 address: "EYEKyCrqTNmVCpdDV8w49XvSKRP9N3EUyF",
        #                 amount: "63.10796020",
        #                 fee: "0.01000000",
        #                 timestamp: 1510819838,
        #                 status: "COMPLETE: d37354f9d02cb24d98c8c4fc17aa42f475530b5727effdf668ee5a43ce667fd6",
        #                 ipAddress: "x.x.x.x"
        #             },
        #             {
        #                 withdrawalNumber: 9290444,
        #                 currency: "ETH",
        #                 address: "0x191015ff2e75261d50433fbd05bd57e942336149",
        #                 amount: "0.15500000",
        #                 fee: "0.00500000",
        #                 timestamp: 1514099289,
        #                 status: "COMPLETE: 0x12d444493b4bca668992021fd9e54b5292b8e71d9927af1f076f554e4bea5b2d",
        #                 ipAddress: "x.x.x.x"
        #             },
        #             {
        #                 withdrawalNumber: 11518260,
        #                 currency: "BTC",
        #                 address: "8JoDXAmE1GY2LRK8jD1gmAmgRPq54kXJ4t",
        #                 amount: "0.20000000",
        #                 fee: "0.00050000",
        #                 timestamp: 1527918155,
        #                 status: "COMPLETE: 1864f4ebb277d90b0b1ff53259b36b97fa1990edc7ad2be47c5e0ab41916b5ff",
        #                 ipAddress: "x.x.x.x"
        #             }
        #         ]
        #     }
        #
        return response

    async def fetch_transactions(self, code=None, since=None, limit=None, params={}):
        """
        fetch history of deposits and withdrawals
        :param str|None code: unified currency code for the currency of the transactions, default is None
        :param int|None since: timestamp in ms of the earliest transaction, default is None
        :param int|None limit: max number of transactions to return, default is None
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: a list of `transaction structure <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        await self.load_markets()
        response = await self.fetch_transactions_helper(code, since, limit, params)
        currency = None
        if code is not None:
            currency = self.currency(code)
        withdrawals = self.safe_value(response, 'withdrawals', [])
        deposits = self.safe_value(response, 'deposits', [])
        withdrawalTransactions = self.parse_transactions(withdrawals, currency, since, limit)
        depositTransactions = self.parse_transactions(deposits, currency, since, limit)
        transactions = self.array_concat(depositTransactions, withdrawalTransactions)
        return self.filter_by_currency_since_limit(self.sort_by(transactions, 'timestamp'), code, since, limit)

    async def fetch_withdrawals(self, code=None, since=None, limit=None, params={}):
        """
        fetch all withdrawals made from an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch withdrawals for
        :param int|None limit: the maximum number of withdrawals structures to retrieve
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        response = await self.fetch_transactions_helper(code, since, limit, params)
        currency = None
        if code is not None:
            currency = self.currency(code)
        withdrawals = self.safe_value(response, 'withdrawals', [])
        transactions = self.parse_transactions(withdrawals, currency, since, limit)
        return self.filter_by_currency_since_limit(transactions, code, since, limit)

    async def fetch_deposits(self, code=None, since=None, limit=None, params={}):
        """
        fetch all deposits made to an account
        :param str|None code: unified currency code
        :param int|None since: the earliest time in ms to fetch deposits for
        :param int|None limit: the maximum number of deposits structures to retrieve
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns [dict]: a list of `transaction structures <https://docs.ccxt.com/en/latest/manual.html#transaction-structure>`
        """
        response = await self.fetch_transactions_helper(code, since, limit, params)
        currency = None
        if code is not None:
            currency = self.currency(code)
        deposits = self.safe_value(response, 'deposits', [])
        transactions = self.parse_transactions(deposits, currency, since, limit)
        return self.filter_by_currency_since_limit(transactions, code, since, limit)

    def parse_transaction_status(self, status):
        statuses = {
            'COMPLETE': 'ok',
            'AWAITING APPROVAL': 'pending',
            'PENDING': 'pending',
            'PROCESSING': 'pending',
            'COMPLETE ERROR': 'failed',
        }
        return self.safe_string(statuses, status, status)

    def parse_transaction(self, transaction, currency=None):
        #
        # deposits
        #
        #     {
        #         "txid": "f49d489616911db44b740612d19464521179c76ebe9021af85b6de1e2f8d68cd",
        #         "type": "deposit",
        #         "amount": "49798.01987021",
        #         "status": "COMPLETE",
        #         "address": "DJVJZ58tJC8UeUv9Tqcdtn6uhWobouxFLT",
        #         "currency": "DOGE",
        #         "timestamp": 1524321838,
        #         "confirmations": 3371,
        #         "depositNumber": 134587098
        #     }
        #
        # withdrawals
        #
        #     {
        #         "fee": "0.00050000",
        #         "type": "withdrawal",
        #         "amount": "0.40234387",
        #         "status": "COMPLETE: fbabb2bf7d81c076f396f3441166d5f60f6cea5fdfe69e02adcc3b27af8c2746",
        #         "address": "1EdAqY4cqHoJGAgNfUFER7yZpg1Jc9DUa3",
        #         "currency": "BTC",
        #         "canCancel": 0,
        #         "ipAddress": "x.x.x.x",
        #         "paymentID": null,
        #         "timestamp": 1523834337,
        #         "canResendEmail": 0,
        #         "withdrawalNumber": 11162900
        #     }
        #
        # withdraw
        #
        #     {
        #         response: 'Withdrew 1.00000000 USDT.',
        #         email2FA: False,
        #         withdrawalNumber: 13449869
        #     }
        #
        timestamp = self.safe_timestamp(transaction, 'timestamp')
        currencyId = self.safe_string(transaction, 'currency')
        code = self.safe_currency_code(currencyId)
        status = self.safe_string(transaction, 'status', 'pending')
        txid = self.safe_string(transaction, 'txid')
        if status is not None:
            parts = status.split(': ')
            numParts = len(parts)
            status = parts[0]
            if (numParts > 1) and (txid is None):
                txid = parts[1]
            status = self.parse_transaction_status(status)
        defaultType = 'withdrawal' if ('withdrawalNumber' in transaction) else 'deposit'
        type = self.safe_string(transaction, 'type', defaultType)
        id = self.safe_string_2(transaction, 'withdrawalNumber', 'depositNumber')
        amount = self.safe_number(transaction, 'amount')
        address = self.safe_string(transaction, 'address')
        tag = self.safe_string(transaction, 'paymentID')
        # according to https://poloniex.com/fees/
        feeCost = self.safe_number(transaction, 'fee', 0)
        if type == 'withdrawal':
            # poloniex withdrawal amount includes the fee
            amount = amount - feeCost
        return {
            'info': transaction,
            'id': id,
            'currency': code,
            'amount': amount,
            'network': None,
            'address': address,
            'addressTo': None,
            'addressFrom': None,
            'tag': tag,
            'tagTo': None,
            'tagFrom': None,
            'status': status,
            'type': type,
            'updated': None,
            'txid': txid,
            'timestamp': timestamp,
            'datetime': self.iso8601(timestamp),
            'fee': {
                'currency': code,
                'cost': feeCost,
            },
        }

    async def fetch_position(self, symbol, params={}):
        """
        fetch data on a single open contract trade position
        :param str symbol: unified market symbol of the market the position is held in, default is None
        :param dict params: extra parameters specific to the poloniex api endpoint
        :returns dict: a `position structure <https://docs.ccxt.com/en/latest/manual.html#position-structure>`
        """
        await self.load_markets()
        market = self.market(symbol)
        request = {
            'currencyPair': market['id'],
        }
        response = await self.privatePostGetMarginPosition(self.extend(request, params))
        #
        #     {
        #         type: "none",
        #         amount: "0.00000000",
        #         total: "0.00000000",
        #         basePrice: "0.00000000",
        #         liquidationPrice: -1,
        #         pl: "0.00000000",
        #         lendingFees: "0.00000000"
        #     }
        #
        # todo unify parsePosition/parsePositions
        return response

    def nonce(self):
        return self.milliseconds()

    def sign(self, path, api='public', method='GET', params={}, headers=None, body=None):
        url = self.urls['api'][api]
        query = self.extend({'command': path}, params)
        if api == 'public':
            url += '?' + self.urlencode(query)
        else:
            self.check_required_credentials()
            query['nonce'] = self.nonce()
            body = self.urlencode(query)
            headers = {
                'Content-Type': 'application/x-www-form-urlencoded',
                'Key': self.apiKey,
                'Sign': self.hmac(self.encode(body), self.encode(self.secret), hashlib.sha512),
            }
        return {'url': url, 'method': method, 'body': body, 'headers': headers}

    def handle_errors(self, code, reason, url, method, headers, body, response, requestHeaders, requestBody):
        if response is None:
            return
        # {"error":"Permission denied."}
        if 'error' in response:
            message = response['error']
            feedback = self.id + ' ' + body
            self.throw_exactly_matched_exception(self.exceptions['exact'], message, feedback)
            self.throw_broadly_matched_exception(self.exceptions['broad'], message, feedback)
            raise ExchangeError(feedback)  # unknown message
