"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/*
 * Creates an Appsync GraphQL API and with multiple tables.
 * Testing for importing, querying, and mutability.
 *
 * Stack verification steps:
 * Add to a table through appsync GraphQL API.
 * Read from a table through appsync API.
 *
 * -- aws appsync list-graphql-apis                 -- obtain apiId               --
 * -- aws appsync get-graphql-api --api-id [apiId]  -- obtain GraphQL endpoint    --
 * -- aws appsync list-api-keys --api-id [apiId]    -- obtain api key             --
 * -- bash verify.integ.graphql.sh [apiKey] [url]   -- shows query and mutation   --
 */
const app = new core_1.App();
const stack = new core_1.Stack(app, 'aws-appsync-integ');
const userPool = new aws_cognito_1.UserPool(stack, 'Pool', {
    userPoolName: 'myPool',
});
const api = new lib_1.GraphQLApi(stack, 'Api', {
    name: 'demoapi',
    schemaDefinition: lib_1.SchemaDefinition.FILE,
    schemaDefinitionFile: path_1.join(__dirname, 'integ.graphql.graphql'),
    authorizationConfig: {
        defaultAuthorization: {
            authorizationType: lib_1.AuthorizationType.USER_POOL,
            userPoolConfig: {
                userPool,
                defaultAction: lib_1.UserPoolDefaultAction.ALLOW,
            },
        },
        additionalAuthorizationModes: [
            {
                authorizationType: lib_1.AuthorizationType.API_KEY,
            },
        ],
    },
});
const noneDS = api.addNoneDataSource('None', 'Dummy data source');
noneDS.createResolver({
    typeName: 'Query',
    fieldName: 'getServiceVersion',
    requestMappingTemplate: lib_1.MappingTemplate.fromString(JSON.stringify({
        version: '2017-02-28',
    })),
    responseMappingTemplate: lib_1.MappingTemplate.fromString(JSON.stringify({
        version: 'v1',
    })),
});
const customerTable = new aws_dynamodb_1.Table(stack, 'CustomerTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const orderTable = new aws_dynamodb_1.Table(stack, 'OrderTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'customer',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    sortKey: {
        name: 'order',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
new aws_dynamodb_1.Table(stack, 'PaymentTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const paymentTable = aws_dynamodb_1.Table.fromTableName(stack, 'ImportedPaymentTable', 'PaymentTable');
const customerDS = api.addDynamoDbDataSource('Customer', 'The customer data source', customerTable);
const orderDS = api.addDynamoDbDataSource('Order', 'The order data source', orderTable);
const paymentDS = api.addDynamoDbDataSource('Payment', 'The payment data source', paymentTable);
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomers',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
customerDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'addCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').is('id'), lib_1.Values.projecting('customer')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'saveCustomerWithFirstOrder',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey
        .partition('order').auto()
        .sort('customer').is('customer.id'), lib_1.Values
        .projecting('order')
        .attribute('referral').is('referral')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
customerDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'removeCustomer',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbDeleteItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const ops = [
    { suffix: 'Eq', op: lib_1.KeyCondition.eq },
    { suffix: 'Lt', op: lib_1.KeyCondition.lt },
    { suffix: 'Le', op: lib_1.KeyCondition.le },
    { suffix: 'Gt', op: lib_1.KeyCondition.gt },
    { suffix: 'Ge', op: lib_1.KeyCondition.ge },
];
for (const { suffix, op } of ops) {
    orderDS.createResolver({
        typeName: 'Query',
        fieldName: 'getCustomerOrders' + suffix,
        requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(op('customer', 'customer')),
        responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
    });
}
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersFilter',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.beginsWith('order', 'order'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
orderDS.createResolver({
    typeName: 'Query',
    fieldName: 'getCustomerOrdersBetween',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbQuery(lib_1.KeyCondition.eq('customer', 'customer').and(lib_1.KeyCondition.between('order', 'order1', 'order2'))),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
paymentDS.createResolver({
    typeName: 'Query',
    fieldName: 'getPayment',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
paymentDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'savePayment',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('payment')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const httpDS = api.addHttpDataSource('http', 'The http data source', 'https://aws.amazon.com/');
httpDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'doPostOnAws',
    requestMappingTemplate: lib_1.MappingTemplate.fromString(`{
    "version": "2018-05-29",
    "method": "POST",
    # if full path is https://api.xxxxxxxxx.com/posts then resourcePath would be /posts
    "resourcePath": "/path/123",
    "params":{
        "body": $util.toJson($ctx.args),
        "headers":{
            "Content-Type": "application/json",
            "Authorization": "$ctx.request.headers.Authorization"
        }
    }
  }`),
    responseMappingTemplate: lib_1.MappingTemplate.fromString(`
    ## Raise a GraphQL field error in case of a datasource invocation error
    #if($ctx.error)
      $util.error($ctx.error.message, $ctx.error.type)
    #end
    ## if the response status code is not 200, then return an error. Else return the body **
    #if($ctx.result.statusCode == 200)
        ## If response is 200, return the body.
        $ctx.result.body
    #else
        ## If response is not 200, append the response to error block.
        $utils.appendError($ctx.result.body, "$ctx.result.statusCode")
    #end
  `),
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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