"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
const aws_cognito_1 = require("@aws-cdk/aws-cognito");
const aws_dynamodb_1 = require("@aws-cdk/aws-dynamodb");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
/*
 * Creates an Appsync GraphQL API and Lambda with IAM Roles.
 * Testing for IAM Auth and grantFullAccess.
 *
 * Stack verification steps:
 * Install dependencies and deploy integration test. Invoke Lambda
 * function with different permissions to test policies.
 *
 * -- bash verify.integ.graphql-iam.sh --start             -- get dependencies/deploy    --
 * -- aws lambda list-functions                            -- obtain testFail/testQuery  --
 * -- aws lambda invoke /dev/stdout --function-name [FAIL] -- fails beacuse no IAM Role` --
 * -- aws lambda invoke /dev/stdout --function-name [Query]-- succeeds with empty get  ` --
 * -- bash verify.integ.graphql-iam.sh --clean             -- clean dependencies/deploy  --
 */
const app = new core_1.App();
const stack = new core_1.Stack(app, 'aws-appsync-integ');
const userPool = new aws_cognito_1.UserPool(stack, 'Pool', {
    userPoolName: 'myPool',
});
const api = new lib_1.GraphQLApi(stack, 'Api', {
    name: 'Integ_Test_IAM',
    schemaDefinition: lib_1.SchemaDefinition.FILE,
    schemaDefinitionFile: path_1.join(__dirname, 'integ.graphql-iam.graphql'),
    authorizationConfig: {
        defaultAuthorization: {
            authorizationType: lib_1.AuthorizationType.USER_POOL,
            userPoolConfig: {
                userPool,
                defaultAction: lib_1.UserPoolDefaultAction.ALLOW,
            },
        },
        additionalAuthorizationModes: [
            {
                authorizationType: lib_1.AuthorizationType.IAM,
            },
        ],
    },
});
const testTable = new aws_dynamodb_1.Table(stack, 'TestTable', {
    billingMode: aws_dynamodb_1.BillingMode.PAY_PER_REQUEST,
    partitionKey: {
        name: 'id',
        type: aws_dynamodb_1.AttributeType.STRING,
    },
    removalPolicy: core_1.RemovalPolicy.DESTROY,
});
const testDS = api.addDynamoDbDataSource('testDataSource', 'Table for Tests"', testTable);
testDS.createResolver({
    typeName: 'Query',
    fieldName: 'getTest',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbGetItem('id', 'id'),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
testDS.createResolver({
    typeName: 'Query',
    fieldName: 'getTests',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbScanTable(),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultList(),
});
testDS.createResolver({
    typeName: 'Mutation',
    fieldName: 'addTest',
    requestMappingTemplate: lib_1.MappingTemplate.dynamoDbPutItem(lib_1.PrimaryKey.partition('id').auto(), lib_1.Values.projecting('test')),
    responseMappingTemplate: lib_1.MappingTemplate.dynamoDbResultItem(),
});
const lambdaIAM = new aws_iam_1.Role(stack, 'LambdaIAM', { assumedBy: new aws_iam_1.ServicePrincipal('lambda') });
api.grant(lambdaIAM, lib_1.IamResource.custom('types/Query/fields/getTests'), 'appsync:graphql');
api.grant(lambdaIAM, lib_1.IamResource.ofType('test'), 'appsync:GraphQL');
api.grantMutation(lambdaIAM, 'addTest');
new aws_lambda_1.Function(stack, 'testQuery', {
    code: aws_lambda_1.Code.fromAsset('verify'),
    handler: 'iam-query.handler',
    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
    environment: { APPSYNC_ENDPOINT: api.graphQlUrl },
    role: lambdaIAM,
});
new aws_lambda_1.Function(stack, 'testFail', {
    code: aws_lambda_1.Code.fromAsset('verify'),
    handler: 'iam-query.handler',
    runtime: aws_lambda_1.Runtime.NODEJS_12_X,
    environment: { APPSYNC_ENDPOINT: api.graphQlUrl },
});
app.synth();
//# sourceMappingURL=data:application/json;base64,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