"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
// Schema Definitions
const type = 'type test {\n  version: String!\n}\n\n';
const query = 'type Query {\n  getTests: [ test! ]!\n}\n\n';
const mutation = 'type Mutation {\n  addTest(version: String!): test\n}\n';
let stack;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
});
describe('testing schema definition mode `code`', () => {
    test('definition mode `code` produces empty schema definition', () => {
        // WHEN
        new appsync.GraphQLApi(stack, 'API', {
            name: 'demo',
            schemaDefinition: appsync.SchemaDefinition.CODE,
        });
        //THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: '',
        });
    });
    test('definition mode `code` generates correct schema with updateDefinition', () => {
        // WHEN
        const api = new appsync.GraphQLApi(stack, 'API', {
            name: 'demo',
            schemaDefinition: appsync.SchemaDefinition.CODE,
        });
        api.updateDefinition(`${type}${query}${mutation}`);
        //THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${type}${query}${mutation}`,
        });
    });
    test('definition mode `code` errors when schemaDefinitionFile is configured', () => {
        // WHEN
        const when = () => {
            new appsync.GraphQLApi(stack, 'API', {
                name: 'demo',
                schemaDefinition: appsync.SchemaDefinition.CODE,
                schemaDefinitionFile: path_1.join(__dirname, 'appsync.test.graphql'),
            });
        };
        //THEN
        expect(when).toThrowError('definition mode CODE is incompatible with file definition. Change mode to FILE/S3 or unconfigure schemaDefinitionFile');
    });
});
describe('testing schema definition mode `file`', () => {
    test('definition mode `file` produces correct output', () => {
        // WHEN
        new appsync.GraphQLApi(stack, 'API', {
            name: 'demo',
            schemaDefinition: appsync.SchemaDefinition.FILE,
            schemaDefinitionFile: path_1.join(__dirname, 'appsync.test.graphql'),
        });
        //THEN
        expect(stack).toHaveResourceLike('AWS::AppSync::GraphQLSchema', {
            Definition: `${type}${query}${mutation}`,
        });
    });
    test('definition mode `file` errors when calling updateDefiniton function', () => {
        // WHEN
        const api = new appsync.GraphQLApi(stack, 'API', {
            name: 'demo',
            schemaDefinition: appsync.SchemaDefinition.FILE,
            schemaDefinitionFile: path_1.join(__dirname, 'appsync.test.graphql'),
        });
        const when = () => { api.updateDefinition('error'); };
        //THEN
        expect(when).toThrowError('API cannot add type because schema definition mode is not configured as CODE.');
    });
    test('definition mode `file` errors when schemaDefinitionFile is not configured', () => {
        // WHEN
        const when = () => {
            new appsync.GraphQLApi(stack, 'API', {
                name: 'demo',
                schemaDefinition: appsync.SchemaDefinition.FILE,
            });
        };
        //THEN
        expect(when).toThrowError('schemaDefinitionFile must be configured if using FILE definition mode.');
    });
});
//# sourceMappingURL=data:application/json;base64,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