"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path_1 = require("path");
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
const appsync = require("../lib");
let stack;
let role;
let api;
beforeEach(() => {
    // GIVEN
    stack = new cdk.Stack();
    role = new iam.Role(stack, 'Role', {
        assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com'),
    });
    api = new appsync.GraphQLApi(stack, 'API', {
        name: 'demo',
        schemaDefinition: appsync.SchemaDefinition.FILE,
        schemaDefinitionFile: path_1.join(__dirname, 'appsync.test.graphql'),
        authorizationConfig: {
            defaultAuthorization: {
                authorizationType: appsync.AuthorizationType.IAM,
            },
        },
    });
});
describe('grant Permissions', () => {
    test('IamResource throws error when custom is called with no arguments', () => {
        // WHEN
        const when = () => {
            api.grant(role, appsync.IamResource.custom(), 'appsync:GraphQL');
        };
        //THEN
        expect(when).toThrowError('At least 1 custom ARN must be provided.');
    });
    test('grant provides custom permissions when called with `custom` argument', () => {
        // WHEN
        api.grant(role, appsync.IamResource.custom('types/Mutation/fields/addTest'), 'appsync:GraphQL');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Mutation/fields/addTest',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grant provides [type parameter]/* permissions when called with `type` argument', () => {
        // WHEN
        api.grant(role, appsync.IamResource.ofType('Mutation'), 'appsync:GraphQL');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Mutation/*',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grant provides fields/[field param] permissions when called with `type` and `field` argument', () => {
        // WHEN
        api.grant(role, appsync.IamResource.ofType('Mutation', 'addTest'), 'appsync:GraphQL');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Mutation/fields/addTest',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grant provides all permissions when called with IamResource.all()', () => {
        // WHEN
        api.grant(role, appsync.IamResource.all(), 'appsync:GraphQL');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/*',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grant provides multiple permissions using one IamResource custom call', () => {
        // WHEN
        api.grant(role, appsync.IamResource.custom('I', 'am', 'custom'), 'appsync:GraphQL');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: [
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/I',
                                    ]]
                            },
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/am',
                                    ]]
                            },
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/custom',
                                    ]]
                            },
                        ],
                    },
                ],
            },
        });
    });
    test('grant provides multiple permissions using one IamResource ofType call', () => {
        // WHEN
        api.grant(role, appsync.IamResource.ofType('I', 'am', 'custom'), 'appsync:GraphQL');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: [
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/I/fields/am',
                                    ]]
                            },
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/I/fields/custom',
                                    ]]
                            },
                        ],
                    },
                ],
            },
        });
    });
});
describe('grantMutation Permissions', () => {
    test('grantMutation provides Mutation/* permissions when called with no `fields` argument', () => {
        // WHEN
        api.grantMutation(role);
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Mutation/*',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grantMutation provides fields/[field param] permissions when called with `fields` argument', () => {
        // WHEN
        api.grantMutation(role, 'addTest');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Mutation/fields/addTest',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grantMutation provides multiple permissions when called with `fields` argument', () => {
        // WHEN
        api.grantMutation(role, 'addTest', 'removeTest');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: [
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/Mutation/fields/addTest',
                                    ]]
                            },
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/Mutation/fields/removeTest',
                                    ]]
                            },
                        ],
                    },
                ],
            },
        });
    });
});
describe('grantQuery Permissions', () => {
    test('grantQuery provides Query/* permissions when called without the `fields` argument', () => {
        // WHEN
        api.grantQuery(role);
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Query/*',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grantQuery provides fields/[field param] permissions when called with `fields` arugment', () => {
        // WHEN
        api.grantQuery(role, 'getTest');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Query/fields/getTest',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grantQuery provides multiple permissions when called with `fields` argument', () => {
        // WHEN
        api.grantQuery(role, 'getTests', 'getTest');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: [
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/Query/fields/getTests',
                                    ]]
                            },
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/Query/fields/getTest',
                                    ]]
                            },
                        ],
                    },
                ],
            },
        });
    });
});
describe('grantSubscription Permissions', () => {
    test('grantSubscription provides Subscription/* permissions when called without `fields` argument', () => {
        // WHEN
        api.grantSubscription(role);
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Subscription/*',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grantSubscription provides fields/[field param] when called with `field` argument', () => {
        api.grantSubscription(role, 'subscribe');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: {
                            'Fn::Join': ['', [
                                    'arn:',
                                    { Ref: 'AWS::Partition' },
                                    ':appsync:',
                                    { Ref: 'AWS::Region' },
                                    ':',
                                    { Ref: 'AWS::AccountId' },
                                    ':apis/',
                                    { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                    '/types/Subscription/fields/subscribe',
                                ]],
                        },
                    },
                ],
            },
        });
    });
    test('grantSubscription provides multiple permissions when called with `fields` argument', () => {
        // WHEN
        api.grantSubscription(role, 'subscribe', 'custom');
        // THEN
        expect(stack).toHaveResourceLike('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: [
                    {
                        Action: 'appsync:GraphQL',
                        Resource: [
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/Subscription/fields/subscribe',
                                    ]]
                            },
                            {
                                'Fn::Join': ['', [
                                        'arn:',
                                        { Ref: 'AWS::Partition' },
                                        ':appsync:',
                                        { Ref: 'AWS::Region' },
                                        ':',
                                        { Ref: 'AWS::AccountId' },
                                        ':apis/',
                                        { 'Fn::GetAtt': ['API62EA1CFF', 'ApiId'] },
                                        '/types/Subscription/fields/custom',
                                    ]]
                            },
                        ],
                    },
                ],
            },
        });
    });
});
//# sourceMappingURL=data:application/json;base64,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